/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  sattree.c:  This file contains functions to load files containing
              NASA 2-line element set from a specified directory or URL,
	      and return a GtkCTree containing satellite names and the
	      time when the element sets were generated (EPOCH).


  Copyright (C)  2001-2003  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>
#include <stdio.h>
#include "util.h"
#include "satlog.h"
#include "sattree.h"

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


extern GtkWidget *app;

/* SATTREE_MODE_SAT */
#define TREE_COLS_SAT 3
static gchar *tree_titles_sat[TREE_COLS_SAT] = {
	N_("File/Satellite"),
	N_("Catnum"),
	N_("Updated")
};

/* SATTREE_MODE_FILE */
#define TREE_COLS_FILE 2
static gchar *tree_titles_file[TREE_COLS_FILE] = {
	N_("File"),
	N_("Info")
};

/* Private function prototypes */
static void sattree_add_local_uri   (GtkWidget *tree, GtkCTreeNode *parent,
				     const gchar *uri, const gchar *filter,
				     sattree_mode mode);

static void sattree_add_local_file  (GtkWidget *tree, GtkCTreeNode *parent,
				     gchar *file);

static void sattree_add_sat_count   (GtkWidget *tree, GtkCTreeNode *parent,
				     gchar *file);

static void sattree_add_web_uri     (GtkCTree *tree, GtkCTreeNode *parent,
				     const gchar *uri, const gchar *filter,
				     sattree_mode mode);

static void sattree_select_row_cb   (GtkCTree *tree, GtkCTreeNode *node,
				     gint column, gpointer data);

static void sattree_unselect_row_cb (GtkCTree *tree, GtkCTreeNode *node,
				     gint column, gpointer data);


GtkWidget *
sattree_load_uris_recursive (gchar **dir, gchar **web, 
			     const gchar *filter,
			     sattree_mode mode)
{
	/* This function lists the specified directories in a GtkCTree. If mode
	   is SATTREE_MODE_SAT, then the satellites from the TLE files will be
	   listed in the tree as well.
	   The directories are read using the 'filter' pattern. If filter NULL,
	   directories will be read using "*". Subdirectories are listed using
	   "*" pattern regardless of what 'filter' is.
	   The catalog number of each satellite is attached to the row, in which
	   the satellite is.
	*/
	GtkWidget *tree;
	GtkCTreeNode *localnode,*webnode,*thisdir;
	GdkPixmap *pixm;
	GdkBitmap *bitm;
	guint i;
	gchar *buff,**vbuff,*iconfile;
	GdkPixmap *pixopen,*pixclosed;
	GdkBitmap *bitopen,*bitclosed;

	/* We need either some local or some web folders */
	g_return_val_if_fail (dir || web, NULL);

	/* log messsage */
	if ((mode != SATTREE_MODE_FILE) && (mode != SATTREE_MODE_SAT)) {
		buff = g_strdup_printf (_("%s: Invalid mode %d. Using SATTREE_MODE_FILE."),
					__FUNCTION__, mode);
		satlog_log (SAT_LOG_CRITICAL, buff);
		g_free (buff);
	}
	else {
		buff = g_strdup_printf (_("%s: Creating new tree in mode %s."),
					__FUNCTION__,
					(mode==1) ? "SATTREE_MODE_FILE" : "SATTREE_MODE_SAT");
		satlog_log (SAT_LOG_DEBUG, buff);
		g_free (buff);
	}

	/* Create directory pixmaps and bitmaps */
	iconfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				G_DIR_SEPARATOR_S, "gpredict-dir-open.xpm", NULL);
	pixopen = gdk_pixmap_create_from_xpm (app->window, &bitopen, 0, iconfile);
	g_free (iconfile);
	iconfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				G_DIR_SEPARATOR_S, "gpredict-dir-closed.xpm", NULL);
	pixclosed = gdk_pixmap_create_from_xpm (app->window, &bitclosed, 0, iconfile);
	g_free (iconfile);

	/* Create a tree and add two ROOT nodes (Local Files and Network) */
	switch (mode) {
	case SATTREE_MODE_SAT:
		tree = gtk_ctree_new_with_titles (TREE_COLS_SAT, 0, tree_titles_sat);
		gtk_ctree_set_line_style (GTK_CTREE (tree), GTK_CTREE_LINES_DOTTED);
		gtk_ctree_set_spacing (GTK_CTREE (tree), 5);
		gtk_ctree_set_indent (GTK_CTREE (tree), 18);
		gtk_clist_set_selection_mode (GTK_CLIST (tree), GTK_SELECTION_EXTENDED);
		/* adjust column width automatically */
		for (i=0; i<TREE_COLS_SAT; i++)
			gtk_clist_set_column_auto_resize (GTK_CLIST (tree), i, TRUE);
		/* connect signals */
		gtk_signal_connect (GTK_OBJECT (tree), "tree-select-row",
				    GTK_SIGNAL_FUNC (sattree_select_row_cb), NULL);
		gtk_signal_connect (GTK_OBJECT (tree), "tree-unselect-row",
				    GTK_SIGNAL_FUNC (sattree_unselect_row_cb), NULL);
		break;
	case SATTREE_MODE_FILE:
	default:
		tree = gtk_ctree_new_with_titles (TREE_COLS_FILE, 0, tree_titles_file);
		gtk_ctree_set_line_style (GTK_CTREE (tree), GTK_CTREE_LINES_DOTTED);
		gtk_ctree_set_spacing (GTK_CTREE (tree), 5);
		gtk_ctree_set_indent (GTK_CTREE (tree), 18);
		gtk_clist_set_selection_mode (GTK_CLIST (tree), GTK_SELECTION_MULTIPLE);
		/* adjust column width automatically */
		for (i=0; i<TREE_COLS_FILE; i++)
			gtk_clist_set_column_auto_resize (GTK_CLIST (tree), i, TRUE);
		break;
	}


	/* add directories recursively */
	if (dir) {
		/* create local node */
		iconfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
					G_DIR_SEPARATOR_S, "gpredict-disk-16.xpm", NULL);
		pixm = gdk_pixmap_create_from_xpm (app->window, &bitm, 0, iconfile);
		g_free (iconfile);
		switch (mode) {
		case SATTREE_MODE_SAT:
			buff = g_strconcat (_("Local Folders"), "<!SEP> <!SEP> ", NULL);
			vbuff = g_strsplit (buff, "<!SEP>", TREE_COLS_SAT);
			localnode = gtk_ctree_insert_node (GTK_CTREE (tree), NULL, NULL, vbuff, 5,
							   pixm, bitm, pixm, bitm,
							   FALSE, FALSE);
			g_free (buff);
			g_strfreev (vbuff);
			for (i=0; dir[i]; i++) {
				/* add the directory node to the tree */
				buff = g_strconcat (dir[i], "<!SEP> <!SEP> ", NULL);
				vbuff = g_strsplit (buff, "<!SEP>", TREE_COLS_SAT);
				g_free (buff);
				thisdir = gtk_ctree_insert_node (GTK_CTREE (tree), localnode, NULL, vbuff, 5,
								 pixclosed, bitclosed, pixopen, bitopen,
								 FALSE, FALSE);
				g_strfreev (vbuff);

				/* don't let users select whole directories */
				gtk_ctree_node_set_selectable (GTK_CTREE (tree), thisdir, FALSE);
			
				/* load the contents into the tree */
				sattree_add_local_uri (tree, thisdir, dir[i], filter, mode);
			}
			break;
		case SATTREE_MODE_FILE:
		default:
			buff = g_strconcat (_("Local Folders"), "<!SEP> ", NULL);
			vbuff = g_strsplit (buff, "<!SEP>", TREE_COLS_FILE);
			localnode = gtk_ctree_insert_node (GTK_CTREE (tree), NULL, NULL, vbuff, 5,
							   pixm, bitm, pixm, bitm,
							   FALSE, FALSE);
			g_free (buff);
			g_strfreev (vbuff);
			for (i=0; dir[i]; i++) {
				/* add the directory node to the tree */
				buff = g_strconcat (dir[i], "<!SEP> ", NULL);
				vbuff = g_strsplit (buff, "<!SEP>", TREE_COLS_FILE);
				g_free (buff);
				thisdir = gtk_ctree_insert_node (GTK_CTREE (tree), localnode, NULL, vbuff, 5,
								 pixclosed, bitclosed, pixopen, bitopen,
								 FALSE, FALSE);
				g_strfreev (vbuff);

				/* don't let users select whole directories */
				gtk_ctree_node_set_selectable (GTK_CTREE (tree), thisdir, FALSE);
			
				/* load the contents into the tree */
				sattree_add_local_uri (tree, thisdir, dir[i], filter, mode);
			}
			break;
		}
		/* The roots should not be selectable */
		gtk_ctree_node_set_selectable (GTK_CTREE (tree), localnode, FALSE);
	}

	/* Network node */
	if (web) {
		iconfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
					G_DIR_SEPARATOR_S, "gpredict-globe-16.xpm", NULL);
		pixm = gdk_pixmap_create_from_xpm (app->window, &bitm, 0, iconfile);
		g_free (iconfile);
		switch (mode) {
		case SATTREE_MODE_SAT:
			buff = g_strconcat (_("Web Folders"), "<!SEP> <!SEP> ", NULL);
			vbuff = g_strsplit (buff, "<!SEP>", TREE_COLS_SAT);
			webnode = gtk_ctree_insert_node (GTK_CTREE (tree), NULL, NULL, vbuff, 5,
							 pixm, bitm, pixm, bitm,
							 FALSE, FALSE);
			g_free (buff);
			g_strfreev (vbuff);
			for (i=0; web[i]; i++) {
				/* add the directory node to the tree */
				buff = g_strconcat (web[i], "<!SEP> <!SEP> ", NULL);
				vbuff = g_strsplit (buff, "<!SEP>", TREE_COLS_SAT);
				g_free (buff);
				thisdir = gtk_ctree_insert_node (GTK_CTREE (tree), webnode, NULL, vbuff, 5,
								 pixclosed, bitclosed, pixopen, bitopen,
								 FALSE, FALSE);
				g_strfreev (vbuff);

				/* don't let users select whole directories */
				gtk_ctree_node_set_selectable (GTK_CTREE (tree), thisdir, FALSE);
				
				/* load the contents into the tree */
				sattree_add_web_uri (GTK_CTREE (tree), thisdir, web[i], filter, mode);
			}
			break;
		case SATTREE_MODE_FILE:
		default:
			buff = g_strconcat (_("Web Folders"), "<!SEP> ", NULL);
			vbuff = g_strsplit (buff, "<!SEP>", TREE_COLS_FILE);
			webnode = gtk_ctree_insert_node (GTK_CTREE (tree), NULL, NULL, vbuff, 5,
							 pixm, bitm, pixm, bitm,
							 FALSE, FALSE);
			g_free (buff);
			g_strfreev (vbuff);
			for (i=0; web[i]; i++) {
				/* add the directory node to the tree */
				buff = g_strconcat (web[i], "<!SEP> ", NULL);
				vbuff = g_strsplit (buff, "<!SEP>", TREE_COLS_FILE);
				g_free (buff);
				thisdir = gtk_ctree_insert_node (GTK_CTREE (tree), webnode, NULL, vbuff, 5,
								 pixclosed, bitclosed, pixopen, bitopen,
								 FALSE, FALSE);
				g_strfreev (vbuff);

				/* don't let users select whole directories */
				gtk_ctree_node_set_selectable (GTK_CTREE (tree), thisdir, FALSE);
				
				/* load the contents into the tree */
				sattree_add_web_uri (GTK_CTREE (tree), thisdir, web[i], filter, mode);
			}
			break;
		}
		/* The roots should not be selectable */
		gtk_ctree_node_set_selectable (GTK_CTREE (tree), webnode, FALSE);
	}

	return tree;
}


static void
sattree_add_local_uri (GtkWidget *tree, GtkCTreeNode *parent,
		       const gchar *uri, const gchar *filter,
		       sattree_mode mode)
{
	/* This function loads the specified local directory recursively
	   into the GtkCTree starting at the specified GtkCTreeNode.
	*/
	GnomeVFSResult result;
	GnomeVFSDirectoryFilter *vfsfilter;
	GnomeVFSFileInfo *fileinfo;
	gchar *msg,*path,*vfsuri;
	GList *list,*node;
	gchar *text,**vbuf,*iconfile;
	GdkPixmap *pixopen,*pixclosed;
	GdkBitmap *bitopen,*bitclosed;
	GtkCTreeNode *myself;
	guint numfiles=0,numdirs=0;

	/* this shouldn't happen, but... */
	g_return_if_fail (uri);
	g_return_if_fail (mode);

	/* Create pixmaps and bitmaps */
	iconfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				G_DIR_SEPARATOR_S, "gpredict-dir-open.xpm", NULL);
	pixopen = gdk_pixmap_create_from_xpm (app->window, &bitopen, 0, iconfile);
	g_free (iconfile);
	iconfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				G_DIR_SEPARATOR_S, "gpredict-dir-closed.xpm", NULL);
	pixclosed = gdk_pixmap_create_from_xpm (app->window, &bitclosed, 0, iconfile);
	g_free (iconfile);

	/****  Load subdirectories if any ****/

	/* create gnome_vfs_uri */
	vfsuri = gnome_vfs_get_uri_from_local_path (uri);

	/* create a filter and load the directory */
	vfsfilter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						    GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						    GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						    "*");
	result = gnome_vfs_directory_list_load (&list, vfsuri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						vfsfilter);
	/* free some memory */
	g_free (vfsuri);
	gnome_vfs_directory_filter_destroy (vfsfilter);

	if (result == GNOME_VFS_OK) {

		/* make a log message */
		msg = g_strconcat (__FUNCTION__, _(": loading subdirs from "), uri, ": GNOME_VFS_OK", NULL);
		satlog_log (SAT_LOG_DEBUG, msg);
		g_free (msg);
		
		/* loop throough the file list */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;  /* fileinfo->name contains filename */

			path = g_strconcat (uri, G_DIR_SEPARATOR_S, fileinfo->name, NULL);
			if (g_file_test (path, G_FILE_TEST_ISDIR)) {
				/* got a new directory */
				numdirs++;

				/* add this dir to parent */
				switch (mode) {
				case SATTREE_MODE_SAT:
					text = g_strconcat (fileinfo->name, "<!SEP> <!SEP> ", NULL);
					vbuf = g_strsplit (text, "<!SEP>", TREE_COLS_SAT);
					g_free (text);
					myself = gtk_ctree_insert_node (GTK_CTREE (tree), parent, NULL, vbuf, 5,
									pixclosed, bitclosed, pixopen, bitopen,
									FALSE, FALSE);
					g_strfreev (vbuf);
					/* don't let users select whole directories */
					gtk_ctree_node_set_selectable (GTK_CTREE (tree), myself, FALSE);

					/* load the contents of this directory (WARNING: RECURSIVE CALL!) */
					sattree_add_local_uri (tree, myself, path, filter, mode);
					break;
				case SATTREE_MODE_FILE:
				default:
					text = g_strconcat (fileinfo->name, "<!SEP> ", NULL);
					vbuf = g_strsplit (text, "<!SEP>", TREE_COLS_FILE);
					g_free (text);
					myself = gtk_ctree_insert_node (GTK_CTREE (tree), parent, NULL, vbuf, 5,
									pixclosed, bitclosed, pixopen, bitopen,
									FALSE, FALSE);
					g_strfreev (vbuf);
					/* don't let users select whole directories */
					gtk_ctree_node_set_selectable (GTK_CTREE (tree), myself, FALSE);

					/* load the contents of this directory (WARNING: RECURSIVE CALL!) */
					sattree_add_local_uri (tree, myself, path, filter, mode);
					break;
				}
			}
			g_free (path);
		}

		/* free memory taken up by file list */
		gnome_vfs_file_info_list_free (list);
	}


	/****  Load files if any ****/

	/* create gnome_vfs_uri */
	vfsuri = gnome_vfs_get_uri_from_local_path (uri);

	/* create a filter and load the directory */
	vfsfilter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						    GNOME_VFS_DIRECTORY_FILTER_NODIRS |
						    GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						    GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						    filter ? filter : "*");
	result = gnome_vfs_directory_list_load (&list, vfsuri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						vfsfilter);
	/* free some memory */
	g_free (vfsuri);
	gnome_vfs_directory_filter_destroy (vfsfilter);

	if (result == GNOME_VFS_OK) {
		GdkPixmap *pixm;
		GdkBitmap *bitm;

		/* make a log message */
		msg = g_strconcat (__FUNCTION__, _(": loading files from "), uri, ": GNOME_VFS_OK", NULL);
		satlog_log (SAT_LOG_DEBUG, msg);
		g_free (msg);

		/* Create pixmaps and bitmaps */
		iconfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
					G_DIR_SEPARATOR_S, "gpredict-file-list.xpm", NULL);
		pixm = gdk_pixmap_create_from_xpm (app->window, &bitm, 0, iconfile);
		g_free (iconfile);

		/* loop throough the file list */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;  /* fileinfo->name contains filename */

			path = g_strconcat (uri, G_DIR_SEPARATOR_S, fileinfo->name, NULL);
			if (g_file_test (path, G_FILE_TEST_ISFILE)) {
				numfiles++;
				switch (mode) {
				case SATTREE_MODE_SAT:
					/* add this file to parent */
					text = g_strconcat (fileinfo->name, "<!SEP> <!SEP> ", NULL);
					vbuf = g_strsplit (text, "<!SEP>", TREE_COLS_SAT);
					g_free (text);
					myself = gtk_ctree_insert_node (GTK_CTREE (tree), parent, NULL, vbuf, 5,
									pixm, bitm, pixm, bitm,
									FALSE, FALSE);
					g_strfreev (vbuf);
					/* don't want users to select this */
					gtk_ctree_node_set_selectable (GTK_CTREE (tree), myself, FALSE);
					
					/* load the contents of this file */
					sattree_add_local_file (tree, myself, path);
					break;
				case SATTREE_MODE_FILE:
				default:
					/* add this file to parent */
					text = g_strconcat (fileinfo->name, "<!SEP> ", NULL);
					vbuf = g_strsplit (text, "<!SEP>", TREE_COLS_FILE);
					g_free (text);
					myself = gtk_ctree_insert_node (GTK_CTREE (tree), parent, NULL, vbuf, 5,
									pixm, bitm, pixm, bitm,
									FALSE, FALSE);
					g_strfreev (vbuf);
					/* don't want users to select this */
					gtk_ctree_node_set_selectable (GTK_CTREE (tree), myself, TRUE);
					
					/* count the number of sats in this file */
					sattree_add_sat_count (tree, myself, path);
					break;
				}
			}
			g_free (path);
		}

		/* free memory taken up by file list */
		gnome_vfs_file_info_list_free (list);
	}

	/* fill in the info field of the parent (if in FILE mode) */
	if (mode == SATTREE_MODE_FILE) {
		text = g_strdup_printf ("%d dirs and %d files", numdirs, numfiles);
		gtk_ctree_node_set_text (GTK_CTREE (tree), parent, 1, text);
		g_free (text);
	}

}



static void
sattree_add_local_file (GtkWidget *tree, GtkCTreeNode *parent, gchar *file)
{
	/* This function read the satellites from the 'file' into to the
	   GtkCTree. It also performs a check whether the file actually
	   contains valid TLE data (the same check as with the engine.
	*/
	FILE *satfile = NULL;
	gchar *msg;
	gint i,x;
	gchar *text,**vbuf;
	GtkCTreeNode *satnode;
	gchar catnum_s[6],*entry,*epoch_s;
	guint catnum,j,epochyear, epochday;
	GDate *epoch;
	gchar name[80],line1[80],line2[80];

	/* Just to prevent unexpected behaviuor */
	g_return_if_fail (file && tree && parent);

	/* open TLE file */
	satfile = fopen (file, "r");
	if (satfile) {

		i = 0;
		/* avoid reading the last sat twice */
		fgets (name, 75, satfile);
		while (!feof (satfile)) {
			/* read lines */
			fgets (line1, 75, satfile);
			fgets (line2, 75, satfile);

			if (KepCheck (line1, line2)) {

				/* fix satname */
				x = strlen (name);
				while ( name[x]==32 || name[x]==0 || name[x]==10 || name[x]==13 ) {
					name[x] = 0;
					x--;
				}

				/* extract catalog number */
				for (j=0; j<6; j++)
					catnum_s[j] = line1[j+2];
				catnum_s[5] = 0;
				catnum = (guint) g_strtod (catnum_s, NULL);

				/* FIXME: check whether we already have a
				   satellite with this catalog number.
				*/

				/* EPOCH */
				epochyear = (guint) g_strtod (SubString (line1, 18, 19), NULL);
				epochday  = (guint) g_strtod (SubString (line1, 20, 31), NULL);
				epoch = g_date_new_dmy (1, 1, 2000 + epochyear);
				g_date_add_days (epoch, epochday);
				epoch_s = g_malloc (50);
				g_date_strftime (epoch_s, 50, "%e %b %Y", epoch);
				
				/* add satelite to the tree */
				entry = g_strconcat (name,
						     "<!SEP>", catnum_s,
						     "<!SEP>", epoch_s,
						     NULL);
				vbuf = g_strsplit (entry, "<!SEP>", TREE_COLS_SAT);
				satnode = gtk_ctree_insert_node (GTK_CTREE (tree),
								 parent, NULL,
								 vbuf, 5,
								 NULL, NULL, NULL, NULL,
								 TRUE, FALSE);
				
				/* attach catalog number; this will provide the same functionality
				   as the flat list (pre-0.4).
				*/
				gtk_ctree_node_set_row_data (GTK_CTREE (tree), satnode,
							     GUINT_TO_POINTER (catnum));

				g_free (epoch_s);
				g_date_free (epoch);
				g_free (entry);
				g_strfreev (vbuf);
				i++;
			}
			fgets (name, 75, satfile);
		}
		fclose (satfile);
		text = g_strdup_printf ("%s: read %d satelites from %s",
					__FUNCTION__, i, file);
		satlog_log (SAT_LOG_INFO, text);
		g_free (text);
	}
	else {
		msg = g_strconcat (__FUNCTION__, _(": could not open file "), file, NULL);
		satlog_log (SAT_LOG_WARNING, msg);
		g_free (msg);
	}
}


static void
sattree_add_sat_count   (GtkWidget *tree, GtkCTreeNode *parent,
			 gchar *file)
{
	/* This function read the satellites from the 'file' into to the
	   GtkCTree. It also performs a check whether the file actually
	   contains valid TLE data (the same check as with the engine.
	*/
	FILE *satfile = NULL;
	gchar *msg;
	gint i=0;
	gchar *text;
	gchar name[80],line1[80],line2[80];

	/* Just to prevent unexpected behaviuor */
	g_return_if_fail (file && tree && parent);

	/* open TLE file */
	satfile = fopen (file, "r");
	if (satfile) {
		/* avoid reading the last sat twice */
		fgets (name, 75, satfile);
		while (!feof (satfile)) {
			/* read lines */
			fgets (line1, 75, satfile);
			fgets (line2, 75, satfile);

			if (KepCheck (line1, line2)) {
				i++;
			}
			fgets (name, 75, satfile);
		}
		fclose (satfile);
		text = g_strdup_printf ("%s: detected %d satelites in %s",
					__FUNCTION__, i, file);
		satlog_log (SAT_LOG_INFO, text);
		g_free (text);

		/* Fill in info field of parent */
		text = g_strdup_printf ("%d satellites", i);
		gtk_ctree_node_set_text (GTK_CTREE (tree), parent, 1, text);
		g_free (text);
	}
	else {
		msg = g_strconcat (__FUNCTION__, _(": could not open file "), file, NULL);
		satlog_log (SAT_LOG_WARNING, msg);
		g_free (msg);
	}
}




/*********** FIXME: MOVE THIS FURTHER UP WHEN IMPLEMENTED! */
static void
sattree_add_web_uri (GtkCTree *tree, GtkCTreeNode *parent, 
		     const gchar *uri, const gchar *filter,
		     sattree_mode mode)
{
	/* This function loads the specified web directory recursively
	   into the GtkCTree starting at the specified GtkCTreeNode.
	*/
}



static void 
sattree_select_row_cb   (GtkCTree *tree, GtkCTreeNode *node,
			 gint column, gpointer data)
{
	GList *sel = tree->clist.selection;
	guint length = 0;
	gchar *title;

	length = g_list_length (sel);
	if (length)
		title = g_strdup_printf ("%s (%d selected)", tree_titles_sat[0], length);
	else
		title = g_strdup_printf ("%s", tree_titles_sat[0]);
	gtk_clist_set_column_title (GTK_CLIST (tree), 0, title);
	g_free (title);
}



static void
sattree_unselect_row_cb (GtkCTree *tree, GtkCTreeNode *node,
			 gint column, gpointer data)
{
	GList *sel = tree->clist.selection;
	guint length = 0;
	gchar *title;

	length = g_list_length (sel);
	if (length)
		title = g_strdup_printf ("%s (%d selected)", tree_titles_sat[0], length);
	else
		title = g_strdup_printf ("%s", tree_titles_sat[0]);
	gtk_clist_set_column_title (GTK_CLIST (tree), 0, title);
	g_free (title);
}
