/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satdata.c:  Data structures and functions to store and retrieve     
              satellite data.                                         

  Copyright (C)  2001-2003  Alexandru Csete & John A. Magliacane.

  Authors:   Alexandru Csete <csete@users.sourceforge.net>
             John Magliacane (original author of tracking code).

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <stdio.h>
#include "defaults.h"
#include "satdata.h"


#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif


sat_t *sat_table[SAT_MAX];    /* Le big data-structure containing EVERYTHING */
guint count = 0;              /* last entry */

/* function prototypes */
static gboolean free_sat (guint);

/* Use this when we are updating
   sats at runtime.
*/
sat_t *dummy;

gboolean UPDATING = FALSE;


gint satdata_init ()
{
	/* This function creates and initializes the
	   hash table. If it returns non-zero value,
	   the hash table could not be created  and
	   the program must terminate. */

	/* intialize the dummy sat */
	if (!dummy)
		dummy = (sat_t *) g_malloc (sizeof (sat_t));
	if (!dummy->name)
		dummy->name = g_strdup ("Dummy");
	dummy->decayed = TRUE;

	return 0;
}


gint satdata_add_sat (sat_t *newsat)
{
	/* This function adds sat to the hash table.
	   It can also be used to update a specific
	   a satellite.
	   Returns the index of newsat or -1 in case
	   of error.
	*/
	gint index;

	index = satdata_get_sat_index (newsat->catnum, NULL);
	if (index != -1) {
		/* copy the contents */
		sat_table[index]->epyear = newsat->epyear;
		sat_table[index]->epday  = newsat->epday;
		sat_table[index]->ephem  = newsat->ephem;
		sat_table[index]->elnum  = newsat->elnum;
		sat_table[index]->incl   = newsat->incl;
		sat_table[index]->raan   = newsat->raan;
		sat_table[index]->eccn   = newsat->eccn;
		sat_table[index]->argper = newsat->argper;
		sat_table[index]->meanan = newsat->meanan;
		sat_table[index]->meanmo = newsat->meanmo;
		sat_table[index]->drag   = newsat->drag;
		sat_table[index]->decac  = newsat->decac;
		sat_table[index]->bstar  = newsat->bstar;
		sat_table[index]->eporb  = newsat->eporb;
		g_free (sat_table[index]->file);
		sat_table[index]->file   = g_strdup (newsat->file);

		/* free newsat */
		g_free (newsat->name);
		g_free (newsat->piece);
		g_free (newsat->file);
		g_free (newsat);
		
		return index;
	}
	else if (count<SAT_MAX) {
		sat_table[count] = newsat;
		return count++;
	}
	return -1;
}



sat_t *satdata_get_sat (guint i)
{
	/* This function returns a pointer to a sat_t
	   structure containing data for the satellite
	*/
	if (UPDATING)
		return dummy;
	else
		return (((i<count) && (i>=0)) ? sat_table[i] : NULL );

}


gint satdata_get_sat_index (guint32 catnum, gchar *name)
{
	/* This function returns the index of the satellite
	   that matches catalog number "catnum" or name
	   "name". If "catnum" is non-zero it is used and
	   "name" is ignored. The function return -1 if the
	   requested satellite is not in the list.
	*/
	guint i;

	if (UPDATING)
		return -1;

	if (catnum || name)
		for (i=0; i<count; i++) {
			if (catnum) {
				if (sat_table[i]->catnum == catnum)
					return i;
			}
			else
				if (!g_strcasecmp (sat_table[i]->name, name))
					return i;
		}
	
	/* sorry... */
	return -1;
}


guint satdata_count_sat ()
{
	/* Returns the number of satellites in the hash table */

	return count;
}


void satdata_remove_sat (guint i)
{
	/* Removes the satellite which has catalog number "key"
	   from the hash table. */

/*	if ( (i>=0) && (i<SAT_MAX) ) {
		free_sat( i );
		}*/

}


gint satdata_close ()
{
	/* Cleans up the hash table entries and destroys
	   the hash table. Returns nonzero value, if the
	   number of satellites deleted is not equal to
	   the original size of the hash table. */

	while (count > 0)
		free_sat (--count);

	return 0;
}



static gboolean
free_sat (guint i)
{
	/* This function frees the memory that has been
	   allocated to an entry in the sat_table. It is
	   called by the satdata_close() function. This
	   function should only be used by the satdata_*
	   functions, and not called from the outside. */

	if (sat_table[i]) {
		g_free (sat_table[i]->name);
		g_free (sat_table[i]->piece);
		if (sat_table[i]->file)
			g_free (sat_table[i]->file);
		g_free (sat_table[i]);
		return 0;
	}
	return 1;
}
