/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 */

#include <math.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <config.h>
#include "../gui.h" /* for gui_option_menu_get_active() */
#include "../modutils.h"

#ifdef HAVE_SAMPLERATE
#include <samplerate.h>
#endif

module modinfo;

typedef struct {
    module_id id;
    shell *shl;
    double base_rate;
    int new_frame_count_lock;
    AFframecount base_frame_count;
    AFframecount new_frame_count;
    GtkSpinButton *rate_control;
    GtkSpinButton *time_control;
    GtkSpinButton *factor_control;
    GtkOptionMenu *algo_control;
    GtkWidget *dialog;
} resample_params;

#define RECALC_FROM_FACTOR 0
#define RECALC_FROM_TIME   1
#define RECALC_FROM_RATE   2
#ifdef HAVE_GNOME2
#define RESAMPLE_GLADE_FILE "resample-2.glade"
#else
#define RESAMPLE_GLADE_FILE "resample.glade"
#endif

module *
module_new() {
    MODULE_INIT(&modinfo, 
		"Resample",
		"Pascal Haakmat",
		"Copyright (C) 2002,2003");
    return &modinfo;
}

void
on_apply_clicked(GtkWidget *w,
                 gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)), 
                          "user_data");
    resample_params *p = (resample_params *)module_state->data;
    p->new_frame_count = gtk_spin_button_get_value(p->time_control);
    action_do(ACTION_MODULE_EXECUTE_NEW(WITH_UNDO, 
                                        p->shl,
                                        p->id));
}

void
on_ok_clicked(GtkWidget *w,
              gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    resample_params *p = (resample_params *)module_state->data;
    on_apply_clicked(w, NULL);
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_close_clicked(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    resample_params *p = (resample_params *)module_state->data;
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_dialog_destroy(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    module_state->is_open = 0;
    free(module_state->data);
}

void
recalc(resample_params *p, int caller) {
    double a, b;
    double factor = (double)p->new_frame_count / (double)p->base_frame_count;

    p->new_frame_count_lock = 1;
    a = gtk_spin_button_get_value(p->rate_control);
    b = factor * p->base_rate;
    if(a != b && caller != RECALC_FROM_RATE)
        gtk_spin_button_set_value(p->rate_control, b);

    a = gtk_spin_button_get_value(p->time_control);
    b = p->new_frame_count;
    if(a != b && caller != RECALC_FROM_TIME) 
        gtk_spin_button_set_value(p->time_control, b);

    a = gtk_spin_button_get_value(p->factor_control);
    b = factor;
    if(a != b && caller != RECALC_FROM_FACTOR) {
        gtk_spin_button_set_value(p->factor_control, b);
    }
    p->new_frame_count_lock = 0;

}

void
on_rate_changed(GtkWidget *w,
                gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    resample_params *p = (resample_params *)module_state->data;
    if(p->new_frame_count_lock)
        return;
    p->new_frame_count = p->base_frame_count * 
        (gtk_spin_button_get_value(p->rate_control) / p->base_rate);
    recalc(p, RECALC_FROM_RATE);
}

void
on_time_changed(GtkWidget *w,
                gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    resample_params *p = (resample_params *)module_state->data;
    if(p->new_frame_count_lock)
        return;
    p->new_frame_count = gtk_spin_button_get_value_as_int(p->time_control);
    recalc(p, RECALC_FROM_TIME);
}

void
on_factor_changed(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    resample_params *p = (resample_params *)module_state->data;    
    if(p->new_frame_count_lock)
        return;
    p->new_frame_count = p->base_frame_count * 
        (1 / gtk_spin_button_get_value(p->factor_control));
    recalc(p, RECALC_FROM_FACTOR);
}

void
on_dialog_focus_in_event(GtkWidget *w,
                         GdkEvent *event,
                         gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    resample_params *p = (resample_params *)module_state->data;    

    if(p->base_frame_count != p->shl->select_end - p->shl->select_start) {
        p->base_frame_count = p->shl->select_end - p->shl->select_start;
        gtk_spin_button_set_value(p->time_control,
                                  p->shl->select_end - p->shl->select_start);
    }
}

void
module_open(module_id id,
            shell *shl, 
            int undo) {
    GladeXML *xml;
    GtkWidget *w, *algos;
    char path[4096];
    int i;
    resample_params *p = mem_calloc(1, sizeof(resample_params));
    
    if(!p) {
        gui_alert("Not enough memory.");
        return;
    }

    snprintf(path, 4096, "%s/%s", dirname(modules[id].fname), RESAMPLE_GLADE_FILE);
    DEBUG("loading interface %s\n", path);
    xml = glade_xml_new(path, "dialog", NULL);
    
    if(!xml) {
        gui_alert("Resample: could not load interface %s.", path);
        free(p);
        return;
    }

    shl->module_state[id].is_open = 1;
    shl->module_state[id].data = p;
    p->id = id;
    p->shl = shl;
    p->base_rate = shl->sr->rate;
    p->new_frame_count_lock = 0;
    p->base_frame_count = shl->select_end - shl->select_start;
    p->rate_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "rate"));
    p->time_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "time"));
    p->factor_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "factor"));
    p->algo_control = GTK_OPTION_MENU(glade_xml_get_widget(xml, "algo"));
    p->dialog = GTK_WIDGET(glade_xml_get_widget(xml, "dialog"));
    g_signal_connect(GTK_OBJECT(p->dialog), "destroy", 
                     G_CALLBACK(on_dialog_destroy), shl);
    g_signal_connect(GTK_OBJECT(p->dialog), "focus_in_event", 
                     G_CALLBACK(on_dialog_focus_in_event), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "ok"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_ok_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "close"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_close_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "apply"));
    gtk_spin_button_set_value(p->rate_control, shl->sr->rate);
    gtk_spin_button_set_value(p->time_control, 
                              shl->select_end - shl->select_start);
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_apply_clicked), shl);
    g_signal_connect(GTK_OBJECT(p->rate_control), "changed", 
                     G_CALLBACK(on_rate_changed), shl);
    g_signal_connect(GTK_OBJECT(p->time_control), "changed", 
                     G_CALLBACK(on_time_changed), shl);
    g_signal_connect(GTK_OBJECT(p->factor_control), "changed", 
                     G_CALLBACK(on_factor_changed), shl);
    g_object_set_data(G_OBJECT(p->dialog), "user_data",
                      GINT_TO_POINTER(&shl->module_state[id]));
    g_object_unref(G_OBJECT(xml));

    algos = gtk_menu_new();
    gtk_widget_show(algos);
    for(i = 0; resample_algo_name(i); i++) {
        w = gtk_menu_item_new_with_label(resample_algo_name(i));
        gtk_widget_show(w);
        gtk_menu_shell_append(GTK_MENU_SHELL(algos), w);
    }
    gtk_option_menu_set_menu(p->algo_control, algos);
}

void
module_close(mod_state *module_state) {
    gtk_object_destroy(GTK_OBJECT(((resample_params *)module_state->data)->dialog));
}

action_group *
module_execute(shell *shl, 
               int undo) {
    AFframecount start = shl->select_start,
        end = shl->select_end;
    int map = shl->select_channel_map;
    int t, id = shl->active_module_id, seltrks = 0, algo;
    AFframecount largest = 0, newlen = 0;
    resample_params *p;
    AFframecount track_newlen[32];
    action_group *undo_ag = NULL;
    snd *del_sr = NULL;
    double factor;

    if(!shl->module_state[id].is_open) { 
        gui_alert("Resample does not have defaults.");
        return NULL;
    }

    p = shl->module_state[id].data;
    algo = gui_option_menu_get_active(p->algo_control);
    factor = (double)(end - start) / (double)p->new_frame_count;
#ifdef HAVE_SAMPLERATE
    if(src_get_name(algo)) {
        if(!src_is_valid_ratio(factor)) {
            gui_alert("Factor is out of range for this method.\nTry the Integer ZOH method.");
            return NULL;
        }
    }
#endif

    DEBUG("resample: algorithm: %d\n", algo);
    
    if(p->new_frame_count == 0) {
        gui_alert("Factor cannot be 0.");
        FAIL("factor == 0\n");
        return NULL;
    }

    rwlock_rlock(&shl->sr->rwl);

    if(undo) {
        del_sr = snd_copy(shl->sr, map, start, end - start);
        if(RESULT_IS_ERROR(shl->sr)) {
            gui_alert("Could not create undo: %s!", snd_error_get(shl->sr));
            rwlock_runlock(&shl->sr->rwl);
            return NULL;
        }
    }

    for(t = 0; t < snd_track_count(shl->sr); t++) {
        if((1 << t) & map) {
            newlen = resample(shl,
                              t,
                              start,
                              end,
                              p->new_frame_count,
                              algo,
                              HONOR_ENVELOPES,
                              !GUARANTEE_NEW_FRAME_COUNT);
            track_newlen[t] = newlen;
            seltrks++;
            if(newlen > largest)
                largest = newlen;
        }
    }

    /* FIXME: this code is ugly. it is necessary because the selection
       model cannot describe selections of different lengths over
       several tracks. To generalize this code such a selection model
       would need to be created. */

    if(largest != 0)
        shl->select_end = shl->select_start + largest;

    if(undo) {
        undo_ag = action_group_new_empty(2 + seltrks);

        if(!undo_ag) {
            snd_destroy(del_sr);
            rwlock_runlock(&shl->sr->rwl);
            return NULL;
        }

        undo_ag->a[seltrks] = 
            ACTION_INSERT_NEW(DONT_UNDO,
                              shl->sr, 
                              del_sr, 
                              map,
                              start);
        undo_ag->a[seltrks + 1] = 
            ACTION_SELECT_NEW(DONT_UNDO,
                              shl,
                              shl->sr,
                              map,
                              start,
                              snd_frame_count(del_sr));

        for(t = 0; t < snd_track_count(shl->sr); t++) {
            if((1 << t) & map) {
                undo_ag->a[--seltrks] =
                    ACTION_DELETE_ON_TRACK_NEW(DONT_UNDO,
                                               shl,
                                               shl->sr,
                                               t,
                                               start,
                                               track_newlen[t]);
            }
        }

    }

    rwlock_runlock(&shl->sr->rwl);
    return undo_ag;
}
