/*
**  LocalMailDelivery.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Jonathan B. Leffert <jonathan@leffert.net>
**          Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "LocalMailDelivery.h"

#include "Constants.h"
#include <Pantomime/LocalFolder.h>

#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>


#ifndef LOCK_SH
#define LOCK_SH 1
#endif
#ifndef LOCK_EX
#define LOCK_EX 2
#endif 
#ifndef LOCK_NB
#define LOCK_NB 4
#endif
#ifndef LOCK_UN
#define LOCK_UN 8
#endif


@implementation LocalMailDelivery


//
//
//
- (id) initWithPathToSpool: (NSString *) thePath
{
  self = [super init];

  [self setPathToSpool: thePath];

  return self;
}


//
//
//
- (void) dealloc
{
  RELEASE(pathToSpool);

  [super dealloc];
}


//
// general methods
//
- (BOOL) needToReadMailspool: (NSString *) theMailspool
{
  NSFileManager *fileManager = [NSFileManager defaultManager];
  NSDictionary *fileAttributes = nil;
  NSNumber *spoolFileSize = nil;

  if ( ![fileManager fileExistsAtPath: theMailspool] )
    {
      return NO;
    }
  
  fileAttributes = [fileManager fileAttributesAtPath: theMailspool
				traverseLink: YES];

  spoolFileSize = [fileAttributes objectForKey: NSFileSize];
  
  if ( [spoolFileSize isEqualToNumber: [NSNumber numberWithInt: 0]] )
    {
      return NO;
    }
  else
    {
      return YES;
    }
}


//
//
//
- (NSArray *) readContentsOfMailspool: (NSString *) theMailspool
{
  NSMutableArray *aMutableArray;
  
  struct flock flock_entry;
  int mailspool_fd;
  FILE *stream;
  
  // we uses fcntl(2) for locking the local mailspool.  this provides a
  // mandatory, kernel-enforced lock that should work over NFS.
  // However, on linux, fcntl(2) and flock(2) are implemented as to ignore
  // each other and if the local delivery agent uses flock(2), we won't
  // block appropriately.  so, if we're on linux, we'll lock the mailspool
  // with both fcntl(2) and flock(2).

  aMutableArray = [[NSMutableArray alloc] init];
  AUTORELEASE(aMutableArray);
  
  if ( (mailspool_fd = open([theMailspool cString], O_RDWR)) != -1 )
    {
      flock_entry.l_type = F_WRLCK;
      flock_entry.l_whence = SEEK_SET;
      flock_entry.l_start = 0;
      flock_entry.l_len = 0;
      flock_entry.l_pid = getpid();
      
      if ( fcntl(mailspool_fd, F_SETLK, &flock_entry) != -1 )
	{
#ifdef __linux__
	  if ( flock(mailspool_fd, LOCK_EX | LOCK_NB) != -1 )
	    {
#endif
	      char aLine[1024];
	      long begin, end;
	
	      // When we close the stream later, it'll close the fd.
	      stream = fdopen(mailspool_fd, "r+");

	      // FIXME - we should unlock things properly
	      if (stream == NULL)
		{
		  return nil;
		}

	      begin = end = 0;
	      memset(aLine, 0, 1024);

	      fseek(stream, begin, SEEK_SET);

	      while (fgets(aLine, 1024, stream) != NULL)
		{
		  if (strncasecmp(aLine, "From ", 5) == 0)
		    {
		      NSData *aData;
		      
		      unsigned long length;
		      char *buf;
		      
		      // We always 'skip' the "From " line
		      begin = ftell(stream);
		      end = ftell(stream);

		      // We read until we reach an other "From " or, the end of the stream
		      while (fgets(aLine, 1024, stream) != NULL)
			{
			  if (strncmp(aLine, "From ", 5) == 0) break;
			  else end = ftell(stream);
			}

		      // We get the length of our message
		      length = end - begin - 1;

		      // We allocate our buffer for the message
		      buf = (char *)malloc(length * sizeof(char));
		      memset(buf, 0, length);
		      
		      // We move our fp to the beginning of the message
		      fseek(stream, begin, SEEK_SET);
		      fread(buf, sizeof(char), length, stream);
		      
		      aData = [[NSData alloc] initWithBytesNoCopy: buf
					      length: length];
		      
		      [aMutableArray addObject: aData];
		      RELEASE(aData);
		      
		      // We reset our fp to the right position (end of previous message)
		      fseek(stream, end, SEEK_SET);
		      memset(aLine, 0, 1024);
		    }
		}
	      	      
	      // now we can read the mailspool
	      //contents = [NSString stringWithContentsOfFile:
	      //			     theMailspool];
	      
	      // We now truncate our file to a length of 0.
	      if ( ftruncate(mailspool_fd, 0) != -1 )
		{
		  // remove the advisory lock
		  flock(mailspool_fd, LOCK_UN);
		  
		  // remove the mandatory lock
		  flock_entry.l_type = F_UNLCK;
		  if ( fcntl(mailspool_fd, F_SETLK, &flock_entry) == -1 )
		    {
		      NSDebugLog(@"Could not remove mandatory file lock for: %@", theMailspool);
		      NSDebugLog(@"Reson: %s", strerror(errno));
		    }
		  // now, close the file
		  if ( fclose(stream) != 0 )
		    {
		      NSDebugLog(@"Could not close file: %@", theMailspool);
		      NSDebugLog(@"Reason: %s", strerror(errno));
		    }		      
		}
	      else
		{
		  NSDebugLog(@"Could not truncate file: %@",
			theMailspool);
		  NSDebugLog(@"Reason: %s", strerror(errno));
		}
#ifdef __linux__
	    }
	  else
	    {
	      // error setting the advisory lock.
	      NSDebugLog(@"Could not set advisory lock on file: %@",
		    theMailspool);
	      NSDebugLog(@"Reason: %s", strerror(errno));
	    }
#endif
	}
      else
	{
	  // error setting the mandatory lock
	  NSDebugLog(@"Could not set mandatory lock on file: %@",
		theMailspool);
	  NSDebugLog(@"Reason: %s", strerror(errno));
	}
    }
  else
    {
      // error while opening file
      NSDebugLog(@"Could not open file: %@", theMailspool);
      NSDebugLog(@"Reason: %s", strerror(errno));
    }

  return aMutableArray;
}


//
// FIXME. This method is slow and ugly. We should combine
// this method and the -readContentsOfMailspool method
// to make something way more faster.
//
- (NSArray *) messagesFromMailspoolFile
{ 
  if ( [self needToReadMailspool: [self pathToSpool]] )
    {
      NSArray *allMessages;
      
      allMessages = [self readContentsOfMailspool: [self pathToSpool]];
      
      if ( allMessages )
	{
	  return allMessages;
	}
    }

  return [NSArray array];
}


//
// access/mutation methods
//
- (NSString *) pathToSpool
{
  return pathToSpool;
}

- (void) setPathToSpool: (NSString *) thePath
{
  RETAIN(thePath);
  RELEASE(pathToSpool);
  pathToSpool = thePath;
}

@end
