//  Gnomoradio - rainbow/rdf-resource.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "rdf-resource.h"
#include <deque>

using namespace Rainbow;
using namespace sigc;
using namespace std;

map<Glib::ustring,RdfResource*> Rainbow::RdfResource::resource_map;

void Rainbow::RdfResource::get_and_do (const Glib::ustring &uri,
				       const sigc::slot<void,xmlpp::Element*,ref_ptr<RdfResource> > &slot)
{
	ref_ptr<RdfResource> resource;
	Glib::ustring base_uri(uri), id;
	remove_tag_from_uri(base_uri, id);
	map<Glib::ustring,RdfResource*>::iterator p = resource_map.find(base_uri);
	if (p != resource_map.end()) {
		resource = ref_ptr<RdfResource>(p->second);
		if (resource->downloaded) { // already downloaded
			map<Glib::ustring,xmlpp::Element*>::iterator el = resource->id_map.find(id);
			signal<void,xmlpp::Element*,ref_ptr<RdfResource> > sig;
			sig.connect(slot);
			sig(el != resource->id_map.end() ? el->second : 0, resource);
			return;
		}
	} else {
		resource = ref_ptr<RdfResource>(new RdfResource(base_uri));

		// make sure it is actually a valid url
		Glib::ustring host, file;
		unsigned short port;
		if (!HttpClient::parse_url(uri, host, port, file)) {
			signal<void,xmlpp::Element*,ref_ptr<RdfResource> > sig;
			sig.connect(slot);
			sig(0, resource);
			return;
		}
	}

	// connect mem_fun to appropriate signal
	map<Glib::ustring,signal<void,xmlpp::Element*, ref_ptr<RdfResource> >*>::iterator sig;
	sig = resource->signal_map.find(id);
	if (sig != resource->signal_map.end()) {
		// found existing signal for id
		sig->second->connect(slot);
	} else {
		// need to create new signal to handle id
		signal<void,xmlpp::Element*,ref_ptr<RdfResource> > *signal_ = new signal<void,xmlpp::Element*,ref_ptr<RdfResource> >;
		signal_->connect(slot);
		resource->signal_map.insert(make_pair(id, signal_));
	}
}

void Rainbow::RdfResource::make_absolute_uri (Glib::ustring &uri,
					      const Glib::ustring &base)
{
	if (uri.size() && uri[0] == '#') {
		Glib::ustring b(base), id;
		remove_tag_from_uri(b, id);
		uri = b + uri;
	}
}

void Rainbow::RdfResource::remove_tag_from_uri (Glib::ustring &uri,
						Glib::ustring &id)
{
	Glib::ustring::size_type pound = uri.find('#');
	if (pound != Glib::ustring::npos) {
		id = uri.substr(pound + 1);
		uri = uri.substr(0, pound);
	}
}

Rainbow::RdfResource::RdfResource (const Glib::ustring &uri)
	: refcnt(0),
	  resource_uri(uri),
	  downloaded(false)
{
	base_uri = resource_uri;
	Glib::ustring tmpid;
	remove_tag_from_uri(base_uri, tmpid);

	resource_map.insert(make_pair(uri, this));

	// cache
	const unsigned int cache_size = 10;
	static deque<ref_ptr<RdfResource> > cache;
	if (cache.size() == cache_size)
		cache.pop_front();
	cache.push_back(ref_ptr<RdfResource>(this));

	alarm.signal_alarm().connect(mem_fun(*this, &RdfResource::get));
	get();
}

void Rainbow::RdfResource::get ()
{
	Glib::ustring host, file;
	unsigned short port;
	HttpClient::parse_url(base_uri, host, port, file); // error case is taken care of above

	http.reset(new HttpClient(host, port));
	http->signal_request_done.connect(mem_fun(*this, &RdfResource::on_downloaded));
	ref();
	http->get(file);
}

Rainbow::RdfResource::~RdfResource ()
{
	// remove all occurances from global map
	for (map<Glib::ustring,RdfResource*>::iterator i = resource_map.begin(); i != resource_map.end(); ++i) {
		if (i->second == this)
			resource_map.erase(i);
	}

	clear_signal_map();
}

void Rainbow::RdfResource::on_downloaded (bool success)
{
	xmlpp::Element *root;
	xmlpp::Node::NodeList nodes;
	xmlpp::Attribute *xml_base;
	string buffer;

	downloaded = true;

	if (!success)
		goto error;

	// this block is for rdf files embedded in something larger
	{
		string::size_type rdf_block_begin = http->get_buffer().find("<rdf:RDF");
		string::size_type rdf_block_end = http->get_buffer().find("</rdf:RDF>");
		if (rdf_block_begin != string::npos
		    && rdf_block_end != string::npos
		    && rdf_block_begin < rdf_block_end) {
			buffer = http->get_buffer().substr(rdf_block_begin,
							   rdf_block_end - rdf_block_begin + 10);
		} else
			buffer = http->get_buffer();
	}

	try {
		xml_tree.parse_memory(buffer);
	} catch (...) {
		goto error;
	}

	// iterate through document and take note of ID tags and about tags
	root = dynamic_cast<xmlpp::Element*>(xml_tree.get_document()->get_root_node());
	if (!root || root->get_name() != "RDF")
		goto error;
	id_map.insert(make_pair(Glib::ustring(), root));
	xml_base = root->get_attribute("base");
	if (xml_base && xml_base->get_value().size())
		base_uri = xml_base->get_value();
	nodes = root->get_children();
	for (xmlpp::Node::NodeList::iterator i = nodes.begin(); i != nodes.end(); ++i) {
		xmlpp::Element *node = dynamic_cast<xmlpp::Element*>(*i);
		if (!node)
			continue;

		xmlpp::Attribute *id = node->get_attribute("ID");
		if (id)
			id_map.insert(make_pair(id->get_value(), node));
		else {
			xmlpp::Attribute *about = node->get_attribute("about");
			if (about)
				about_map.insert(make_pair(about->get_value(), node));
		}
	}

 error:
	// call all signals
	map<Glib::ustring,signal<void,xmlpp::Element*,ref_ptr<RdfResource> >*>::iterator signal;
	for (signal = signal_map.begin(); signal != signal_map.end(); ++signal) {
		map<Glib::ustring,xmlpp::Element*>::iterator el = id_map.find(signal->first);
		signal->second->emit(el != id_map.end() ? el->second : 0, ref_ptr<RdfResource>(this));
	}
	clear_signal_map();

	http.reset();
	alarm.set_relative(3600 * 12); // FIXME: make this time interval configurable
	unref();
}

void Rainbow::RdfResource::clear_signal_map ()
{
	map<Glib::ustring,signal<void,xmlpp::Element*,ref_ptr<RdfResource> >*>::iterator signal;
	for (signal = signal_map.begin(); signal != signal_map.end(); ++signal)
		delete signal->second;
	signal_map.clear();
}

xmlpp::Element *Rainbow::RdfResource::get_secondary_info (const Glib::ustring &uri)
{
	map<Glib::ustring,xmlpp::Element*>::iterator p = about_map.find(uri);
	return p != about_map.end() ? p->second : 0;
}
