//  Gnomoradio - rainbow/rainbow-hub.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>

#include <sstream>
#include <string>
#include <set>
#include <map>

#include <cassert>
#include <unistd.h>

#include "rainbow/util.h"

#define CRLF "\r\n"

/*
things we need to limit:
number of requests per minute per user -- some sort of priority system
make sure files are actually sharable
*/

using namespace std;

const int server_port = 18373;
const int max_pending_connections = 25;

const int max_urls_per_user = 100;
const int max_urls_per_query = 25;
const int max_statement_length = 2000;

class Client;

static fd_set fds;
static Client *client_fd_map[FD_SETSIZE];

class Resource
{
public:
	static Resource *find (const string &url);
	static Resource *create (const string &url)
		{
			Resource *r = find(url);
			return r ? r : new Resource(url);
		}

	// only to be called from Client
	void add_client (Client *client)
		{ clients.insert(client); }
	void remove_client (Client *client)
		{
			set<Client*>::iterator p = clients.find(client);
			assert(p != clients.end());
			clients.erase(p);

			if (clients.size() == 0) {
				resource_map.erase(my_iter);
				delete this;
			}
		}

	int clients_size () const { return clients.size(); }
	set<Client*>::iterator clients_begin () { return clients.begin(); }
	set<Client*>::iterator clients_end () { return clients.end(); }

	unsigned int offset () { if (m_offset >= clients_size()) m_offset = 0; return m_offset++; }

private:
	Resource (const string &url) : m_url(url), m_offset(0)
		{ my_iter = resource_map.insert(make_pair(url, this)).first; }

	unsigned int m_offset;
	string m_url;
	set<Client*> clients;

	map<string,Resource*>::iterator my_iter;
	static map<string,Resource*> resource_map;
};

class Client
{
public:
	Client (int socket, struct sockaddr_in *clientname)
		: fd(socket),
		  initialized(false)
		{
			if (fd >= FD_SETSIZE) {
				cerr << "Out of file descriptors" << endl;
				exit(1);
			}
			client_fd_map[fd] = this;

			FD_SET(fd, &fds);

			ip = inet_ntoa(clientname->sin_addr);
		}

	~Client ();

	void process_request ();
	string get_address() { return address; }

private:
	void add_resource (const string &url);
	void remove_resource (const string &url);
	int resources_size () const { return resources.size(); }

	set<Resource*> resources;
	string buffer;
	string ip, address;
	bool initialized;
	int fd;
	//time_t last_ping;
};

map<string,Resource*> Resource::resource_map;

Resource *Resource::find (const string &url)
{
	map<string,Resource*>::iterator p = resource_map.find(url);
	return (p != resource_map.end())
		? p->second : 0;
}

Client::~Client ()
{
	// remove resources
	set<Resource*>::iterator i;
	for (i = resources.begin(); i != resources.end(); ++i)
		(*i)->remove_client(this);
	
	close(fd);
	FD_CLR(fd, &fds);
}

void Client::process_request ()
{
	const int buf_size = 16384;
	char buf[buf_size];

	int r = read(fd, buf, buf_size);
	if (r < 0) {
		cerr << "Read error" << endl;
		delete this;
	} else if (r == 0) {
		// EOF
		delete this;
	} else {
		// normal operation
		buffer.append(buf, r);
		string::size_type p;
		while ((p = buffer.find("\n")) != string::npos) {
			string line = buffer.substr(0, p);
			buffer = buffer.substr(p + 1);
			if (line.size() == 0)
				return;
			string::size_type line_end = (buffer[buffer.size() - 1] == '\r') ? buffer.size() - 3 : buffer.size() - 2;
			if (line_end < 0)
				return;
			if (!initialized) {
				// handshake line
				if (line.substr(0, 12) != "RAINBOW/1.0 ") {
					delete this;
					return;
				}
				address = ip + ':' + line.substr(12);
				initialized = true;
				continue;
			}
			string url = line.substr(1, line_end);
			switch (line[0]) {
			case '+':
				if (resources_size() < max_urls_per_user)
					add_resource(url);
				break;
			case '-':
				remove_resource(url);
				break;
			case '?':
			{
				Resource *r = Resource::find(url);
				string reply;
				if (r) {
					unsigned int locations = min(r->clients_size(), max_urls_per_query);
					ostringstream length;
					length << locations;
					reply = length.str() + CRLF;
					set<Client*>::iterator c = r->clients_begin();

					unsigned int offset = r->offset();
					for (unsigned int i = 0; i < offset; ++i)
						++c;

					for (unsigned int i = 0; i < locations; ++i) {
						reply += (*c)->get_address();
						reply += CRLF;

						if (++c == r->clients_end())
							c = r->clients_begin();
					}
				} else {
					reply = "0" CRLF;
				}
				ssize_t bytes_written = 0;
				while (bytes_written < reply.length())
					bytes_written += write(fd,
							       reply.data() + bytes_written,
							       reply.length() - bytes_written);
			} break;
			default:
				delete this;
				return;
			};
		}
		if (buffer.size() > max_statement_length)
			delete this;
	}
}

void Client::add_resource (const string &url)
{
	Resource *r = Resource::create(url);
	resources.insert(r);
	r->add_client(this);
}

void Client::remove_resource (const string &url)
{
	Resource *r = Resource::find(url);
	if (r == 0)
		return;

	set<Resource*>::iterator p = resources.find(r);
	assert(p != resources.end());

	r->remove_client(this);
	resources.erase(p);
}

int main ()
{
	// create socket
	struct sockaddr_in servername;
	int sock = socket(PF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		cerr << "Cannot create socket" << endl;
		return 1;
	}

	// bind to address
	servername.sin_family = AF_INET;
	servername.sin_port = htons(server_port);
	servername.sin_addr.s_addr = htonl(INADDR_ANY);
	if (bind(sock, (struct sockaddr *) &servername, sizeof(servername)) < 0) {
		cerr << "Cannot bind to port " << server_port << endl;
		return 1;
	}

	// listen on port
	if (listen(sock, max_pending_connections) < 0) {
		cerr << "Cannot listen to port" << endl;
		return 1;
	}

	// initialize fd_set
	FD_ZERO(&fds);
	FD_SET(sock, &fds);

	// work requests
	for (;;) {
		fd_set ready_fds = fds;
		if (select(FD_SETSIZE, &ready_fds, 0, 0, 0) < 0) {
			cerr << "Cannot select" << endl;
			return 1;
		}

		for (int i = 0; i < FD_SETSIZE; ++i) {
			if (FD_ISSET(i, &ready_fds)) {
				// file descriptor is active
				if (i == sock) {
					// pending connection
					struct sockaddr_in clientname;
					socklen_t size = sizeof(clientname);
					int s = accept(sock, (struct sockaddr *) &clientname, &size);
					if (s < 0) {
						cerr << "Can't accept connection" << endl;
						return 1;
					} else if (s >= FD_SETSIZE) {
						close(s);
					} else {
						new Client(s, &clientname);
					}
				} else {
					client_fd_map[i]->process_request();
				}
			}
		}
	}

	return 0;
}
