/*  Gnometab -- a guitar tablature editor for GNOME
    Copyright (C) 2001  William L. Guelker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "tab_canvas.h"
#include "interface.h"
#include "tab_objects.h"
#include "chord_builder.h"
#include "chord_canvas.h"
#include "dialogs.h"
#include "chord_lib.h"
#include "tab_canvas_print.h"
#include "tab_doc.h"

/* about box */

static void 
show_about_box(void)
{
    GtkWidget* about_box;
    const gchar *authors[] = {    "William L. Guelker", NULL    };
    const gchar version[] = "0.7.4";
    const gchar copyright[] = "Copyright 2002 William L. Guelker";
    const gchar comments[] = "A guitar tablature editor for GNOME.";
    GdkPixbuf *pixbuf = NULL;
    
    pixbuf = gdk_pixbuf_new_from_file(PACKAGE_PIXMAPS_DIR"/gtab-icon.png", NULL);
    
    about_box = gnome_about_new("Gnometab", version,
                                copyright,
                                comments,
                                authors,
                                NULL, 
                                NULL,
                                pixbuf);
    
    gtk_window_set_type_hint(GTK_WINDOW(about_box), GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_position(GTK_WINDOW(about_box), GTK_WIN_POS_CENTER_ON_PARENT);
    gtk_widget_show(about_box);
    
    
}

/* icons */

static gchar *pointer_icon[]={
    "21 21 3 1",
    ". c None",
    "a c #303030",
    "# c #ffffff",
    ".....................",
    ".##..................",
    ".#a##................",
    "..#aa##..............",
    "..#aaaa##............",
    "...#aaaaa##..........",
    "...#aaaaaaa##........",
    "....#aaaaaaaa##......",
    "....#aaaaaaaaaa##....",
    ".....#aaaaaaaaaaa#...",
    ".....#aaaaaaaaaa#....",
    "......#aaaaaaaa#.....",
    "......#aaaaaaa#......",
    ".......#aaaaaaa#.....",
    ".......#aaaa#aaa#....",
    "........#aa#.#aaa#...",
    "........#a#...#aaa#..",
    ".........#.....#a#...",
    "................#....",
    ".....................",
    "....................."
};

static gchar *text_icon[]={
    "21 21 19 1",
    ". c None",
    "c c #303030",
    "e c #323232",
    "i c #434343",
    "f c #4b4b4b",
    "b c #4c4c4c",
    "g c #4f4f4f",
    "d c #505050",
    "h c #555555",
    "a c #585858",
    "j c #636363",
    "k c #646464",
    "m c #656565",
    "l c #666666",
    "o c #676767",
    "p c #696969",
    "n c #6a6a6a",
    "# c #707070",
    "q c #d1d1d1",
    ".....................",
    "........#####a.......",
    ".......bcccccca......",
    "......deaaaaafca.....",
    ".....gca......hca....",
    ".....ia.......aca....",
    "..............jca....",
    ".............kca.....",
    "............jcl......",
    "...........mcn.......",
    "..........jco........",
    ".........kcn.........",
    "........jco..........",
    ".......acn...........",
    "......aca............",
    ".....acp.............",
    "....accq.............",
    "....accaaaaaaaaaa....",
    "....accccccccccca....",
    "....aaaaaaaaaaaaa....",
    "....................."
};

static gchar *bend_icon[]={
    "21 21 3 1",
    ". c None",
    "a c #303030",
    "# c #585858",
    ".....................",
    ".....................",
    "..............##.....",
    ".............####....",
    "............#aaaa#...",
    "...........#aaaaaa#..",
    "..............aa.....",
    "..............a#.....",
    "..............a#.....",
    "..............a#.....",
    ".............#a......",
    ".............#a......",
    ".............a#......",
    "............#a.......",
    "............a#.......",
    "...........#a........",
    ".........#aa#........",
    "......##aa#..........",
    "..##aaaa#............",
    "..#aa#...............",
    "....................."
};

static gchar *bar_icon[]={
    "21 21 3 1",
    ". c None",
    "a c #303030",
    "# c #bf0841",
    "...........##....a...",
    "..........##....aa...",
    ".........##....aa....",
    ".........#....aa.....",
    "........##....a..a...",
    "........#.......aa...",
    ".......##......aa....",
    ".......#......aa.....",
    "......##......aa.....",
    "......##......aa.....",
    "......##......aa.....",
    "......##......a..a...",
    "......##........aa...",
    "......##.......aa....",
    ".......##.....aa.....",
    "........#.....aa.....",
    "........##....aa.....",
    ".........#....aa.....",
    ".........##...aa.....",
    "..........##..aa.....",
    "...........##.aa....."
};

static gchar *slur_icon[]={
    "21 21 4 1",
    ". c None",
    "a c #303030",
    "b c #585858",
    "# c #bf0841",
    ".....................",
    ".........####........",
    ".....############....",
    "...###..........###..",
    "..##..............##.",
    ".##..................",
    "......aa.........aa..",
    ".....aa.........aa...",
    "....aa.........aa....",
    "...aa.........aa.....",
    "...aa.........aa.....",
    "...aa.........aa.....",
    "...aa.........aa.....",
    "...aa.........aa.....",
    "...aaaaaaaaaaaaa.....",
    "...aaaaaaaaaaaaa.....",
    "...aa.........aa.....",
    "...aa.........aa.....",
    "...aaaaaaaaaaaaa.....",
    "...bbbbbbbbbbbbb.....",
    "....................."
};

static gchar *measure_icon[]={
    "21 21 2 1",
    ". c None",
    "# c #303030",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    ".....#####....###....",
    ".....#####....###....",
    "...#########..###....",
    "...#########..###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###....",
    "..............###...."
};

static gchar *legato_icon[]={
    "21 21 3 1",
    ". c None",
    "a c #303030",
    "# c #585858",
    "........##...........",
    ".........#a..........",
    "..........aa.........",
    "...........aa........",
    "............a#.......",
    "...........aa........",
    "..........aa.........",
    ".........aa..........",
    "........#a...........",
    ".........aa..........",
    "..........aa.........",
    "...........aa........",
    "............a#.......",
    "...........aa........",
    "..........aa.........",
    ".........aa..........",
    "........#a...........",
    ".........aa..........",
    "..........aa.........",
    "...........a#........",
    "............##......."
};

static gchar *strum_icon[]={
    "21 21 2 1",
    ". c None",
    "# c #303030",
    ".....................",
    ".....................",
    "....##....##....##...",
    "....##...####...##...",
    "....##..######..##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "....##....##....##...",
    "..######..##..######.",
    "...####...##...####..",
    "....##....##....##...",
    ".....................",
    "....................."
};

static gchar *rhythm_icon[]={
    "21 21 3 1",
    ". c None",
    "# c #303030",
    "a c #585858",
    ".....................",
    ".......#.........#...",
    "......#a........#a...",
    ".....#a........#a....",
    "....#a........#a.....",
    "....#a........#a.....",
    "....#a........#a.....",
    "....#a........#a.....",
    "....#a........#a.....",
    "....#a........#a.....",
    "....#a........#a.....",
    "....#a........#a.....",
    "....#aaaaaaaaa#a.....",
    "....###########a.....",
    "....#aaaaaaaaa#a.....",
    "....#a........#a.....",
    "....#a........#a.....",
    "....#aaaaaaaaa#a.....",
    "....###########a.....",
    "....aaaaaaaaaaaa.....",
    "....................."
};

static gchar *rest_icon[]={
    "21 21 3 1",
    ". c None",
    "# c #303030",
    "a c #585858",
    ".......##............",
    "........##...........",
    ".........##..........",
    "..........##.........",
    "...........##........",
    "..........####.......",
    ".........#####.......",
    "........#####........",
    ".......#####.........",
    "........###..........",
    ".........##..........",
    "..........##.........",
    "...........##........",
    "........aaa.##.......",
    ".......aa#####.......",
    ".......a##...........",
    ".......a#a...........",
    ".......a##...........",
    "........a##..........",
    ".........a##.........",
    "...........##........"
};

static gchar *repeat_open_icon[]={
    "21 21 2 1",
    ". c None",
    "# c #303030",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.##......",
    "......##..##.##......",
    "......##..##.........",
    "......##..##.........",
    "......##..##.##......",
    "......##..##.##......",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##.........",
    "......##..##........."
};

static gchar *repeat_close_icon[]={
    "21 21 2 1",
    ". c None",
    "# c #303030",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    "......##.##..##......",
    "......##.##..##......",
    ".........##..##......",
    ".........##..##......",
    "......##.##..##......",
    "......##.##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......",
    ".........##..##......"
};

static gchar *timesig_icon[]={
    "21 21 2 1",
    ". c None",
    "# c #303030",
    ".....................",
    ".....................",
    "...........###.......",
    "..........####.......",
    ".........##.##.......",
    "........##..##.......",
    ".......#######.......",
    "............##.......",
    "............##.......",
    "............##.......",
    "............##.......",
    ".....................",
    "...........###.......",
    "..........####.......",
    ".........##.##.......",
    "........##..##.......",
    ".......#######.......",
    "............##.......",
    "............##.......",
    "............##.......",
    "............##......."
};

static gchar *comment_icon[] = {
    "21 21 3 1",
    "     g None",
    ".    g #010101",
    "+    g #FFFFFF",
    "        ....         ",
    "      ..++++..       ",
    "     .++++++++.      ",
    "    .++++++++++.     ",
    "   .+++++..+++++.    ",
    "  .++++++..++++++.   ",
    "  .++++++..++++++.   ",
    " .+++++++..+++++++.  ",
    " .+++++++..+++++++.  ",
    " .+++++++..+++++++.  ",
    " .+++++++..+++++++.  ",
    " .+++++++..+++++++.  ",
    "  .++++++..++++++.   ",
    "  .++++++++++++++.   ",
    "   .+++++..+++++..   ",
    "    .++++..++++...   ",
    "     .++++++++.....  ",
    "      ..++++.......  ",
    "        .....   .... ",
    "                  .. ",
    "                     "
};

static gchar *chord_icon[] = {
    "21 21 4 1",
    "     g None",
    ".    g #010101",
    "+    g #FFFFFF",
    "@    g #000000",
    "                     ",
    "                     ",
    "  ................   ",
    "  .++.++.++.++.++.   ",
    "  .++.++.++.++.++.   ",
    "  ................   ",
    "  .++.++.++.++.++.   ",
    "  .++.++.++.++.++.   ",
    "  ................   ",
    "  .++.++.++.++.++.   ",
    "  .++.++.++.++.++.   ",
    "  ................   ",
    "  .++.++.++.++.++.   ",
    "  .++.++.++.++.++.   ",
    "  ................   ",
    "  .++@++.++.++.++.   ",
    "  .++.++.++.++.++.   ",
    "  ..............@.   ",
    "                     ",
    "                     ",
    "                     "
};

static gchar *props_icon[]={
    "21 21 10 1",
    ". c None",
    "a c #010101",
    "# c #303030",
    "f c #585858",
    "e c #808080",
    "g c #a0a0a0",
    "d c #c3c3c3",
    "h c #caa636",
    "b c #f8e008",
    "c c #ffffff",
    "..########aaaaaaaa..b",
    "..#cccccccccccccca.bb",
    "..#ccccccccccccccabbb",
    "..#ccccccccccccccbbbb",
    "..#c#d#e#d#efgccbbbbb",
    "..#dd#e#e#efgfdbbbbb.",
    "..#cccccccccccbbbbb..",
    "..#ccccccccccbbbbb...",
    "..#ccccccccchbbbba...",
    "..#cccccccchhhbbca...",
    "..#c#e#f#edhhhhcca...",
    "..#dg#f#f#ahhcccca...",
    "..#cccccccccccccc#...",
    "..#cccccccccccccc#...",
    "..#cccccccccccccc#...",
    "..#cccccccccccccc#...",
    "..#cccccccccccccc#...",
    "..#cccccccccccccc#...",
    "..#cccccccccccccc#...",
    "..#cccccccccccccc#...",
    "..################..."
};

static gchar *lib_chord_icon[] = {
    "21 21 6 1",
    "     g None",
    ".    g #010101",
    "+    g #F8E008",
    "@    g #FFFFFF",
    "#    g #CAA636",
    "$    g #000000",
    "                     ",
    "                     ",
    "  ................  +",
    "  .@@.@@.@@.@@.@@. ++",
    "  .@@.@@.@@.@@.@@.+++",
    "  ...............++++",
    "  .@@.@@.@@.@@.@+++++",
    "  .@@.@@.@@.@@.+++++ ",
    "  ............+++++  ",
    "  .@@.@@.@@.@+++++   ",
    "  .@@.@@.@@.#++++.   ",
    "  .........###++..   ",
    "  .@@.@@.@@####@@.   ",
    "  .@@.@@.@.##@.@@.   ",
    "  ................   ",
    "  .@@$@@.@@.@@.@@.   ",
    "  .@@.@@.@@.@@.@@.   ",
    "  ..............$.   ",
    "                     ",
    "                     ",
    "                     "
};

GtkWidget*
create_pixmap                          (GtkWidget       *widget, gchar **data)
{
    GtkWidget *pixmap;
    GdkColormap *colormap;
    GdkPixmap *gdkpixmap;
    GdkBitmap *mask;

    colormap = gtk_widget_get_colormap (widget);
    
    gdkpixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL, (gchar **) data);
    
    pixmap = gtk_image_new_from_pixmap (gdkpixmap, mask);
    gdk_pixmap_unref (gdkpixmap);
    gdk_bitmap_unref (mask);
    return pixmap;
}

static void
on_save_as1_activate                   (GtkMenuItem     *menuitem,
                                        GtabApp         *app)
{
    GtkWidget *save_as;
    gint response;
    gboolean success = FALSE;
    gchar *filename;
    
    
    while (!success) {
        save_as = gtk_file_selection_new(_("Save As"));
        response = gtk_dialog_run(GTK_DIALOG(save_as));
           
        switch (response) {
            case GTK_RESPONSE_OK:
                filename = g_strdup(gtk_file_selection_get_filename
                                            (GTK_FILE_SELECTION
                                            (save_as)));
                success = save_new_tab_file(app->current_doc, filename);
                g_free(filename);
                break;
            default:
                success = TRUE;
                break;
        }
        gtk_widget_destroy(save_as);
    }
       
}


static void
on_exit1_activate                      (GtkMenuItem     *menuitem,
                                        GtabApp         *app)
{
    if (doc_save_check(app->current_doc, TRUE)) {
        take_care_of_business(app);
        
        gtk_main_quit();
    }
}

static void
on_about1_activate                     (GtkMenuItem     *menuitem,
                                        GtabApp         *app)
{
    show_about_box();

}


static void
on_gnometab_destroy_event              (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        GtabApp         *app)
{
    gboolean save_test;
    
    save_test = doc_save_check(app->current_doc, FALSE);
    
    take_care_of_business(app);
            
    gtk_main_quit();
      
}

static void
on_btn_ptr_clicked                     (GtkButton         *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Pointer");
    app->current_tool = TOOL_POINTER;
}

static void
on_btn_num_clicked                     (GtkButton         *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Text Entry");
    app->current_tool = TOOL_NUM;
}


static void
on_btn_bend_clicked                    (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Bend");
    app->current_tool = TOOL_BEND;
}


static void
on_btn_bar_clicked                     (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Bar");
    app->current_tool = TOOL_BAR;
}


static void
on_btn_slur_clicked                    (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Slur");
    app->current_tool = TOOL_SLUR;
}

static void
on_btn_save_clicked                    (GtkButton       *button,
                                        GtabApp         *app)
{
    GtkWidget *save_as;
    gint response;
    gchar *filename;
    gboolean success = FALSE;
    gint i = 0;
    
    if ( ((GtabDoc* )app->current_doc)->filename == NULL )
    {
        while (!success) {
            save_as = gtk_file_selection_new(_("Save As"));
            response = gtk_dialog_run(GTK_DIALOG(save_as));
            
            switch (response) {
                case GTK_RESPONSE_OK:
                    filename = g_strdup(gtk_file_selection_get_filename
                                                (GTK_FILE_SELECTION
                                                (save_as)));
                    success = save_new_tab_file(app->current_doc, filename);
                    g_free(filename);
                    break;
                default:
                    success = TRUE;
                    break;
            }
            gtk_widget_destroy(save_as);
        }
        
        
    }
    else
    {
        i = save_tab_file( ((GtabDoc* )app->current_doc) );
        if (i)    {
            // fprintf(stderr, "Saved file %s\n", current_filename);
            gnome_appbar_set_status(GNOME_APPBAR
                                    (app->appbar), "File Saved.");
        }    
    }
}


static void
on_btn_open_clicked                    (GtkButton       *button,
                                        GtabApp         *app)
{
    GtkWidget *load_file;
    gboolean result = TRUE;
    
    if (((GtabDoc* )app->current_doc)->changed == TRUE) {
        result = doc_save_check((GtabDoc* )app->current_doc, TRUE);
    }
    
    if (result) {
        load_file = create_file_load_select(((GtabDoc* )app->current_doc));
        gtk_widget_show(load_file);
    }
}

static void
on_btn_new_clicked                     (GtkButton       *button,
                                        GtabApp         *app)
{
    gboolean result;
    
    result = doc_save_check(app->current_doc, TRUE);
    
    if (result) {
        refresh_canvas(((GtabDoc* )app->current_doc), TRUE);
        set_title(app->current_doc);
    }
}

static void
on_prefs_activate                        (gpointer         *object,
                                         GtabApp         *app)
{
    GtabDialogPrefs *dialog;
    GConfClient     *client;
    gchar             *chord_lib;
    gchar             *tab_font;
    gchar             *tab_font_large;
    gchar             *tab_font_small;
    gchar             *tab_font_tiny;
    gchar             *chrd_bldr_font;
    gchar             *chrd_bldr_font_small;
    gboolean         result;
    
    client = gconf_client_get_default();
    
    chords_above_staff = gconf_client_get_bool(client, GCONF_CHORDS_ABOVE_STAFF, NULL);
    chord_lib = gconf_client_get_string(client, GCONF_CHORD_LIB, NULL);
    tab_font = gconf_client_get_string(client, GCONF_TAB_FONT, NULL);
    tab_font_large = gconf_client_get_string(client, GCONF_TAB_FONT_LARGE, NULL);
    tab_font_small = gconf_client_get_string(client, GCONF_TAB_FONT_SMALL, NULL);
    tab_font_tiny = gconf_client_get_string(client, GCONF_TAB_FONT_TINY, NULL);
    chrd_bldr_font = gconf_client_get_string(client, GCONF_CHRD_BLDR_FONT, NULL);
    chrd_bldr_font_small = gconf_client_get_string(client, GCONF_CHRD_BLDR_FONT_SMALL, NULL);
    
    dialog = create_prefs_popup(app);
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->chords_above_staff),
                                chords_above_staff);
    
    /* here the callback sets dialog->cas_is_changed; have to change it back */
    
    dialog->cas_is_changed = FALSE;
    
    if (chord_lib != NULL) {
        gnome_file_entry_set_filename(GNOME_FILE_ENTRY(dialog->current_chord_lib), chord_lib);
        g_free(chord_lib);
    }
    if (tab_font != NULL) {
        result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(dialog->tab_font), tab_font);
        g_free(tab_font);
    }
    if (tab_font_large != NULL) {
        result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(dialog->tab_font_large), tab_font_large);
        g_free(tab_font_large);
    }
    if (tab_font_small != NULL) {
        result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(dialog->tab_font_small), tab_font_small);
        g_free(tab_font_small);
    }
    if (tab_font_tiny != NULL) {
        result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(dialog->tab_font_tiny), tab_font_tiny);
        g_free(tab_font_tiny);
    }
    if (chrd_bldr_font != NULL) {
        result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(dialog->chrd_bldr_font), chrd_bldr_font);
        g_free(chrd_bldr_font);
    }
    if (chrd_bldr_font_small != NULL) {
        result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(dialog->chrd_bldr_font_small), chrd_bldr_font_small);
        g_free(chrd_bldr_font_small);
    }
    
    gtk_widget_show(dialog->dialog);
    
}

static void
on_btn_measure_clicked                 (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar),
                                                    "Current Tool: Measure");
    app->current_tool = TOOL_MEASURE;

}


static void
on_btn_legato_clicked                  (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                                    "Current Tool: Legato");
    app->current_tool = TOOL_LEGATO;

}

static void
on_btn_strum_clicked                   (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar),
                                                    "Current Tool: Strum");
    app->current_tool = TOOL_STRUM;
}

static void
on_btn_chord_clicked                   (GtkButton       *button,
                                        GtabApp         *app)
{
    
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar),
                                                        "Current Tool: Chord");
    app->current_tool = TOOL_CHORD;
}


static void
on_btn_rhythm_clicked                  (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar),
                                                        "Current Tool: Rhythm");
    app->current_tool = TOOL_RHYTHM;
    
}


static void
on_btn_print_clicked                   (GtkButton       *button,
                                        GtabApp         *app)
{
    sync_loaded_doc( ((GtabDoc *)app->current_doc) );
    tab_canvas_print( ((GtabDoc *)app->current_doc), FALSE);

}

static void
on_print_preview_clicked               (GtkButton       *button,
                                        GtabApp         *app)
{
    sync_loaded_doc( ((GtabDoc *)app->current_doc) );
    tab_canvas_print( ((GtabDoc *)app->current_doc), TRUE);

}

static void
on_btn_properties_clicked              (GtkButton       *button,
                                        GtabApp         *app)
{
    GtabDialogProps *props_popup;
    gchar *title;
    gchar *artist;
    gchar *tuning;
   
    props_popup = create_props_popup( ((GtabDoc *)app->current_doc) );
    if (((GtabDoc *)app->current_doc)->properties != NULL)    {
        title = get_prop(((GtabDoc *)app->current_doc)->doc_props, "title");
        artist = get_prop(((GtabDoc *)app->current_doc)->doc_props, "artist");
        tuning = get_prop(((GtabDoc *)app->current_doc)->doc_props, "tuning");
        
        if (title != NULL) {
            gtk_entry_set_text(GTK_ENTRY(props_popup->title), title);
            g_free(title);
        }
        if (artist != NULL) {
            gtk_entry_set_text(GTK_ENTRY(props_popup->artist), artist);
            g_free(artist);
        }
        if (tuning != NULL) {
            gtk_entry_set_text(GTK_ENTRY(props_popup->tuning), tuning);
            g_free(tuning);
        }
        
    }
    gtk_widget_show(props_popup->dialog);
}

static void
on_btn_timsig_clicked                  (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                            "Current Tool: Time Signature");
    app->current_tool = TOOL_TIMESIG;

}

static void
on_btn_prev_page_clicked               (GtkButton       *button,
                                        GtabApp         *app)
{
    switch_page(((GtabDoc *)app->current_doc), gnome_appbar_get_progress(GNOME_APPBAR
                                        (app->appbar)), 
                                        FALSE);
}


static void
on_btn_next_page_clicked               (GtkButton       *button,
                                        GtabApp         *app)
{
    switch_page(((GtabDoc *)app->current_doc), gnome_appbar_get_progress(GNOME_APPBAR
                                        (app->appbar)), 
                                        TRUE);
}


static void
on_btn_exit_clicked                    (GtkButton       *button,
                                        GtabApp         *app)
{
    if (doc_save_check(app->current_doc, TRUE)) {
        take_care_of_business(app);
        
        gtk_main_quit();
    }
}

static void
on_btn_comment_clicked                 (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                                    "Current Tool: Comment");
    app->current_tool = TOOL_COMMENT;
}


static void
on_btn_repeat_open_clicked             (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                                    "Current Tool: Repeat Bar");
    app->current_tool = TOOL_REPEAT_OPEN;
}


static void
on_btn_repeat_close_clicked            (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                                    "Current Tool: Repeat Bar");
    app->current_tool = TOOL_REPEAT_CLOSE;
}




static void
on_btn_rest_clicked                    (GtkButton       *button,
                                        GtabApp         *app)
{
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar),
                                        "Current Tool: Rest");
    app->current_tool = TOOL_REST;
}


static void
on_btn_lib_chord_clicked               (GtkButton       *button,
                                        GtabApp         *app)
{
    gboolean     value;
    
    value = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
    
    if (value == FALSE)    {
        
        gtk_widget_destroy(((GtabChordLib *)app->chord_lib)->scrolledwindow);
        ((GtabChordLib *)app->chord_lib)->scrolledwindow = NULL;
        if (((GtabChordLib *)app->chord_lib)->selected_chord != NULL) {
            ((GtabChordLib *)app->chord_lib)->highlights = NULL;
        }
        g_list_free(((GtabChordLib *)app->chord_lib)->obj_counter);
        ((GtabChordLib *)app->chord_lib)->obj_counter = NULL;
        ((GtabChordLib *)app->chord_lib)->visible = FALSE;
    }
    else    {
        if (!((GtabChordLib *)app->chord_lib)->visible) {
            render_chord_lib_in_pane(app);
        }
    
    }
        
}

static void
on_btn_zoomout_clicked            (gpointer         *object,
                                    GtabApp     *app)
{
    if (app->zoom_level > 0.8) {
        app->zoom_level = app->zoom_level - 0.1;
        
        gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(((GtabDoc *)app->current_doc)->tab_canvas), 
                                        app->zoom_level);
    }
}

static void
on_btn_zoomin_clicked            (gpointer         *object,
                                    GtabApp     *app)
{
    if (app->zoom_level < 1.4) {
        
        app->zoom_level = app->zoom_level + 0.1;
        
        gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(((GtabDoc *)app->current_doc)->tab_canvas), 
                                        app->zoom_level);
    }
}

static void
on_menu_ptr_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_ptr), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Pointer");
    app->current_tool = TOOL_POINTER;
}

static void
on_menu_num_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_num), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Text Entry");
    app->current_tool = TOOL_NUM;
}

static void
on_menu_bend_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_bend), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Bend");
    app->current_tool = TOOL_BEND;
}

static void
on_menu_bar_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_bar), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Bar");
    app->current_tool = TOOL_BAR;
}

static void 
on_menu_slur_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_slur), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Slur");
    app->current_tool = TOOL_SLUR;
}

static void 
on_menu_measure_clicked         (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_measure), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Measure");
    app->current_tool = TOOL_MEASURE;
}

static void 
on_menu_legato_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_legato), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Legato");
    app->current_tool = TOOL_LEGATO;
}

static void 
on_menu_strum_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_strum), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Strum");
    app->current_tool = TOOL_STRUM;
}

static void 
on_menu_rhythm_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_rhythm), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Rhythm");
    app->current_tool = TOOL_RHYTHM;
}

static void 
on_menu_rest_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_rest), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Rest");
    app->current_tool = TOOL_REST;
}

static void 
on_menu_timsig_clicked             (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_timsig), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Time Signature");
    app->current_tool = TOOL_TIMESIG;
}

static void 
on_menu_repeat_open_clicked     (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_repeat_open), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Repeat Bar");
    app->current_tool = TOOL_REPEAT_OPEN;
}

static void 
on_menu_repeat_close_clicked     (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_repeat_close), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Repeat Bar");
    app->current_tool = TOOL_REPEAT_CLOSE;
}

static void 
on_menu_comment_clicked     (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_comment), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Comment");
    app->current_tool = TOOL_COMMENT;
}

static void 
on_menu_chord_clicked     (gpointer         *object,
                                    GtabApp     *app)
{
    if (num_entry_bool) return;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(app->btn_chord), TRUE);
    gnome_appbar_set_status(GNOME_APPBAR(app->appbar), 
                                        "Current Tool: Chord");
    app->current_tool = TOOL_CHORD;
}

/*  RECENT FILE LIST STUFF */

static void
on_recent_file_activate (gpointer   *object,
                                    GtabApp *app)
{
    static gchar *filename;
    gboolean file_exists;
    gboolean result = TRUE;
    
    if (((GtabDoc* )app->current_doc)->changed == TRUE) {
        result = doc_save_check((GtabDoc* )app->current_doc, TRUE);
    }
    
    if (!result) return;
        
    filename = g_object_get_data(G_OBJECT(object), "label");
    
    file_exists = g_file_test(filename,
                             G_FILE_TEST_EXISTS && G_FILE_TEST_IS_REGULAR);
    if (file_exists) {
        if (((GtabDoc *)app->current_doc)->filename != NULL) 
                g_free(((GtabDoc *)app->current_doc)->filename);
            
        ((GtabDoc *)app->current_doc)->filename = g_strdup(filename);
        file_pre_load(((GtabDoc *)app->current_doc));
    }
    else {
        ((GtabDoc *)app->current_doc)->filename = NULL;
        gtab_warn(app, "This file is not valid!");
    }
    
    g_free(filename);
    
}

static gchar*
recent_item_label(const gchar *name, const gint accel_number)
{
    gchar *label, *menuname;
    
    menuname = g_path_get_basename(name);
    
    label = g_strdup_printf ("_%d. %s", accel_number, menuname);
    
    g_free (menuname);
    
    return label;
}

static void
recent_menu_separator(GtabApp *app)
{
    static GnomeUIInfo info[] = {
        GNOMEUIINFO_SEPARATOR
    };
    gnome_app_insert_menus(GNOME_APP(app->gnometab), "File/Print...", info);
}

static void 
recent_menu_item_create (GtabApp *app, gchar *name, gint num) 
{
    static GnomeUIInfo info[] = {
        { GNOME_APP_UI_ITEM, NULL, NULL, on_recent_file_activate, NULL },
        GNOMEUIINFO_END
    };
    
    info[0].hint = name;
    info[0].label = recent_item_label(name, num);
    
    gnome_app_insert_menus_with_data(GNOME_APP(app->gnometab), "File/Print...", info, app);
    
    g_object_set_data(G_OBJECT(info[0].widget), "label", name);
    
    gnome_app_install_menu_hints(GNOME_APP(app->gnometab), info);
}

static void
recent_menu_fill (GtabApp *app)
{
    GConfClient *client;
    GSList *l;
    gint i, j;
    gboolean file_exists;
    
    client = gconf_client_get_default();
    
    if (app->history_files) g_slist_free(app->history_files);
        
    app->history_n = gconf_client_get_int(client, GCONF_HISTORY_N, NULL);
    app->history_files = gconf_client_get_list(client, GCONF_HISTORY_FILES, 
                                                GCONF_VALUE_STRING, NULL);
    
    /* test files */
    
    for (l = app->history_files; l; l = l->next) {
        file_exists = g_file_test((gchar *)l->data,
                             G_FILE_TEST_EXISTS && G_FILE_TEST_IS_REGULAR);
        if (!file_exists) g_slist_remove(app->history_files, l->data);
    }
        
        
        
    i = j = g_slist_length(app->history_files);
    
    if (app->history_n < 1) app->history_n = 5;
        
    for (l = app->history_files; l; l = l->next) {
        
        recent_menu_item_create(app, (gchar *)l->data, i);
        i--;
       
    }
    if (j > 0) recent_menu_separator(app);
}

static void
recent_menu_clear (GtabApp *app)
{
    gint i;
    
    i = g_slist_length(app->history_files);
    
    if (i > 0) {
        
        i++;
        
        gnome_app_remove_menu_range(GNOME_APP(app->gnometab), "File/Print...", 
                    1, i);
    }
}

static void 
refresh_recent_menu(GtabApp *app)
{
    GConfClient *client;
    gboolean test;
    GSList *l;
    GSList *newlist = NULL;
    gint i;
    
    client = gconf_client_get_default();
    
    if (app->history_n < 1) app->history_n = 5;
        
    l = app->history_files;
    
    for (i = 0; i < app->history_n; i++) {
        if (l) {
            newlist = g_slist_prepend(newlist, l->data);
            l = l->next;
        }
        
    }
    
    test = gconf_client_set_list(client, GCONF_HISTORY_FILES, GCONF_VALUE_STRING,
                                newlist, NULL);
    test = gconf_client_set_int(client, GCONF_HISTORY_N, app->history_n, NULL);
    
    
    recent_menu_fill(app);
}

void 
add_to_history(GtabApp *app, gchar *filename)
{
    gboolean exists = FALSE;
    GSList *l;
    gpointer data = NULL;
    
    recent_menu_clear(app);
    
    for (l = app->history_files; l; l = l->next) {
        if (g_strcasecmp(l->data, filename) == 0) {
            exists = TRUE;
            data = l->data;
        }
        
    }
    
    if (exists) {
        app->history_files = g_slist_remove(app->history_files, data);
    }
    app->history_files = g_slist_reverse(app->history_files);
    app->history_files = g_slist_prepend(app->history_files, filename);
    
    refresh_recent_menu(app);
}

/* MENUS AND TOOLBARS  */

static GnomeUIInfo file1_menu_uiinfo[] =
{
    GNOMEUIINFO_MENU_NEW_ITEM (N_("_New File"), NULL, on_btn_new_clicked, NULL),
    GNOMEUIINFO_MENU_OPEN_ITEM (on_btn_open_clicked, NULL),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_SAVE_ITEM (on_btn_save_clicked, NULL),
    GNOMEUIINFO_MENU_SAVE_AS_ITEM (on_save_as1_activate, NULL),
    GNOMEUIINFO_SEPARATOR,
    {
        GNOME_APP_UI_ITEM, N_("Print Previe_w"), 
        NULL,
        (gpointer) on_print_preview_clicked, NULL, NULL,
        GNOME_APP_PIXMAP_STOCK, GTK_STOCK_PRINT_PREVIEW,
        GDK_P, GDK_SHIFT_MASK | GDK_CONTROL_MASK, NULL
    },
    GNOMEUIINFO_MENU_PRINT_ITEM (on_btn_print_clicked, NULL),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_EXIT_ITEM (on_exit1_activate, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo edit1_menu_uiinfo[] =
{
    GNOMEUIINFO_MENU_PREFERENCES_ITEM(on_prefs_activate, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo view_menu_uiinfo[] =
{
    {
        GNOME_APP_UI_ITEM, N_("Zoom _In"),
        N_("Zoom in on the canvas"),
        (gpointer) on_btn_zoomin_clicked, NULL, NULL,
        GNOME_APP_PIXMAP_STOCK, GTK_STOCK_ADD,
        GDK_plus, GDK_CONTROL_MASK, NULL
    },
    {
        GNOME_APP_UI_ITEM, N_("Zoom _Out"),
        N_("Zoom out from the canvas"),
        (gpointer) on_btn_zoomout_clicked, NULL, NULL,
        GNOME_APP_PIXMAP_STOCK, GTK_STOCK_REMOVE,
        GDK_minus, GDK_CONTROL_MASK, NULL
    },
    GNOMEUIINFO_END
};

static GnomeUIInfo tools_menu_uiinfo[] =
{   
    {
        GNOME_APP_UI_ITEM, 
        N_("_Pointer"),
        N_("Select Objects"),
        (gpointer )on_menu_ptr_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        pointer_icon,
        GDK_quoteleft,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("_Text Entry"),
        N_("Enter text on the canvas"),
        (gpointer )on_menu_num_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        text_icon,
        GDK_1,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("_Bend"),
        N_("Enter a bend"),
        (gpointer )on_menu_bend_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        bend_icon,
        GDK_2,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("B_ar"),
        N_("Enter a bar"),
        (gpointer )on_menu_bar_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        bar_icon,
        GDK_3,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("_Slur"),
        N_("Enter a slur"),
        (gpointer )on_menu_slur_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        slur_icon,
        GDK_4,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("_Measure"),
        N_("Enter a Measure Bar"),
        (gpointer )on_menu_measure_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        measure_icon,
        GDK_5,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("_Legato"),
        N_("Enter a legato"),
        (gpointer )on_menu_legato_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        legato_icon,
        GDK_6,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("Str_um"),
        N_("Enter a strum pattern"),
        (gpointer )on_menu_strum_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        strum_icon,
        GDK_7,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("R_hythm"),
        N_("Enter rhythm"),
        (gpointer )on_menu_rhythm_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        rhythm_icon,
        GDK_8,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("R_est"),
        N_("Enter a rest"),
        (gpointer )on_menu_rest_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        rest_icon,
        GDK_9,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("Time Si_gnature"),
        N_("Enter Time Signature"),
        (gpointer )on_menu_timsig_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        timesig_icon,
        GDK_0,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("_Repeat Bar Open"),
        N_("Open repeat bars"),
        (gpointer )on_menu_repeat_open_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        repeat_open_icon,
        GDK_bracketleft,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("Repeat Bar _Close"),
        N_("Close repeat bars"),
        (gpointer )on_menu_repeat_close_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        repeat_close_icon,
        GDK_bracketright,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("C_omment"),
        N_("Enter a comment"),
        (gpointer )on_menu_comment_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        comment_icon,
        GDK_semicolon,
        GDK_CONTROL_MASK,
        NULL 
    },
    
    {
        GNOME_APP_UI_ITEM, 
        N_("Chor_d"),
        N_("Create a chord box"),
        (gpointer )on_menu_chord_clicked,
        NULL, NULL, 
        GNOME_APP_PIXMAP_DATA,
        chord_icon,
        GDK_quoteright,
        GDK_CONTROL_MASK,
        NULL 
    },
   
    GNOMEUIINFO_END
};

static GnomeUIInfo help1_menu_uiinfo[] =
{
    GNOMEUIINFO_MENU_ABOUT_ITEM (on_about1_activate, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo menubar1_uiinfo[] =
{
    GNOMEUIINFO_MENU_FILE_TREE (file1_menu_uiinfo),
    GNOMEUIINFO_MENU_EDIT_TREE (edit1_menu_uiinfo),
    GNOMEUIINFO_MENU_VIEW_TREE (view_menu_uiinfo),
    {
        GNOME_APP_UI_SUBTREE_STOCK,
        N_("_Tools"), NULL,
        tools_menu_uiinfo,
        NULL, NULL,
        (GnomeUIPixmapType) 0,
        NULL, 0,
        (GdkModifierType) 0,
        NULL
    },
    GNOMEUIINFO_MENU_HELP_TREE (help1_menu_uiinfo),
    GNOMEUIINFO_END
};

static GnomeUIInfo toolbar1_data[] = {
    { GNOME_APP_UI_ITEM, 
      N_("New"), N_("New File"), 
      on_btn_new_clicked, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_NEW,
      0, 0, NULL },

    { GNOME_APP_UI_ITEM, 
      N_("Open"), N_("Open File"), 
      on_btn_open_clicked, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_OPEN,
      0, 0, NULL },
    
    { GNOME_APP_UI_ITEM, 
      N_("Save"), N_("Save File"), 
      on_btn_save_clicked, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_SAVE, 
      0, 0, NULL },

    { GNOME_APP_UI_ITEM, 
      N_("Preferences"), N_("Set Gnometab Preferences"), 
      on_prefs_activate, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_PREFERENCES, 
      0, 0, NULL },
        
    GNOMEUIINFO_SEPARATOR,
    
    { GNOME_APP_UI_ITEM, 
      N_("Print Preview"), N_("Preview Print Output"), 
      on_print_preview_clicked, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_PRINT_PREVIEW, 
      0, 0, NULL },
      
    { GNOME_APP_UI_ITEM, 
      N_("Print"), N_("Print Tablature"), 
      on_btn_print_clicked, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_PRINT, 
      0, 0, NULL },
      
    GNOMEUIINFO_SEPARATOR,
      
    {    GNOME_APP_UI_ITEM,
        N_("Zoom Out"), N_("Zoom Out"),
        on_btn_zoomout_clicked, NULL, NULL,
        GNOME_APP_PIXMAP_STOCK, GTK_STOCK_REMOVE,
        0, 0, NULL },
      
    {    GNOME_APP_UI_ITEM,
        N_("Zoom In"), N_("Zoom In"),
        on_btn_zoomin_clicked, NULL, NULL,
        GNOME_APP_PIXMAP_STOCK, GTK_STOCK_ADD,
        0, 0, NULL },
    
    GNOMEUIINFO_SEPARATOR,
        
    { GNOME_APP_UI_ITEM, 
      N_("Previous"), N_("Previous Page"), 
      on_btn_prev_page_clicked, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_GO_BACK, 
      0, 0, NULL },
    
    { GNOME_APP_UI_ITEM, 
      N_("Next"), N_("Next Page"), 
      on_btn_next_page_clicked, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_GO_FORWARD, 
      0, 0, NULL },
    
    GNOMEUIINFO_SEPARATOR,
    
    { GNOME_APP_UI_ITEM, 
      N_("Quit"), N_("Quit"), 
      on_btn_exit_clicked, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GTK_STOCK_QUIT, 
      0, 0, NULL },
      
    GNOMEUIINFO_END
};

GtabApp*
create_gnometab (void)
{
    GtabApp     *app;
    GtkWidget *toolbar1;
    GtkWidget *tmp_toolbar_icon;
    GtkWidget *button;
    BonoboDockItemBehavior behavior;
    GConfClient *client;
    
    client = gconf_client_get_default();
    
    behavior = BONOBO_DOCK_ITEM_BEH_EXCLUSIVE;
    
    if (! gconf_client_get_bool(client, "/desktop/gnome/interface/toolbar_detachable", NULL)) {
        behavior |= BONOBO_DOCK_ITEM_BEH_LOCKED;
    }
    
    app = g_new0(GtabApp, 1);
    
    app->gnometab = gnome_app_new ("Gnometab", _("Gnometab"));
        
    gnome_app_create_menus_with_data (GNOME_APP (app->gnometab), menubar1_uiinfo, app);

    /* store a pointer to the tools menu so it can be deactivated easily */
    app->toolmenu = menubar1_uiinfo[3].widget;
    
    toolbar1 = gtk_toolbar_new ();
    gnome_app_fill_toolbar_with_data(GTK_TOOLBAR(toolbar1), toolbar1_data, NULL, app);
    gtk_widget_show (toolbar1);
        
    gnome_app_add_toolbar (GNOME_APP (app->gnometab), GTK_TOOLBAR (toolbar1), "toolbar1",
                                behavior,
                                BONOBO_DOCK_TOP, 1, 0, 0);
 
    app->toolbox = gtk_toolbar_new ();
    gtk_toolbar_set_style(GTK_TOOLBAR(app->toolbox), GTK_TOOLBAR_ICONS);
    
    gtk_widget_show (app->toolbox);
        
    gnome_app_add_toolbar (GNOME_APP (app->gnometab), GTK_TOOLBAR(app->toolbox), "toolbar2",
                                behavior,
                                BONOBO_DOCK_TOP, 2, 0, 0);
    tmp_toolbar_icon = create_pixmap (app->gnometab, pointer_icon);
    app->btn_ptr = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                NULL,
                                _("Pointer"),
                                _("Select objects"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show(app->btn_ptr);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (app->btn_ptr), TRUE);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_ptr), FALSE);
                                
    tmp_toolbar_icon = create_pixmap (app->gnometab, text_icon);
    app->btn_num = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Text"),
                                _("Enter numbers or text"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_num);
    
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_num), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, bend_icon);
    app->btn_bend = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Bend"),
                                _("Create a bend"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_bend);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_bend), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, bar_icon);
    app->btn_bar = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Bar"),
                                _("Create a bar"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_bar);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_bar), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, slur_icon);
    app->btn_slur = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Slur"),
                                _("Create a slur"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_slur);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_slur), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, measure_icon);
    app->btn_measure = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Measure"),
                                _("Divide measures"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_measure);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_measure), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, legato_icon);
    app->btn_legato = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Lgto\t"),
                                _("Create a legato"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_legato);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_legato), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, strum_icon);
    app->btn_strum = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Strum"),
                                _("Create a strum stroke"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_strum);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_strum), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, rhythm_icon);
    app->btn_rhythm = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Rhythm"),
                                _("Create rhythm notation"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_rhythm);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_rhythm), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, rest_icon);
    app->btn_rest = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Rest"),
                                _("Create a rest"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_rest);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_rest), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, timesig_icon);
    app->btn_timsig = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Time"),
                                _("Create the time signature"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_timsig);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_timsig), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, repeat_open_icon);
    app->btn_repeat_open = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("RepOpen"),
                                _("Open repeat bars"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_repeat_open);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_repeat_open), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, repeat_close_icon);
    app->btn_repeat_close = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("RepClose"),
                                _("Close repeat bars"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_repeat_close);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_repeat_close), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, comment_icon);
    app->btn_comment = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Comment"),
                                _("Make a comment"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_comment);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_comment), FALSE);

    tmp_toolbar_icon = create_pixmap (app->gnometab, chord_icon);
    app->btn_chord = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                app->btn_ptr,
                                _("Chord"),
                                _("Build a chord"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_chord);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_chord), FALSE);

    gtk_toolbar_append_space (GTK_TOOLBAR (app->toolbox));
  
    tmp_toolbar_icon = create_pixmap(app->gnometab, props_icon);
    
    button = gtk_toolbar_append_element(GTK_TOOLBAR(app->toolbox),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Properties"),
                                _("Set Document Properties"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
            
    gtk_widget_show(button);
    g_signal_connect(G_OBJECT(button), "clicked",
                    G_CALLBACK(on_btn_properties_clicked),
                    app);
                    
    gtk_toolbar_append_space (GTK_TOOLBAR (app->toolbox));
    
    tmp_toolbar_icon = create_pixmap (app->gnometab, lib_chord_icon);
    app->btn_lib_chord = gtk_toolbar_append_element (GTK_TOOLBAR (app->toolbox),
                                GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
                                NULL,
                                _("LibChord"),
                                _("Select a chord from your library"), NULL,
                                tmp_toolbar_icon, NULL, NULL);
    gtk_widget_show (app->btn_lib_chord);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (app->btn_lib_chord), FALSE);
                    
    app->paned_window = gtk_hpaned_new();
    gtk_widget_show(app->paned_window);
    gnome_app_set_contents (GNOME_APP (app->gnometab), app->paned_window);

    app->appbar = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_NEVER);
    gtk_widget_show (app->appbar);
    gnome_app_set_statusbar (GNOME_APP (app->gnometab), app->appbar);

    gnome_app_install_menu_hints (GNOME_APP (app->gnometab), menubar1_uiinfo);
    recent_menu_fill(app);
    
    g_signal_connect (GTK_OBJECT (app->gnometab), "delete_event",
                      GTK_SIGNAL_FUNC (on_gnometab_destroy_event),
                      app);
      
    g_signal_connect (GTK_OBJECT (app->btn_ptr), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_ptr_clicked),
                      app);
  
    g_signal_connect (GTK_OBJECT (app->btn_num), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_num_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_bend), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_bend_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_bar), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_bar_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_slur), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_slur_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_measure), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_measure_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_legato), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_legato_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_strum), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_strum_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_rhythm), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_rhythm_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_rest), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_rest_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_timsig), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_timsig_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_repeat_open), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_repeat_open_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_repeat_close), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_repeat_close_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_comment), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_comment_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_chord), "clicked",
                      GTK_SIGNAL_FUNC (on_btn_chord_clicked),
                      app);
    g_signal_connect (GTK_OBJECT (app->btn_lib_chord), "toggled",
                      GTK_SIGNAL_FUNC (on_btn_lib_chord_clicked),
                      app);

    app->current_doc = NULL;
    
    return app;
}

