/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: edit.c,v 1.8 2002/01/13 02:51:16 dengen Exp $
 *
 */

#include <errno.h>
#include <fcntl.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "display_cache.h"
#include "gnoise.h"
#include "edit.h"
#include "gtkwaveset.h"


extern void      ladspa_plug_run(float *io_buf, chnl_indx channels, smpl_indx samples);
extern chnl_indx ladspa_plug_channels(void);

/* prototypes */
static void	*edit_undo(void *thread_arg);

static void	*edit_dispatcher(void *thread_arg);
static void	 edit_display_update(undo_t *undo);
static gboolean  edit_shiftleft(undo_t *undo);
static gboolean  edit_shiftright(undo_t *undo);
static gboolean  edit_undo_save_data(undo_t *undo);
static gboolean	 edit_selection_save(undo_t *undo);
static gboolean  edit_file_resize(GtkWaveSet *ws, gint32 new_end);

static gboolean  edit_cut(undo_t *undo);
static gboolean  edit_copy(undo_t *undo);
static gboolean  edit_paste(undo_t *undo);
static gboolean  edit_fade_linear_in(undo_t *undo);
static gboolean  edit_fade_linear_out(undo_t *undo);
static gboolean  edit_interp_linear(undo_t *undo);
static gboolean  edit_reverse(undo_t *undo);
static gboolean  edit_normalize(undo_t *undo);
static gboolean  edit_mute(undo_t *undo);
static gboolean  edit_insert_silence(undo_t *undo);
#ifdef ENABLE_LADSPA
static gboolean  edit_plugin(undo_t *undo);
#endif

static undo_t		*clipboard;
static pthread_mutex_t	edit_mutex = PTHREAD_MUTEX_INITIALIZER;
static const edit_t	edit_table[] =
{
    /* cmd                 name              disp   undoable save   func
     *                                       updt            data
     */
    {EDIT_UNDO,            "Undo",           TRUE,  FALSE,   FALSE, (gboolean(*)(undo_t *))edit_undo},

    {EDIT_SELECTION_SAVE,  "Save selection", FALSE, FALSE,   FALSE, edit_selection_save},
    {EDIT_MUTE,            "Mute",           TRUE,  TRUE,    TRUE,  edit_mute},
    {EDIT_FADE_LINEAR_IN,  "Fade in",        TRUE,  TRUE,    TRUE,  edit_fade_linear_in},
    {EDIT_FADE_LINEAR_OUT, "Fade out",       TRUE,  TRUE,    TRUE,  edit_fade_linear_out},
    {EDIT_INTERP_LINEAR,   "Linear Interp",  TRUE,  TRUE,    TRUE,  edit_interp_linear},
    {EDIT_REVERSE,         "Reverse",        TRUE,  TRUE,    TRUE,  edit_reverse},
    {EDIT_NORMALIZE,       "Normalize",      TRUE,  TRUE,    TRUE,  edit_normalize},
    {EDIT_CUT,             "Cut",            TRUE,  TRUE,    TRUE,  edit_cut},
    {EDIT_COPY,            "Copy",           FALSE, FALSE,   TRUE,  edit_copy},
    {EDIT_PASTE,           "Paste",          TRUE,  TRUE,    FALSE, edit_paste},
    {EDIT_INSERT_SILENCE,  "Insert silence", TRUE,  TRUE,    FALSE, edit_insert_silence},
#   ifdef ENABLE_LADSPA
    {EDIT_PLUGIN,          "Plugin",         TRUE,  TRUE,    TRUE,  edit_plugin},
#   endif
};



/*
 * FUNC: main entry point to this module, creates an undo, create it's file,
 *	 and then thread off the cmd dispatcher
 *   IN: cmd is the edit cmd to do
 *	 cmd_arg is usually a wave set, but may be cmd specific
 *	 thread is if the cmd should be run as a seperate thread
 *	 new_thread is the thread created
 *  OUT: TRUE if it was a valid cmd and we could start the thread,
 *	 FALSE otherwise
 * ASMP: only one edit command (thread) can be active at one time,
 *	 this is enforced with the edit_mutex
 */
gboolean edit_cmd(GtkWaveSet *ws, edit_enum_t cmd, void *cmd_arg,
		  gboolean thread, pthread_t *new_thread)
{
    undo_t	*undo = NULL;

    if (cmd >= EDIT_LAST)
	goto err1;

    /* serialize edits */
    if (pthread_mutex_trylock(&edit_mutex) < 0)
    {
	log("EDIT", "Edit already in progress!\n");
	goto err1;
    }

    /* EDIT_UNDO only needs a waveset since it don't create an undo's but
     * frees one
     */
    if (edit_table[cmd].cmd == EDIT_UNDO)
    {
	pthread_create(new_thread, NULL, (void *(*)(void *))edit_table[cmd].func, (void *)ws);
	return TRUE;
    }

    undo = calloc(1, sizeof(undo_t));
    if (undo == NULL)
    {
	log("EDIT CMD", "Unable to allocate undo\n");
	goto err1;
    }

    /* set up the undo */
    undo->ws = ws;
    undo->cmd_arg = cmd_arg;
    if (!prefs.disable_undo && edit_table[cmd].undoable)
	undo->next = ws->sb->undo;
    undo->cmd = cmd;
    undo->size_change = 0;
    undo->ref = 1;
    if (prefs.selection_bond)
    {
	chnl_indx channel;

	undo->selection_set.channel_start	= CHANNEL_ALL;
	undo->dcache.start			= ws->wd[0]->selection_start;
	undo->dcache.end			= ws->wd[0]->selection_end;
	undo->display			= undo->dcache;
	undo->selection_set.min_smp		= ws->wd[0]->selection_start;
	undo->selection_set.max_smp		= ws->wd[0]->selection_end;
	undo->selection_set.min_sel		= undo->dcache.end - undo->dcache.start;
	undo->selection_set.max_sel		= undo->dcache.end - undo->dcache.start;
	for(channel = 0; channel < ws->sb->info.channels; channel++)
	{
	    undo->selection_set.selection[channel].start = ws->wd[channel]->selection_start;
	    undo->selection_set.selection[channel].end = ws->wd[channel]->selection_end;
	}
    }
    else
    {
	chnl_indx channel;
	undo->selection_set.channel_start	= 127;
	undo->selection_set.channel_end	= 0;
	undo->selection_set.min_smp		= ws->sb->info.samples;
	undo->selection_set.max_smp		= -1;
	undo->selection_set.min_sel		= ws->sb->info.samples;
	undo->selection_set.max_sel		= 0;

	for(channel = 0; channel < ws->sb->info.channels; channel++)
	{
	    undo->selection_set.selection[channel].start = -1;
	    if (ws->wd[channel]->selection_start > 0)
	    {
		undo->selection_set.channel_start = MIN(undo->selection_set.channel_start, channel);
		undo->selection_set.channel_end   = MAX(undo->selection_set.channel_end, channel);
		undo->selection_set.min_smp	  = MIN(undo->selection_set.min_smp, ws->wd[channel]->selection_start);
		undo->selection_set.max_smp	  = MAX(undo->selection_set.max_smp, ws->wd[channel]->selection_end);

		undo->selection_set.selection[channel].start = ws->wd[channel]->selection_start;
		undo->selection_set.selection[channel].end   = ws->wd[channel]->selection_end;

		/* check if this is the smallest selection */
		if (undo->selection_set.min_sel != 0)
		{
		    undo->selection_set.min_sel =
			MIN(undo->selection_set.min_sel,
			    (undo->selection_set.selection[channel].end -
			     undo->selection_set.selection[channel].start));
		}

		/* check if this is the largest selection */
		undo->selection_set.max_sel =
		    MAX(undo->selection_set.max_sel,
			(undo->selection_set.selection[channel].end -
			 undo->selection_set.selection[channel].start));
	    } else {
		undo->selection_set.min_sel	  = 0;
	    }
	}
	undo->dcache.start = undo->selection_set.min_smp;
	undo->dcache.end   = undo->selection_set.max_smp;
	undo->display      = undo->dcache;
    }

    if (!prefs.disable_undo && edit_table[cmd].cmd != EDIT_SELECTION_SAVE)
    {
	/* generate undofile */
	sprintf(undo->filename, "undo%04d-XXXXXX", ws->sb->undo_cnt);
	undo->fd = mkstemp(undo->filename);
	if (undo->fd < 0)
	{
	    log("EDIT CMD", "Unable to make temp file %s %s\n", undo->filename,
		strerror(errno));
	    goto err2;
	}

	if (write(undo->fd, undo, sizeof(undo_t)) < 0)
	{
	    log("UNDO CREATE", "Unable to write undo header %s\n", strerror(errno));
	    goto err3;
	}
    }


    if (thread)
    {
	/* when the edit function is done, it will release the edit_mutex */
	pthread_create(new_thread, NULL, edit_dispatcher, (void *)undo);
	return(TRUE);
    }
    else
	return (gboolean)edit_dispatcher(undo);

err3:
    close(undo->fd);
    unlink(undo->filename);
err2:
    free(undo);
err1:
    return(FALSE);
}



/*
 * FUNC: new thread that does the work of creating the undo file
 *	 and calling through the jump table to do the work of
 *	 the edit.
 *   IN: the undo allocated to this operation
 *  OUT: to the caller or reaper of this thread a boolean indication of success
 */
static void * edit_dispatcher(void *thread_arg)
{
    undo_t *undo = (undo_t *)thread_arg;
    const edit_t *e_op = &edit_table[undo->cmd];

    if (!prefs.disable_undo)
    {
	if (e_op->save_data)
	{
	    if (!edit_undo_save_data(undo))
	    {
		log("EDIT", "unable to save undo for cmd %s\n", e_op->name);
		goto err2;
	    }

	    close(undo->fd);
	}

	/* if it's undoable, link into chain */
	if (e_op->undoable)
	{
	    undo->ws->sb->undo = undo;
	    undo->ws->sb->undo_cnt++;
	}
    }

    if (e_op->func(undo))
    {
	if (e_op->display_update)
	    edit_display_update(undo);
    }
    else
    {
	if (!prefs.disable_undo && e_op->undoable)
	{
	    log("EDIT", "edit cmd %s failed, undoing...\n", e_op->name);
	    edit_undo(undo->ws);
	}
	goto err1;
    }

    if (edit_thread != 0)
	queue_cmd(CMD_REAP_EDIT_THREAD, NULL);

    pthread_mutex_unlock(&edit_mutex);
    return((void *)TRUE);

err2:
    close(undo->fd);
    unlink(undo->filename);
    free(undo);
err1:
    pthread_mutex_unlock(&edit_mutex);
    return((void *)FALSE);
}



/*
 * FUNC: regenerate display cache and redraw region that was modified
 *   IN: undo describing region
 * ASMP: the edit_mutex is locked
 * NOTE: runs as part of non-gui thread
 */
static void edit_display_update(undo_t *undo)
{
    static selection_t dcache_sel;
    static selection_t display_sel;

    dcache_sel  = undo->dcache;
    display_sel = undo->display;
    log("EDIT DISPUPDT", "dcache:%d-%d display:%d-%d\n",
	dcache_sel.start, dcache_sel.end,
	display_sel.start, display_sel.end);

    dcache_update(undo->ws->sb,
		  dcache_sel.start * undo->ws->sb->info.channels,
		  dcache_sel.end   * undo->ws->sb->info.channels, FALSE);
    queue_cmd(CMD_DISPLAY_UPDATE, &display_sel);
}



/*
 * FUNC: shift region right of undo's area left to where undo starts.
 *   IN: undo describing region to be shifted over
 *  OUT: the new ending sample or -1 on failure
 */
static gint32 edit_shiftleft(undo_t *undo)
{
    GtkWaveSet		*ws = undo->ws;
    selection_set_t	*sel_set = &undo->selection_set;
    gint8		channel;
    gint32		old_end	= ws->sb->info.samples;
    gint32		new_end = old_end + undo->size_change;

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gint32 smp;
	gint32 dst_smp	= sel_set->selection[channel].start;
	gint32 src_smp	= sel_set->selection[channel].end;
	gint32 src_indx;
	gint32 dst_indx;

	if (dst_smp < 0)
	{
	    // new_end = old_end;
	    continue;
	}

	log("SHIFT LEFT", "%d-%d to %d-%d\n", src_smp, old_end, dst_smp,
	    dst_smp+(old_end-src_smp));

	// move region after chunk to be cut to where chunk started
	for(smp = src_smp; smp <= old_end; smp++, dst_smp++)
	{
	    src_indx =     smp * ws->sb->info.channels;
	    dst_indx = dst_smp * ws->sb->info.channels;

	    switch(ws->sb->info.sample_bits)
	    {
		case 16:
		    ((gint16 *)ws->sb->data)[dst_indx + channel] =
		    ((gint16 *)ws->sb->data)[src_indx + channel];
		    break;
	    }
	}

	// fill ending part with silence
	for( ; dst_smp <= old_end; dst_smp++)
	{
	    dst_indx = dst_smp * ws->sb->info.channels;

	    switch(ws->sb->info.sample_bits)
	    {
		case 16:
		    ((gint16 *)ws->sb->data)[dst_indx + channel] = 0;
		    break;
	    }
	}

	//new_end = MAX(new_end, (old_end - shft_len));
    }

    edit_file_resize(ws, new_end);

    // make display cache be regenerated to eof
    // FIXME!! optimize: should only regen necessary part
    undo->dcache.end  = new_end;
    undo->display.end = old_end;
    undo->size_change = new_end - old_end;
    log("SHIFT LEFT", "size change:%d\n", undo->size_change);

    return(new_end);
}



/*
 * FUNC: shift region right of undo's area right by the size of undo
 *   IN: undo describing size to be shifted over
 *  OUT: boolean success value
 */
static gboolean edit_shiftright(undo_t *undo)
{
    GtkWaveSet		*ws	= undo->ws;
    selection_set_t	*sel_set= &undo->selection_set;
    gint8		channel;
    gint32		old_end;
    gint32		new_end;


    log("SHIFT RIGHT", "size_change:%d\n", undo->size_change);

    // determine if we should change the file size
    old_end = ws->sb->info.samples;
    if (undo->size_change != 0)
    {
	/* grow file */
	new_end = old_end + undo->size_change;
	edit_file_resize(ws, new_end);
    } else {
	new_end = old_end;
    }

    undo->dcache.end  = new_end;
    undo->display.end = new_end;

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gint32 smp;
	gint32 src_smp;
	gint32 dst_smp	= new_end;
	gint32 src_start;
	gint32 src_indx;
	gint32 dst_indx;

	src_start = sel_set->selection[channel].start;
	if (src_start < 0)
	    continue;

	src_smp = new_end - (sel_set->selection[channel].end - sel_set->selection[channel].start);

	log("SHIFT RIGHT", "%d-%d to %d-%d\n", src_start, src_smp,
	    dst_smp-(src_smp-src_start), dst_smp);

	// shift region right
	for(smp = src_smp; smp >= src_start; smp--, dst_smp--)
	{
	    src_indx =     smp * ws->sb->info.channels;
	    dst_indx = dst_smp * ws->sb->info.channels;

	    switch(ws->sb->info.sample_bits)
	    {
		case 16:
		    ((gint16 *)ws->sb->data)[dst_indx + channel] = 
		    ((gint16 *)ws->sb->data)[src_indx + channel];
		    break;
	    }
	}
    }

    return(TRUE);
}



/*
 * FUNC: resize the file underlying the mmap and potentially the display cache
 *	 too...
 */
static gboolean edit_file_resize(GtkWaveSet *ws, smpl_indx new_end)
{
    snd_file_resize(ws->sb, new_end);

    if (!dcache_resize(ws->sb))
	return(FALSE);
    return(TRUE);
}


/*
 * FUNC: actually implment actions undo describes modifying the data of ws
 */
static gboolean edit_do_undo(undo_t *undo, GtkWaveSet *ws, gboolean has_data)
{
    selection_set_t	*sel_set;

    undo->fd = open(undo->filename, O_RDONLY);
    if (undo->fd < 0)
    {
	log("UNDO", "Unable to open %s %s\n", undo->filename,
	    strerror(errno));
	goto err1;
    }

    // skip past region info in undo file
    if (lseek(undo->fd, sizeof(undo_t), SEEK_CUR) < 0)
    {
	log("UNDO", "Unable to fseek %s\n", strerror(errno));
	goto err2;
    }

    sel_set = &undo->selection_set;

    // see what kind of edit it was
    if (undo->size_change > 0)
    {
	undo->size_change *= -1;
	edit_shiftleft(undo);
	goto ok;
    }

    if (undo->size_change < 0)
    {
	undo->size_change *= -1;
	edit_shiftright(undo);
    }
    
    if (!has_data)
	goto ok;

    // read actual data within regions

    // if we're doing all channels we can optimize
    if (sel_set->channel_start == CHANNEL_ALL)
    {
	gint32 data_start_indx;
	gint32 data_end_indx;
	char   *data_start = NULL;
	char   *data_end = NULL;

	data_start_indx = sel_set->selection[0].start * ws->sb->info.channels;
	data_end_indx   = sel_set->selection[0].end   * ws->sb->info.channels;
	
	switch(ws->sb->info.sample_bits)
	{
	    case 8:
		data_start = &((gint8 *)ws->sb->data)[data_start_indx];
		data_end   = &((gint8 *)ws->sb->data)[data_end_indx + ws->sb->info.channels];
		break;

	    case 16:
		data_start = (char *)&((gint16 *)ws->sb->data)[data_start_indx];
		data_end   = (char *)&((gint16 *)ws->sb->data)[data_end_indx + ws->sb->info.channels];		break;

	}
	
	if (read(undo->fd, data_start, data_end-data_start) < 0)
	{
	    log("UNDO CREATE", "Unable to read undo data %s\n", strerror(errno));
	    goto err2;
	}
    }
    else
    {
	gint8	channel;
	gint32	sample;

	// restore each channel seperately
	for(channel = sel_set->channel_start; channel <= sel_set->channel_end; channel++)
	{
	    gint32	start = sel_set->selection[channel].start;
	    gint32	end   = sel_set->selection[channel].end;
	    char	*buf;
	    gint8	*cur8;
	    gint16	*cur16;
	    size_t	bufsize;

	    // verify this channel had a valid selection
	    if (start < 0)
		continue;


	    bufsize = ((end - start + 1) * (ws->sb->info.sample_bits/8));
	    buf = malloc(bufsize);
	    if (buf == NULL)
	    {
		goto err2;
	    }

	    if (read(undo->fd, buf, bufsize) < 0)
	    {
		log("UNDO", "read failed %s\n", strerror(errno));
		goto err2;
	    }

	    cur16 = (gint16 *)buf;
	    cur8  = (gint8  *)buf;
	    for(sample =  start * ws->sb->info.channels;
		sample <= end   * ws->sb->info.channels;
		sample+=ws->sb->info.channels)
	    {

		switch(ws->sb->info.sample_bits)
		{
		    case 8:
			((gint8 *)ws->sb->data)[sample + channel] = *cur8++;
			break;
		    case 16:
			((gint16 *)ws->sb->data)[sample + channel] = *cur16++;
			break;
		}
	    }
	    free(buf);
	}
    }

ok:
    edit_display_update(undo);
    close(undo->fd);
    return(TRUE);

err2:
    close(undo->fd);
err1:
    return(FALSE);
}



/*
 * FUNC: undo a command
 *   IN: undo to be undone
 *  OUT: boolean indicator of success to thread reaper
 */
static void *edit_undo(void *thread_arg)
{
    GtkWaveSet		*ws = GTK_WAVE_SET(thread_arg);
    undo_t		*undo;

    undo = ws->sb->undo;
    if (undo == NULL)
    {
	log("UNDO", "No undo available\n");
	goto err2;
    }

    edit_do_undo(undo, ws, edit_table[undo->cmd].save_data);

    /* unlink undo from front of chain */
    ws->sb->undo = undo->next;
    ws->sb->undo_cnt--;

    if (unlink(undo->filename) < 0)
	log("UNDO", "undo ok, but unable to unlink undofile %s\n", undo->filename);

    log("UNDO", "%s %-10.10s restored\n", undo->filename, edit_table[undo->cmd].name);

    undo->ref--;
    if (undo->ref == 0)
	free(undo);

    status("Undone");
    pthread_mutex_unlock(&edit_mutex);
    queue_cmd(CMD_REAP_EDIT_THREAD, NULL);
    return((void *)TRUE);

err2:
    pthread_mutex_unlock(&edit_mutex);
    status("Undo failed...");
    return((void *)FALSE);
}



/*
 * FUNC: save selected region
 *   IN: undo describing region to save
 *  OUT: TRUE on success, FALSE on failure
 */
static gboolean	edit_selection_save(undo_t *undo)
{
    int			fd;
    char		*filename = undo->cmd_arg;
    selection_set_t	*sel_set = &undo->selection_set;
    smpl_indx		smp;
    snd_info_t		si;
    smpl_indx		smp_per_blk;
    guint32		smp_blks;
    guint32		smp_remain_bytes;
    guint32		bytes_per_smp;

    if(undo->selection_set.channel_start != CHANNEL_ALL)
    {
	status("Cannot save discontigous selection");
	return FALSE;
    }

    status("Saving selection");

    si = undo->ws->sb->info;
    si.samples = sel_set->max_smp - sel_set->min_smp + 1;
    log("EDIT", "save as samples:%d\n", si.samples);

    /* open file to write to */
    fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0644);
    if (fd < 0)
    {
	log("EDIT", "Unable to open file %s %s\n", filename, strerror(errno));
	return FALSE;
    }

    /* seek past where header will go */
    lseek(fd, sizeof(wave_header_t) + sizeof(data_chunk_t), SEEK_SET);

    bytes_per_smp = (si.sample_bits/8) * si.channels;
    smp_per_blk = 4096 / bytes_per_smp;
    smp_blks = si.samples / smp_per_blk;
    smp_remain_bytes = si.samples * bytes_per_smp % smp_per_blk;
    if (smp_remain_bytes)
	smp_blks++;

    for(smp = sel_set->min_smp; smp_blks; smp+=smp_per_blk, smp_blks--)
    {
	void *buf = NULL;
	smpl_indx data_indx;
	int rc;

	data_indx = smp * si.channels;
	switch(si.sample_bits)
	{
	    case 8:
		buf = &((guint8 *)undo->ws->sb->data)[data_indx];
		break;

	    case 16:
		buf = &((gint16 *)undo->ws->sb->data)[data_indx];
		break;
	}

	if (smp_blks == 0)
	    rc = write(fd, buf, smp_remain_bytes);
	else
	    rc = write(fd, buf, 4096);

	if (rc < 0)
	{
	    log("EDIT", "write of smp:%d blk:%d failed %s\n", smp, smp_blks,
		strerror(errno));
	    close(fd);
	    return FALSE;
	}
    }

    snd_file_header_save(fd, si);
    close(fd);
    status("Saved");
    return TRUE;
}



/*
 * FUNC: save region described by undo to the already opened
 *	 undo->fd.
 *   IN: undo describing region to save
 *  OUT: TRUE on success, FALSE on failure
 */
static gboolean edit_undo_save_data(undo_t *undo)
{
    selection_set_t	*sel_set = &undo->selection_set;
    GtkWaveSet		*ws = undo->ws;

    status("Saving undo...");

    // if we're doing all channels we can optimize
    if (sel_set->channel_start == CHANNEL_ALL)
    {
	gint32 data_start_indx;
	gint32 data_end_indx;
	char   *data_start = NULL;
	char   *data_end = NULL;

	data_start_indx = sel_set->selection[0].start * ws->sb->info.channels;
	data_end_indx   = sel_set->selection[0].end   * ws->sb->info.channels;

	switch(ws->sb->info.sample_bits)
	{
	    case 8:
		data_start = &(((gint8 *)ws->sb->data)[data_start_indx]);
		data_end   = &(((gint8 *)ws->sb->data)[data_end_indx + ws->sb->info.channels]);
		break;

	    case 16:
		data_start = (char *)&(((gint16 *)ws->sb->data)[data_start_indx]);
		data_end   = (char *)&(((gint16 *)ws->sb->data)[data_end_indx + ws->sb->info.channels]);
		break;
	}

	log("UNDO SAVE DATA", "%s %-10.10s %d bytes chan:all\n", undo->filename,
	    edit_table[undo->cmd].name, data_end-data_start);

	if (write(undo->fd, data_start, data_end-data_start) < 0)
	{
	    goto err1;
	}
    }
    else
    {
	gint8  channel;
	gint32 sample;

	// save each channel seperately
	for(channel = sel_set->channel_start; channel <= sel_set->channel_end; channel++)
	{
	    gint32	start = sel_set->selection[channel].start;
	    gint32	end   = sel_set->selection[channel].end;
	    gint8	*cur8;
	    gint16	*cur16;
	    size_t	bufsize;
	    char	*buf;

	    if (start < 0)
		continue;

	    bufsize = ((end - start + 1) * (ws->sb->info.sample_bits/8));
	    buf = malloc(bufsize);
	    if (buf == NULL)
	    {
		goto err1;
	    }


	    cur16 = (gint16 *)buf;
	    cur8 =  (gint8  *)buf;
	    for(sample =  start * ws->sb->info.channels;
		sample <= end   * ws->sb->info.channels;
		sample+=ws->sb->info.channels)
	    {
		switch(ws->sb->info.sample_bits)
		{
		    case 8:
			*cur8++ = ((gint8 *)ws->sb->data)[sample + channel];
			break;
		    case 16:
			*cur16++ = ((gint16 *)ws->sb->data)[sample + channel];
			break;
		}
	    }

	    log("UNDO SAVE DATA", "%s %-10.10s %d bytes (%d-%d) chan:%2d \n",
		undo->filename, edit_table[undo->cmd].name, bufsize,
		start, end, channel);

	    if (write(undo->fd, buf, bufsize) < 0)
	    {
		log("UNDO SAVE DATA", "write failed %s\n", strerror(errno));
	    }
	    free(buf);
	}
    }

    return(TRUE);

err1:
    log("UNDO SAVE DATA", "Unable to write %s\n", strerror(errno));
    return(FALSE);
}



/*
 * FUNC: remove all undo's from the wave_set and unlink the undo files
 *   IN: wave_set to clear of undo's
 */
void edit_undos_free(GtkWaveSet *ws)
{
    undo_t	*undo;
    undo_t	*next;

    /* serialize with any other edits */
    pthread_mutex_lock(&edit_mutex);

    undo = ws->sb->undo;
    while(undo != NULL)
    {
	log("UNDOFREE", "unlinking undofile %s\n", undo->filename);
	if (unlink(undo->filename) < 0)
	{
	    log("UNDOFREE", "unable to unlink %s %s\n", undo->filename,
		strerror(errno));
	}

	// remove from list
	next = undo->next;
	undo->ws->sb->undo = next;
	undo->ws->sb->undo_cnt--;
	free(undo);
	undo = next;
    };

    pthread_mutex_unlock(&edit_mutex);
}



static gboolean edit_insert_silence(undo_t *undo)
{
    selection_set_t	*sel_set = &undo->selection_set;

    status("Inserting silence...");

    undo->size_change = sel_set->max_smp - sel_set->min_smp;
    edit_shiftright(undo);
    edit_mute(undo);

    status("Insert silence finished");
    return(TRUE);
}



/*
 * FUNC: copy selection to clipboard
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_copy(undo_t *undo)
{
    selection_set_t	*sel_set = &undo->selection_set;

    /* check to get rid of clipboards previous undo if the clipboard
     * was the last (and only) reference to it
     */
    if (clipboard && clipboard->ref == 1)
    {
	log("COPY", "Free last clipboard's undo file\n");
	unlink(undo->filename);
    }

    undo->size_change = -(sel_set->max_smp - sel_set->min_smp);
    log("COPY", "copied %d samples to clipboard\n", undo->size_change);
    clipboard = undo;
    return(TRUE);
}



/*
 * FUNC: insert clipboard region into the file at the beginning
 *	 of the selection
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_paste(undo_t *undo)
{
    gint32		diff;
    gint8 		channel;
    undo_t		pseudo_undo;

    if (!clipboard)
    {
	log("PASTE", "Nothing on the clipboard!\n");
	return(FALSE);
    }

    status("Pasting...");

    // adjust the paste's undo so it can be undone
    undo->size_change = ABS(clipboard->size_change);
    diff = undo->selection_set.min_smp - clipboard->selection_set.min_smp;
    undo->selection_set = clipboard->selection_set;
    undo->selection_set.min_smp += diff;
    undo->selection_set.max_smp += diff;
    for(channel = 0; channel < MAX_CHANNELS; channel++)
    {
	undo->selection_set.selection[channel].start += diff;
	undo->selection_set.selection[channel].end   += diff;
    }


    // create pseudo undo to pass to edit_do_undo...
    pseudo_undo = *undo;
    strcpy(pseudo_undo.filename, clipboard->filename);
    pseudo_undo.size_change = -undo->size_change;
    log("PASTE", "diff:%d undo size:%d\n", diff, undo->size_change);
    edit_do_undo(&pseudo_undo, undo->ws, TRUE);
    edit_display_update(&pseudo_undo);

    status("Paste finished");
    return(TRUE);
}



/*
 * FUNC: remove a region from the file
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_cut(undo_t *undo)
{
    status("Cutting...");

    /* if the clipboard had a ref count of 1 the original undo
     * is no longer needed to be kept around
     */
    if (clipboard && clipboard->ref == 1)
	free(clipboard);
    
    clipboard = undo;
    undo->ref++;
    undo->size_change = -(undo->selection_set.min_sel);
    log("CUT", "size change:%d min_sel %d\n", undo->size_change, undo->selection_set.min_sel);
    edit_shiftleft(undo);

    status("Cut finished");
    return(TRUE);
}



/*
 * FUNC: fade in
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_fade_linear_in(undo_t *undo)
{
    gint8	channel;
    GtkWaveSet		*ws = undo->ws;
    selection_set_t	*sel_set = &undo->selection_set;

    status("Fading in...");

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gint32	start = sel_set->selection[channel].start;
	gint32	end   = sel_set->selection[channel].end;
	gint32	len   = end - start;
	gint32	smp;
	gint32	data_indx;
	gfloat	percent;

	if (start < 0)
	    continue;

	for(smp = start; smp < end; smp++)
	{
	    data_indx = smp * ws->sb->info.channels;
	    percent = ((gfloat)(smp-start))/len;
	    switch(ws->sb->info.sample_bits)
	    {
		gint8 smp8;

		case 8:
		    smp8 = ((guint8 *)ws->sb->data)[data_indx + channel] - 128;
		    smp8 *= percent;
		    ((guint8 *)ws->sb->data)[data_indx + channel] = smp8 + 128;
		    break;

		case 16:
		    ((gint16 *)ws->sb->data)[data_indx + channel] *= percent;
		    break;
	    }
	}
    }

    status("Fade in finished");
    return(TRUE);
}



/*
 * FUNC: fade out
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_fade_linear_out(undo_t *undo)
{
    gint8	channel;
    GtkWaveSet		*ws = undo->ws;
    selection_set_t	*sel_set = &undo->selection_set;

    status("Fading out...");

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gint32	start = sel_set->selection[channel].start;
	gint32	end   = sel_set->selection[channel].end;
	gint32	len   = end - start;
	gint32	smp;
	gint32	data_indx;
	gfloat	percent;

	if (start < 0)
	    continue;

	for(smp = start; smp < end; smp++)
	{
	    data_indx = smp * ws->sb->info.channels;
	    percent = (gfloat)1.0-(gfloat)(smp-start)/len;
	    switch(ws->sb->info.sample_bits)
	    {
		gint8 smp8;

		case 8:
		    smp8 = ((guint8 *)ws->sb->data)[data_indx + channel] - 128;
		    smp8 *= percent;
		    ((guint8 *)ws->sb->data)[data_indx + channel] = smp8 + 128;
		    break;

		case 16:
		    ((gint16 *)ws->sb->data)[data_indx + channel] *= percent;
		    break;
	    }
	}
    }

    status("Fade out finished");
    return(TRUE);
}



/*
 * FUNC: linearly interpolate between beginning sample and ending sample
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_interp_linear(undo_t *undo)
{
    gint8     channel;
    GtkWaveSet                *ws = undo->ws;
    selection_set_t   *sel_set = &undo->selection_set;

    status("Interpolating...");

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gint32  start = sel_set->selection[channel].start;
	gint32  end   = sel_set->selection[channel].end;
	gint32  len   = end - start;
	gint32  smp;
	gint32  data_indx;
	gint16  start16 = 0, end16 = 0;
	gint8   start8 = 0, end8 = 0;
	gfloat  percent;

	if (start < 0)
	    continue;

	switch(ws->sb->info.sample_bits)
	{
	    case 8:
		start8 = ((guint8 *)ws->sb->data)[start*ws->sb->info.channels+channel] - 128;
		end8 = ((guint8 *)ws->sb->data)[end*ws->sb->info.channels+channel] - 128;
		break;

	    case 16:
		start16 = ((gint16 *)ws->sb->data)[start*ws->sb->info.channels+channel];
		end16 = ((gint16 *)ws->sb->data)[end*ws->sb->info.channels+channel];
		break;
	}

	for(smp = start; smp < end; smp++)
	{
	    data_indx = smp * ws->sb->info.channels;
            percent = ((gfloat)(smp-start))/len;
	    switch(ws->sb->info.sample_bits)
	    {
		case 8:
		    ((guint8 *)ws->sb->data)[data_indx + channel] = 
			(guint8)(start8*(1-percent)+end8*percent + 128);
		    break;

    		case 16:
		    ((gint16 *)ws->sb->data)[data_indx + channel] =
			start16*(1-percent)+end16*percent;
		    break;
	    }
	}
    }

    status("Linear interpolation finished");
    return(TRUE);
}



/*
 * FUNC: normalize
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_normalize(undo_t *undo)
{
    chnl_indx		channel;
    GtkWaveSet		*ws = undo->ws;
    selection_set_t	*sel_set = &undo->selection_set;
    gint16		max16 = 0;
    gint8		max8 = 0;
    gfloat		scale = 1.0;

    status("Normalize scanning...");

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gint32	start = sel_set->selection[channel].start;
	gint32	end   = sel_set->selection[channel].end;
	gint32	smp;
	gint32	data_indx;

	if (start < 0)
	    continue;

	for(smp = start; smp < end; smp++)
	{
	    data_indx = smp * ws->sb->info.channels;
	    switch(ws->sb->info.sample_bits)
	    {
		case 8:
		    max8 = MAX(max8, ABS(((guint8 *)ws->sb->data)[data_indx + channel] - 128));
		case 16:
		    max16 = MAX(max16, ABS(((gint16 *)ws->sb->data)[data_indx + channel]));
		    break;
	    }
	}
    }

    switch(ws->sb->info.sample_bits)
    {
	case 8:
	    scale = (gfloat)128.0/max8;
	    break;
	case 16:
	    scale = (gfloat)32767.0/max16;
	    break;
    }

    status("Normalizing...");

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gint32	start = sel_set->selection[channel].start;
	gint32	end   = sel_set->selection[channel].end;
	gint32	smp;
	gint32	data_indx;

	if (start < 0)
	    continue;

	for(smp = start; smp < end; smp++)
	{
	    data_indx = smp * ws->sb->info.channels;
	    switch(ws->sb->info.sample_bits)
	    {
		gint8 smp8;

		case 8:
		    smp8 = ((guint8 *)ws->sb->data)[data_indx + channel] - 128;
		    smp8 *= scale;
		    ((guint8 *)ws->sb->data)[data_indx + channel] = smp8 + 128;
		    break;

		case 16:
		    ((gint16 *)ws->sb->data)[data_indx + channel] *= scale;
		    break;
	    }
	}
    }

    return(TRUE);
}



/*
 * FUNC: mute
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_mute(undo_t *undo)
{
    gint8		channel;
    gint32		data_indx;
    gint32		smp;
    GtkWaveSet		*ws = undo->ws;
    selection_set_t	*sel_set = &undo->selection_set;

    status("Muting...");

    for(smp = sel_set->min_smp; smp < sel_set->max_smp; smp++)
    {
	for(channel = 0; channel < ws->sb->info.channels; channel++)
	{
	    if ((smp >= sel_set->selection[channel].start) &&
	        (smp <  sel_set->selection[channel].end))
	    {
		data_indx = smp * ws->sb->info.channels;
		switch(ws->sb->info.sample_bits)
		{
		    case 8:
			((guint8 *)ws->sb->data)[data_indx + channel] = 128;
			break;

		    case 16:
			((gint16 *)ws->sb->data)[data_indx + channel] = 0;
			break;
		}
	    }
	}
    }

    status("Mute finished");
    return(TRUE);
}



/*
 * FUNC: reverse
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_reverse(undo_t *undo)
{
    gint8		channel;
    GtkWaveSet		*ws = undo->ws;
    selection_set_t	*sel_set = &undo->selection_set;

    status("Reversing...");

    for(channel = 0; channel < ws->sb->info.channels; channel++)
    {
	gint32	start = sel_set->selection[channel].start;
	gint32	end   = sel_set->selection[channel].end;
	gint32	smp;
	gint32	data_indx1;
	gint32	data_indx2;
	gint16  tmp16;
	gint8   tmp8;

	if (start < 0)
	    continue;

	for(smp = start; smp < end-((end-start)/2); smp++)
	{
	    data_indx1 = smp * ws->sb->info.channels;
	    data_indx2 = (end - (smp-start)) * ws->sb->info.channels;

	    switch(ws->sb->info.sample_bits)
	    {
		case 8:
		    tmp8 = ((gint8 *)ws->sb->data)[data_indx1 + channel];
		    ((gint8 *)ws->sb->data)[data_indx1 + channel] =
			((gint8 *)ws->sb->data)[data_indx2 + channel];
		    ((gint8 *)ws->sb->data)[data_indx2 + channel] = tmp8;
		    break;

		case 16:
		    tmp16 = ((gint16 *)ws->sb->data)[data_indx1 + channel];
		    ((gint16 *)ws->sb->data)[data_indx1 + channel] =
			((gint16 *)ws->sb->data)[data_indx2 + channel];
		    ((gint16 *)ws->sb->data)[data_indx2 + channel] = tmp16;
		    break;
	    }
	}
    }

    status("Reverse finished");
    return(TRUE);
}



#ifdef ENABLE_LADSPA
/*
 * FUNC: call ladspa plugin
 *   IN: undo describing region to work on
 *  OUT: boolean indication of success
 */
static gboolean edit_plugin(undo_t *undo)
{
    chnl_indx		channel;
    chnl_indx		plug_channels;
    gint32		data_indx;
    gint32		smp;
    GtkWaveSet		*ws = undo->ws;
    selection_set_t	*sel_set = &undo->selection_set;
    ulong		plug_indx;
    float		*plug_io = NULL;

    plug_channels = ladspa_plug_channels();

    /* process a disjoint selection one channel at a time */
    if (sel_set->channel_start != CHANNEL_ALL)
	plug_channels = 1;

    /* allocate i/o for size of largest selection */
    plug_io = malloc(sel_set->max_sel *	sizeof (float) * plug_channels);
    if (plug_io == NULL)
    {
	log("EDIT", "unable to allocate i/o space for plugin\n");
	return FALSE;
    }

    log("EDIT", "plugin buf:%p chan:%d samples:%d\n", plug_io, plug_channels, sel_set->max_sel);


    /* do each channel seperately */
    if (plug_channels == 1)
    {
	for(channel = 0; channel < ws->sb->info.channels; channel++)
	{
	    smpl_indx channel_size;

	    /* skip channels without a selection */
	    if (sel_set->selection[channel].start == -1)
		continue;

	    log("EDIT", "plugin: running against channel %d\n", channel);

	    channel_size = sel_set->selection[channel].end -
		   	sel_set->selection[channel].start;

	    /* convert samples to floats for plugin */
	    for(smp = sel_set->selection[channel].start, plug_indx = 0;
		smp < sel_set->selection[channel].end; smp++)
	    {
		data_indx = smp * ws->sb->info.channels;
		switch(ws->sb->info.sample_bits)
		{
		    case 8:
			plug_io[plug_indx++] = ((gint8 *)ws->sb->data)[data_indx + channel];
			break;

		    case 16:
			plug_io[plug_indx++] = ((gint16 *)ws->sb->data)[data_indx + channel];
			break;
		}
	    }

	    /* run the plugin on the data */
	    ladspa_plug_run(plug_io, plug_channels, channel_size);

	    /* convert processed data back from floats */
	    for(smp = sel_set->selection[channel].start, plug_indx = 0;
		smp < sel_set->selection[channel].end; smp++)
	    {
		data_indx = smp * ws->sb->info.channels;
		switch(ws->sb->info.sample_bits)
		{
		    case 8:
			((gint8 *)ws->sb->data)[data_indx + channel] = plug_io[plug_indx++];
			break;

		    case 16:
			((gint16 *)ws->sb->data)[data_indx + channel] = plug_io[plug_indx++];
			break;
		}
	    }
	}
    }
    else if (sel_set->channel_end - sel_set->channel_start +1 <= plug_channels)
    {
	smpl_indx channel_size;

	channel_size = sel_set->max_smp - sel_set->min_smp;

	log("EDIT", "plugin: running against channels %d-%d (total:%d)\n",
	    sel_set->channel_start, sel_set->channel_end, sel_set->channel_end - sel_set->channel_end + 1);


	/* convert samples to floats for plugin */
	for(smp = sel_set->min_smp, plug_indx = 0;
	    smp < sel_set->max_smp; smp++)
	{
	    for(channel = 0; channel < ws->sb->info.channels; channel++)
	    {
		data_indx = smp * ws->sb->info.channels;
		switch(ws->sb->info.sample_bits)
		{
		    case 8:
			plug_io[plug_indx++] = ((gint8 *)ws->sb->data)[data_indx + channel];
			break;

		    case 16:
			plug_io[plug_indx++] = ((gint16 *)ws->sb->data)[data_indx + channel];
			break;
		}
	    }
	}

	/* run the plugin on the data */
	ladspa_plug_run(plug_io, plug_channels, channel_size);

	/* convert processed data back from floats */
	for(smp = sel_set->min_smp, plug_indx = 0;
	    smp < sel_set->max_smp; smp++)
	{
	    for(channel = 0; channel < ws->sb->info.channels; channel++)
	    {
		data_indx = smp * ws->sb->info.channels;
		switch(ws->sb->info.sample_bits)
		{
		    case 8:
			((gint8 *)ws->sb->data)[data_indx + channel] = plug_io[plug_indx++];
			break;

		    case 16:
			((gint16 *)ws->sb->data)[data_indx + channel] = plug_io[plug_indx++];
			break;
		}
	    }
	}
    }
    else
    {
	log("EDIT", "plugin: more channels selected than plugin supports\n");
    }

    free(plug_io);

    return(TRUE);
}
#endif /* ENABLE_LADSPA */
