/******************************************************************************\
 gnofin/history.h   $Revision: 1.3 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#ifndef HISTORY_H
#define HISTORY_H

#include <gtk/gtkobject.h>

/* a general history interface */

#ifdef G_DISABLE_CHECKS
# define HISTORY_CAN_UNDO(history)  ((history)->can_undo)
# define HISTORY_CAN_REDO(history)  ((history)->can_redo)
#else
# define HISTORY_CAN_UNDO(history)  ((history) ? (history)->can_undo : FALSE)
# define HISTORY_CAN_REDO(history)  ((history) ? (history)->can_redo : FALSE)
#endif

typedef struct _History          History;
typedef struct _HistorySignals   HistorySignals;
typedef struct _HistoryBatchItem HistoryBatchItem;

typedef void (* HistoryFunc) (gpointer parent_data, gpointer item_data);

/* design notes:
 *
 * this class implements a simple history with infinite undo/redo.
 * the user can define the following callbacks:
 *
 *   undo_item  - try to undo a specific history item
 *   redo_item  - try to redo a specific history item
 *   free_item  - free memory associated with history item
 *
 * thus, the user is left to define what it means to "undo" a
 * history item, where a history item is just an abstract name
 * for some "remembered" operation that changes the state of 
 * the program.
 *
 * adding items to the history is done by calling history_remember.
 */

struct _HistorySignals
{
  HistoryFunc undo_item;
  HistoryFunc redo_item;
  HistoryFunc free_item;
  HistoryFunc dump_item;
};

struct _History
{
  GList *items;	   /* list of history items, user defined */
  GList *current;  /* points to next UNDO item */

  gpointer parent_data;

  HistoryBatchItem *batch_item;
  guint16 batch_mode;

  guint can_undo : 1;
  guint can_redo : 1;
};

History *history_new  (gpointer parent_data);
void history_destroy  (History *history);

void history_init     (History *history, gpointer parent_data);
void history_undo     (History *history);
void history_redo     (History *history);
void history_remember (History *history, HistorySignals *signals, gpointer item);
void history_clear    (History *history);
void history_dump     (History *history);

/* When batch mode is enabled, history items are remembered as one item.  
 * Calls to undo/redo treat the entire batch as one item.  enter() may be
 * called within a enter/leave pair, provided there is a matching leave().
 */
void history_enter_batch_mode (History *history);
void history_leave_batch_mode (History *history);

#endif

// vim: ts=8 sw=2
