/*
 *    misc.c  --  Miscellaneous helper functions (non-optimized version)
 *
 *    Copyright (C) 2001, 2002, 2003, 2004
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "misc.h"

#ifndef __OPTIMIZE__

/* ---------------------------------------------------------------------- */

/*
 * Hamming weight (number of bits that are ones).
 */
unsigned int hweight32(unsigned int w) 
{
	w = (w & 0x55555555) + ((w >>  1) & 0x55555555);
	w = (w & 0x33333333) + ((w >>  2) & 0x33333333);
	w = (w & 0x0F0F0F0F) + ((w >>  4) & 0x0F0F0F0F);
	w = (w & 0x00FF00FF) + ((w >>  8) & 0x00FF00FF);
	w = (w & 0x0000FFFF) + ((w >> 16) & 0x0000FFFF);
	return w;
}

unsigned int hweight16(unsigned short w)
{
	w = (w & 0x5555) + ((w >> 1) & 0x5555);
	w = (w & 0x3333) + ((w >> 2) & 0x3333);
	w = (w & 0x0F0F) + ((w >> 4) & 0x0F0F);
	w = (w & 0x00FF) + ((w >> 8) & 0x00FF);
	return w;
}

unsigned int hweight8(unsigned char w)
{
	w = (w & 0x55) + ((w >> 1) & 0x55);
	w = (w & 0x33) + ((w >> 2) & 0x33);
	w = (w & 0x0F) + ((w >> 4) & 0x0F);
	return w;
}

/* ---------------------------------------------------------------------- */

/*
 * Parity function. Return one if `w' has odd number of ones, zero otherwise.
 */
int parity(unsigned int w)
{
	return hweight32(w) & 1;
}

/* ---------------------------------------------------------------------- */

/*
 * Reverse order of bits.
 */
unsigned int rbits32(unsigned int w)
{
	w = ((w >>  1) & 0x55555555) | ((w <<  1) & 0xAAAAAAAA);
	w = ((w >>  2) & 0x33333333) | ((w <<  2) & 0xCCCCCCCC);
	w = ((w >>  4) & 0x0F0F0F0F) | ((w <<  4) & 0xF0F0F0F0);
	w = ((w >>  8) & 0x00FF00FF) | ((w <<  8) & 0xFF00FF00);
	w = ((w >> 16) & 0x0000FFFF) | ((w << 16) & 0xFFFF0000);
	return w;
}

unsigned short rbits16(unsigned short w)
{
	w = ((w >> 1) & 0x5555) | ((w << 1) & 0xAAAA);
	w = ((w >> 2) & 0x3333) | ((w << 2) & 0xCCCC);
	w = ((w >> 4) & 0x0F0F) | ((w << 4) & 0xF0F0);
	w = ((w >> 8) & 0x00FF) | ((w << 8) & 0xFF00);
	return w;
}

unsigned char rbits8(unsigned char w)
{
	w = ((w >> 1) & 0x55) | ((w << 1) & 0xFF);
	w = ((w >> 2) & 0x33) | ((w << 2) & 0xCC);
	w = ((w >> 4) & 0x0F) | ((w << 4) & 0xF0);
	return w;
}

/* ---------------------------------------------------------------------- */

/*
 * Integer base-2 logarithm
 */
int log2(unsigned int x)
{
	int y = 0;

	x >>= 1;

	while (x) {
		x >>= 1;
		y++;
	}

	return y;
}

/* ---------------------------------------------------------------------- */

/*
 * Gray encoding and decoding (8 bit)
 */
unsigned char grayencode(unsigned char data)
{
	unsigned char bits = data;

	bits ^= data >> 1;
	bits ^= data >> 2;
	bits ^= data >> 3;
	bits ^= data >> 4;
	bits ^= data >> 5;
	bits ^= data >> 6;
	bits ^= data >> 7;

	return bits;
}

unsigned char graydecode(unsigned char data)
{
	return data ^ (data >> 1);
}

/* ---------------------------------------------------------------------- */

/*
 * Hamming window function
 */
double hamming(double x)
{
        return 0.54 - 0.46 * cos(2 * M_PI * x);
}

/* ---------------------------------------------------------------------- */

/*
 * Sinc etc...
 */
double sinc(double x)
{
	return (fabs(x) < 1e-10) ? 1.0 : (sin(M_PI * x) / (M_PI * x));
}

double cosc(double x)
{
	return (fabs(x) < 1e-10) ? 0.0 : ((1.0 - cos(M_PI * x)) / (M_PI * x));
}

/* ---------------------------------------------------------------------- */

float clamp(float x, float min, float max)
{
	return (x < min) ? min : ((x > max) ? max : x);
}

/* ---------------------------------------------------------------------- */

float decayavg(float average, float input, float weight)
{
	return input * (1.0 / weight) + average * (1.0 - (1.0 / weight));
}

/* ---------------------------------------------------------------------- */

#endif				/* __OPTIMIZE__ */
