/*  Glimmer - misc.c
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "declarations.h"
#include "misc.h"
#include "main.h"
#include "settings.h"
#include "widgets/gdsfile.h"
#include "widgets/gdstearaway.h"

gchar *
replace_percent_tokens (const gchar * string)
{
    gchar *path;
    gchar *filename;
    gchar *file;
    gchar *new_string = NULL;
    gchar *ptr;

    path = gds_file_get_path (cur_file);
    filename = gds_file_get_filename (cur_file);
    file = gds_file_get_filename_nix_extension (cur_file);
    new_string = g_strdup (string);
    while (1) {
        ptr =
            g_str_replace (new_string, "%term%", general_preferences.terminal);
        if (ptr) {
            g_free (new_string);
            new_string = ptr;
        } else {
            break;
        }
    }
    while (1) {
        ptr = g_str_replace (new_string, "%make%", general_preferences.make);
        if (ptr) {
            g_free (new_string);
            new_string = ptr;
        } else {
            break;
        }
    }
    while (1) {
        ptr = g_str_replace (new_string, "%1", path);
        if (ptr) {
            g_free (new_string);
            new_string = ptr;
        } else {
            break;
        }
    }
    while (1) {
        ptr = g_str_replace (new_string, "%2", filename);
        if (ptr) {
            g_free (new_string);
            new_string = ptr;
        } else {
            break;
        }
    }
    while (1) {
        ptr = g_str_replace (new_string, "%3", file);
        if (ptr) {
            g_free (new_string);
            new_string = ptr;
        } else {
            break;
        }
    }
    //FIXME: should we release the strings in this routine?
    return (new_string);
}

gchar *
get_path_from_filename (gchar * filename)
{
    gchar *file = NULL;
    gchar *text = NULL;
    gchar *value = NULL;

    file = g_strdup (filename);
    text = strrchr (file, '/');
    if (text) {
        *text = '\0';
        text = file;
        value = g_strdup (text);
    }
    g_free (file);
    return (value);
}

gchar *
get_file_from_filename (gchar * filename)
{
    gchar *text = NULL;

    text = strrchr (filename, '/');
    if (text)
        text++;
    else
        text = filename;
    return (g_strdup (text));
}

gchar *
get_file_wo_extension (gchar * filename)
{
    gchar *file = NULL;
    gchar *text = NULL;
    gchar *extension = NULL;
    gchar *value = NULL;

    file = g_strdup (filename);
    text = strrchr (file, '/');
    if (text) {
        text++;
    } else {
        text = file;
    }
    extension = strrchr (text, '.');
    if (extension) {
        *extension = '\0';
    }
    value = g_strdup (text);
    g_free (file);
    return (value);
}

gchar *
g_str_replace (const gchar * input, const gchar * replace, const gchar * with)
{
    gchar *text = NULL;
    gchar *start = NULL;
    gchar *end = NULL;
    gchar *value = NULL;

    text = g_strdup (input);
    start = strstr (text, replace);
    if (!start)
        return (NULL);
    end = start + strlen (replace);
    *start = '\0';
    value = g_strconcat (text, with, end, NULL);
    g_free (text);
    return (value);
}

gchar *
g_str_replace_bad_chars (const gchar * input)
{
    gchar *ptr;
    gchar *new_string;
    gint i, j;
    gchar last = 0;
    new_string = g_new0 (char, strlen (input) + 1);

    for (i = 0, j = 0; i < strlen (input); last = input[i], i++, j++) {
        if ((last == '\\') && (input[i] == 't' || input[i] == 'n')) {
            j--;
            if (input[i] == 't')
                new_string[j] = '\t';
            else if (input[i] == 'n')
                new_string[j] = '\n';
            continue;
        }
        new_string[j] = input[i];
    }
    ptr = g_strdup (new_string);
    g_free (new_string);
    return (ptr);
}

GList *
build_glist_from_file (gchar * filename, guint max)
{
    gchar full_path[256] = "";
    FILE *file;
    gchar buffer[256];
    gpointer temp;
    GList *list = NULL;
    guint iterations = 0;

    g_snprintf (full_path, sizeof (full_path), "%s/." PACKAGE "/%s",
                getenv ("HOME"), filename);

    file = fopen (full_path, "r");
    if (!file)
        return (list);

    while (fgets (buffer, sizeof (buffer), file) && iterations < max) {
        if (!strcmp (buffer, "\n"))
            continue;
        buffer[strlen (buffer) - 1] = '\0';
        if (check_list_for_string (list, buffer))
            continue;
        temp = g_new (char, strlen (buffer) + 1);

        strcpy ((gchar *) temp, buffer);
        list = g_list_append (list, temp);
        iterations++;
    }
    return (list);
}


void
add_string_to_file (gchar * filename, const gchar * string)
{
    gchar full_path[256] = "";
    gchar temp_file[256] = "";
    FILE *file;
    FILE *tmp_file;
    gchar buffer[256];

    g_snprintf (full_path, sizeof (full_path), "%s/." PACKAGE "/%s",
                getenv ("HOME"), filename);
    g_snprintf (temp_file, sizeof (temp_file), "%s/." PACKAGE "/%s",
                getenv ("HOME"), "yadda.tmp");

    file = fopen (full_path, "a");
    if (!file)
        return;
    fclose (file);

    rename (full_path, temp_file);

    file = fopen (full_path, "w");
    if (!file)
        return;

    fprintf (file, "%s\n", string);

    tmp_file = fopen (temp_file, "r");
    if (!tmp_file) {
        fclose (file);
        return;
    }

    while (fgets (buffer, sizeof (buffer), tmp_file) != NULL) {
        fprintf (file, "%s", buffer);
    }
    fclose (file);
    fclose (tmp_file);
    remove (temp_file);
}

/* 0 for none, + for it's position in the list - use position-1 to index*/
gint
check_list_for_string (GList * list, gchar * string)
{
    GList *current = NULL;
    gint return_value = 0;
    gint iterations = 0;

    current = g_list_first (list);
    while (current) {
        iterations++;
        if (!strcmp (string, (gchar *) current->data)) {
            return_value = iterations;
            break;
        }
        current = current->next;
    }
    return (return_value);
}

gint
get_entry_length (GtkEntry * entry)
{
    gchar *text;

    text = gtk_entry_get_text (entry);
    return (strlen (text));
}

/* return g_new()'d text string...must be g_free()'d later */
gchar *
get_entry_text (GtkEntry * entry)
{
    gchar *text;
    gint length;

    length = get_entry_length (entry);
    text = g_new (char, length + 1);
    strcpy (text, gtk_entry_get_text (entry));
    return (text);
}

gint
get_selection_length (GtkExText * text)
{
    gint length;

    g_return_val_if_fail (text != NULL, -1);
    if (text->has_selection) {
        length = text->selection_end_pos - text->selection_start_pos;
        return (length);
    } else {
        return (-1);
    }
}

gchar *
get_selection_text (GtkExText * text)
{
    gchar *chars;
    gint start;
    gint end;

    if (text->has_selection) {
        start = text->selection_start_pos;
        end = text->selection_end_pos;
        chars = gtk_extext_get_chars (text, start, end);
        return (chars);
    } else {
        return (NULL);
    }
}

GdsFile *
get_nth_file_proper (gint number)
{
    GdsFile *set = NULL;
    gint iterations = 0;
    gint current = -1;

    if (number < 0)
        return (NULL);
    while (iterations < total_files) {
        set = (GdsFile *) g_list_nth_data (files_list, iterations);
        if (!set)
            break;
        if (!GDS_TEAR_AWAY (set->tear_away)->torn)
            current++;
        if (current == number)
            break;
        iterations++;
    }
    if (g_list_index (files_list, set) >= 0)
        return (set);
    return (NULL);
}

gint
get_nth_file_number (GdsFile * gds_file)
{
    GdsFile *check = NULL;
    gint iterations = 0;
    gint current = 0;

    while (iterations < total_files) {
        check = (GdsFile *) g_list_nth_data (files_list, iterations);
        if (check == gds_file)
            break;
        if (!check)
            break;
        if (!GDS_TEAR_AWAY (check->tear_away)->torn)
            current++;
        iterations++;
    }
    return (current);
}

void
display_message (const gchar * text, gint type)
{
    if (type == FLASH)
        gnome_app_flash (GNOME_APP (window), text);
    if (type == ERROR)
        gnome_app_error (GNOME_APP (window), text);
    if (type == WARNING)
        gnome_app_warning (GNOME_APP (window), text);
}

gint
get_which_radio_active (GSList * list)
{
    GSList *first;
    GSList *current;
    gint iterations = 0;
    GtkToggleButton *toggle;

    first = g_slist_copy (list);
    first = current = g_slist_reverse (first);
    while (current) {
        toggle = (GtkToggleButton *) current->data;
        if (toggle->active)
            break;
        iterations++;
        current = current->next;
    }
    g_slist_free (first);
    return (iterations);
}

void
set_which_radio_active (GSList * list, gint which)
{
    GSList *first;
    GtkToggleButton *toggle;

    first = g_slist_copy (list);
    first = g_slist_reverse (first);
    toggle = (GtkToggleButton *) g_slist_nth_data (first, which);
    gtk_toggle_button_set_active (toggle, TRUE);
    g_slist_free (first);
}

GList *
build_file_list (void)
{
    GList *temp;

    temp = g_list_copy (files_list);
    return (temp);
}

gchar *
localize_uri_string (gchar * string)
{
    gchar find[] = "file://";
    gchar *temp;

    temp = strstr (string, find);
    if (temp)
        temp += strlen (find);
    else
        temp = string;
    return (g_strdup (temp));
}

void
build_stock_pixmap (gchar * label, gchar ** xpm_data, gint size)
{
    GnomePixmap *pixmap;
    GnomeStockPixmapEntry *entry;
    gint state;

    for (state = 0; state < 3; state++) {
        pixmap =
            (GnomePixmap *) gnome_pixmap_new_from_xpm_d_at_size (xpm_data, size,
                                                                 size);
        entry = g_malloc0 (sizeof (GnomeStockPixmapEntry));
        entry->type = GNOME_STOCK_PIXMAP_TYPE_GPIXMAP;
        entry->gpixmap.width = size;
        entry->gpixmap.height = size;
        entry->gpixmap.label = g_strdup (label);
        entry->gpixmap.pixmap = pixmap;
        switch (state) {
        case 0:
            gnome_stock_pixmap_register (label, GNOME_STOCK_PIXMAP_REGULAR,
                                         entry);
            break;
        case 1:
            gnome_stock_pixmap_register (label, GNOME_STOCK_PIXMAP_FOCUSED,
                                         entry);
            break;
        case 2:
            modify_pixmap_state (window, GTK_STATE_INSENSITIVE, &pixmap);
            gnome_stock_pixmap_register (label, GNOME_STOCK_PIXMAP_DISABLED,
                                         entry);
            break;
        }
    }
}

void
modify_pixmap_state (GtkWidget * window, GtkStateType state,
                     GnomePixmap ** inout_pixmap)
{
    GdkGC *gc;
    GdkWindow *pixmap = (*inout_pixmap)->pixmap;
    gint w, h, x, y;
    GdkGCValues vals;
    GdkVisual *visual;
    GdkImage *image;
    GdkColorContext *cc;
    GdkColormap *cmap;
    gint32 red, green, blue;
    GtkStyle *style;
    int use_stripple = -1;

    g_return_if_fail (window != NULL);

    /* values for use_stripple:
     * 0 - decide upon the value of cc->mode
     * 1 - always stripple
     * 2 - always shade */
    use_stripple = gnome_config_get_int ("/Gnome/Icons/Shade=0");

    gdk_window_get_size (pixmap, &w, &h);
    visual = gtk_widget_get_visual (GTK_WIDGET (*inout_pixmap));
    cmap = gtk_widget_get_colormap (GTK_WIDGET (*inout_pixmap));
    gc = gdk_gc_new (pixmap);

    cc = gdk_color_context_new (visual, cmap);
    if (((cc->mode != GDK_CC_MODE_TRUE) &&
         (cc->mode != GDK_CC_MODE_MY_GRAY) &&
         (use_stripple != 2)) || (use_stripple == 1)) {
        gdk_color_context_free (cc);
        style = gtk_widget_get_style (window);
        gdk_gc_set_foreground (gc, &style->bg[state]);
        for (y = 0; y < h; y++) {
            for (x = y % 2; x < w; x += 2) {
                gdk_draw_point (pixmap, gc, x, y);
            }
        }
        gdk_gc_destroy (gc);
        return;
    }

    image = gdk_image_get (pixmap, 0, 0, w, h);
    gdk_gc_get_values (gc, &vals);
    style = gtk_widget_get_style (window);
    red = style->bg[state].red;
    green = style->bg[state].green;
    blue = style->bg[state].blue;
    for (y = 0; y < h; y++) {
        for (x = 0; x < w; x++) {
            GdkColor c;
            int failed;

            c.pixel = gdk_image_get_pixel (image, x, y);
            gdk_color_context_query_color (cc, &c);
            c.red = (((gint32) c.red - red) >> 1)
                + red;
            c.green = (((gint32) c.green - green) >> 1)
                + green;
            c.blue = (((gint32) c.blue - blue) >> 1)
                + blue;
            c.pixel = gdk_color_context_get_pixel (cc, c.red,
                                                   c.green, c.blue, &failed);
            gdk_image_put_pixel (image, x, y, c.pixel);
        }
    }
    gdk_draw_image (pixmap, gc, image, 0, 0, 0, 0, w, h);
    gdk_image_destroy (image);
    gdk_gc_destroy (gc);
    gdk_color_context_free (cc);
}
