/*
 *  newsurface.c
 *  newsurface
 *
 *  Created by Anthony S. Mitchell and Joshua J. McKinnon.
 *  Contact:  Joshua.McKinnon@une.edu.au
 *  Copyright (c) 2004 University of New England. All rights reserved.
 *
 * This file contains (almost) all of the routines required to calculate
 * w(r) for the Hirshfeld surface.
 *
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "space.h"
#include "numeric.h"
#include "hirshfeld.h"
#include "molsurf.h"

#define MAXATOMS 54
#define NPOINTS 700
/* spline limits (in Angstroms) */
#define MAXR 7.4084815
/* crystal environment cutoff */
#define HFS_RMAX 10.0

gboolean hfs_rho_init = FALSE;
struct rho_pak
{
gdouble x[NPOINTS];
gdouble y[NPOINTS];
gdouble y2[NPOINTS];
} atomdata[MAXATOMS];

/************************************************************/
/* initialization routine for Hirshfeld surface calculation */
/************************************************************/
void hfs_init(void)
{
gint i, j;
gdouble x, delta, big, small;

/* only compute spline once */
if (hfs_rho_init)
  return;

/* setup data array for spline fit */
for (i=0 ; i<MAXATOMS ; i++)
  {
  for (j=0 ; j<NPOINTS ; j++)
    {
    x = (gdouble) j * MAXR / (gdouble) NPOINTS;		
/*** JJM: evaldens at sqrt(r) ***/
    atomdata[i].x[j] = x*x;
    atomdata[i].y[j] = rho[i][j];
    }

/***  use a crude numerical derivative at the first point  ***/
/*
  delta = 0.0001;
  big = evaldens(atom[i],0.0) - evaldens(atom[i],sqrt(delta));
  big = big/delta;
*/
 
/***  use a crude numerical derivative at the last point  ***/
/*
  delta = 0.0001;
  small = evaldens(atom[i],sqrt(maxr)) - evaldens(atom[i],(sqrt(maxr)+delta));
  small = small/delta;
*/

/* FIXME - I've just used 0.0 for the end point derivatives */
  spline(atomdata[i].x, atomdata[i].y, NPOINTS, 0.0, 0.0, atomdata[i].y2);
  }

hfs_rho_init = TRUE;
return;        
}

/***********************************/ 
/* generate atoms around selection */
/***********************************/ 
GSList *hfs_calc_env(GSList *select, struct model_pak *model)
{
gboolean flag, primary;
gint i, imax[3];
gdouble r2, dx[3];
GSList *ilist, *list, *list2, *nlist;
struct image_pak *image;
struct core_pak *core, *copy, *test;

/* checks */
g_assert(model != NULL);
g_assert(select != NULL);

/* construct periodic image vectors */
for (i=0 ; i<model->periodic ; i++)
  imax[i] = HFS_RMAX/model->pbc[i] + 1;

model->image_limit[0] = imax[0];
model->image_limit[1] = imax[0]+1;
model->image_limit[2] = imax[1];
model->image_limit[3] = imax[1]+1;
model->image_limit[4] = imax[2];
model->image_limit[5] = imax[2]+1;
space_make_images(CREATE, model);
calc_coords(REFRESH, model);

/* original + image iteration */
nlist = NULL;
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

/* mark cores in the central molecule */
  if (g_slist_find(select, core))
    primary = TRUE;
  else
    primary = FALSE;

/* add the atom */
  ilist = NULL;
  do
    {
/* add coordinates */
/* NB: probably don't need to worry about shells, as this is */
/* a temporary model and the algorithm will ignore them anyway */
    copy = dup_core(core);
    if (ilist)
      {
/* image translation */
      image = (struct image_pak *) ilist->data;
      ARR3ADD(copy->rx, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      ilist = model->images;

/* set cartesian coords */
    ARR3SET(copy->x, copy->rx);
    copy->primary = primary;

/* test if core satisfies distance condition */
    flag = FALSE;
    for (list2=select ; list2 ; list2=g_slist_next(list2))
      {
      test = (struct core_pak *) list2->data;

      ARR3SET(dx, test->rx);
      ARR3SUB(dx, copy->rx);
      r2 = VEC3MAGSQ(dx);
      if (r2 < HFS_RMAX*HFS_RMAX)
        {
        flag = TRUE;
        break;
        }
      }
    if (flag)
      nlist = g_slist_prepend(nlist, copy);
    }
  while (ilist);
  }

/* reset source model's images */
space_make_images(INITIAL, model);

return(nlist);
}

/************************************/
/* retrieve an interpolated density */
/************************************/
gdouble hfs_calc_density(gint n, gdouble r2)
{
gdouble value;

g_assert(n < MAXATOMS);

if (r2 > atomdata[n].x[NPOINTS-1])
  return(0.0);

splint(atomdata[n].x, atomdata[n].y, atomdata[n].y2, NPOINTS, r2, &value);
return(value);
}

/******************************************/
/* compute the weight function at a point */
/******************************************/
gdouble hfs_calc_wf(gdouble *x, GSList *core_list)
{
gdouble r2;
gdouble weight_r, r[3];
gdouble rhoNugget, rhoMol, rhoXtal;
GSList *list;
struct core_pak *core;

/* checks */
g_assert(core_list != NULL);
g_assert(hfs_rho_init == TRUE);

/* add up the contribution from the molecule */
rhoMol=0.0;
for (list=core_list ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (!core->primary)
    continue;

/* compute cartesian distance squared (in Angs) */
  ARR3SET(r, core->x);
  ARR3SUB(r, x);
  r2 = VEC3MAGSQ(r);

  rhoNugget = hfs_calc_density(core->atom_code, r2);
  rhoMol += rhoNugget;
  }

/* add up the total contribution */
rhoXtal=0.0;
for (list=core_list ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

/* compute cartesian distance squared (in Angs) */
  ARR3SET(r, core->x);
  ARR3SUB(r, x);
  r2 = VEC3MAGSQ(r);

  rhoNugget = hfs_calc_density(core->atom_code, r2);
  rhoXtal += rhoNugget;
  }

/* calc ratio */
if (rhoXtal != 0.0)
  weight_r = rhoMol/rhoXtal;
else
  weight_r = 0.0;

#define STRICT 1
#if STRICT
g_assert(weight_r <= 1.0);
#else
if (weight_r > 1.0)
  weight_r = 1.0;
#endif

return(weight_r);
}


/* TODO - surface distance and gradient functions using weightfunction calls */

