/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <string.h>
#include <strings.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "model.h"
#include "parse.h"
#include "spatial.h"
#include "matrix.h"

/* main structures */
extern struct sysenv_pak sysenv;

/* triggers */
enum 
{
XML_INACTIVE,
XML_SYSTEM, XML_SYMMETRY, XML_CRYSTAL, XML_CELL, XML_CORE, XML_SHELL,
XML_CELL_A, XML_CELL_B, XML_CELL_C,
XML_CELL_ALPHA, XML_CELL_BETA, XML_CELL_GAMMA,
XML_SPATIAL, XML_VERTEX,
XML_TRANSFORM, XML_MATRIX, XML_VECTOR, XML_SCALAR,
XML_PICTURE
};

gint xml_context = XML_INACTIVE;
struct model_pak *xml_model=NULL;
struct core_pak *xml_core=NULL;
struct shel_pak *xml_shell=NULL;
GHashTable *xml_system_table;



/* TEMP - parse diffax sfc data */
void parse_sfc(FILE *fp)
{
gint i, n;
gchar *line, *type, **buff;
gdouble *sfc;
GSList *list;

while ((line = file_read_line(fp)))
  {
  type = g_strstrip(g_strndup(line, 6));

  if (elem_symbol_test(type))
    {
/* tokenize everything after the atom type */
    buff = tokenize(&line[6], &n);

printf("[%s] ", type);
    list = NULL;
    for (i=0 ; i<n ; i++)
      {
      sfc = g_malloc(sizeof(gdouble));
      *sfc = str_to_float(*(buff+i));

printf("[%f] ", *sfc);
      list = g_slist_prepend(list, sfc);
      }
    list = g_slist_reverse(list);
printf("\n");

    g_hash_table_insert(sysenv.sfc_table, type, list);

    g_strfreev(buff);
    }
  g_free(line);
  }
}


/************************************/
/* find or allocate a model pointer */
/************************************/
void xml_parse_system(const gchar **names, const gchar **values)
{
gint i;
gchar *key;

/* get the system id value */
i=0;
while (*(names+i))
  {
  if (g_ascii_strncasecmp(*(names+i), "id\0", 3) == 0)
    {
/* first time exception - model already allocated */
    if (xml_model)
      {
printf("initial system: %s\n", *(values+i));
      key = g_strdup(*(values+i));
      g_hash_table_insert(xml_system_table, key, xml_model);
      }
    else
      {
/* check id against existing models */
      xml_model = g_hash_table_lookup(xml_system_table, *(values+i)); 
      if (!xml_model)
        {
printf("adding system: %s\n", *(values+i));
/* id not found - allocate new model & add to hash table */
        xml_model = model_new();
        key = g_strdup(*(values+i));
        g_hash_table_insert(xml_system_table, key, xml_model);
        }
      else
printf("existing system: %s\n", *(values+i));
      }
    }
  i++;
  }
/* got all we want */
xml_context = XML_INACTIVE;
}

/***************************************/
/* process symmetry element attributes */
/***************************************/
void xml_parse_symmetry(const gchar **names, const gchar **values)
{
gint i;

g_assert(xml_model != NULL);

/* process attributes */
i=0;
while (*(names+i))
  {
/* only init the cell parsing trigger if it's the full cell's data */
  if (g_ascii_strncasecmp(*(names+i), "spacegroup", 10) == 0)
    {
    xml_model->sginfo.spacename = g_strdup(*(values+i));
    }
  i++;
  }
}

/********************************************/
/* start cell reading if we get a full cell */
/********************************************/
void xml_parse_fullcell(const gchar **names, const gchar **values)
{
gint i;

/* process attributes */
i=0;
xml_context = XML_INACTIVE;
while (*(names+i))
  {
  if (g_ascii_strncasecmp(*(names+i), "dictRef\0", 8) == 0)
    {
/* CURRENT - only enter cell parsing context if it's the full cell's data */
    if (g_ascii_strncasecmp(*(values+i), "gulp:fullcell\0", 16) == 0)
      {
      xml_context = XML_CELL;
      xml_model->periodic = 3;
      }

    if (g_ascii_strncasecmp(*(values+i), "gulp:surface\0", 13) == 0)
      {
      xml_context = XML_CELL;
      xml_model->periodic = 2;
      }

    }
  i++;
  }
}

/***************************/
/* process cell attributes */
/***************************/
void xml_parse_cell(const gchar **names, const gchar **values)
{
gint i;

g_assert(xml_model != NULL);

/* process attributes (if any) */
i=0;
while (*(names+i))
  {
/* TODO - CHECK that the units are angs and degrees */
  if (g_ascii_strncasecmp(*(names+i), "dictRef\0", 8) == 0)
    {
/* TODO - split (remove namespace) and check a/b/etc. only? */
/* cell value require the text callback */
    if (g_ascii_strncasecmp(*(values+i), "cml:a\0", 6) == 0)
      xml_context = XML_CELL_A;
    if (g_ascii_strncasecmp(*(values+i), "cml:b\0", 6) == 0)
      xml_context = XML_CELL_B;
    if (g_ascii_strncasecmp(*(values+i), "cml:c\0", 6) == 0)
      xml_context = XML_CELL_C;
    if (g_ascii_strncasecmp(*(values+i), "cml:alpha\0", 10) == 0)
      xml_context = XML_CELL_ALPHA;
    if (g_ascii_strncasecmp(*(values+i), "cml:beta\0", 9) == 0)
      xml_context = XML_CELL_BETA;
    if (g_ascii_strncasecmp(*(values+i), "cml:gamma\0", 10) == 0)
      xml_context = XML_CELL_GAMMA;
    }
  i++;
  }
}

/*******************/
/* parse atom info */
/*******************/
void xml_parse_atom(const gchar **names, const gchar **values)
{
gint i, n;
gchar **buff;

g_assert(xml_model != NULL);

/* init structure */
if (!xml_core)
  {
  xml_core = new_core("X", xml_model);
  xml_model->cores = g_slist_prepend(xml_model->cores, xml_core);
  }

/* process attributes (if any) */
i=0;
while (*(names+i))
  {
  if (g_ascii_strncasecmp(*(names+i), "elementType\0", 12) == 0)
    {
/* FIXME - potential overflow bug here, since label is an array */
    strcpy(xml_core->label, *(values+i));
    template_core(xml_core->label, xml_core, xml_model);
    }
  if (g_ascii_strncasecmp(*(names+i), "xyzf", 4) == 0)
    {
    buff = tokenize(*(values+i), &n);
    if (n > 2)
      {
      xml_core->x[0] = str_to_float(*(buff+0));
      xml_core->x[1] = str_to_float(*(buff+1));
      xml_core->x[2] = str_to_float(*(buff+2));
      xml_model->fractional = TRUE;
      }
    }
  if (g_ascii_strncasecmp(*(names+i), "xf", 2) == 0)
    {
    xml_core->x[0] = str_to_float(*(values+i));
/* FIXME - inefficient to repeat this for all atoms */
/* but may be necessary if every atom is allowed to be cart or fract */
    xml_model->fractional = TRUE;
    }
  if (g_ascii_strncasecmp(*(names+i), "x3", 2) == 0)
    {
    xml_core->x[0] = str_to_float(*(values+i));
/* FIXME - inefficient to repeat this for all atoms */
/* but may be necessary if every atom is allowed to be cart or fract */
    xml_model->fractional = FALSE;
    }

  if (g_ascii_strncasecmp(*(names+i), "yf", 2) == 0 ||
      g_ascii_strncasecmp(*(names+i), "y3", 2) == 0) 
    {
    xml_core->x[1] = str_to_float(*(values+i));
    }
  if (g_ascii_strncasecmp(*(names+i), "zf", 2) == 0 ||
      g_ascii_strncasecmp(*(names+i), "z3", 2) == 0) 
    {
    xml_core->x[2] = str_to_float(*(values+i));
    }
  i++;
  }
}

/********************/
/* parse shell info */
/********************/
void xml_parse_shell(const gchar **names, const gchar **values)
{
gint i, n;
gchar **buff;

g_assert(xml_model != NULL);
g_assert(xml_core != NULL);

/* init structure */
if (!xml_shell)
  {
  xml_shell = new_shell(xml_core->label, xml_model);
  xml_model->shels = g_slist_prepend(xml_model->shels, xml_shell);
  }

/* process attributes (if any) */
i=0;
while (*(names+i))
  {
  if (g_ascii_strncasecmp(*(names+i), "xyzf", 4) == 0)
    {
    buff = tokenize(*(values+i), &n);
    if (n > 2)
      {
      xml_core->x[0] = str_to_float(*(buff+0));
      xml_core->x[1] = str_to_float(*(buff+1));
      xml_core->x[2] = str_to_float(*(buff+2));
      xml_model->fractional = TRUE;
      }
    }
  if (g_ascii_strncasecmp(*(names+i), "xf", 2) == 0)
    {
    xml_core->x[0] = str_to_float(*(values+i));
/* FIXME - inefficient to repeat this for all atoms */
/* but may be necessary if every atom is allowed to be cart or fract */
    xml_model->fractional = TRUE;
    }
  if (g_ascii_strncasecmp(*(names+i), "x3", 2) == 0)
    {
    xml_core->x[0] = str_to_float(*(values+i));
/* FIXME - inefficient to repeat this for all atoms */
/* but may be necessary if every atom is allowed to be cart or fract */
    xml_model->fractional = FALSE;
    }

  if (g_ascii_strncasecmp(*(names+i), "yf", 2) == 0 ||
      g_ascii_strncasecmp(*(names+i), "y3", 2) == 0) 
    {
    xml_core->x[1] = str_to_float(*(values+i));
    }
  if (g_ascii_strncasecmp(*(names+i), "zf", 2) == 0 ||
      g_ascii_strncasecmp(*(names+i), "z3", 2) == 0) 
    {
    xml_core->x[2] = str_to_float(*(values+i));
    }
  i++;
  }
}

/*******************/
/* spatial parsing */
/*******************/
struct spatial_pak *xml_spatial;

/************************/
/* create a new spatial */
/************************/
void xml_parse_spatial(const gchar **names, const gchar **values)
{
gint i;

g_assert(xml_model != NULL);

xml_spatial = g_malloc(sizeof(struct spatial_pak));
xml_spatial->type = -1;
xml_spatial->method = -1;
xml_spatial->data = NULL;
xml_model->spatial = g_slist_prepend(xml_model->spatial, xml_spatial);

i=0;
while (*(names+i))
  {
  if (g_ascii_strncasecmp(*(names+i), "type", 4) == 0)
    xml_spatial->type = str_to_float(*(values+i));

  if (g_ascii_strncasecmp(*(names+i), "method", 6) == 0)
    xml_spatial->method = str_to_float(*(values+i));

  i++;
  }
}

/********************/
/* populate spatial */
/********************/
void xml_parse_vertex(const gchar **names, const gchar **values)
{
gint i=0;
struct vec_pak *vec;

g_assert(xml_spatial != NULL);

vec = g_malloc(sizeof(struct vec_pak));
xml_spatial->data = g_slist_prepend(xml_spatial->data, vec);

while (*(names+i))
  {
  if (g_ascii_strncasecmp(*(names+i), "red", 3) == 0)
    vec->colour[0] = str_to_float(*(values+i));
  if (g_ascii_strncasecmp(*(names+i), "green", 5) == 0)
    vec->colour[1] = str_to_float(*(values+i));
  if (g_ascii_strncasecmp(*(names+i), "blue", 4) == 0)
    vec->colour[2] = str_to_float(*(values+i));

  if (g_ascii_strncasecmp(*(names+i), "x3", 2) == 0)
    vec->x[0] = str_to_float(*(values+i));
  if (g_ascii_strncasecmp(*(names+i), "y3", 2) == 0)
    vec->x[1] = str_to_float(*(values+i));
  if (g_ascii_strncasecmp(*(names+i), "z3", 2) == 0)
    vec->x[2] = str_to_float(*(values+i));

  if (g_ascii_strncasecmp(*(names+i), "nx", 2) == 0)
    vec->n[0] = str_to_float(*(values+i));
  if (g_ascii_strncasecmp(*(names+i), "ny", 2) == 0)
    vec->n[1] = str_to_float(*(values+i));
  if (g_ascii_strncasecmp(*(names+i), "nz", 2) == 0)
    vec->n[2] = str_to_float(*(values+i));

  i++;
  }
}

/**************************/
/* transformation parsing */
/**************************/
struct transform_pak *xml_transform=NULL;

/*******************************/
/* create a new transformation */
/*******************************/
void xml_parse_transform(const gchar **names, const gchar **values)
{
gint i=0;

g_assert(xml_model != NULL);

xml_transform = g_malloc(sizeof(struct transform_pak));
xml_model->transform_list = g_slist_append(xml_model->transform_list, xml_transform);

while (*(names+i))
  {
  if (g_ascii_strncasecmp(*(names+i), "id", 2) == 0)
    xml_transform->id = str_to_float(*(values+i));

  i++;
  }
}

/****************/
/* process text */
/****************/
void xml_parse_text(GMarkupParseContext *context,
                    const gchar *text,
                    gsize text_len,  
                    gpointer user_data,
                    GError **error)
{
gint i;
gchar **buff;

switch (xml_context)
  {
/* specific cell contexts */
  case XML_CELL_A:
    xml_model->pbc[0] = str_to_float(text);
/* revert to general cell value context */
    xml_context = XML_CELL;
    break;
  case XML_CELL_B:
    xml_model->pbc[1] = str_to_float(text);
/* revert to general cell value context */
    xml_context = XML_CELL;
    break;
  case XML_CELL_C:
    xml_model->pbc[2] = str_to_float(text);
/* revert to general cell value context */
    xml_context = XML_CELL;
    break;
  case XML_CELL_ALPHA:
    xml_model->pbc[3] = D2R * str_to_float(text);
/* revert to general cell value context */
    xml_context = XML_CELL;
    break;
  case XML_CELL_BETA:
    xml_model->pbc[4] = D2R * str_to_float(text);
/* revert to general cell value context */
    xml_context = XML_CELL;
    break;
  case XML_CELL_GAMMA:
    xml_model->pbc[5] = D2R * str_to_float(text);
/* revert to general cell value context */
    xml_context = XML_CELL;
    break;

  case XML_PICTURE:
    xml_model->picture_list = g_slist_append(xml_model->picture_list,
                                         g_strstrip(g_strdup(text)));
    break;

  case XML_MATRIX:
/* FIXME - bit of a hack here */
    if (xml_transform)
      {
      buff = g_strsplit(text, " ", 10);
      for (i=0 ; i<9 ; i++)
        xml_transform->matrix[i] = str_to_float(*(buff+i));
      g_strfreev(buff);
      }
    break;
 
  case XML_VECTOR:
/* FIXME - bit of a hack here */
    if (xml_transform)
      {
      buff = g_strsplit(text, " ", 4);
      for (i=0 ; i<3 ; i++)
        xml_transform->vector[i] = str_to_float(*(buff+i));
      g_strfreev(buff);
      }
    break;

  case XML_SCALAR:
/* FIXME - bit of a hack here */
    if (xml_transform)
      {
      buff = g_strsplit(text, " ", 2);
      xml_transform->scalar = str_to_float(*(buff+0));
      g_strfreev(buff);
      }
    break;
  }
}

/**************************/
/* element start callback */
/**************************/
void xml_start_element(GMarkupParseContext *context,
                       const gchar *element_name,
                       const gchar **attribute_names,
                       const gchar **attribute_values,
                       gpointer current_model,
                       GError **error)
{
struct model_pak *model;

model = (struct model_pak *) current_model;

/* context switching control */
if (g_ascii_strncasecmp(element_name, "system\0", 7) == 0)
  xml_context = XML_SYSTEM;
if (g_ascii_strncasecmp(element_name, "crystal\0", 8) == 0)
  xml_context = XML_CRYSTAL;
if (g_ascii_strncasecmp(element_name, "symmetry\0", 9) == 0)
  xml_context = XML_SYMMETRY;
if (g_ascii_strncasecmp(element_name, "atom\0", 5) == 0)
  xml_context = XML_CORE;
if (g_ascii_strncasecmp(element_name, "particle\0", 9) == 0)
  xml_context = XML_SHELL;
if (g_ascii_strncasecmp(element_name, "spatial\0", 8) == 0)
  xml_context = XML_SPATIAL;
if (g_ascii_strncasecmp(element_name, "vertex\0", 7) == 0)
  xml_context = XML_VERTEX;
if (g_ascii_strncasecmp(element_name, "picture\0", 8) == 0)
  xml_context = XML_PICTURE;
if (g_ascii_strncasecmp(element_name, "transform\0", 10) == 0)
  xml_context = XML_TRANSFORM;
if (g_ascii_strncasecmp(element_name, "matrix\0", 7) == 0)
  xml_context = XML_MATRIX;
if (g_ascii_strncasecmp(element_name, "vector\0", 7) == 0)
  xml_context = XML_VECTOR;
if (g_ascii_strncasecmp(element_name, "scalar\0", 7) == 0)
  xml_context = XML_SCALAR;

/* context attribute parsing */
switch (xml_context)
  {
  case XML_SYSTEM:
    xml_parse_system(attribute_names, attribute_values);
    return;

  case XML_CRYSTAL:
    xml_parse_fullcell(attribute_names, attribute_values);
    return;

  case XML_CELL:
    xml_parse_cell(attribute_names, attribute_values);
    return;

  case XML_CORE:
    xml_parse_atom(attribute_names, attribute_values);
    return;

  case XML_SHELL:
    xml_parse_shell(attribute_names, attribute_values);
    return;

  case XML_SYMMETRY:
    xml_parse_symmetry(attribute_names, attribute_values);
    return;

  case XML_SPATIAL:
    xml_parse_spatial(attribute_names, attribute_values);
    return;

  case XML_VERTEX:
    xml_parse_vertex(attribute_names, attribute_values);
    return;

  case XML_TRANSFORM:
    xml_parse_transform(attribute_names, attribute_values);
    return;
  }
}

/************************/
/* element end callback */
/************************/
void xml_end_element(GMarkupParseContext *context,
                     const gchar *element_name,
                     gpointer current_model,
                     GError **error)
{
/* reset parsing context and related data */
if (g_ascii_strncasecmp(element_name, "system\0", 7) == 0)
  {
  xml_model = NULL;
  xml_context = XML_INACTIVE;
  }
if (g_ascii_strncasecmp(element_name, "crystal\0", 8) == 0)
  {
  xml_context = XML_INACTIVE;
  }
if (g_ascii_strncasecmp(element_name, "atom\0", 5) == 0)
  {
  xml_core = NULL;
  xml_context = XML_INACTIVE;
  }
if (g_ascii_strncasecmp(element_name, "particle\0", 5) == 0)
  {
  xml_shell = NULL;
  xml_context = XML_CORE;
  }
if (g_ascii_strncasecmp(element_name, "spatial\0", 8) == 0)
  {
  xml_context = XML_INACTIVE;
  xml_spatial->data = g_slist_reverse(xml_spatial->data);
  xml_spatial = NULL;
  }
if (g_ascii_strncasecmp(element_name, "vertex\0", 7) == 0)
  xml_context = XML_SPATIAL;
if (g_ascii_strncasecmp(element_name, "picture\0", 8) == 0)
  xml_context = XML_INACTIVE;
if (g_ascii_strncasecmp(element_name, "transform\0", 10) == 0)
  {
  xml_model->num_frames++;
  xml_context = XML_INACTIVE;
  xml_transform = NULL;
  }

if (g_ascii_strncasecmp(element_name, "matrix\0", 7) == 0)
  xml_context = XML_INACTIVE;
if (g_ascii_strncasecmp(element_name, "vector\0", 7) == 0)
  xml_context = XML_INACTIVE;
if (g_ascii_strncasecmp(element_name, "scalar\0", 7) == 0)
  xml_context = XML_INACTIVE;
}

/******************************/
/* read a single model's data */
/******************************/
gint read_xml_frame(FILE *fp, struct model_pak *model)
{
gchar *line;
GMarkupParser xml_parser;
GMarkupParseContext *xml_context;
GError *error;

xml_system_table = g_hash_table_new_full(&g_str_hash, &hash_strcmp, &g_free, NULL);

/* TODO - think more about this... */
xml_model = model;

/* setup context parse (ie callbacks) */
xml_parser.start_element = &xml_start_element;
xml_parser.end_element = &xml_end_element;
xml_parser.text = &xml_parse_text;
xml_parser.passthrough = NULL;
xml_parser.error = NULL;
xml_context = g_markup_parse_context_new(&xml_parser, 0, model, NULL);

/* read in blocks (lines) of text */
line = file_read_line(fp);
while (line)
  {
/* parse the line */
  if (!g_markup_parse_context_parse(xml_context, line, strlen(line), &error))
    printf("read_xml() : parsing error.\n");

  g_free(line);
  line = file_read_line(fp);
  }

/* cleanup */
if (!g_markup_parse_context_end_parse(xml_context, &error))
  printf("read_xml() : errors occurred reading file.\n");

g_markup_parse_context_free(xml_context);
g_hash_table_destroy(xml_system_table);

return(0);
}

/*****************************/
/* the main reading function */
/*****************************/
#define DEBUG_READ_XML 0
gint read_xml(gchar *filename, struct model_pak *model)
{
gint i, n, num_models=0;
FILE *fp;

fp = fopen(filename, "rt");
if (!fp)
  return(1);

/* TODO - store current file position in frame list */
/* TODO - alloc new model pointers for the read */
read_xml_frame(fp, model);
num_models++; 

fclose(fp);

#if DEBUG_READ_XML
printf("read_xml() : found %d models.\n", num_models);
#endif

/*
n = sysenv.num_models;
for (i=0 ; i<num_models ; i++)
  {
  n--;
  model = model_ptr(n, RECALL);

#if DEBUG_READ_XML
printf("read_xml() : prep'ing model %p (%d)\n", model, n);
#endif

  prep_model(model);
  }
*/

prep_model(model);

return(0);
}

/*****************************/
/* the main writing function */
/*****************************/
gint write_xml(gchar *filename, struct model_pak *model)
{
gint i;
gdouble vec[3];
GSList *list, *list2;
struct core_pak *core;
struct vec_pak *v;
struct spatial_pak *spatial;
struct transform_pak *transform;
FILE *fp;

fp = fopen(filename, "wt");
if (!fp)
  return(1);

fprintf(fp, "<system>\n");

/* periodicity */
if (model->periodic)
  {
/* TODO - 1D */
  switch (model->periodic)
    {
    case 2:
      fprintf(fp, "  <crystal dictRef=\"gulp:surface\">\n");
      break;

    default:
      fprintf(fp, "  <crystal dictRef=\"gulp:fullcell\">\n");
      break;
    }
  fprintf(fp, "    <cell dictRef=\"cml:a\"> %f </cell>\n", model->pbc[0]);
  fprintf(fp, "    <cell dictRef=\"cml:b\"> %f </cell>\n", model->pbc[1]);
  fprintf(fp, "    <cell dictRef=\"cml:c\"> %f </cell>\n", model->pbc[2]);
  fprintf(fp, "    <cell dictRef=\"cml:alpha\"> %f </cell>\n",
                                           R2D*model->pbc[3]);
  fprintf(fp, "    <cell dictRef=\"cml:beta\"> %f </cell>\n",
                                          R2D*model->pbc[4]);
  fprintf(fp, "    <cell dictRef=\"cml:gamma\"> %f </cell>\n",
                                           R2D*model->pbc[5]);
  fprintf(fp, "  </crystal>\n");
  }

/* cores */
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  ARR3SET(vec, core->x);
  vecmat(model->latmat, vec);
  fprintf(fp, "  <atom elementType=\"%s\" x3=\"%f\" y3=\"%f\" z3=\"%f\">",
          core->label, vec[0], vec[1], vec[2]);
  fprintf(fp, "</atom>\n");
/* TODO - shells as particles */
  }

/* spatial data */
for (list=model->spatial ; list ; list=g_slist_next(list))
  {
  spatial = (struct spatial_pak *) list->data;

  fprintf(fp, "<spatial Type=\"%d\" Method=\"%d\">\n",
                      spatial->type, spatial->method);

  for (list2=spatial->data ; list2 ; list2=g_slist_next(list2))
    {
    v = (struct vec_pak *) list2->data;

    fprintf(fp, "  <vertex red=\"%f\" green=\"%f\" blue=\"%f\"",
                 v->colour[0], v->colour[1], v->colour[2]);
 
    fprintf(fp, " x3=\"%f\" y3=\"%f\" z3=\"%f\"",
                 v->x[0], v->x[1], v->x[2]);

    fprintf(fp, " nx=\"%f\" ny=\"%f\" nz=\"%f\"></vertex>\n",
                 v->n[0], v->n[1], v->n[2]);
    }

  fprintf(fp, "</spatial>\n");
  }

/* transform data */
for (list=model->transform_list ; list ; list=g_slist_next(list))
  {
  transform = (struct transform_pak *) list->data;

  fprintf(fp, "  <transform id=\"%d\">\n", transform->id);

  fprintf(fp, "    <matrix>");
  for (i=0 ; i<9 ; i++)
    fprintf(fp, "%f ", transform->matrix[i]);
  fprintf(fp, "</matrix>\n");

  fprintf(fp, "    <vector>");
  for (i=0 ; i<3 ; i++)
    fprintf(fp, "%f ", transform->vector[i]);
  fprintf(fp, "</vector>\n");

  fprintf(fp, "    <scalar>");
  fprintf(fp, "%f ", transform->scalar);
  fprintf(fp, "</scalar>\n");
  
  fprintf(fp, "</transform>\n");
  }


/* pictures */
for (list=model->picture_list ; list ; list=g_slist_next(list))
  fprintf(fp, "<picture> %s </picture>\n", (gchar *) list->data);


fprintf(fp, "</system>\n");

fclose(fp);
return(0);
}

