from DisplayTarget import DisplayTarget
from utils import svg
from utils.datatypes import *
from utils import Unit
from utils import dialog
from xml import sax

import gtk

try:
    import gnomevfs
except ImportError:
    import gnome.vfs as gnomevfs



#
# Target where sensors can draw on.
#
class TargetCanvas(DisplayTarget):

    def __init__(self, parent):

        # a mini DOM for accessing the SVG data
        self.__dom = None
        
        # the previous size of the widget; used to detect resizings
        self.__old_size = (0, 0)

        # the size of the image
        self.__image_size = (100, 100)


        DisplayTarget.__init__(self, parent)

        self.__widget = gtk.Image()
        self.__widget.connect("expose-event", self.__on_expose)
        self.__widget.show()

        # the "graphics" property is not readable because otherwise it could
        # be used to spy out files on the user's system after loading them into
        # "uri"
        self._register_property("graphics", TYPE_STRING,
                                self._setp_graphics, None)
        self._register_property("dom", TYPE_OBJECT,
                                None, self._getp_dom)
        self._register_property("uri", TYPE_STRING,
                                self._setp_uri, self._getp)

        self._setp("graphics", "")


    def get_widget(self): return self.__widget


    #
    # Detects size changes and redraws the image in that case.
    #
    def __on_expose(self, src, event):

        w, h = self.__widget.size_request()
        if ((w, h) != self.__old_size):
            self.__redraw()
            self.__old_size = (w, h)
        

    #
    # Transforms the given coordinates into buffer space.
    #
    def __transform_coords(self, x, y):

        width, height = self.get_geometry()[2:4]
        tx = (width.as_px() / 2.0)  * (1.0 + float(x))
        ty = (height.as_px() / 2.0) * (1.0 - float(y))

        return (tx, ty)


    def __make_style(self, foreground, fill):

        s = "stroke:" + foreground
        if (fill): s+= ";fill:" + foreground
        else: s+= ";fill:none"
        out = "style='%s'" % s
        return out


    #
    # Performs the given drawing operations. This is used for backwards
    # compatibility. New code should directly send SVG data.
    #
    def __draw_svg(self, commands):

        w, h = self.get_geometry()[2:4]
        out = "<svg width='%d' height='%d'>" % (w.as_px(), h.as_px())
        current_fg = "rgb(0, 0, 0)"
        for c in commands:
            if (not c.strip()): continue
            parts = c.split()

            cmd, args = parts[0], parts[1:]

            if (cmd == "color"):
                color = args[0]
                gdkcolor = gtk.gdk.color_parse(color)
                current_fg = "rgb(%d, %d, %d)" \
                 % (gdkcolor.red >> 8, gdkcolor.green >> 8, gdkcolor.blue >> 8)

            elif (cmd == "line"):
                x1, y1, x2, y2 = args
                x1, y1 = self.__transform_coords(x1, y1)
                x2, y2 = self.__transform_coords(x2, y2)
                style = self.__make_style(current_fg, False)
                out += "<line x1='%f' y1='%f' x2='%f' y2='%f' %s/>" \
                       % (x1, y1, x2, y2, style)

            elif (cmd == "polygon"):
                fill = int(args[-1])
                style = self.__make_style(current_fg, fill)
                points = [ self.__transform_coords(args[i], args[i+1])
                           for i in range(0,len(args)-1, 2) ]
                if (points): path = "M%f %f " % (points.pop(0))
                while (points):
                    path += "L%f %f " % (points.pop(0))
                out += "<path d='%s' %s/>" % (path, style)

            elif (cmd == "rectangle"):
                x1, y1, x2, y2, fill = args
                style = self.__make_style(current_fg, fill)
                x1, y1 = self.__transform_coords(x1, y1)
                x2, y2 = self.__transform_coords(x2, y2)
                w = x2 - x1
                h = y2 - y1
                out += "<rect x='%f' y='%f' width='%f' height='%f' %s/>" \
                       % (x, y, w, h, style)

            #end if
        #end for
        out += "</svg>"
        self.__render(out)


    #
    # Renders the given SVG data.
    #
    def __render(self, data):

        if (not data): return

        self.__dom = DOMBuilder(data).get_dom()
        if (not self.__dom): return
        
        self.__dom.set_update_handler(self.__redraw)

        # check if the SVG has dynamic or static size
        if (self.__dom.has_key("width") and self.__dom.has_key("height")):
            try:
                width = int(float(self.__dom["width"]))
                height = int(float(self.__dom["height"]))
                self.__image_size = (width, height)
            except:
                # FIXME: convert SVG units to gDesklets units in case units
                # are used
                pass

        self.__redraw()
        

    #
    # Redraws the canvas.
    #
    def __redraw(self):

        if (not self.__dom): return

        w, h = self.__widget.size_request()
        imgw, imgh = self.__image_size
        
        # crappy SVG needs the size to be given; just set it here so that it
        # dynamically takes the correct size
        self.__dom["width"] = str(w)
        self.__dom["height"] = str(h)

        # so that's why the XML parser inserted an empty <g> node... :)
        g = self.__dom.get_children()[0]
        g["transform"] = "scale(%f, %f)" % (float(w) / imgw,
                                            float(h) / imgh)

        svg.render(self.__widget, w, h, str(self.__dom))
        

    #
    # "graphics" property.
    #
    def _setp_graphics(self, key, value):

        # native SVG
        if (value and value.lstrip()[0] == "<"):
            self.__render(value)

        # legacy graphics language
        else:
            value = value.split(",")
            self.__draw_svg(value)

        self._setp(key, value)


    #
    # Returns the DOM object of the graphics.
    #
    def _getp_dom(self, key): return self.__dom


    #
    # Loads SVG from the given URI.
    #
    def _setp_uri(self, key, value):

        try:
            uri = self._get_display().get_full_path(value)
            data = gnomevfs.read_entire_file(uri)
        except gnomevfs.Error:
            import sys
            print >>sys.stderr, "Could not open image %s." % (uri)
            return

        self.__render(data)
        self._setp(key, value)



#
# Class for nodes of the DOM tree.
#
class DOMNode:

    def __init__(self, name):

        self.__name = name
        self.__attrs = {}
        self.__children = []
        self.__update_handler = None
        self.__id_table = {}


    #
    # Sets the ID table for quickly accessing a node by its ID.
    #
    def set_id_table(self, table): self.__id_table = table


    #
    # Returns whether the given key exists.
    #
    def has_key(self, key): return (key in self.__attrs)


    #
    # Returns the node with the given ID.
    #
    def get(self, ident):

        try:
            return self.__id_table[ident]
        except KeyError:
            dialog.warning(_("No such element: %s") % ident,
                           _("The element with ID <b>%s</b> does not "
                             "exist in the SVG image.") % ident,
                           False)
            return None


    #
    # Sets the handler for updating the tree.
    #
    def set_update_handler(self, handler):

        self.__update_handler = handler

    #
    # Updates the tree.
    #
    def update(self):

        if (self.__update_handler): self.__update_handler()
        

    def add_child(self, child): self.__children.append(child)
    def get_children(self): return self.__children[:]

    def __getitem__(self, key):

        try:
            return self.__attrs[key]
        except KeyError:
            dialog.warning(_("No such property: %s") % key,
                           _("The SVG element <b>%s</b> does not have "
                             "the <b>%s</b> property.") % (self.__name, key),
                           False)
            return None
        
    def __setitem__(self, key, value): self.__attrs[key] = value


    #
    # Returns a XML representation of this node.
    #
    def __str__(self):

        attrs = [ "%s=\"%s\"" % (k, v) for k, v in self.__attrs.items() ]
        attrs = " ".join(attrs)
        children = [ str(c) for c in self.get_children() ]
        children = "\n".join(children)
        contents = self.__attrs.get("PCDATA", "")
        contents += children
        if (contents):
            out = "<%s %s>%s</%s>" % (self.__name, attrs, contents, self.__name)
        else:
            out = "<%s %s/>" % (self.__name, attrs)

        return out
    

#
# Class for building up a DOM tree by parsing XML.
#
class DOMBuilder(sax.handler.ContentHandler):

    def __init__(self, xml):

        self.__dom = None
        self.__node_stack = []
        self.__current_node = None
        self.__current_chars = ""
        self.__id_table = {}

        sax.handler.ContentHandler.__init__(self)

        try:
            sax.parseString(xml, self)
        except sax._exceptions.SAXParseException, e:
            import traceback; traceback.print_exc()
            import sys; print >>sys.stderr, "Parse Error: %s" % (e)
            # abort if a parse error occured
            from utils import dialog
            dialog.warning("Invalid XML",
                           "XML parse error in line %d, column %d: %s"
                      % (e.getLineNumber(), e.getColumnNumber(), e.getMessage()))
            return
        except:
            import traceback; traceback.print_exc()
            return

        self.__dom.set_id_table(self.__id_table)


    def get_dom(self): return self.__dom
    

    def startElement(self, name, attrs):

        parent = self.__current_node
        self.__current_node = DOMNode(name)
        self.__current_chars = ""
        for key, value in attrs.items():
            self.__current_node[key] = value

        if (parent): parent.add_child(self.__current_node)
        else:
            # we insert an empty <g> node here; it can be used later if needed
            self.__dom = self.__current_node
            new_node = DOMNode("g")
            self.__dom.add_child(new_node)
            self.__current_node = new_node

        if ("id" in attrs.keys()):
            self.__id_table[attrs["id"]] = self.__current_node

        self.__node_stack.append(self.__current_node)
        

    def endElement(self, name):

        if (self.__current_chars):
            self.__current_node["PCDATA"] = self.__current_chars

        self.__node_stack.pop(-1)
        if (self.__node_stack):
            self.__current_node = self.__node_stack[-1]


    def characters(self, content):

        self.__current_chars += content
