#ifndef CLIENTS_H
#define CLIENTS_H

/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdint.h>
#include <sys/types.h>

#include <abz/aton.h>

#include "stats.h"

struct bandwidth
{
   uint64_t rate;
   uint64_t peak;
};

struct counter
{
   struct network network;
   unsigned int inrule,outrule;
   struct counter *next;
};

struct client
{
   char *name;
   struct bandwidth input,output;
   struct counter *counter;
   struct stats stats;
   struct client *next;
};

/*
 * Initialize client counters. Returns 0 if successful, -1 if some
 * error occurred. You can check which error occurred by calling
 * abz_get_error().
 */
extern int client_create (struct client **clients,const char *table,const char *chain);

/*
 * Destroy client counters.
 */
extern void client_destroy (struct client **clients);

/*
 * Add a client. Return 0 if successful, -1 if some error occurred.
 * You can check which error occurred by calling abz_get_error().
 */
extern int client_add (struct client **clients,
					   const char *name,
					   const struct bandwidth *input,
					   const struct bandwidth *output,
					   const struct network *network,
					   size_t n);

/*
 * Commit changes and sort list of clients alphabetically. Return 0
 * if successful, -1 if some error occurred. You can check which error
 * occurred by calling abz_get_error().
 */
extern int client_commit (struct client **clients);

/*
 * Update client counters. This function should be called at least
 * once per second. Returns 0 if successful, -1 if some error occurred.
 */
extern int client_update (struct client *clients);

/*
 * Calculate average rate over the specified period. Return 0 if
 * successful, -1 if some error occurred. You can check which error
 * occurred by calling abz_get_error().
 */
extern int client_calc_rate (struct rate *avg,const struct client *client,time_t period);

/*
 * Reset client counters. Returns 0 if successful, -1 if some error
 * occurred by calling abz_get_error().
 */
extern int client_reset (struct client *clients);

#endif	/* #ifndef CLIENTS_H */
