//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BoundaryExpression.hpp,v 1.4 2004/04/16 16:25:41 delpinux Exp $

#ifndef BOUNDARY_EXPRESSION_HPP
#define BOUNDARY_EXPRESSION_HPP

#include <Expression.hpp>

#include <Vector3Expression.hpp>
#include <Variable.hpp>

#include <set>
#include <list>

/*!
  \class BoundaryExpression

  This class defines the base class of Boundary expressions (which
  are used for boundary boundarys)

  \author Stephane Del Pino
*/
class Boundary;
class BoundaryExpression
  : public Expression
{
public:
  enum BoundaryType {
    undefined,
    surfaceMesh,
    povray,
    references,
    list,
    variable
  };

private:
  BoundaryExpression::BoundaryType __boundaryType;

protected:
  ReferenceCounting<Boundary> __boundary;

public:
  const BoundaryExpression::BoundaryType& boundaryType() const
  {
    return __boundaryType;
  }

  ReferenceCounting<Boundary> boundary();

  BoundaryExpression(const BoundaryExpression& e);

  BoundaryExpression(const BoundaryExpression::BoundaryType& t);

  virtual ~BoundaryExpression();
};

/*!
  \class BoundaryExpressionSurfaceMesh

  This class defines the class of Surface Mesh Boundary expressions.

  \author Stephane Del Pino
 */
class BoundaryExpressionSurfaceMesh
  : public BoundaryExpression
{
private:
  ReferenceCounting<MeshExpression> __surfaceMeshExpression;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  BoundaryExpressionSurfaceMesh(ReferenceCounting<MeshExpression> m);

  BoundaryExpressionSurfaceMesh(const BoundaryExpressionSurfaceMesh& m);

  ~BoundaryExpressionSurfaceMesh();
};

/*!
  \class BoundaryExpressionPOVRay

  This class defines the class of POVRay Boundary expressions.

  \author Stephane Del Pino
 */
class BoundaryExpressionPOVRay
  : public BoundaryExpression
{
private:
  ReferenceCounting<Vector3Expression> __povrayReference;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  BoundaryExpressionPOVRay(ReferenceCounting<Vector3Expression> r)
    : BoundaryExpression(BoundaryExpression::povray),
      __povrayReference(r)
  {
    ;
  }

  BoundaryExpressionPOVRay(const BoundaryExpressionPOVRay& r)
    : BoundaryExpression(r),
      __povrayReference(r.__povrayReference)
  {
    ;
  }

  ~BoundaryExpressionPOVRay()
  {
    ;
  }
};

/*!
  \class BoundaryExpressionReferences

  This class defines the class of Referenced Boundary expressions.

  \author Stephane Del Pino
 */
class BoundaryExpressionReferences
  : public BoundaryExpression
{
public:
  typedef std::set<ReferenceCounting<RealExpression> > ReferencesSet;

private:
  ReferencesSet __references;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  void add(ReferenceCounting<RealExpression> reference)
  {
    __references.insert(reference);
  }

  BoundaryExpressionReferences()
    : BoundaryExpression(BoundaryExpression::references)
  {
    ;
  }

  BoundaryExpressionReferences(BoundaryExpressionReferences& r)
    : BoundaryExpression(r),
      __references(r.__references)
  {
    ;
  }

  ~BoundaryExpressionReferences()
  {
    ;
  }
};

/*!
  \class BoundaryExpressionReferences

  This class defines the class of Referenced Boundary expressions.

  \author Stephane Del Pino
 */
class BoundaryExpressionList
  : public BoundaryExpression
{
public:
  typedef std::list<ReferenceCounting<BoundaryExpression> > List;

private:
  List __boundaries;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  void add(ReferenceCounting<BoundaryExpression> boundary)
  {
    __boundaries.push_back(boundary);
  }

  BoundaryExpressionList()
    : BoundaryExpression(BoundaryExpression::list)
  {
    ;
  }

  BoundaryExpressionList(BoundaryExpressionList& l)
    : BoundaryExpression(l),
      __boundaries(l.__boundaries)
  {
    ;
  }

  ~BoundaryExpressionList()
  {
    ;
  }
};

#endif // BOUNDARY_EXPRESSION_HPP

