//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BiConjugateGradientStabilized.hpp,v 1.1.1.1 2003/02/17 16:32:51 delpinux Exp $

#ifndef _BI_CONJUGATE_GRADIENT_STAB_HPP_
#define _BI_CONJUGATE_GRADIENT_STAB_HPP_

/*!
  \class BiConjugateGradientStabilized
  Bi-Conjugate Gradient Stabilized.

  \par Resolve the linear system \f$ Au=b \f$ using the preconditionner P.

  \author Stphane Del Pino.
*/

#include <BiConjugateGradientStabilizedOptions.hpp>
#include <GetParameter.hpp>

class BiConjugateGradientStabilized
{
 private:
  real_t epsilon;
  int    max_iter;

  GetParameter<BiConjugateGradientStabilizedOptions> __options;

 public:
  template <typename VectorType,
	    typename MatrixType,
	    typename PreconditionerType>
  BiConjugateGradientStabilized(const VectorType& b,
				const MatrixType& A, 
				const PreconditionerType& P,
				VectorType& x)
  {
    epsilon = __options.value().epsilon();
    max_iter = __options.value().maxiter();

    VectorType r_k_1(b.size());
    A.timesX(x,r_k_1);
    r_k_1 = b - r_k_1;

    real_t residu = Norm(r_k_1);

    if(residu == 0)
      return;

    real_t resid0 = residu;

    VectorType rTilda_0 = r_k_1;
    VectorType p_k = r_k_1;

    VectorType s_k(x.size());

    VectorType Ap_k(x.size());
    VectorType As_k(x.size());

    VectorType r_k(x.size());

    ffout(2) << "initial residu: " << resid0 << '\n';
    for (int i=1; i<= max_iter; ++i) {
      ffout(3) << "biCGstab iteration: " << i << " relative residu: "
	       << residu/resid0 << '\n';
      A.timesX(p_k,Ap_k);
      real_t alpha_k = (r_k_1*rTilda_0) / (Ap_k*rTilda_0);

      s_k = r_k_1 - alpha_k*Ap_k;
      A.timesX(s_k,As_k);

      real_t w_k = (As_k * s_k) / (As_k * As_k);

      x += alpha_k * p_k + w_k * s_k;
      r_k = s_k - (w_k * As_k);

      real_t beta_k = (r_k * rTilda_0)/(r_k_1 * rTilda_0) * (alpha_k / w_k);

      p_k -= w_k*Ap_k;
      p_k *= beta_k;
      p_k += r_k;

      if ((residu = (Norm(r_k)))/resid0<epsilon) {
	if (i==1) {
	  resid0=residu;
	} else {
	  ffout(2) << "leaving biCGstab:\n";
	  ffout(2) << "\tresidu = " << residu << '\n';
	  ffout(2) << "\tepsilon= " << epsilon << '\n';
	  return;
	}
      }

      r_k_1 = r_k;
    }

    fferr(2) << "### NOT CONVERGED ###\nleaving biCGstab:\n";
    fferr(2) << "\tresidu      = " << residu << '\n';
    fferr(2) << "\tepsilon     = " << epsilon << '\n';
  }

};
  
#endif // _BI_CONJUGATE_GRADIENT_STAB_HPP_

