/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-smime.c,v 1.6 2003/12/10 23:16:45 hoa Exp $
 */

#include "etpan-cfg-smime.h"
#include "etpan-cfg-common.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-errors.h"
#include "etpan-db-read.h"

struct etpan_smime_config *
etpan_smime_config_new(char * cert_dir, char * CA_dir,
    char * private_keys_dir, int CA_check, int auto_extract_cert)
{
  struct etpan_smime_config * config;

  config = malloc(sizeof(struct etpan_smime_config));
  if (config == NULL)
    goto err;

  config->cert_dir = cert_dir;
  config->CA_dir = CA_dir;
  config->private_keys_dir = private_keys_dir;
  config->CA_check = CA_check;
  config->auto_extract_cert = auto_extract_cert;
  
  return config;

 err:
  return NULL;
}

void etpan_smime_config_free(struct etpan_smime_config * config)
{
  if (config->private_keys_dir != NULL)
    free(config->private_keys_dir);
  if (config->cert_dir != NULL)
    free(config->cert_dir);
  if (config->CA_dir != NULL)
    free(config->CA_dir);
  free(config);
}

int etpan_smime_config_read(char * filename,
    struct etpan_smime_config ** result)
{
  struct etpan_db * db;
  int res;
  struct etpan_smime_config * smime_config;
  int r;
  unsigned int i;
  char * CA_dir;
  char * cert_dir;
  char * private_keys_dir;
  int CA_check;
  int auto_extract_cert;
  
  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  CA_dir = NULL;
  cert_dir = NULL;
  private_keys_dir = NULL;
  CA_check = 1;
  auto_extract_cert = 1;
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    char * new_val;
    chash * entry;
    
    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    new_val = etpan_db_entry_get_value(entry, "CA-dir");
    if (new_val != NULL)
      CA_dir = new_val;

    new_val = etpan_db_entry_get_value(entry, "cert-dir");
    if (new_val != NULL)
      cert_dir = new_val;

    new_val = etpan_db_entry_get_value(entry, "private-keys-dir");
    if (new_val != NULL)
      private_keys_dir = new_val;

    new_val = etpan_db_entry_get_value(entry, "CA-check");
    if (new_val != NULL)
      CA_check = strtoul(new_val, NULL, 10);

    new_val = etpan_db_entry_get_value(entry, "auto-extract-cert");
    if (new_val != NULL)
      auto_extract_cert = strtoul(new_val, NULL, 10);
  }
  
  r = etpan_cfg_strdup(&CA_dir, CA_dir);
  if (r != NO_ERROR) {
    res = r;
    goto free_db;
  }

  etpan_cfg_strdup(&cert_dir, cert_dir);
  if (r != NO_ERROR) {
    res = r;
    goto free_CA_dir;
  }

  etpan_cfg_strdup(&private_keys_dir, private_keys_dir);
  if (r != NO_ERROR) {
    res = r;
    goto free_cert_dir;
  }
  
  smime_config = etpan_smime_config_new(cert_dir, CA_dir,
      private_keys_dir, CA_check, auto_extract_cert);
  if (smime_config == NULL) {
    res = ERROR_MEMORY;
    goto free_private_keys_dir;
  }
  
  * result = smime_config;
  
  etpan_db_free(db);

  return NO_ERROR;

 free_private_keys_dir:
  free(private_keys_dir);
 free_cert_dir:
  free(cert_dir);
 free_CA_dir:
  free(CA_dir);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}
