/*-------------------------------------------------------------------------
Tab.java -- Symbol Table Management
Compiler Generator Coco/R,
Copyright (c) 1990, 2004 Hanspeter Moessenboeck, University of Linz
extended by M. Loeberbauer & A. Woess, Univ. of Linz
ported from C# to Java by Wolfgang Ahorner
with improvements by Pat Terry, Rhodes University

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

As an exception, it is allowed to write an extension of Coco/R that is
used as a plugin in non-free software.

If not otherwise stated, any source code generated by Coco/R (other than
Coco/R itself) does not fall under the GNU General Public License.
------------------------------------------------------------------------*/

package Coco;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.TreeMap;


class Position { // position of source code stretch (e.g. semantic action, resolver expressions)
	public int beg;      // start relative to the beginning of the file
	public int len;      // length of stretch
	public int col;      // column number of start position

	public Position(int beg, int len, int col) {
		this.beg = beg; this.len = len; this.col = col;
	}
}

class SymInfo { // output attribute of symbols in the ATG
	String name;
	int kind;		// 0 = ident, 1 = string
}

//---------------------------------------------------------------------
// Symbols
//---------------------------------------------------------------------

class Symbol implements Comparable {
	public static ArrayList terminals = new ArrayList();
	public static ArrayList pragmas = new ArrayList();
	public static ArrayList nonterminals = new ArrayList();

	// token kinds
	public static final int fixedToken    = 0; // e.g. 'a' ('b' | 'c') (structure of literals)
	public static final int classToken    = 1; // e.g. digit {digit}   (at least one char class)
	public static final int litToken      = 2; // e.g. "while"
	public static final int classLitToken = 3; // e.g. letter {letter} but without literals that have the same structure

	public int      n;           // symbol number
	public int      typ;         // t, nt, pr, unknown, rslv /* ML 29_11_2002 slv added */ /* AW slv --> rslv */
	public String   name;        // symbol name
	public Node     graph;       // nt: to first node of syntax graph
	public int      tokenKind;   // t:  token kind (fixedToken, classToken, ...)
	public boolean  deletable;   // nt: true if nonterminal is deletable
	public boolean  firstReady;  // nt: true if terminal start symbols have already been computed
	public BitSet   first;       // nt: terminal start symbols
	public BitSet   follow;      // nt: terminal followers
	public BitSet   nts;         // nt: nonterminals whose followers have to be added to this sym
	public int      line;        // source text line number of item in this node
	public Position attrPos;     // nt: position of attributes in source text (or null)
	public Position semPos;      // pr: pos of semantic action in source text (or null)
															 // nt: pos of local declarations in source text (or null)
	public String   retType;     // AH - nt: Type of output attribute (or null)
	public String   retVar;      // AH - nt: Name of output attribute (or null)

	public Symbol(int typ, String name, int line) {
		if (name.length() == 2 && name.charAt(0) == '"') {
			Parser.SemErr("empty token not allowed"); name = "???";
		}
		this.typ = typ; this.name = name; this.line = line;
		switch (typ) {
			case Node.t:  n = terminals.size(); terminals.add(this); break;
			case Node.pr: pragmas.add(this); break;
			case Node.nt: n = nonterminals.size(); nonterminals.add(this); break;
		}
	}

	public static Symbol Find(String name) {
		Symbol s;
		//foreach (Symbol s in terminals)
		for (int i = 0; i < terminals.size(); i++) {
			s = (Symbol)terminals.get(i);
			if (s.name.compareTo(name) == 0) return s;
		}
		//foreach (Symbol s in nonterminals)
		for (int i = 0; i < nonterminals.size(); i++) {
			s = (Symbol)nonterminals.get(i);
			if (s.name.compareTo(name) == 0) return s;
		}
		return null;
	}

	public int compareTo(Object x) {
		return name.compareTo(((Symbol)x).name);
	}

}


//---------------------------------------------------------------------
// Syntax graph (class Node, class Graph)
//---------------------------------------------------------------------

class Node {
	public static ArrayList nodes = new ArrayList();
	public static String[] nTyp =
	{"    ", "t   ", "pr  ", "nt  ", "clas", "chr ", "wt  ", "any ", "eps ",  /* AW 03-01-14 nTyp[0]: " " --> "    " */
	"sync", "sem ", "alt ", "iter", "opt ", "rslv"};

	// constants for node kinds
	public static final int t    =  1;  // terminal symbol
	public static final int pr   =  2;  // pragma
	public static final int nt   =  3;  // nonterminal symbol
	public static final int clas =  4;  // character class
	public static final int chr  =  5;  // character
	public static final int wt   =  6;  // weak terminal symbol
	public static final int any  =  7;  //
	public static final int eps  =  8;  // empty
	public static final int sync =  9;  // synchronization symbol
	public static final int sem  = 10;  // semantic action: (. .)
	public static final int alt  = 11;  // alternative: |
	public static final int iter = 12;  // iteration: { }
	public static final int opt  = 13;  // option: [ ]
	public static final int rslv = 14;  // resolver expr  /* ML */ /* AW 03-01-13 renamed slv --> rslv */

	public static final int normalTrans  = 0;		// transition codes
	public static final int contextTrans = 1;

	public int      n;				// node number
	public int      typ;			// t, nt, wt, chr, clas, any, eps, sem, sync, alt, iter, opt, rslv
	public Node     next;			// to successor node
	public Node     down;			// alt: to next alternative
	public Node     sub;			// alt, iter, opt: to first node of substructure
	public boolean  up;				// true: "next" leads to successor in enclosing structure
	public Symbol   sym;			// nt, t, wt: symbol represented by this node
	public int      val;			// chr:  ordinal character value
														// clas: index of character class
	public int      code;			// chr, clas: transition code
	public BitSet set;				// any, sync: the set represented by this node
	public Position pos;			// nt, t, wt: pos of actual attributes
														// sem:       pos of semantic action in source text
	public int      line;			// source text line number of item in this node
	public State    state;		// DFA state corresponding to this node
														// (only used in DFA.ConvertToStates)
	public String retVar;			// AH 20040206 - nt: name of output attribute (or null)

	public Node(int typ, Symbol sym, int line) {
		this.typ = typ; this.sym = sym; this.line = line;
		n = nodes.size();
		nodes.add(this);
	}

	public Node(int typ, Node sub) {
		this(typ, null, 0);
		this.sub = sub;
	}

	public Node(int typ, int val, int line) {
		this(typ, null, line);
		this.val = val;
	}

	public static boolean DelGraph(Node p) {
		return p == null || DelNode(p) && DelGraph(p.next);
	}

	public static boolean DelSubGraph(Node p) {
		return p == null || DelNode(p) && (p.up || DelSubGraph(p.next));
	}

	public static boolean DelAlt(Node p) {
		return p == null || DelNode(p) && (p.up || DelAlt(p.next));
	}

	public static boolean DelNode(Node p) {
		if (p.typ == nt) return p.sym.deletable;
		else if (p.typ == alt) return DelAlt(p.sub) || p.down != null && DelAlt(p.down);
		else return p.typ == eps || p.typ == iter || p.typ == opt || p.typ == sem
			|| p.typ == sync || p.typ == rslv;
	}

	//----------------- for printing ----------------------

	static int Ptr(Node p, boolean up) {
		if (p == null) return 0;
		else if (up) return -p.n;
		else return p.n;
	}

	static String Pos(Position pos) {
		if (pos == null) return "     ";
		else return Trace.formatString(Integer.toString(pos.beg), 5);
	}

	public static String Name(String name) {
		return (name + "           ").substring(0, 12);
		// found no simpler way to get the first 12 characters of the name
		// padded with blanks on the right
	}

	public static void PrintNodes() {
		Trace.WriteLine("Graph nodes:");
		Trace.WriteLine("----------------------------------------------------");
		Trace.WriteLine("   n type name          next  down   sub   pos  line");
		Trace.WriteLine("                               val  code");
		Trace.WriteLine("----------------------------------------------------");
		//foreach (Node p in nodes) {
		Node p;
		for (int i = 0; i < nodes.size(); i++) {
			p = (Node)nodes.get(i);
			Trace.Write(Integer.toString(p.n), 4);
			Trace.Write(" " + nTyp[p.typ] + " ");
			if (p.sym != null) {
				Trace.Write(Name(p.sym.name), 12);
				Trace.Write(" ");
			} else if (p.typ == Node.clas) {
				CharClass c = (CharClass)CharClass.classes.get(p.val);
				Trace.Write(Name(c.name), 12);
				Trace.Write(" ");
			} else Trace.Write("             ");
			Trace.Write(Integer.toString(Ptr(p.next, p.up)), 5);
			Trace.Write(" ");
			switch (p.typ) {
				case t: case nt: case wt:
					Trace.Write("             ");
					Trace.Write(Pos(p.pos), 5);
					break;
				case chr:
					Trace.Write(Integer.toString(p.val), 5);
					Trace.Write(" ");
					Trace.Write(Integer.toString(p.code), 5);
					Trace.Write("       ");
					break;
				case clas:
					Trace.Write("      ");
					Trace.Write(Integer.toString(p.code), 5);
					Trace.Write("       ");
					break;
				case alt: case iter: case opt:
					Trace.Write(Integer.toString(Ptr(p.down, false)), 5);
					Trace.Write(" ");
					Trace.Write(Integer.toString(Ptr(p.sub, false)), 5);
					Trace.Write("       ");
					break;
				case sem:
					Trace.Write("             ");
					Trace.Write(Pos(p.pos), 5);
					break;
				case eps: case any: case sync:
					Trace.Write("                  "); break;
			}
			Trace.WriteLine(Integer.toString(p.line), 5);
		}
		Trace.WriteLine();
	}

}


class Graph {
	static Node dummyNode = new Node(Node.eps, null, 0);

	public Node l;	// left end of graph = head
	public Node r;	// right end of graph = list of nodes to be linked to successor graph

	public Graph() {}
	public Graph(Node p) {
		l = p; r = p;
	}

	public static void MakeFirstAlt(Graph g) {
		g.l = new Node(Node.alt, g.l); g.l.line = g.l.sub.line; /* AW 2002-03-07 make line available for error handling */
		g.l.next = g.r;
		g.r = g.l;
	}

	public static void MakeAlternative(Graph g1, Graph g2) {
		g2.l = new Node(Node.alt, g2.l); g2.l.line = g2.l.sub.line;
		Node p = g1.l; while (p.down != null) p = p.down;
		p.down = g2.l;
		p = g1.r; while (p.next != null) p = p.next;
		p.next = g2.r;
	}

	public static void MakeSequence(Graph g1, Graph g2) {
		Node p = g1.r.next; g1.r.next = g2.l; // link head node
		while (p != null) {  // link substructure
			Node q = p.next; p.next = g2.l; p.up = true;
			p = q;
		}
		g1.r = g2.r;
	}

	public static void MakeIteration(Graph g) {
		g.l = new Node(Node.iter, g.l);
		Node p = g.r;
		g.r = g.l;
		while (p != null) {
			Node q = p.next; p.next = g.l; p.up = true;
			p = q;
		}
	}

	public static void MakeOption(Graph g) {
		g.l = new Node(Node.opt, g.l);
		g.l.next = g.r;
		g.r = g.l;
	}

	public static void Finish(Graph g) {
		Node p = g.r;
		while (p != null) {
			Node q = p.next; p.next = null; p = q;
		}
	}

	public static void SetContextTrans(Node p) { // set transition code in the graph rooted at p
		DFA.hasCtxMoves = true;
		while (p != null) {
			if (p.typ == Node.chr || p.typ == Node.clas) {
				p.code = Node.contextTrans;
			} else if (p.typ == Node.opt || p.typ == Node.iter) {
				SetContextTrans(p.sub);
			} else if (p.typ == Node.alt) {
				SetContextTrans(p.sub); SetContextTrans(p.down);
			}
			if (p.up) break;
			p = p.next;
		}
	}

	public static void DeleteNodes() {
		Node.nodes = new ArrayList();
		dummyNode = new Node(Node.eps, null, 0);
	}

	public static Graph StrToGraph(String str) {
		String s = DFA.Unescape(str.substring(1, str.length()-1));
		if (s.length() == 0) Parser.SemErr("empty token not allowed");
		Graph g = new Graph();
		g.r = dummyNode;
		for (int i = 0; i < s.length(); i++) {
			Node p = new Node(Node.chr, (int)s.charAt(i), 0);
			g.r.next = p; g.r = p;
		}
		g.l = dummyNode.next; dummyNode.next = null;
		return g;
	}

}


//----------------------------------------------------------------
// Bit sets
//----------------------------------------------------------------

class Sets {

	public static int First(BitSet s) {
		int max = s.size();
		for (int i=0; i<max; i++)
			if (s.get(i)) return i;
		return -1;
	}

	public static int Elements(BitSet s) {
		int max = s.size();
		int n = 0;
		for (int i=0; i<max; i++)
			if (s.get(i)) n++;
		return n;
	}

	public static boolean Equals(BitSet a, BitSet b) {
		int max = a.size();
		for (int i=0; i<max; i++)
			if (a.get(i) != b.get(i)) return false;
		return true;
	}

	public static boolean Includes(BitSet a, BitSet b) {	// a > b ?
		int max = a.size();
		for (int i=0; i<max; i++)
			if (b.get(i) && ! a.get(i)) return false;
		return true;
	}

	public static boolean Intersect(BitSet a, BitSet b) { // a * b != {}
		int max = a.size();
		for (int i=0; i<max; i++)
			if (a.get(i) && b.get(i)) return true;
		return false;
	}

	public static void Subtract(BitSet a, BitSet b) { // a = a - b
		BitSet c = (BitSet)b.clone();
		//a.and(c.not());
		c.flip(0, c.size());	// c.not
		a.and(c);
	}

	public static void PrintSet(BitSet s, int indent) {
		int col, len;
		Symbol sym;

		ArrayList t = Symbol.terminals;
		int sz = t.size();
		col = indent;

		//foreach (Symbol sym in Symbol.terminals) {
		for (int i = 0; i < sz; i++) {
			sym = (Symbol)t.get(i);
			if (s.get(sym.n)) {
				len = sym.name.length();
				if (col + len >= 80) {
					Trace.WriteLine();
					for (col = 1; col < indent; col++) Trace.Write(" ");
				}
				Trace.Write(sym.name + " ");
				col += len + 1;
			}
		}
		if (col == indent) Trace.Write("-- empty set --");
		Trace.WriteLine();
	}

}


//---------------------------------------------------------------------
// Character class management
//---------------------------------------------------------------------

class CharClass {
	public static ArrayList classes = new ArrayList();
	public static int dummyName = 'A';

	public static final int charSetSize = 256;  // must be a multiple of 16

	public int n;       	// class number
	public String name;		// class name
	public BitSet set;		// set representing the class

	public CharClass(String name, BitSet s) {
		if (name.compareTo("#") == 0) name = "#" + (char)dummyName++;
		this.n = classes.size(); this.name = name; this.set = s;
		classes.add(this);
	}

	public static CharClass Find(String name) {
		//foreach (CharClass c in classes)
		CharClass c;
		for (int i = 0; i < classes.size(); i++) {
			c = (CharClass)classes.get(i);
			if (c.name.compareTo(name) == 0) return c;
		}
		return null;
	}

	public static CharClass Find(BitSet s) {
		//foreach (CharClass c in classes)
		CharClass c;
		for (int i = 0; i < classes.size(); i++) {
			c = (CharClass)classes.get(i);
			if (Sets.Equals(s, c.set)) return c;
		}
		return null;
	}

	public static BitSet Set(int i) {
		return ((CharClass)classes.get(i)).set;
	}

	static String Ch(int ch) {
		if (ch < ' ' || ch >= 127 || ch == '\'' || ch == '\\') return Integer.toString(ch);
		else return ("'" + (char)ch + "'");
	}

	static void WriteCharSet(BitSet s) {
		int i = 0, len = s.size();
		while (i < len) {
			while (i < len && !s.get(i)) i++;
			if (i == len) break;
			int j = i;
			while (i < len && s.get(i)) i++;
			if (j < i-1) Trace.Write(Ch(j) + ".." + Ch(i-1) + " ");
			else Trace.Write(Ch(j) + " ");
		}
	}

	public static void WriteClasses () {
		//foreach (CharClass c in classes) {
		CharClass c;
		for (int i = 0; i < classes.size(); i++) {
			c = (CharClass)classes.get(i);
			Trace.Write(c.name + ": ", -10);
			WriteCharSet(c.set);
			Trace.WriteLine();
		}
		Trace.WriteLine();
	}

}

//-----------------------------------------------------------
// Symbol table management routines
//-----------------------------------------------------------

public class Tab {
	public static Position semDeclPos;		    // position of global semantic declarations
	public static BitSet ignored;					    // characters ignored by the scanner
	public static boolean[] ddt = new boolean[10];	// debug and test switches
	public static Symbol gramSy;					    // root nonterminal; filled by ATG
	public static Symbol eofSy;						    // end of file symbol
	public static Symbol noSym;						    // used in case of an error
	public static BitSet allSyncSets;			    // union of all synchronisation sets
	public static String nsName;        	    // namespace for generated files
	public static String frameDir = null;	    // directory containing the frame files
	public static Hashtable literals;         // symbols that are used as literals

	static BitSet visited;								    // mark list for graph traversals
	static Symbol curSy;									    // current symbol in computation of sets

	//---------------------------------------------------------------------
	//  Symbol set computations
	//---------------------------------------------------------------------

	/* Computes the first set for the given Node. */
	static BitSet First0(Node p, BitSet mark) {
		BitSet fs = new BitSet(Symbol.terminals.size());
		while (p != null && !mark.get(p.n)) {
			mark.set(p.n);
			switch (p.typ) {
				case Node.nt: {
					if (p.sym.firstReady) fs.or(p.sym.first);
					else fs.or(First0(p.sym.graph, mark));
					break;
				}
				case Node.t: case Node.wt: {
					fs.set(p.sym.n); break;
				}
				case Node.any: {
					fs.or(p.set); break;
				}
				case Node.alt: {
					fs.or(First0(p.sub, mark));
					fs.or(First0(p.down, mark));
					break;
				}
				case Node.iter: case Node.opt: {
					fs.or(First0(p.sub, mark));
					break;
				}
			}
			if (!Node.DelNode(p)) break;
			p = p.next;
		}
		return fs;
	}

	public static BitSet First(Node p) {
		BitSet fs = First0(p, new BitSet(Node.nodes.size()));
		if (ddt[3]) {
			Trace.WriteLine();
			if (p != null) Trace.WriteLine("First: node = " + p.n);
			else Trace.WriteLine("First: node = null");
			Sets.PrintSet(fs, 0);
		}
		return fs;
	}


	static void CompFirstSets() {
		Symbol sym;
		ArrayList nt = Symbol.nonterminals;
		int sz = nt.size();
		//foreach (Symbol sym in Symbol.nonterminals) {
		for (int i = 0; i < sz; i++) {
			sym = (Symbol)nt.get(i);
			sym.first = new BitSet(Symbol.terminals.size());
			sym.firstReady = false;
		}
		//foreach (Symbol sym in Symbol.nonterminals) {
		for (int i = 0; i < sz; i++) {
			sym = (Symbol)nt.get(i);
			sym.first = First(sym.graph);
			sym.firstReady = true;
		}
	}

	static void CompFollow(Node p) {
		while (p != null && !visited.get(p.n)) {
			visited.set(p.n);
			if (p.typ == Node.nt) {
				BitSet s = First(p.next);
				p.sym.follow.or(s);
				if (Node.DelGraph(p.next))
					p.sym.nts.set(curSy.n);
			} else if (p.typ == Node.opt || p.typ == Node.iter) {
				CompFollow(p.sub);
			} else if (p.typ == Node.alt) {
				CompFollow(p.sub); CompFollow(p.down);
			}
			p = p.next;
		}
	}

	static void Complete(Symbol sym) {
		if (!visited.get(sym.n)) {
			visited.set(sym.n);
			Symbol s;
			ArrayList nt = Symbol.nonterminals;
			//foreach (Symbol s in Symbol.nonterminals) {
			for (int i = 0; i < nt.size(); i++) {
				s = (Symbol)nt.get(i);
				if (sym.nts.get(s.n)) {
					Complete(s);
					sym.follow.or(s.follow);
					if (sym == curSy) sym.nts.clear(s.n);
				}
			}
		}
	}

	static void CompFollowSets() {
		Symbol sym;

		ArrayList nt = Symbol.nonterminals;
		int sz = nt.size();

		//foreach (Symbol sym in Symbol.nonterminals) {
		for (int i = 0; i < sz; i++) {
			sym = (Symbol)nt.get(i);
			sym.follow = new BitSet(Symbol.terminals.size());
			sym.nts = new BitSet(sz);
		}
		gramSy.follow.set(eofSy.n);
		visited = new BitSet(Node.nodes.size());
		//foreach (Symbol sym in Symbol.nonterminals) { // get direct successors of nonterminals
		for (int i = 0; i < sz; i++) {
			sym = (Symbol)nt.get(i);
			curSy = sym;
			CompFollow(sym.graph);
		}
		//foreach (Symbol sym in Symbol.nonterminals) { // add indirect successors to followers
		for (int i = 0; i < sz; i++) {
			sym = (Symbol)nt.get(i);
			visited = new BitSet(sz);
			curSy = sym;
			Complete(sym);
		}
	}

	static Node LeadingAny(Node p) {
		if (p == null) return null;
		Node a = null;
		if (p.typ == Node.any) a = p;
		else if (p.typ == Node.alt) {
			a = LeadingAny(p.sub);
			if (a == null) a = LeadingAny(p.down);
		}
		else if (p.typ == Node.opt || p.typ == Node.iter) a = LeadingAny(p.sub);
		else if (Node.DelNode(p) && !p.up) a = LeadingAny(p.next);
		return a;
	}

	static void FindAS(Node p) { // find ANY sets
		Node a;
		while (p != null) {
			if (p.typ == Node.opt || p.typ == Node.iter) {
				FindAS(p.sub);
				a = LeadingAny(p.sub);
				if (a != null) Sets.Subtract(a.set, First(p.next));
			} else if (p.typ == Node.alt) {
				BitSet s1 = new BitSet(Symbol.terminals.size());
				Node q = p;
				BitSet h;
				while (q != null) {
					FindAS(q.sub);
					a = LeadingAny(q.sub);
					if (a != null) {
						h = First(q.down);
						h.or(s1);
						Sets.Subtract(a.set, h);
					}
					else
						s1.or(First(q.sub));
					q = q.down;
				}
			}
			if (p.up) break;
			p = p.next;
		}
	}

	static void CompAnySets() {
		Symbol sym;

		ArrayList nt = Symbol.nonterminals;
		int sz = nt.size();
		//foreach (Symbol sym in Symbol.nonterminals)
		for (int i = 0; i < sz; i++) {
			sym = (Symbol)nt.get(i);
			FindAS(sym.graph);
		}
	}

	public static BitSet Expected(Node p, Symbol curSy) {
		BitSet s = First(p);
		if (Node.DelGraph(p)) s.or(curSy.follow);
		return s;
	}

	// does not look behind resolvers; only called during LL(1) test and in CheckRes
	public static BitSet Expected0(Node p, Symbol curSy) {
		if (p.typ == Node.rslv) return new BitSet(Symbol.terminals.size());
		else return Expected(p, curSy);
	}

	static void CompSync(Node p) {
		while (p != null && !visited.get(p.n)) {
			visited.set(p.n);
			if (p.typ == Node.sync) {
				BitSet s = Expected(p.next, curSy);
				s.set(eofSy.n);
				allSyncSets.or(s);
				p.set = s;
			} else if (p.typ == Node.alt) {
				CompSync(p.sub); CompSync(p.down);
			} else if (p.typ == Node.opt || p.typ == Node.iter)
				CompSync(p.sub);
			p = p.next;
		}
	}

static void CompSyncSets() {
		Symbol sym;
		ArrayList nt = Symbol.nonterminals;

		allSyncSets = new BitSet(Symbol.terminals.size());
		allSyncSets.set(eofSy.n);
		visited = new BitSet(Node.nodes.size());
		//foreach (Symbol sym in Symbol.nonterminals) {
		for (int i = 0; i < nt.size(); i++) {
			sym = (Symbol)nt.get(i);
			curSy = sym;
			CompSync(curSy.graph);
		}
	}

	public static void SetupAnys() {
		Node p;
		//foreach (Node p in Node.nodes)
		for (int i = 0; i < Node.nodes.size(); i++) {
			p = (Node)Node.nodes.get(i);
			if (p.typ == Node.any) {
				p.set = new BitSet(Symbol.terminals.size());
				p.set.set(0, p.set.size());
				p.set.clear(eofSy.n);
			}
		}
	}

	// [Re]7.4.1
	public static void CompDeletableSymbols() {
		boolean changed;
		Symbol sym;

		ArrayList nt = Symbol.nonterminals;
		int sz = nt.size();

		do {
			changed = false;
			//foreach (Symbol sym in Symbol.nonterminals)
			for (int i = 0; i < sz; i++) {
				sym = (Symbol)nt.get(i);
				if (!sym.deletable && sym.graph != null && Node.DelGraph(sym.graph)) {
					sym.deletable = true; changed = true;
				}
			}
		} while (changed);
		//foreach (Symbol sym in Symbol.nonterminals)
		for (int i = 0; i < sz; i++) {
			sym = (Symbol)nt.get(i);
			if (sym.deletable) System.out.println("  " + sym.name + " deletable");
		}
	}

	public static void RenumberPragmas() {
		int n = Symbol.terminals.size();
		Symbol sym;
		//foreach (Symbol sym in Symbol.pragmas)
		for (int i = 0; i < Symbol.pragmas.size(); i++) {
			sym = (Symbol)Symbol.pragmas.get(i);
			sym.n = n++;
		}
	}

	public static void CompSymbolSets() {
		CompDeletableSymbols();
		CompFirstSets();
		CompFollowSets();
		CompAnySets();
		CompSyncSets();
		if (ddt[1]) {
			Trace.WriteLine();
			Trace.WriteLine("First & follow symbols:");
			Trace.WriteLine("----------------------"); Trace.WriteLine();
			Symbol sym;
			ArrayList nt = Symbol.nonterminals;
			//foreach (Symbol sym in Symbol.nonterminals) {
			for (int i = 0; i < nt.size(); i++) {
				sym = (Symbol)nt.get(i);
				Trace.WriteLine(sym.name);
				Trace.Write("first:   "); Sets.PrintSet(sym.first, 10);
				Trace.Write("follow:  "); Sets.PrintSet(sym.follow, 10);
				Trace.WriteLine();
			}
		}
		if (ddt[4]) {
			Trace.WriteLine();
			Trace.WriteLine("ANY and SYNC sets:");
			Trace.WriteLine("-----------------");
			//foreach (Node p in Node.nodes)
			Node p;
			for (int i = 0; i < Node.nodes.size(); i++) {
				p = (Node)Node.nodes.get(i);
				if (p.typ == Node.any || p.typ == Node.sync) {
					Trace.Write(Integer.toString(p.n), 4);
					Trace.Write(" ");
					Trace.Write(Node.nTyp[p.typ], 4);
					Trace.Write(": ");
					Sets.PrintSet(p.set, 11);
				}
			}
		}
	}

	//---------------------------------------------------------------------
	//  Grammar checks
	//---------------------------------------------------------------------

	public static boolean GrammarOk() {
		boolean ok = NtsComplete()
			&& AllNtReached()
			&& NoCircularProductions()
			&& AllNtToTerm()
			&& ResolversOk();
		if (ok) CheckLL1();
		return ok;
	}

	//--------------- check for circular productions ----------------------

	static void GetSingles(Node p, ArrayList singles) {
		if (p == null) return;  // end of graph
		if (p.typ == Node.nt) {
			if (p.up || Node.DelGraph(p.next)) singles.add(p.sym);
		} else if (p.typ == Node.alt || p.typ == Node.iter || p.typ == Node.opt) {
			if (p.up || Node.DelGraph(p.next)) {
				GetSingles(p.sub, singles);
				if (p.typ == Node.alt) GetSingles(p.down, singles);
			}
		}
		if (!p.up && Node.DelNode(p)) GetSingles(p.next, singles);
	}

	public static boolean NoCircularProductions() {
		class CNode {	// node of list for finding circular productions
			public Symbol left, right;

			public CNode (Symbol l, Symbol r) {
				left = l; right = r;
			}
		}

		boolean ok, changed, onLeftSide, onRightSide;
		ArrayList list = new ArrayList();
		Symbol sym;
		for (int i = 0; i < Symbol.nonterminals.size(); i++) {
			sym = (Symbol)Symbol.nonterminals.get(i);
			ArrayList singles = new ArrayList();
			GetSingles(sym.graph, singles); // get nonterminals s such that sym-->s
			Symbol s;
			for (int j = 0; j < singles.size(); j++) {
				s = (Symbol)singles.get(j);
				list.add(new CNode(sym, s));
			}
		}
		do {
			changed = false;
			for (int i = 0; i < list.size(); i++) {
				CNode n = (CNode)list.get(i);
				onLeftSide = false; onRightSide = false;
				CNode m;
				for (int j = 0; j < list.size(); j++) {
					m = (CNode)list.get(j);
					if (n.left == m.right) onRightSide = true;
					if (n.right == m.left) onLeftSide = true;
				}
				if (!onLeftSide || !onRightSide) {
					list.remove(n); i--; changed = true;
				}
			}
		} while(changed);
		ok = true;
		CNode n;
		for (int i = 0; i < list.size(); i++) {
			n = (CNode)list.get(i);
			ok = false; Errors.count++;
			System.out.println("  " + n.left.name + " --> " + n.right.name);
		}
		return ok;
	}

	//--------------- check for LL(1) errors ----------------------

	static void LL1Error(int cond, Symbol sym) {
		System.out.print("  LL1 warning in " + curSy.name + ": ");
		if (sym != null) System.out.print(sym.name + " is ");
		switch (cond) {
			case 1: System.out.println("start of several alternatives"); break;
			case 2: System.out.println("start & successor of deletable structure"); break;
			case 3: System.out.println("an ANY node that matches no symbol"); break;
			case 4: System.out.println("contents of [...] or {...} must not be deletable"); break;
		}
	}

	static void CheckOverlap(BitSet s1, BitSet s2, int cond) {
		Symbol sym;
		for (int i = 0; i < Symbol.terminals.size(); i++) {
			sym = (Symbol)Symbol.terminals.get(i);
			if (s1.get(sym.n) && s2.get(sym.n)) LL1Error(cond, sym);
		}
	}

	static void CheckAlts(Node p) {
		BitSet s1, s2;
		while (p != null) {
			if (p.typ == Node.alt) {
				Node q = p;
				s1 = new BitSet(Symbol.terminals.size());
				while (q != null) { // for all alternatives
					s2 = Expected0(q.sub, curSy);
					CheckOverlap(s1, s2, 1);
					s1.or(s2);
					CheckAlts(q.sub);
					q = q.down;
				}
			} else if (p.typ == Node.opt || p.typ == Node.iter) {
				if (Node.DelSubGraph(p.sub)) LL1Error(4, null); // e.g. [[...]]
				else {
					s1 = Expected0(p.sub, curSy);
					s2 = Expected(p.next, curSy);
					CheckOverlap(s1, s2, 2);
				}
				CheckAlts(p.sub);
			} else if (p.typ == Node.any) {
				if (Sets.Elements(p.set) == 0) LL1Error(3, null);
				// e.g. {ANY} ANY or [ANY] ANY
			}
			if (p.up) break;
			p = p.next;
		}
	}

	public static void CheckLL1() {
		Symbol sym;
		for (int i = 0; i < Symbol.nonterminals.size(); i++) {
			sym = (Symbol)Symbol.nonterminals.get(i);
			curSy = sym;
			CheckAlts(curSy.graph);
		}
	}

	//------------- check if resolvers are legal  --------------------

	static boolean resOk;

	static void ResErr(Node p, String msg) {
		Errors.Error(p.line, p.pos.col, msg);
		resOk = false;
	}

	static void CheckRes(Node p, boolean rslvAllowed) {
		while (p != null) {
			switch (p.typ) {
				case Node.alt:
					BitSet expected = new BitSet(Symbol.terminals.size());
					for (Node q = p; q != null; q = q.down)
						expected.or(Expected0(q.sub, curSy));
					BitSet soFar = new BitSet(Symbol.terminals.size());
					for (Node q = p; q != null; q = q.down) {
						if (q.sub.typ == Node.rslv) {
							BitSet fs = First(q.sub.next);
							if (Sets.Intersect(fs, soFar))
								ResErr(q.sub, "Resolver will never be evaluated. " +
									"Place it at previous conflicting alternative.");
							if (!Sets.Intersect(fs, expected))
								ResErr(q.sub, "Misplaced resolver: no LL(1) conflict.");
						} else soFar.or(First(q.sub));
						CheckRes(q.sub, true);
					}
					break;
				case Node.iter: case Node.opt:
					if (p.sub.typ == Node.rslv) {
						BitSet fs = First(p.sub.next);
						BitSet fsNext = Expected(p.next, curSy);
						if (!Sets.Intersect(fs, fsNext))
							ResErr(p.sub, "Misplaced resolver: no LL(1) conflict.");
					}
					CheckRes(p.sub, true);
					break;
				case Node.rslv:
					if (!rslvAllowed)
						ResErr(p, "Misplaced resolver: no alternative.");
					break;
			}
			if (p.up) break;
			p = p.next;
			rslvAllowed = false;
		}
	}

	public static boolean ResolversOk() {
		resOk = true;
		Symbol sym;
		//foreach (Symbol sym in Symbol.nonterminals) {
		for (int i = 0; i < Symbol.nonterminals.size(); i++) {
			sym = (Symbol)Symbol.nonterminals.get(i);
			curSy = sym;
			CheckRes(curSy.graph, false);
		}
		return resOk;
	}

	//------------- check if every nts has a production --------------------

	public static boolean NtsComplete() {
		boolean complete = true;
		Symbol sym;
		for (int i = 0; i < Symbol.nonterminals.size(); i++) {
			sym = (Symbol)Symbol.nonterminals.get(i);
			if (sym.graph == null) {
				complete = false; Errors.count++;
				System.out.println("  No production for " + sym.name);
			}
		}
		return complete;
	}

	//-------------- check if every nts can be reached  -----------------

	static void MarkReachedNts(Node p) {
		while (p != null) {
			if (p.typ == Node.nt && !visited.get(p.sym.n)) { // new nt reached
				visited.set(p.sym.n);
				MarkReachedNts(p.sym.graph);
			} else if (p.typ == Node.alt || p.typ == Node.iter || p.typ == Node.opt) {
				MarkReachedNts(p.sub);
				if (p.typ == Node.alt) MarkReachedNts(p.down);
			}
			if (p.up) break;
			p = p.next;
		}
	}

	public static boolean AllNtReached() {
		boolean ok = true;
		visited = new BitSet(Symbol.nonterminals.size());
		visited.set(gramSy.n);
		MarkReachedNts(gramSy.graph);
		Symbol sym;
		for (int i = 0; i < Symbol.nonterminals.size(); i++) {
			sym = (Symbol)Symbol.nonterminals.get(i);
			if (!visited.get(sym.n)) {
				ok = false; Errors.count++;
				System.out.println("  " + sym.name + " cannot be reached");
			}
		}
		return ok;
	}

	//--------- check if every nts can be derived to terminals  ------------

	static boolean IsTerm(Node p, BitSet mark) { // true if graph can be derived to terminals
		while (p != null) {
			if (p.typ == Node.nt && !mark.get(p.sym.n)) return false;
			if (p.typ == Node.alt && !IsTerm(p.sub, mark)
			&& (p.down == null || !IsTerm(p.down, mark))) return false;
			if (p.up) break;
			p = p.next;
		}
		return true;
	}

	public static boolean AllNtToTerm() {
		boolean changed, ok = true;
		BitSet mark = new BitSet(Symbol.nonterminals.size());
		Symbol sym;
		// a nonterminal is marked if it can be derived to terminal symbols
		do {
			changed = false;
			for (int i = 0; i < Symbol.nonterminals.size(); i++) {
				sym = (Symbol)Symbol.nonterminals.get(i);
				if (!mark.get(sym.n) && IsTerm(sym.graph, mark)) {
					mark.set(sym.n); changed = true;
				}
			}
		} while (changed);
		for (int i = 0; i < Symbol.nonterminals.size(); i++) {
			sym = (Symbol)Symbol.nonterminals.get(i);
			if (!mark.get(sym.n)) {
				ok = false; Errors.count++;
				System.out.println("  " + sym.name + " cannot be derived to terminals");
			}
		}
		return ok;
	}

	/*---------------------------------------------------------------------
	  Utility functions
	---------------------------------------------------------------------*/

	static int Num(Node p) {
		if (p == null) return 0; else return p.n;
	}

	static String[] tKind = {"fixedToken", "classToken", "litToken", "classLitToken"};

	static void PrintSym(Symbol sym) {
		Trace.Write(Integer.toString(sym.n), 3);
		Trace.Write(" ");
		Trace.Write(Node.Name(sym.name), -14);
		Trace.Write(" ");
		Trace.Write(Node.nTyp[sym.typ], 2);
		if (sym.attrPos==null) Trace.Write(" false "); else Trace.Write(" true  ");
		if (sym.typ == Node.nt) {
			Trace.Write(Integer.toString(Num(sym.graph)), 5);
			if (sym.deletable) Trace.Write(" true  "); else Trace.Write(" false ");
		} else
			Trace.Write("            ");
		Trace.Write(Integer.toString(sym.line), 5);
		Trace.WriteLine(" " + tKind[sym.tokenKind]);
	}

	public static void PrintSymbolTable() {
		Trace.WriteLine("Symbol Table:");
		Trace.WriteLine("------------"); Trace.WriteLine();
		Trace.WriteLine(" nr name           typ  hasAt graph  del   line tokenKind");
		//foreach (Symbol sym in Symbol.terminals)
		Symbol sym;
		for (int i = 0; i < Symbol.terminals.size(); i++) {
			sym = (Symbol)Symbol.terminals.get(i);
			PrintSym(sym);
		}
		//foreach (Symbol sym in Symbol.pragmas)
		for (int i = 0; i < Symbol.pragmas.size(); i++) {
			sym = (Symbol)Symbol.pragmas.get(i);
			PrintSym(sym);
		}
		//foreach (Symbol sym in Symbol.nonterminals)
		for (int i = 0; i < Symbol.nonterminals.size(); i++) {
			sym = (Symbol)Symbol.nonterminals.get(i);
			PrintSym(sym);
		}
		Trace.WriteLine();
		Trace.WriteLine("Literal Tokens:");
		Trace.WriteLine("--------------");
		java.util.Set es = literals.entrySet();
		java.util.Iterator iter = es.iterator();
		Map.Entry me = null;
		//foreach (DictionaryEntry e in literals) {
		while (iter.hasNext()) {
			me = (Map.Entry)iter.next();
			Trace.WriteLine("_" + ((Symbol)me.getValue()).name + " = " + me.getKey() + ".");
		}
		Trace.WriteLine();
	}

	public static void XRef() {
		TreeMap tab = new TreeMap();
		// collect lines where symbols have been defined
		//foreach (Symbol sym in Symbol.nonterminals) {
		Symbol sym;
		for (int i = 0; i < Symbol.nonterminals.size(); i++) {
			sym = (Symbol)Symbol.nonterminals.get(i);
			ArrayList list = (ArrayList)tab.get(sym);
			if (list == null) {list = new ArrayList(); tab.put(sym, list);}
			list.add(new Integer(- sym.line));
		}
		// collect lines where symbols have been referenced
		//foreach (Node n in Node.nodes) {
		Node n;
		for (int i = 0; i < Node.nodes.size(); i++) {
			n = (Node)Node.nodes.get(i);
			if (n.typ == Node.t || n.typ == Node.wt || n.typ == Node.nt) {
				ArrayList list = (ArrayList)tab.get(n.sym);
				if (list == null) {list = new ArrayList(); tab.put(n.sym, list);}
				list.add(new Integer(n.line));
			}
		}
		// print cross reference list
		Trace.WriteLine();
		Trace.WriteLine("Cross reference list:");
		Trace.WriteLine("--------------------"); Trace.WriteLine();
		//foreach (Symbol sym in tab.Keys) {
		java.util.Set ks = tab.keySet();
		java.util.Iterator iter = ks.iterator();
		while (iter.hasNext()) {
			sym = (Symbol)iter.next();
			Trace.Write("  ");
			Trace.Write(Node.Name(sym.name), -12);
			ArrayList list = (ArrayList)tab.get(sym);
			int col = 14;
			//foreach (int line in list) {
			Integer line;
			for (int j = 0; j < list.size(); j++) {
				line = (Integer)list.get(j);
				if (col + 5 > 80) {
					Trace.WriteLine();
					for (col = 1; col <= 14; col++) Trace.Write(" ");
				}
				Trace.Write(line.toString(), 5); col += 5;
			}
			Trace.WriteLine();
		}
		Trace.WriteLine(); Trace.WriteLine();
	}

	public static void SetDDT (String s) {
		char ch;
		s = s.toUpperCase();
		for (int i = 0; i < s.length(); i++) {
			ch = s.charAt(i);
			if ('0' <= ch && ch <= '9') ddt[ch - '0'] = true;
			else switch (ch) {
				case 'A' : ddt[0] = true; break; // trace automaton
				case 'F' : ddt[1] = true; break; // list first/follow sets
				case 'G' : ddt[2] = true; break; // print syntax graph
				case 'I' : ddt[3] = true; break; // trace computation of first sets
				case 'J' : ddt[4] = true; break; // print ANY and SYNC sets
				case 'P' : ddt[8] = true; break; // print statistics
				case 'S' : ddt[6] = true; break; // list symbol table
				case 'X' : ddt[7] = true; break; // list cross reference table
				default : break;
			}
		}
	}

	public static void Init () {
		eofSy = new Symbol(Node.t, "EOF", 0);
		literals = new Hashtable();
	}

}
