;;;
;;; BOUNCE.LISP
;;;
;;; Translation of "bounce.c" by Brian Paul
;;; Changed to use RGB mode.
;;;

(eval-when (:compile-toplevel :load-toplevel)
  (require :gl)
  (require :xlib))

(defpackage :bounce-glx
  (:use :cl :gl :xlib)
  (:export :main))
(in-package :bounce-glx)

(defun sind (x)
  (coerce (sin (* x (/ pi 180.))) 'single-float))

(defun cosd (x)
  (coerce (cos (* x (/ pi 180.))) 'single-float))

(defvar ball)
(defvar window)
(defvar Zrot)
(defvar Zstep)
(defvar Xpos)
(defvar Ypos)
(defvar Xvel)
(defvar Yvel)
(defvar Xmin)
(defvar Xmax)
(defvar Ymin)
(defvar Ymax)
(defvar G)
(defvar vel0)

(defvar *animate?*)
(defvar *display* nil)
(defvar *window*)
(defvar *debug* nil)

(setq Zrot 0.0
      Zstep 0.6
      Xpos 0.0 Ypos 1.0
      Xvel 0.02 Yvel 0.0
      Xmin -4.0 Xmax 4.0
      Ymin -3.8 Ymax 4.0
      G -0.001
      vel0 -100.0)

(defun make-ball ()
  (let ((da 18.0) (db 18.0)
	(radius 1.0)
	(color 0)
	(x 0.0) (y 0.0) (z 0.0))
    
    (setq ball (glGenLists 1))
    (glNewList ball GL_COMPILE)
    
    (setq color 0)
    (do ((a -90.0 (+ a da))) ((> (+ a da) 90.0))
      ;;
      (glBegin GL_QUAD_STRIP)
      (do ((b 0.0 (+ b db))) ((> b 360.0))
	
	(format t "b:~a, color:~a~%" b color)
	(if (> color 0)
	  (glColor3f 1.0 0.0 0.0)
	  (glColor3f 1.0 1.0 1.0))
	
	(setq x (* (cosd b) (cosd a)))
	(setq y (* (sind b) (cosd a)))
	(setq z (sind a))
	(glVertex3f x y z)
	
	(setq x (* radius (cosd b) (cosd (+ a da))))
	(setq y (* radius (sind b) (cosd (+ a da))))
	(setq z (* radius (sind (+ a da))))
	(glVertex3f x y z)
	
	(setq color (- 1 color)))
      
      (glEnd))
    
    (glEndList)))

(defun reshape (width height)
  (when *debug* (format t "RESHAPE. WIDTH:~a, HEIGHT:~a~%" width height))
  (glViewport 0 0 width height)
  (glMatrixMode GL_PROJECTION)
  (glLoadIdentity)
  (glOrtho -6d0 6d0 -6d0 6d0 -6d0 6d0)
  (glMatrixMode GL_MODELVIEW))

(defun idle ()
  (when *debug* (format t "Callback IDLE.~%"))
  (setq zrot (+ zrot zstep))
  
  (setq xpos (+ xpos xvel))
  (when (>= xpos xmax)
    (setq xpos xmax)
    (setq xvel (- xvel))
    (setq zstep (- zstep)))
  ;;
  (when (<= xpos xmin)
    (setq xpos xmin)
    (setq xvel (- xvel))
    (setq zstep (- zstep)))
  ;;
  (setq ypos (+ ypos yvel))
  (setq yvel (+ yvel g))
  (when (< ypos ymin)
    (setq ypos ymin)
    (when (= vel0 -100.0) (setq vel0 (abs yvel)))
    (setq yvel vel0))
  (draw))

(defun draw ()
  (when *debug* (format t "Callback DRAW.~%"))
  (glClear GL_COLOR_BUFFER_BIT)
  (glColor3f 0.0 1.0 1.0)
  (glBegin GL_LINES)
  (do ((i -5 (+ i 1))) ((> i 5))
    (glVertex2i i -5) (glVertex2i i 5))
  ;;
  (do ((i -5 (+ i 1))) ((> i 5))
    (glVertex2i -5 i) (glVertex2i 5 i))
  ;;
  (do ((i -5 (+ i 1))) ((> i 5))
    (glVertex2i i -5) (glVertex2f (* i 1.15) -5.9))
  ;;
  (glVertex2f -5.3 -5.35)  (glVertex2f 5.3 -5.35)
  (glVertex2f -5.75 -5.9)  (glVertex2f 5.75 -5.9)
  (glEnd)
  
  (glPushMatrix)
  (glTranslatef Xpos Ypos 0.0)
  (glScalef 2.0 2.0 2.0)
  (glRotatef 8.0 0.0 0.0 1.0)
  (glRotatef 90.0 1.0 0.0 0.0)
  (glRotatef Zrot 0.0 0.0 1.0)
  
  (glCallList ball)
  (glPopMatrix)
  (glFlush)
  (when *debug* (format t "swapbuffers~%"))
  (glXSwapBuffers *display* *window*)
  (when *debug* (format t "exit DRAW~%")))

(defun event-loop (display)
  (let ((done? nil)
	(debug t)
	(event (make-xevent)))
    ;;
    ;; Main event loop
    (loop
      ;;
      ;; If we are animating, calc and redraw each frame until an event occurs
      (when *animate?*
	(when debug (format t "Animate...~%"))
	(loop
	  (idle)
	  (when (> (xpending display) 0)
	    (return))))
      ;;
      ;; Handle events.  If we are not animating, we wait here for event.
      (when debug (format t "Waiting for event...~%"))
      (xnextevent display event)
      (let ((event-type (xanyevent-type event)))
	(when debug (format t "Event:~a~%" event-type))
	(cond
	 ;;
	 ;; Expose
	 ((eq event-type expose)
	  ;;
	  ;; Gobble all other expose events
	  (loop
	    (when (zerop (xpending display))
	      (return))
	    (xnextevent display event)
	    (let ((event-type (xanyevent-type event)))
	      (unless (eq event-type expose)
		(xputbackevent display event)
		(return)))
	    (when debug (format t "Gobble event:~a~%" event-type)))
	  (draw))
	 ;;
	 ;; Resize
	 ((eq event-type configurenotify)
	  (reshape (xconfigureevent-width event)
		   (xconfigureevent-height event)))
	 ((eq event-type buttonpress)
	  (let ((button (xbuttonevent-button event)))
	    (when debug (format t "Button:~a~%" button))
	    (cond ((eq button button1)
		   (setf *animate?* (not *animate?*)))
		  ((eq button button3)
		   (setf done? t)))))))
      ;;
      (when done? (return)))
    (free-xevent event)))

(defun create-gl-window (display width height)
  ;; Create a double buffered, RGBA window
  ;; Warning ... this code will probably only work for True Color visual.
  (let* ((screen (XDefaultScreen display))
	 (root (XRootWindow display screen))
	 ;; Setup a byte-array of integers (C ints), terminated by "None"
	 (attrib (make-array 9
			     :element-type
			     #+cmu '(signed-byte 32) #-cmu 'fixnum
			     :initial-contents
			     (list GLX_RGBA GLX_RED_SIZE 1
				   GLX_GREEN_SIZE 1 GLX_BLUE_SIZE 1
				   GLX_DOUBLEBUFFER None)))
	 (visinfo (glXChooseVisual display screen attrib)))
    (when (zerop visinfo)
      (error "CREATE-GL-WINDOW: Couldn't get an RGB, double-buffered visual"))
    (let ((attr (make-xsetwindowattributes)))
      (set-xsetwindowattributes-background_pixel! attr 0)
      (set-xsetwindowattributes-border_pixel! attr 0)
      (set-xsetwindowattributes-colormap!
       attr (XcreateColormap display root
			     (XVisualInfo-visual visinfo) AllocNone))
      (set-xsetwindowattributes-event_mask!
       attr (+ StructureNotifyMask ExposureMask ButtonPressMask))
      (let* ((mask (+ CWBackPixel CWBorderPixel CWColormap CWEventMask))
	     (window (XCreateWindow display root 0 0 width height
				    0
				    (XVisualInfo-depth visinfo)
				    InputOutput
				    (XVisualInfo-visual visinfo)
				    mask attr))
	     (glxcontext (glXCreateContext display visinfo NULL 1)))
	(glXMakeCurrent display window glxcontext)
	(XMapWindow display window)
	window))))

(defun main ()
  (unless *display* (setq *display* (xopendisplay "")))
  ;;(setq *display* (xopendisplay ""))
  (setq *window* (create-gl-window *display* 300 300))
  ;; Create display list for ball
  (make-ball)
  ;; some GL init
  (glcullface GL_BACK)
  (glenable GL_CULL_FACE)
  (gldisable GL_DITHER)
  (glshademodel GL_FLAT)
  ;; Initial state of animation.
  (setf *animate?* t)
  ;; start event loop
  (event-loop *display*)
  ;; Cleanup and exit
  (xdestroywindow *display* *window*)
  ;; Closing and reopening the display seems to be a problem on some systems;
  ;; instead flush all events so that the window will close
  (let ((event (make-xevent)))
    (do ()
	((zerop (xpending *display*)))
      (xnextevent *display* event))
    (free-xevent event))
  ;;(xclosedisplay *display*)
  )
