#ifndef _PRPARTEDITOR_H_
#define _PRPARTEDITOR_H_

#include "atom.h"
#include "position.h"


class Table;
class Note;
class Part;
class Track;
class Reference;
class Event;

enum EditorType { NOTEEDITOR, PREDITOR, SCOREEDITOR, MASTEREDITOR, EVENTEDITOR, AUDIOEDITOR, DRUMEDITOR };


class PrPartEditor
{
 protected:
  Table *   _parts;       // An entry is always the first part in a track, so the number of parts is the number of displayed tracks
  Reference * _current; // pointing to the part currently edited
  Position  _leftPos;    // left pos
  Position  _rightPos;   // (obsolete: caution: this has to be updated manualy, e.g. in rewind())
  int       _current_length;  // current length value (192 = 1/8, etc)
  int       _current_enh;
  int       _current_vel;
  int       _current_dot;
  int       _current_tuplet;
  int       _current_tuplet_dur;
  char *    _cLeft;
  char *    _cRight;
  int       _bars;
  int       _grid;
  int       _resolution;
  bool      _no_overlap;
  bool      _color;
  bool      _speaker;
  EditorType _type;

  void adjustEventParameters();
  bool handles(Track*) const;

 public:
  PrPartEditor(EditorType type);
  PrPartEditor(Part * p);
  PrPartEditor(Track * t);
  virtual ~PrPartEditor();

  /** returns the part currently edited
   */
  Part * part() const;

  int partIndex();

  /** returns the part i
   */
  Part * part(int i);

  /** sets the current part
   */
  Part * setPart(int i);

  /** returns the number of parts edited by this editor
   */
  int parts() const;

  /** returns the name of the part currently edited
   */
  const char * name() const;

  /** moves one track down and returns the current part
   */
  Part * moveDown();

  /** moves one track up and returns the current part
   */
  Part * moveUp();

  /** returns the left position as char
   */
  char * cLeft();

  /** returns the right position as char
   */
  char * cRight();

  /** returns the left position as Position
   */
  Position left() const { return _leftPos; }

  /** returns the right position as Position
   */
  Position right() const { return _rightPos; }

  /** moves the positions one bar to the left
   */
  void moveleft(int i=1);

  /** moves the positions one bar to the right
   */
  void moveright(int i=1);

  /** moves the selection one element to the left
   */
  void moveselleft();

  /** moves the selection one element to the right
   */
  void moveselright();

  /** moves the positions to the specified position offset.
   */
  void wind(Position p);

  /** moves the positions to the specified bar offset
   */
  void wind(int bar);


  /** zooms in
   */
  void zoomin();

  /** zooms out
   */
  void zoomout();

  /** returns the currently selected number of bars viewed
   */
  int numberOfBars() const { return _bars; }

  void adjustRightPos();

  /** returns the current length value in ticks
   */
  int length() const { return _current_length; }

  /** returns the current length value in ticks, including increase through dots.
   */
  int totalLength() const;

  /** returns the current length value as an ordered int (1, ..., 7)
   */
  int lengthOrd() const;

  /** sets the length value as ticks
   */
  void setLength(int);

  /** sets the length value as an ordered int (1, ..., 7)
   */
  void setLengthOrd(int);

  /** returns the number of dots
   */
  int dot() const { return _current_dot; }

  /** sets the number of dots
   */
  void setDot(int);

  /** returns the base of the tuplet (3 for triplets, etc)
   */
  int tuplet() const { return _current_tuplet; }

  /** returns the tuplets total duration
   */
  int tupletDuration() const { return _current_tuplet_dur; }

  /** sets the tuplet base and the duration. If the duration is omitted, the default values
   * are used: a quarter (384) for triplets, a dotted quarter (576) for quintlets
   */
  void setTuplet(int base, int duration=-1);

  /** returns the enharmonic shift value (-2=bb, -1=b, 0=nat, 1=#, 2=x).
   */
  int enharmonicShift() const { return _current_enh; }

  /** sets the enharmonic shift value (-2=bb, -1=b, 0=nat, 1=#, 2=x).
   */
  void setEnharmonicShift(int);

  int velocity() const { return _current_vel; }

  void setVelocity(int);

  /** implement this for your own editor
   */
  virtual void update()=0;

  /** if a note exists at the specified pitch and position (up to the note's end) this note is returned, or zero else
   */
  Note * getGrabNote(long pos, int pitch, bool & tail);

  /** if an event exists at the position this event is returned, or zero else
   */
  Event * getGrabEvent(long pos);

  /** the specified position is quantized and returned
   */
  long snap(long pos) const;

  int grid() const { return _grid; }

  void setGrid(int);

  int resolution() const { return _resolution; }

  void setPartResolution(int);

  bool noOverlap() const { return _no_overlap; }

  void setNoOverlap(bool);

  virtual void setCursorPosition(long ticks)=0;

  bool color() const { return _color; }

  bool speaker() const { return _speaker; }

  void setColor(bool);

  void setSpeaker(bool);

  /** if the selection contains exactly one event, it is returned by this method. Null otherwise
   */
  Event * currentEvent();

  char ** actionList();

  bool isOfType(EditorType) const;

  /** enables and disables the gui buttons for the GuiEvents: changes, undo, redo, selection and memory
   */
  virtual void gui(GuiEvents ev, bool flag)=0;

  /** prints the editor's content
   */
  virtual void print()=0;

  /** the editor's settings
   */
  virtual void settings()=0;

  /** the editor's width
   */
  virtual int editorWidth() const;

  /** is this a score editor?
   */
  virtual bool isScoreEditor() const;

};


#endif

