/* cmd-diff.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "libfsutils/rmrf.h"
#include "libfsutils/tmp-files.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/patch-logs.h"
#include "libarch/local-cache.h"
#include "libarch/make-changeset.h"
#include "libarch/changeset-report.h"
#include "libarch/inode-sig.h"
#include "libarch/chatter.h"
#include "libarch/archive.h"
#include "commands/cmdutils.h"
#include "commands/cmd.h"
#include "commands/diff.h"
#include "commands/version.h"


/* __STDC__ prototypes for static functions */
static void make_changeset_callback (void * ign, char * fmt, va_list ap);



static t_uchar * usage = N_("[options] [revision] [-- limit...]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string and exit.")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("Change to DIR first.")) \
  OP (opt_output_dir, "o", "output DIR", 1, \
      N_("Save changeset in DIR.")) \
  OP (opt_verbose, "v", "verbose", 0, \
      N_("Verbose changeset report.")) \
  OP (opt_quiet, "q", "quiet", 0, \
      N_("Suppress progress information")) \
  OP (opt_nodiffs, "s", "summary", 0, \
      N_("Do not include diffs in the output.")) \
  OP (opt_hardlinks, 0, "link", 0, \
      N_("hardlink unchanged files to revision library"))\
  OP (opt_unescaped, 0, "unescaped", 0, \
      N_("show filenames in unescaped form"))


t_uchar arch_cmd_diff_help[] = N_("report about local changes in a project tree\n"
                                   "Generate a patch report describing the differences between the\n"
                                   "current project tree (or DIR if it is supplied) and\n"
                                   "the project trees current revision (or REVISION if supplied).\n"
                                   "\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_diff (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * dir = 0;
  t_uchar * output = 0;
  int nodiffs = 0;
  int keep = 0;
  int verbose = 0;
  int quiet = 0;
  int exit_status = 0;
  int link_same = 0;
  int escape_classes = arch_escape_classes;

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      if ((argc > 1) && !str_cmp ("--", argv[1]))
        break;

      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_diff_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;


        case opt_dir:
          {
            lim_free (0, dir);
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_output_dir:
          {
            lim_free (0, output);
            output = str_save (0, option->arg_string);
            keep = 1;
            break;
          }

        case opt_nodiffs:
          {
            nodiffs = 1;
            break;
          }

        case opt_quiet:
          {
            quiet = 1;
	    nodiffs = 1;
            break;
          }

        case opt_verbose:
          {
            verbose = 1;
            break;
          }

	case opt_hardlinks:
	  {
	    link_same = 1;
	    break;
	  }

	case opt_unescaped:
	  {
	    escape_classes = 0;
	    break;
	  }
        }
    }

  {
    int argn;
    arch_project_tree_t tree;
    t_uchar * fqvsn = 0;
    t_uchar * output_dir = 0;
    t_uchar * rvsnspec = 0;
    t_uchar * archive = 0;
    t_uchar * revision = 0;
    t_uchar * log_file = 0;
    rel_table limits = 0;
    int commandline_rvsn = 0;

    arch_project_tree_init (& tree, dir);

    if (!tree.root)
      {
        safe_printfmt (2, _("%s: not in a project tree\n  dir: %s\n"),
                       argv[0], dir);
        exit (2);
      }

    argn = 1;

    if ((argc > argn) && str_cmp (argv[argn], "--"))
      {
        fqvsn = str_save (0, argv[argn]);
        ++argn;
        commandline_rvsn = (arch_valid_package_name (fqvsn, arch_maybe_archive,
          arch_req_patch_level, 0));
      }
    else
      {
        fqvsn = arch_tree_version (tree.root);
        if (!fqvsn)
          {
            safe_printfmt (2, _("%s: no tree-version set\n  tree: %s\n"), argv[0], tree.root);
            exit (2);
          }
      }

    rvsnspec = arch_fqrvsn_from_tree_and_input (argv[0], fqvsn, tree.root);
    lim_free (0, fqvsn);

    if (argc > argn)
      {
        if (str_cmp (argv[argn], "--"))
          goto usage_error;

        ++argn;

        if (argc <= argn)
          goto usage_error;

        while (argc > argn)
          {
            rel_add_records (&limits, rel_make_record (argv[argn], 0), 0);
            ++argn;
          }
      }

    if (!arch_valid_package_name (rvsnspec, arch_maybe_archive, arch_req_patch_level, 0))
      {
        safe_printfmt (2, "%s: illegal revision name: %s\n",
                       argv[0], rvsnspec);
        exit (2);
      }

    archive = arch_parse_package_name (arch_ret_archive, NULL, rvsnspec);
    revision = arch_parse_package_name (arch_ret_non_archive, 0, rvsnspec);

    if (commandline_rvsn)
      arch_check_revision_local (archive, revision);

    output_dir = arch_diff_default_output_dir (tree.root, output);

    /****************************************************************
     * The heart of the matter.
     */
    {
      t_uchar * orig_tree = 0;
      struct arch_make_changeset_report make_report = {0, };
      struct arch_changeset_report report = {0, };
      assoc_table inode_shortcut = 0;

      if (!quiet)
        arch_chatter (1, "* looking for %s/%s to compare with\n", archive, revision);

      /* this should use a connection cache or something */
      orig_tree = arch_find_or_make_local_copy ((quiet ? -1 : 1), tree.root, 0, 0, archive, revision);

      if (!orig_tree)
        {
          safe_printfmt (2, "%s: no local copies to compare to (%s/%s)\n  consider `add-pristine --help'\n",
                         argv[0], archive, revision);
          exit (2);
        }


      if (!quiet)
        {
          arch_chatter (1, "* comparing to %s/%s\n", archive, revision);
          make_report.callback = make_changeset_callback;
        }


      arch_read_inode_sig_ids (0, &inode_shortcut, tree.root, archive, revision);
      arch_make_changeset (&make_report, orig_tree, tree.root, output_dir, arch_unspecified_id_tagging, arch_inventory_unrecognized, limits, inode_shortcut, link_same, escape_classes);
      if (!quiet)
          safe_printfmt(1, "\n");
      arch_evaluate_changeset (&report, output_dir);
      exit_status = arch_any_changes (&report);

      if (exit_status && (!nodiffs || verbose))
        {
          if (!quiet)
            safe_printfmt (1, "\n");

          if (verbose)
            arch_print_changeset (1, &report, !nodiffs, escape_classes);
          else
            arch_print_changeset_diffs (1, &report, escape_classes);
        }

      arch_free_make_changeset_report_data (&make_report);
      arch_free_changeset_report_data (&report);
      lim_free (0, orig_tree);

      /* remove the changeset info unless the user requested it persist */
      if (!keep)
        rmrf_file (output_dir);

      free_assoc_table (inode_shortcut);
    }

    lim_free (0, rvsnspec);
    arch_project_tree_finalise (&tree);
    lim_free (0, archive);
    lim_free (0, revision);
    lim_free (0, log_file);
    lim_free (0, output_dir);
    rel_free_table (limits);
  }


  lim_free (0, dir);
  lim_free (0, output);

  exit (exit_status);

  return 0;
}

char *
arch_diff_default_output_dir(char const *tree_root, t_uchar *filename)
{
  t_uchar * dir_root = 0;
  t_uchar * final_destination = 0;

  if (! tree_root)
    {
      dir_root = str_save(0, ".");
    }
  else
    {
      dir_root = str_save(0, tree_root);
    }

  if (! filename)
    {
      final_destination = tmp_file_name(dir_root, ",,changes");
      rmrf_file(final_destination);
    }
  else
    {
      if ( filename[0] != '/')
        final_destination = file_name_in_vicinity(0, dir_root, filename);
      else
        final_destination = file_name_in_vicinity(0, "", filename);
    }

  lim_free (0, dir_root);
    
  return final_destination;
}




static void
make_changeset_callback (void * ign, char * fmt, va_list ap)
{
  safe_printfmt(1, ".");
  safe_flush (1);
}




/* tag: Tom Lord Fri May 23 14:06:15 2003 (what-changed.c)
 */
