/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Mp3 (actually audio, not just mp3) doc/view
 *	class implementations
 *
 *	by Tony Sideris	(04:32AM Aug 02, 2001)
 *================================================*/
#include "arson.h"

#include <qtextstream.h>
#include <qcheckbox.h>
#include <qfile.h>
#include <qlabel.h>
#include <qregexp.h>

#include <klocale.h>

#include "processmgr.h"
#include "filewriter.h"
#include "audiowriter.h"
#include "tempfile.h"
#include "audiofile.h"
#include "audiodoc.h"
#include "konfig.h"
#include "listwnd.h"
#include "cdinfo.h"

/*========================================================*/
/*	Document class implementation
 *========================================================*/

class arsonAudioProgress : public ArsonProgressDisplay
{
public:
	arsonAudioProgress (void)
		: ArsonProgressDisplay(i18n("Minutes"), 60) { }

	virtual QString fullText (int length) const
	{
		return i18n("Total time: %1 / %2 %3")
			.arg(arsonDisplayTime(length))
			.arg(maxProgress() / mod())
			.arg(suffix());
	}

	virtual int maxProgress (void) const
	{
		return (ACONFIG.cdlenMin() * mod());
	}
};

/*========================================================*/

ArsonMp3Doc::ArsonMp3Doc (QWidget *parent, const char *name)
	: ArsonFileListDoc(parent, name), m_pCdInfo(NULL)
{
	setProgressInfo((new arsonAudioProgress));
}

/*========================================================*/

QString ArsonMp3Doc::propDocType (void) const
{
	return "audio";
}

/*========================================================*/

void ArsonMp3Doc::deleteContents (void)
{
	ArsonFileListDoc::deleteContents();

	delete m_pCdInfo;
	m_pCdInfo = NULL;
}

/*========================================================*/

void ArsonMp3Doc::delItem (QListViewItem *ptr)
{
	ArsonFileListFileItem *pi = (ArsonFileListFileItem *) item(ptr);

	if (pi && !pi->isDir())
		ArsonTempFile::deleteTemporaryFor(pi->local());

	ArsonFileListDoc::delItem(ptr);
}

/*========================================================*/

ArsonFileListItem *ArsonMp3Doc::createFileItem (const KURL &url) const
{
//	try {
		ArsonMp3ListItem *ptr = new ArsonMp3ListItem(url);
		return ptr;
/*
	}
	catch (ArsonError &err) {
		err.report();
		return NULL;
	}
*/
}

/*========================================================*/

ArsonProcessMgr *ArsonMp3Doc::createCdWriter (ArsonProcessUI *pUI)
{
	ArsonAudioWriter *ptr;

	if (ACONFIG.programPref(PROGGRP_AUDIOCD) == "cdrecord")
		ptr = new ArsonTaoWriter(pUI);
	else
	{
		ArsonDaoWriter *pd = new ArsonDaoWriter(pUI);
		pd->setCdInfo(m_pCdInfo);
		ptr = pd;
	}

	for (ItemIterator it (this); it; ++it)
		if (ArsonMp3ListItem *pi = (ArsonMp3ListItem *) it.item())
			ptr->addTrack(pi);

	return ptr;
}

/*========================================================*/

void ArsonMp3Doc::buildFileFilter (ArsonFileFilter &filter)
{
	ArsonAudioFile::audioFilter(filter);
}

/*========================================================*/

void ArsonMp3Doc::connectTo (ArsonActionConnector &ac)
{
	ArsonFileListDoc::connectTo(ac);

	ac.connect("list_edit_info", SLOT(editCdInfo()));
}

void ArsonMp3Doc::uiUpdated (ArsonActionEnabler &en)
{
	en.enable("list_edit_info", listWnd() && count() > 0);

	ArsonFileListDoc::uiUpdated(en);
}

/*========================================================*/

void ArsonMp3Doc::editCdInfo (void)
{
	if (count() > 0)
	{
		createCdInfo(m_pCdInfo);

		Assert(m_pCdInfo != NULL);
		m_pCdInfo->edit();
	}
	else
		arsonErrorMsg(
			i18n("Document empty. Add something first."));
}

/*========================================================*/
/**
 *	Used to fill the tracklist of a ArsonCdInfo object
 *	with the contents of an Audio Document.
 */

class trackVisiter : public ArsonListVisiter
{
public:
	trackVisiter (ArsonCdInfo *ptr) : m_info(ptr) { }

	virtual bool visit (QListViewItem *pi, ArsonListItem *pl)
	{
		ArsonCdInfo::Track track;
		ArsonMp3ListItem *pa = (ArsonMp3ListItem *) pl;
		const ArsonCdInfo::Track::Time time (pa->length());
		QString title (pa->title());
		
		track.setTrackNo(m_info->trackCount() + 1);

		/**
		 *	Guess the track title using the filename,
		 *	strip leading numbers (track number), then
		 *	leading spaces and punct, then the file
		 *	extension.
		 */
		if (title.isEmpty())
		{
			int pos;
			title = pa->getURL().fileName();

			title.replace(QRegExp("^[0-9]+"), "");
			title.replace(QRegExp("^[ -:]+"), "");

			if ((pos = title.findRev('.')))
				title.truncate(pos);

			//	Oops, went to far, give up
			if (title.isEmpty())
				title = pa->getURL().fileName();
		}

		track.setTitle(title);
		track.setTime(time);

		m_info->addTrack(track);
		return true;
	}

	ArsonCdInfo *m_info;
};

/*========================================================*/
/*	Verify that a ArsonCdInfo object exists for this
 *	document, and that it is update-to-date with the
 *	current contents of the document.
 *========================================================*/

void ArsonMp3Doc::createCdInfo (const ArsonCdInfo *init)
{
	ArsonCdInfo *ptr = new ArsonCdInfo(this);

	QObject::connect(ptr, SIGNAL(changed()),
		this, SLOT(slotModified()));

	if (init)
		ptr->merge(*init);

	if (!init || isModified())
	{
		trackVisiter tv (ptr);

		//	Reset the tracklist
		ptr->setTracks(ArsonCdInfo::TRACKS());
		visitAll(tv);
	}

	delete m_pCdInfo;
	m_pCdInfo = ptr;
}

/*========================================================*/
/*	Custom progress dialog (with Normalize option)
 *========================================================*/

class audioProgress : public ArsonCdWriterProgress
{
public:
	audioProgress (QWidget *parent, ArsonMp3Doc *pDoc)
		: ArsonCdWriterProgress(parent, "audio"), m_pDoc(pDoc),
		m_pCdText(NULL), m_pDrvFlags(NULL)
	{
		QLabel *pl;
		const QString modes[] = {
			i18n("None"),
			i18n("Batch Mode"),
			i18n("Mix Mode"),
		};

		pl = new QLabel(i18n("&Normalization:"), ctrlParent(), "normlabel");
		m_pNormalize = new ArsonProgressCombobox(
			ctrlParent(), "normmeth");

		for (int index = 0; index < ARRSIZE(modes); ++index)
			m_pNormalize->insertItem(modes[index]);

		m_pNormalize->setCurrentItem(0);
		pl->setSizePolicy(
			QSizePolicy(
				QSizePolicy::Maximum,
				QSizePolicy::Minimum));
		pl->setBuddy(m_pNormalize);

      m_pOnTheFly = addCheckbox(
			i18n("&On the fly (requires cdrecord as Audio Burner)"), optOnTheFly);
		m_pAudioMaster = addCheckbox(
			i18n("&Audio Master (requires cdrecord and Yamaha Drive)"), optAudMaster);
		m_pByteSwap = addCheckbox(
			i18n("&Byte swap"), optByteSwap);

		QObject::connect(m_pOnTheFly, SIGNAL(toggled(bool)),
			m_pNormalize, SLOT(setDisabled(bool)));
		QObject::connect(m_pOnTheFly, SIGNAL(toggled(bool)),
			pl, SLOT(setDisabled(bool)));

		layoutRow() << pl << m_pNormalize;

		if (pDoc->cdInfo())
		{
			m_pCdText = new ArsonProgressCheckbox(
				i18n("Write CD-Text, driver flags: "), ctrlParent(), optCdText);
			m_pDrvFlags = new ArsonProgressLineedit(ctrlParent(), optDrvFlags);
			m_pDrvFlags->setText("0x10");
			m_pDrvFlags->setEnabled(false);

			layoutRow() << m_pCdText << m_pDrvFlags;

			QObject::connect(m_pCdText, SIGNAL(toggled(bool)),
				m_pDrvFlags, SLOT(setEnabled(bool)));
		}
	}

protected:
	inline bool usingCdrecord (void) const {
		return (
			ACONFIG.programPref(PROGGRP_AUDIOCD) == "cdrecord");
	}

	inline bool onTheFly (void) const {
		return (usingCdrecord() && m_pOnTheFly->isChecked());
	}

	virtual void processOpts (ArsonProcessOpts &opts)
	{
		ArsonCdWriterProgress::processOpts(opts);
		opts.addLong(optNormal, m_pNormalize->currentItem());
		opts.addBool(optByteSwap, m_pByteSwap->isChecked());
		opts.addBool(optOnTheFly, onTheFly());
		opts.addBool(optAudMaster, m_pAudioMaster->isChecked());

		if (m_pCdText)
		{
			opts.addBool(optCdText, m_pCdText->isChecked());
			opts.addString(optDrvFlags, m_pDrvFlags->text());
		}
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return m_pDoc->createCdWriter(ui());
	}

	virtual void reconfigure (void)
	{
		const bool onthefly = onTheFly();
		const bool normalize = (
			ACONFIG.program(ArsonConfig::PROGRAM_NORMALIZE)->valid() &&
			(!usingCdrecord() || !onthefly));

		ArsonCdWriterProgress::reconfigure();

		m_pNormalize->setEnabled(normalize);
		((QLabel *) ctrlParent()->child("normlabel"))
			->setEnabled(normalize);

		m_pOnTheFly->setEnabled(usingCdrecord());
		m_pAudioMaster->setEnabled(usingCdrecord());

		if (m_pCdText)
		{
			m_pDrvFlags->setEnabled(!usingCdrecord());
			m_pCdText->setEnabled(!usingCdrecord());
		}
	}

	QComboBox *m_pNormalize;
	QCheckBox *m_pOnTheFly;
	QCheckBox *m_pByteSwap;
	QCheckBox *m_pAudioMaster;
	QCheckBox *m_pCdText;
	QLineEdit *m_pDrvFlags;
	ArsonMp3Doc *m_pDoc;
};

/*========================================================*/
/**
 *	Check the CdInfo object for all info necessary for
 *	a CD-Text writing. Return TRUE if the object is valid.
 */

bool checkCdInfo (const ArsonCdInfo *pi)
{
	if (pi->title().isEmpty())
		return false;

	for (int index = 0; index < pi->trackCount(); ++index)
	{
		const ArsonCdInfo::Track &track = pi->track(index);

		if (track.title().isEmpty())
			return false;
	}

	return true;
}

ArsonProgress *ArsonMp3Doc::createProgress (QWidget *parent)
{
	if (m_pCdInfo && !checkCdInfo(m_pCdInfo))
	{
		if (!arsonWarning(
				i18n("The CD-Text information is not complete. CD-Text will probably not write correctly. Continue?"), QString::null))
			return NULL;
	}

	return new audioProgress(parent, this);
}

/*========================================================*/
/*	List item type
 *========================================================*/

ArsonMp3ListItem::ArsonMp3ListItem (const KURL &url)
	: ArsonFileListFileItem(url)
{
	ArsonAudioFile af (local());

	if (!af.initialize())
	{
		Trace("ArsonAudioFile::initialize failed\n");

		throw ArsonError(
			i18n("Not a valid audio file (%1)")
			.arg(url.filename()));
	}

	m_length = af.length();
	m_title = af.title();
}

/*========================================================*/

QString ArsonMp3ListItem::displayTime (void) const
{
	/*	Total byte length of decoded track:
	 *	SAMPPERSEC * CHANS * (BITSPERSAMP / BITSPERBYTE) * LENGTHINSEC
	 */
//	const uint bytes = (44100 * (2 * (16 / 8))) * m_length;

	return arsonDisplayTime(m_length);/*
 +
		QString(" (%1)").arg(arsonByteSize(bytes));*/
}

/*========================================================*/

void ArsonMp3ListItem::refresh (QListViewItem *pi, ArsonDocWidget *pd)
{
	static QPixmap pm;

	if (pm.isNull())
		pm = loadIcon("sound");

	pi->setPixmap(0, pm);
	pi->setText(0, display());
	pi->setText(1, title());
	pi->setText(2, displayTime());
}

/*========================================================*/

ArsonListWnd *ArsonMp3Doc::createListWnd (void)
{
	return new ArsonMp3List(this);
}

/*========================================================*/

bool ArsonMp3Doc::onStartElement (const QString &name,
	const QXmlAttributes &attr, ArsonListInserter &pos)
{
	if (name == "cdtext")
	{
		createCdInfo();

		m_pCdInfo->setTitle(attr.value("title"));
		m_pCdInfo->setArtist(attr.value("artist"));
		m_pCdInfo->setGenre(attr.value("genre"));
	}
	else if (name == "track" && m_pCdInfo)
	{
		ArsonCdInfo::Track track;
		
		track.setTitle(attr.value("title"));
		track.setArtist(attr.value("artist"));

		m_pCdInfo->addTrack(track);
	}
	else
		return ArsonFileListDoc::onStartElement(name, attr, pos);

	return true;
}

bool ArsonMp3Doc::onEndElement (const QString &name, ArsonListInserter &pos)
{
	if (name == "cdtext") {}
	else if (name == "track") {}
	else
		return ArsonFileListDoc::onEndElement(name, pos);

	return true;
}

/*========================================================*/

class audioFileWriter : public ArsonXmlFileListWriter
{
public:
	audioFileWriter (ArsonFileListDoc *pd, const KURL &url, const ArsonCdInfo *pi)
		: ArsonXmlFileListWriter(pd, url), m_info(pi) { }

	virtual void endDocument (void)
	{
		if (m_info)
		{
			ArsonXmlTag tag (xml(), "cdtext");

			tag.addAttribute("artist", m_info->artist());
			tag.addAttribute("title", m_info->title());
			tag.addAttribute("genre", m_info->genre());

			tag.begin();

			for (int index = 0; index < m_info->trackCount(); ++index)
			{
				ArsonXmlTag track (xml(), "track");

				track.addAttribute("artist", m_info->track(index).artist(m_info));
				track.addAttribute("title", m_info->track(index).title());

				track.doit();
			}

			tag.end();
		}

		ArsonXmlFileListWriter::endDocument();
	}

private:
	const ArsonCdInfo *m_info;
};

ArsonBaseWriter *ArsonMp3Doc::createFileWriter (const KURL &url)
{
	return new audioFileWriter(this, url, m_pCdInfo);
}

/*========================================================*/
/*	TOC file class implementation
 *========================================================*/

ArsonTocFile::ArsonTocFile (void)
	: m_bTemp(false)
{
	//	Nothing...
}

ArsonTocFile::~ArsonTocFile (void)
{
	//	Cleanup temporary TOC file
	if (m_bTemp)
		QFile::remove(QFile::encodeName(m_filename));
}

/*========================================================*/

bool ArsonTocFile::addFile (const QString &fname)
{
	m_files.append(fname);
	return true;
}

/*========================================================*/

bool ArsonTocFile::writeFile (const char *filename)
{
	int index;
	QFile *pf = NULL;
	ArsonTempFile *pt = NULL;
	QTextStream *ps = NULL;

	//	Open a text stream
	if (!filename)
	{
		pt = new ArsonTempFile("arson", "toc");
		ps = pt->textStream();

		m_filename = pt->name();
		m_bTemp = true;
	}
	else
	{
		pf = new QFile(QFile::encodeName(filename));
		m_filename = filename;

		if (!pf->open(IO_WriteOnly))
		{
			m_strError = i18n("Failed to open output file %1")
				.arg(filename);
			return false;
		}

		ps = new QTextStream(pf);
	}

	writeHeader(*ps);

	for (index = 0; index < size(); index++)
		writeTrack(*ps, index, m_files[index]);
	
	//	Clean up
	if (pt)
		delete pt;
	else
	{
		pf->close();
		delete ps;
		delete pf;
	}

	return true;
}

/*========================================================*/

void ArsonTocFile::writeHeader (QTextStream &ts)
{
	ts << "CD_DA\n\n";

	if (m_pCdInfo)
	{
		QString artist (m_pCdInfo->artist());

		if (m_pCdInfo->variousArtistDisk() || artist.isEmpty())
			artist = i18n("Various Artists");

		ts << "CD_TEXT {\n"
		   << "  LANGUAGE_MAP { 0: EN }\n"
		   << "  LANGUAGE 0 {\n"
		   << "    TITLE " << cdTextString(m_pCdInfo->title()) << "\n"
		   << "    PERFORMER " << cdTextString(artist) << "\n";

		if (!m_pCdInfo->genre().isEmpty())
			ts << "      GENRE " << cdTextString(m_pCdInfo->genre()) << "\n";

		ts << "  }\n}\n\n";
	}
}

void ArsonTocFile::writeTrack (QTextStream &ts, int track, const QString &filename)
{
	ts << "TRACK AUDIO\n";

	if (m_pCdInfo)
	{
		ts << "CD_TEXT {\n"
		   << "  LANGUAGE 0 {\n"
		   << "    TITLE " << cdTextString(m_pCdInfo->track(track).title()) << "\n"
		   << "    PERFORMER " << cdTextString(m_pCdInfo->track(track).artist(m_pCdInfo)) << "\n"
		   << "  }\n}\n";
	}

	ts << "FILE " << cdTextString(filename) << " 0\n\n";
}

/*========================================================*/

QString ArsonTocFile::cdTextString (const QString &in) const
{
	QString result ("\"");

	for (int index = 0; index < in.length(); ++index)
	{
		const unsigned char ch = (unsigned char) in[index];

		if (ch >= 32 || ch <= 127)
		{
			switch (ch)
			{
//			case '\'':
			case '"':
				result += "\\";
				break;
			}

			result += ch;
		}
		else 	//	Translate non-ASCII characters to \OCT (octal code)
			result += QString().sprintf("\\%03o", ch);
	}

	return result + "\"";
}

/*========================================================*/
