/* $Id: ArkVBuffer.h,v 1.14 2003/03/12 23:45:31 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_VERTEXBUFFER_H
#define ARK_VERTEXBUFFER_H

#include <Ark/ArkTypes.h>
#include <Ark/ArkMath.h>
#include <vector>

namespace Ark
{

   /// Colors are RGBA even when they have no alpha channel (4byte alignment)
   struct ARK_DLL_API RGBA
   {
	 /// Red component of the colo.r
	 uchar R;
	 /// Green component of the color.
	 uchar G;
	 /// Blue component of the color.
	 uchar B;
	 /// Alpha component of the color (ie transparency).
	 uchar A;
   };

   /**
    * A vertex buffer is a list of vertices, packed in a certain way
    * (interleaved) so that we are able to use the graphic card 
    * specialized transformation unit, if there's one.
    *
    * They should be used to store every big list of vertex (ie more than
    * 10), especially for models, and world renderers. You really have to
    * look at the Renderer class to understand completely how to use them.
    */
   class ARK_DLL_API VertexBuffer
   { 
       // Uses a vector to uchar as private data storage
      std::vector< uchar > m_Buffer;
    
      public:
      /**
       * Those flags should be combined (or'ed) together to specify which data
       * the vertex buffer will include for each vertex, in a call to
       * VertexBuffer::SetFormat(). I've tried to keep it simple, so a lot of
       * features are not present : more than two texture passes, specular
       * lighting, etc.
       *
       * \arg \c VB_HAS_COORD:
       * This flag is compulsory since a vertex buffer has a meaning only when it
       * actually contains vectors...
       *
       * \arg \c VB_HAS_NORMAL:
       * Should be set if there are normal vectors for each of the vertices. Those
       * are used in the lighting pass.
       *
       * \arg \c VB_HAS_COLOR:
       * When doing smooth shading, it specifies that all vertices have an
       * initial color, when unlit (this one will be changed by lighting).
       *
       * \arg \c VB_HAS_UVn:
       * Set if the vertex buffer contains texture mapping coordinates for each
       * of the vertices. Those textures coords will only apply to the nth
       * texture of the material. If the object is multitextured, then you should
       * set the VB_HAS_UVn flag for each of the texture unit you're using.
       */
      enum VertexFormat
      {
	  VB_HAS_COORD  = (1<<0),
	  VB_HAS_NORMAL = (1<<1),
	  VB_HAS_COLOR  = (1<<2),
	  VB_HAS_UV0    = (1<<3),
	  VB_HAS_UV1    = (1<<4)
      };

	 /**
	  * This creates an empty vertex buffer. You have to call the
	  * SetFormat() and Reserve() functions before you're really able to
	  * access vertex data.
	  */
	 VertexBuffer ();

	 /**
	  * Deletes the vertex buffer.
	  */
	 ~VertexBuffer();


	 /**
	  * Remaps the vertices with the given order, ie indice n will be
	  * mapped to newIndices[n].
	  */
	 void Remap (std::vector<size_t> newIndices);

	 /**
	  * Sets the format of the vertex buffer. This function must be
	  * called before any access function since it computes the size
	  * each vertex will take in memory.
	  * Be aware that if you call this function after any access or
	  * reserve function, all data will be lost.
	  *
	  * \param format Mask of or'ed VB_HAS_xx values specifying which data
	  *        will be included for each vertex in the buffer.
	  * \param desired_vertex_size If the computed size is lower than this
	  *        value then the vertex buffer is padded with zero's in order
	  *        to reach the desired size.
	  */
	 void SetFormat(int format, unsigned int desired_vertex_size = 0);

	 /**
	  * Returns a bitfield representing the current format. Before calling
	  * any access function, you should ensure that it is "supported" by
	  * the current format : 
	  * \code
               if (vb.Format() & VB_HAS_COLOR)
	         vb.Color3(i) = color;
	  * \endcode
	  */
	 inline VertexFormat Format() const
	    {return VertexFormat(m_Format);}

	 /**
	  * Returns the size (in bytes) a vertex takes in memory. It depends
	  * on the format (ie on the data contained in the buffer).
	  */
	 inline unsigned int VertexSize() const
	    {return m_VertexSize;}

	 /**
	  * Resize the buffer for \c n vertices. 
	  *
	  * This function should be called
	  * before calling any access function.
	  */
	 void Resize(unsigned int n);

	 /** 
	  * Return the number of vertices contained & initialised in the 
	  * vertex buffer.
	  */
	 inline unsigned int Size() const
	    {return m_Size;} 

	 /** 
	  * Returns the texture coordinate to use.
	  */
	 inline int GetTextureCoordinateOffset() const
	    {return m_UVOffset;} 

	 /** 
	  * Sets the texture coordinate to use.
	  */
	 inline void SetTextureCoordinateOffset(int offset)
	    { m_UVOffset = offset;} 

      public:
	 /// Get the \c nth primary texture coordinate.
	 const Vector2& UV0 (size_t n) const;
	 /// Get the \c nth primary texture coordinate.
	 Vector2& UV0 (size_t n);
	 /// Get the \c nth secondary texture coordinate.
	 const Vector2& UV1 (size_t n) const;
	 /// Get the \c nth secondary texture coordinate.
	 Vector2& UV1 (size_t n);
	 /// Get the \c nth 3D coordinate.
	 const Vector3& Coord (size_t n) const;
	 /// Get the \c nth 3D coordinate.
	 Vector3& Coord (size_t n);
	 /// Get the \c nth normal.
	 const Vector3& Normal (size_t n) const;
	 /// Get the \c nth normal.
	 Vector3& Normal (size_t n);
	 /// Get the \c nth vertex color.
	 const RGBA& Color4 (size_t n) const;
	 /// Get the \c nth vertex color.
	 RGBA& Color4 (size_t n);

	 /**
	  * Get a pointer on the first primary texture coordinate. Be aware
	  * that the different values are interleaved : UV0P()[1] != UV0 (1),
	  * this function is to be used by the Renderer (especially the
	  * openGL renderer) to upload informations to the graphic card.
	  */
	 Vector2 *UV0P () const;

	 /// Get a pointer on the first secondary texture coordinate.
	 Vector2 *UV1P () const;
 
	 /// Get a pointer on the first 3D coordinate.
	 Vector3 *CoordP () const;

	 /// Get a pointer on the first normal.
	 Vector3 *NormalP () const;

	 /// Get a pointer on the first vertex color.
	 RGBA *Color4P () const;

	 /// Copy provides an empty VertexBuffer.
	 VertexBuffer (const VertexBuffer &) {}

	 /// Copy provides an empty VertexBuffer.
	 VertexBuffer & operator = (const VertexBuffer&)
	 { m_Buffer.clear(); SetFormat(0, 0); return *this; }
	 
      private:
	 /// Format of this vertex buffer. Ored VB_HAS_* values
	 int m_Format;

	 /**
	  * Size of this vertex buffer : the number of vertices contained
	  * in the list. (not actually the number of bytes).
	  */
	 unsigned int m_Size;

	 /// Size in byte of a single vertex.
	 unsigned int m_VertexSize;

	 unsigned int m_IndexNormal;
	 unsigned int m_IndexCoord;
	 unsigned int m_IndexColor4;
	 unsigned int m_IndexUV0;
	 unsigned int m_IndexUV1;

	 /// Offset for UV to have alternate texture coordinate
	 int m_UVOffset;

   };
}

#endif
