// ---------------------------------------------------------------------------
// - Library.cpp                                                             -
// - standard object library - shared library class implementation           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "String.hpp"
#include "Vector.hpp"
#include "Method.hpp"
#include "Library.hpp"
#include "Exception.hpp"
#include "cshl.hpp"

namespace aleph {
  // the list of registered library
  struct s_rlib {
    // the static library name
    String d_name;
    // the entry point
    void*  p_entry;
    // next entry in list
    s_rlib* p_next;
    // create a new register library
    s_rlib (const String& name, void* entry) {
      d_name  = name;
      p_entry = entry;
      p_next  = nilp;
    }
  };

  // the static list of registered libraries
  static s_rlib* aleph_rlib = nilp;

  // this procedure returns a registerd library handle by name
  static s_rlib* find_rlib (const String& name) {
    s_rlib* rlib = aleph_rlib;
    while (rlib != nilp) {
      if (rlib->d_name == name) return rlib;
      rlib = rlib->p_next;
    }
    return rlib;
  }

  // this procedure register a new library
  static void register_rlib (const String& name, void* entry) {
    // look for an existing library
    s_rlib* rlib = find_rlib (name);
    if (rlib != nilp) return;
    // create a new entry
    rlib = new s_rlib (name, entry);
    rlib->p_next = aleph_rlib;
    aleph_rlib = rlib;
  }

  // this function compute the dynamic entry point by adding the "dli" prefix
  // and mapping the '-' character to '_' character
  static String get_entry_point (const String& name) {
    String data;
    long   len = name.length ();
    for (long i = 0; i < len; i++) {
      char c = name[i];
      if ((c == '-') || (c == '+')) c = '_';
      data = data + c;
    }
    String result = "dli_";
    result = result + data;
    return result;
  }

  // create a new shared library object

  Library::Library (const String& name) {
    // save library name
    d_name = name;
    // check if the name is a registered library
    s_rlib* rlib = find_rlib (name);
    if (rlib == nilp) {
      char* lname = name.tochar ();
      p_handle = c_dlopen (lname);
      delete [] lname;
      if (p_handle == nilp) 
	throw Exception ("name-error", "cannot open dynamic library", name);
      p_rlib = nilp;
    } else {
      p_handle = c_dlmain ();
      p_rlib   = rlib;
    }
  }

  // delete this shared library by closing it

  Library::~Library (void) {
    c_dlclose (p_handle);
  }
  
  // return the class name

  String Library::repr (void) const {
    return "Library";
  }

  // return the library name

  String Library::getname (void) const {
    return d_name;
  }

  // register a new static library by name and entry point

  void Library::stareg (const String& name, void* entry) {
    register_rlib (name, entry);
  }

  // try to find a symbol by name in this library

  void* Library::find (const String& name) const {
    char* sname = name.tochar ();
    void* sym   = c_dlsym (sname, p_handle);
    delete [] sname;
    if (sym == nilp) 
      throw Exception ("library-error", "cannot find symbol", name);
    return sym;
  }

  // initialize a dynamic library

  Object* Library::dlinit (Runnable* robj, Vector* argv) {
    void* sym = nilp;
    // check for static or dynamic library
    if (p_rlib == nilp) {
      // get the entry point by name
      String entry = get_entry_point (d_name);
      sym = find (entry);
    } else {
      sym = p_rlib->p_entry;
    }
    // map the symbol and call it
    typedef Object* (*t_func) (Runnable*, Vector*);
    t_func func = (t_func) sym;
    return func (robj, argv);
  }

  // create a new shared library in a generic way

  Object* Library::mknew (Vector* argv) {
    long len = (argv == nilp) ? 0 : argv->length ();
    if (len != 0)
      throw Exception ("argument-error", 
		       "invalid number of arguments with library");
    String name = argv->getstring (0);
    return new Library (name);
  }
}
