// ---------------------------------------------------------------------------
// - Exception.hpp                                                           -
// - standard object library - exception class definition                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_EXCEPTION_HPP
#define ALEPH_EXCEPTION_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The Exception class is the base class for any exception generated by 
  /// the standard object library. The Exception holds a short string as a 
  /// descriptor and eventually a message. An object can be attached to the 
  /// exception for further reference.
  /// @author amaury darsch

  class Exception : public virtual Object {
  private:
    /// the exception id
    String d_eid;
    /// the exception reason
    String d_reason;
    /// the exception client object
    Object* p_object;
    /// the file name
    String d_fname;
    /// the line number
    long d_lnum;
    /// the newline flag
    bool d_nlf;
    /// the abort flag
    bool d_abf;

  public:
    /// create a exception with an id
    /// @param eid the exception id
    Exception (const String& eid);

    /// create a exception with an id and a reason
    /// @param eid    the exception id
    /// @param reason the reason for this exception
    Exception (const String& eid, const String& reason);

    /// create an exception with an id, a reason and a name
    /// @param eid    the exception eid
    /// @param reason the reason for this exception
    /// @param name   the name for the reason
    Exception (const String& eid, const String& reason, const String& name);

    /// create an exception with an eid an object
    /// @param eid    the exception id
    /// @param object the exception client object
    Exception (const String& eid, Object* object);

    /// create an exception with an id, a reason and an object
    /// @param eid    the exception id
    /// @param reason the reason for this exception
    /// @param object the exception client object
    Exception (const String& eid, const String& reason, Object* object);

    /// copy constructor for this exception class.
    /// @param that the exception to copy
    Exception (const Exception& that);

    /// destroy this exception class
    ~Exception (void);

    /// @return the class name
    String repr (void) const;

    /// @return the exception id
    String geteid (void) const {
      return d_eid;
    }

    /// @return the exception reason
    String getval (void) const {
      return d_reason;
    }

    /// @return the exception object
    Object* getobj (void) const {
      return p_object;
    }

    /// set the new line flag
    void setnlf (const bool flag);

    /// @return the new line flag
    bool getnlf (void) const {
      return d_nlf;
    }

    /// set the abort flag
    void setabf (const bool flag);

    /// @return the new line flag
    bool getabf (void) const {
      return d_abf;
    }

    /// set the exception file name
    /// @param fname the file name to set
    void setfname (const String& fname);

    /// @return the exxception file name
    String getfname (void) const {
      return d_fname;
    }

    /// set the exception line number
    /// @param lnum the line number to set
    void setlnum (const long lnum);

    /// @return the exception line number
    long getlnum (void) const {
      return d_lnum;
    }

    /// evaluate an exception member by quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to evaluate
    Object* eval (Runnable* robj, Nameset* nset, const long quark);

  private:
    // make the assignment operator private
    Exception& operator = (const Exception&);
  };
}

#endif
