// ---------------------------------------------------------------------------
// - Time.hpp                                                                -
// - aleph:sys library - time class definition                               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_TIME_HPP
#define  ALEPH_TIME_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The Time class is a simple class used to retreive the system time and
  /// date. When the class is constructed, the current time is stored in the
  /// instance and various methods can be called to get a time or date.
  /// The UTC and local time (corrected for daylight saving time) is 
  /// available. The basic time is the UTC time in seconds since the epoch.
  /// @author amaury darsch

  class Time : public Object {
  public:
    /// the time info query fields
    enum t_tdata {SECS, MINS, HOUR, MDAY, YMON, YEAR, WDAY, YDAY};

  private:
    /// the current time since the epoch
    long d_tclk;
    /// the utc info
    struct s_tinfo* p_uinfo;
    /// the local time info
    struct s_tinfo* p_linfo;

  public:
    /// create the current time
    Time (void);

    /// create a time class from a time counter
    /// @param tval the time counter value
    Time (const long tval);

    /// destroy this time class
    ~Time (void);

    /// @return the class name
    String repr (void) const;

    /// add a certain time in seconds to the time
    /// @param tval the time value to add
    void add (const long tval);

    /// return the time since the epoch
    long gettime (void) const;

    /// return a particular time info
    /// @param utc return in utc if true
    /// @param the time info to query
    long getinfo (const bool utc, const t_tdata type) const;

    /// format the date in mm/dd/yy format
    /// utc return in utc if true
    String formatdate (const bool utc) const;

    /// format the time in hh:mm:ss format
    /// utc return in utc if true
    String formattime (const bool utc) const;

    /// format the time for a cookie time
    String fmtcookie (void) const;

    /// format the utc time in rfc 822 format
    String formatrfc (void) const;

    /// generate a new time class
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this time with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
    
  private:
    // make the copy constructor private
    Time (const Time&);
    // make the assignment operator private
    Time& operator = (const Time&);
  };
}

#endif
