// ---------------------------------------------------------------------------
// - Address.cpp                                                             -
// - aleph:net library - address class implementation                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Address.hpp"
#include "Integer.hpp"
#include "Exception.hpp"
#include "cnet.hpp"

namespace aleph {
  // the supported address quark
  static const long QUARK_GETNAME     = String::intern ("get-name");
  static const long QUARK_GETCANON    = String::intern ("get-canonical-name");
  static const long QUARK_GETIPADDR   = String::intern ("get-ip-address");
  static const long QUARK_GETIPVECTOR = String::intern ("get-ip-vector");

  // convert a byte array into a vector
  static Vector* get_vector_address (const t_byte* addr) {
    if (addr == nilp) return nilp;
    long len = (long) addr[0] + 1;
    Vector* result = new Vector;
    for (long i = 1; i < len; i++) result->append (new Integer (addr[i]));
    return result;
  }

  // create an empty address

  Address::Address (void) {
    p_addr = nilp;
  }

  // create a new address by name

  Address::Address (const String& name) {
    d_name = name;
    char* data = d_name.tochar ();
    p_addr = c_ipaddr (data);
    delete [] data;
    // check for good address
    if (p_addr == nilp) 
      throw Exception ("address-error", "cannot resolve address", d_name);
  }

  // create a new address by bytes

  Address::Address (const t_byte* addr) {
    d_name = "";
    p_addr = c_cpaddr (addr);
    d_name = getcanon ();
  }

  // copy construct an address

  Address::Address (const Address& that) {
    that.rdlock ();
    d_name = that.d_name;
    p_addr = c_cpaddr (that.p_addr);
    that.unlock ();
  }

  // destroy this address instance

  Address::~Address (void) {
    delete [] p_addr;
  }

  // return the class name

  String Address::repr (void) const {
    return "Address";
  }

  // assign an address to this one

  Address& Address::operator = (const Address& that) {
    // clean the old address first
    wrlock ();
    delete [] p_addr;
    // lock and copy
    that.rdlock ();
    d_name = that.d_name;
    p_addr = c_cpaddr (that.p_addr);
    // unlock eveything
    that.unlock ();
    unlock ();
    return *this;
  }

  // return the address canonical name

  String Address::getcanon (void) const {
    rdlock ();
    try {
      char* data = (d_name.length () == 0) ? c_iprepr (p_addr) : 
	            d_name.tochar();
      char* name = c_ipcanon (data);
      String result = name;
      delete [] data;
      delete [] name;
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the address representation

  String Address::getaddr (void) const {
    rdlock ();
    try {
      char* data = c_iprepr (p_addr);
      String result = data;
      delete [] data;
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // create a new address instance in a generic way
 
  Object* Address::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Address (name);
    }
    throw Exception ("argument-error",
                     "too many argument with address constructor");
  }
 
  // apply an address member with a set of arguments and a quark
 
  Object* Address::apply (Runnable* robj, Nameset* nset, const long quark,
			  Vector* argv) {

    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
 
    // check for 0 arguments
    if (argc == 0) {
      if (quark == QUARK_GETNAME)   return new String (d_name);
      if (quark == QUARK_GETCANON)  return new String (getcanon ());
      if (quark == QUARK_GETIPADDR) return new String (getaddr  ());
      if (quark == QUARK_GETIPVECTOR) {
	rdlock ();
	Vector* result = get_vector_address (p_addr);
	unlock ();
	return result;
      } 
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
