// ---------------------------------------------------------------------------
// - Aod.cpp                                                                 -
// - aleph object database - command class implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Aod.hpp"
#include "System.hpp"
#include "Integer.hpp"
#include "Function.hpp"
#include "Aodcalls.hpp"
#include "InputFile.hpp"
#include "InputMapped.hpp"
#include "Exception.hpp"
#include "OutputFile.hpp"
#include "PrintTable.hpp"

namespace aleph {

  // query default prompts
  static const char* DEFAULT_PROMPT1 = "(aod)";
  static const char* DEFAULT_PROMPT2 = "....>";
    
  // this function initialize the aod nameset
  static Nameset* initialize_aod (Aod* ctrl) {
    // create the aod nameset
    Nameset* aod = ctrl->mknset ("aod", ctrl->getgset ());
    // bind the aod commands
    aod->symcst ("open",         new Function (aod_open));
    aod->symcst ("save",         new Function (aod_save));
    aod->symcst ("info",         new Function (aod_info));
    aod->symcst ("create",       new Function (aod_create));
    aod->symcst ("import",       new Function (aod_import));
    aod->symcst ("export",       new Function (aod_export));
    aod->symcst ("select",       new Function (aod_select));
    aod->symcst ("insert",       new Function (aod_insert));
    aod->symcst ("table-info",   new Function (aod_tinfo));
    aod->symcst ("table-report", new Function (aod_tblrpt));
    return aod;
  }

  // create a default data commands

  Aod::Aod (void) {
    // initialize the commander
    p_dc  = nilp;
    p_tbl = nilp;
    if (p_term != nilp) {
      p_term->setpprompt (DEFAULT_PROMPT1);
      p_term->setsprompt (DEFAULT_PROMPT2);
    }
    // bind the aod nameset
    Object::iref (p_cset = initialize_aod (this));
  }

  // create a stream based commander

  Aod::Aod (Input* is,Output* os,Output* es) : Interp (is,os,es) {
    // initialize the query
    p_dc  = nilp;
    p_tbl = nilp;
    // bind the aod nameset
    Object::iref (p_cset = initialize_aod (this));
  }
  
  // destroy this commander

  Aod::~Aod (void) {
    if (p_cset != nilp) p_cset->clear ();
    Object::dref (p_cset);
    Object::dref (p_tbl);
    Object::dref (p_dc);
  }

  // return the class name

  String Aod::repr (void) const {
    return "Aod";
  }

  // open an collection by name
  
  Collection* Aod::open (const String& fname) {
    wrlock ();
    // clean the old collection
    Object::dref (p_dc);
    try {
      // check for a file name
      if (System::isfile (fname) == true) {
	InputMapped is (fname);
	p_dc = new Collection (&is);
      } else {
	p_dc = new Collection (fname);
      }
      Object::iref (p_dc);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
    return p_dc;
  }

  // save a collection by name

  void Aod::save (const String& fname) {
    rdlock ();
    if (p_dc == nilp) {
      unlock ();
      throw Exception ("aod-error", "data collection is not open");
    }
    try {
      // try to open the file
      OutputFile os (fname);
      p_dc->write (os);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }
  
  // print some information about the collection

  void Aod::info (Output* os) const {
    rdlock ();
    if (p_dc == nilp) {
      unlock ();
      throw Exception ("aod-error", "data collection is not open");
    }
    try {
      // create a print table
      PrintTable result (2);
      long row = result.add ();
      result.setsize (0, 40);
      result.set (row, 0, "Object");
      result.set (row, 1, "Value");
      // get collection name
      String cname = p_dc->getname ();
      row = result.add ();
      result.set (row, 0, "Collection name");
      result.set (row, 1, cname);
      // get the current table
      String tname = (p_tbl == nilp) ? "none" : p_tbl->getname ();
      row = result.add ();
      result.set (row, 0, "Current table");
      result.set (row, 1, tname);
      // print the table
      if (os != nilp) result.format (*os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // print collection table contents

  void Aod::tinfo (Output* os) const {
    rdlock ();
    if (p_dc == nilp) {
      unlock ();
      throw Exception ("aod-error", "data collection is not open");
    }
    try {
      // create a print table
      PrintTable result (3);
      long row = result.add ();
      result.set (row, 0, "Table index");
      result.set (row, 1, "Table name");
      result.set (row, 2, "Table length");
      // collect table info
      long len = p_dc->length ();
      for (long i = 0; i < len; i++) {
	Table* tbl = p_dc->tget (i);
	if (tbl == nilp) continue;
	Integer ival (i);
	row = result.add ();
	result.set (row, 0, ival.tostring ());
	result.set (row, 1, tbl->getname ());
	ival = tbl->length ();
	result.set (row, 2, ival.tostring ());
      }
      if (os != nilp) result.format (*os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // create a new table and make it the default

  Table* Aod::create (void) {
    wrlock ();
    if (p_dc == nilp) {
      unlock ();
      throw Exception ("aod-error", "data collection is not open");
    }
    // remove the old table and create a new one
    Object::dref (p_tbl);
    Object::iref (p_tbl = new Table);
    // attach it to the collection
    p_dc->tadd (p_tbl);
    unlock ();
    return p_tbl;
  }

  // import data into the default table

  void Aod::import (const String& fname) {
    wrlock ();
    if (p_dc == nilp) {
      unlock ();
      throw Exception ("aod-error", "data collection is not open");
    }
    if (p_tbl == nilp) {
      unlock ();
      throw Exception ("aod-error", "no current table is set");
    }
    InputFile* is = nilp;
    try {
      // try to open the file
      Object::iref (is = new InputFile (fname));
      p_tbl->import (is);
      unlock ();
    } catch (Exception& e) {
      e.setfname (fname);
      Object::dref (is);
      unlock ();
      throw;
    } catch (...) {
      Object::dref (is);
      unlock ();
      throw;
    }
  }

  // select a table by index

  void Aod::select (const long index) {
    wrlock ();
    if (p_dc == nilp) {
      unlock ();
      throw Exception ("aod-error", "data collection is not open");
    }
    try {
      Table* tbl = p_dc->tget (index);
      Object::iref (tbl);
      Object::dref (p_tbl);
      p_tbl = tbl;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // insert element to the current table

  void Aod::insert (Vector* elems) {
    wrlock ();
    if (p_dc == nilp) {
      unlock ();
      throw Exception ("aod-error", "data collection is not open");
    }
    if (p_tbl == nilp) {
      unlock ();
      throw Exception ("aod-error", "no current table is set");
    }
    try {
      p_tbl->add (elems);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // print a table content

  void Aod::tblrpt (Output* os, long max, long start, bool flag) const {
    rdlock ();
    if (p_dc == nilp) {
      unlock ();
      throw Exception ("aod-error", "data collection is not open");
    }
    if (p_tbl == nilp) {
      unlock ();
      throw Exception ("aod-error", "no current table is set");
    }
    try {
      if (os != nilp) p_tbl->report (*os, max, start, flag);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
}
