/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLAttributeStatement.cpp - SAML attribute statement implementation

   Scott Cantor
   5/23/02

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;


SAMLAttributeStatement::SAMLAttributeStatement(SAMLSubject* subject, const Iterator<SAMLAttribute*>& attributes)
    : SAMLSubjectStatement(subject)
{
    RTTI(SAMLAttributeStatement);
    while (attributes.hasNext())
        m_attributes.push_back(static_cast<SAMLAttribute*>(attributes.next()->setParent(this)));
}

SAMLAttributeStatement::SAMLAttributeStatement(DOMElement* e) : SAMLSubjectStatement(e)
{
    RTTI(SAMLAttributeStatement);
    fromDOM(e);
}

SAMLAttributeStatement::SAMLAttributeStatement(istream& in) : SAMLSubjectStatement(in)
{
    RTTI(SAMLAttributeStatement);
    fromDOM(m_document->getDocumentElement());
}

SAMLAttributeStatement::~SAMLAttributeStatement()
{
    for (vector<SAMLAttribute*>::const_iterator i=m_attributes.begin(); i!=m_attributes.end(); i++)
        delete (*i);
}

void SAMLAttributeStatement::fromDOM(DOMElement* e)
{
    if (SAMLConfig::getConfig().strict_dom_checking) {
        if (XMLString::compareString(XML::SAML_NS,e->getNamespaceURI()))
            throw MalformedException("SAMLAttributeStatement::fromDOM() missing saml namespace on root element");

        if (XMLString::compareString(L(AttributeStatement),e->getLocalName())) {
            auto_ptr<saml::QName> type(saml::QName::getQNameAttribute(e,XML::XSI_NS,L(type)));
            if ((XMLString::compareString(L(Statement),e->getLocalName()) && XMLString::compareString(L(SubjectStatement),e->getLocalName())) ||
                    !type.get() || XMLString::compareString(XML::SAML_NS,type->getNamespaceURI()) ||
                    XMLString::compareString(L(AttributeStatementType),type->getLocalName()))
                throw MalformedException("SAMLAttributeStatement::fromDOM() requires saml:AttributeStatement at root");
        }
    }

    DOMElement* a = XML::getFirstChildElement(e,XML::SAML_NS,L(Attribute));
    while (a) {
        try {
            SAMLAttribute* attr=SAMLAttribute::getInstance(a);
            m_attributes.push_back(static_cast<SAMLAttribute*>(attr->setParent(this)));
        }
        catch (SAMLException& ex) {
            SAML_log.warn("exception while instantiating a SAMLAttribute: %s",ex.what());
        }
        a=XML::getNextSiblingElement(a,XML::SAML_NS,L(Attribute));
    }
    checkValidity();
}

void SAMLAttributeStatement::setAttributes(const Iterator<SAMLAttribute*>& attributes)
{
    while (m_attributes.size())
        removeAttribute(0);
    while (attributes.hasNext())
        addAttribute(attributes.next());
}

void SAMLAttributeStatement::addAttribute(SAMLAttribute* attribute)
{
    if (attribute) {
        attribute->setParent(this);
        m_attributes.push_back(attribute);
        ownStrings();
        setDirty();
    }
    else
        throw MalformedException("attribute cannot be null");
}

void SAMLAttributeStatement::removeAttribute(unsigned long index)
{
    SAMLAttribute* kill=m_attributes[index];
    m_attributes.erase(m_attributes.begin()+index);
    delete kill;
    ownStrings();
    setDirty();
}

DOMElement* SAMLAttributeStatement::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* s = doc->createElementNS(XML::SAML_NS, L(AttributeStatement));
    if (xmlns) {
        s->setAttributeNS(XML::XMLNS_NS,L(xmlns),XML::SAML_NS);
        s->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,xsi),XML::XSI_NS);
        s->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,xsd),XML::XSD_NS);
    }
    return s;
}

DOMNode* SAMLAttributeStatement::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLSubjectStatement::toDOM(doc,xmlns);
    DOMElement* s=static_cast<DOMElement*>(m_root);

    if (m_bDirty) {
        for (vector<SAMLAttribute*>::const_iterator i=m_attributes.begin(); i!=m_attributes.end(); i++)
            s->appendChild((*i)->toDOM(m_root->getOwnerDocument(),false));
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(s,XML::SAML_NS);
        DECLARE_NAMESPACE(s,xsi,XML::XSI_NS);
        DECLARE_NAMESPACE(s,xsd,XML::XSD_NS);
    }
    return s;
}

void SAMLAttributeStatement::checkValidity() const
{
    SAMLSubjectStatement::checkValidity();
    if (m_attributes.empty())
        throw MalformedException("AttributeStatement is invalid, requires at least one attribute");
}


SAMLObject* SAMLAttributeStatement::clone() const
{
    return new SAMLAttributeStatement(static_cast<SAMLSubject*>(m_subject->clone()),getAttributes().clone());
}
