/* -------------------------------------------------------------------------- */
/* Copyright 2002-2010, OpenNebula Project Leads (OpenNebula.org)             */
/*                                                                            */
/* Licensed under the Apache License, Version 2.0 (the "License"); you may    */
/* not use this file except in compliance with the License. You may obtain    */
/* a copy of the License at                                                   */
/*                                                                            */
/* http://www.apache.org/licenses/LICENSE-2.0                                 */
/*                                                                            */
/* Unless required by applicable law or agreed to in writing, software        */
/* distributed under the License is distributed on an "AS IS" BASIS,          */
/* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   */
/* See the License for the specific language governing permissions and        */
/* limitations under the License.                                             */
/* -------------------------------------------------------------------------- */


#include "Leases.h"
#include "NebulaLog.h"

/* ************************************************************************** */
/* ************************************************************************** */
/* Lease class                                                                */
/* ************************************************************************** */
/* ************************************************************************** */

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

void Leases::Lease::to_string(string &_ip,
                              string &_mac) const
{
    mac_to_string(mac, _mac);

    ip_to_string(ip, _ip);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

int Leases::Lease::ip_to_number(const string& _ip, unsigned int& i_ip)
{
    istringstream iss;
    size_t        pos=0;
    int           count = 0;
    unsigned int  tmp;

    string ip = _ip;

    while ( (pos = ip.find('.')) !=  string::npos )
    {
        ip.replace(pos,1," ");
        count++;
    }

    if (count != 3)
    {
        return -1;
    }

    iss.str(ip);

    i_ip = 0;

    for (int i=0;i<4;i++)
    {
        iss >> dec >> tmp >> ws;

        i_ip <<= 8;
        i_ip += tmp;
    }

    return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

void Leases::Lease::ip_to_string(const unsigned int i_ip, string& ip)
{
    unsigned int    temp_byte;
    ostringstream	oss;

    // Convert the IP from unsigned int to string

    for (int index=0;index<4;index++)
    {
        temp_byte =   i_ip;
        temp_byte >>= (24-index*8);
        temp_byte &=  255;

        oss << temp_byte;

        if(index!=3)
        {
            oss << ".";
        }
    }

    ip = oss.str();
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

int Leases::Lease::mac_to_number(const string& _mac, unsigned int i_mac[])
{
    istringstream iss;
    size_t        pos=0;
    int           count = 0;
    unsigned int  tmp;

    string mac = _mac;

    while ( (pos = mac.find(':')) !=  string::npos )
    {
        mac.replace(pos,1," ");
        count++;
    }

    if (count != 5)
    {
        return -1;
    }

    iss.str(mac);

    i_mac[PREFIX] = 0;
    i_mac[SUFFIX] = 0;

    iss >> hex >> i_mac[PREFIX] >> ws >> hex >> tmp >> ws;
    i_mac[PREFIX] <<= 8;
    i_mac[PREFIX] += tmp;

    for (int i=0;i<4;i++)
    {
        iss >> hex >> tmp >> ws;

        i_mac[SUFFIX] <<= 8;
        i_mac[SUFFIX] += tmp;
    }

    return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

void Leases::Lease::mac_to_string(const unsigned int i_mac[], string& mac)
{
    ostringstream	oss;
    unsigned int	temp_byte;

    oss.str("");

    for (int i=5;i>=0;i--)
    {
        if ( i < 4 )
        {
            temp_byte =   i_mac[SUFFIX];
            temp_byte >>= i*8;
        }
        else
        {
            temp_byte  = i_mac[PREFIX];
            temp_byte  >>= (i%4)*8;
        }

        temp_byte  &= 255;

        oss.width(2);
        oss.fill('0');
        oss << hex << temp_byte;

        if(i!=0)
        {
            oss << ":";
        }
    }

    mac = oss.str();
}


ostream& operator<<(ostream& os, Leases::Lease& _lease)
{
	string xml;

    os << _lease.to_xml(xml);

    return os;
}

string& Leases::Lease::to_str(string& str) const
{
    string ip;
    string mac;

    ostringstream os;

    to_string(ip,mac);

    ip = "IP = " + ip;
    mac = "MAC = " + mac;

    os.width(20);
    os << left << ip;

    os.width(24);
    os << left << mac;

    os << left << " USED = " << used;
    os << left << " VID = "  << vid;

    str = os.str();

    return str;
}


string& Leases::Lease::to_xml(string& str) const
{
    string ip;
    string mac;

    ostringstream os;

    to_string(ip,mac);

    os <<
        "<LEASE>" <<
            "<IP>"<< ip << "</IP>" <<
            "<MAC>" << mac << "</MAC>" <<
            "<USED>" << used << "</USED>" <<
            "<VID>" << vid << "</VID>" <<
        "</LEASE>";

    str = os.str();

    return str;
}

/* ************************************************************************** */
/* ************************************************************************** */
/* Leases class                                                               */
/* ************************************************************************** */
/* ************************************************************************** */

const char * Leases::table        = "leases";

const char * Leases::db_names     = "(oid,ip,mac_prefix,mac_suffix,vid,used)";

const char * Leases::db_bootstrap = "CREATE TABLE IF NOT EXISTS leases ("
                "oid INTEGER, ip BIGINT, mac_prefix BIGINT, mac_suffix BIGINT,"
                "vid INTEGER, used INTEGER, PRIMARY KEY(oid,ip))";

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

int Leases::select_cb(void *nil, int num, char **values, char **names)
{
    if (    (values[OID] == 0)        ||
            (values[IP]  == 0)        ||
            (values[MAC_PREFIX] == 0) ||
            (values[MAC_SUFFIX] == 0) ||
            (values[VID] == 0)        ||
            (values[USED]== 0)        ||
            (num != LIMIT ))
    {
        return -1;
    }

    unsigned int mac[2];
    unsigned int ip;
    int          vid;
    bool         used;

    istringstream iss;

    iss.str(values[IP]);
    iss >> ip;

    iss.clear();
    iss.str(values[MAC_PREFIX]);
    iss >> mac[Lease::PREFIX];

    iss.clear();
    iss.str(values[MAC_SUFFIX]);
    iss >> mac[Lease::SUFFIX];

    iss.clear();
    iss.str(values[VID]);
    iss >> vid;

    iss.clear();
    iss.str(values[USED]);
    iss >> used;

    leases.insert(make_pair(ip,new Lease(ip,mac,vid,used)));

    return 0;
}

/* -------------------------------------------------------------------------- */

int Leases::select(SqlDB * db)
{
    ostringstream   oss;
    int             rc;

    set_callback(static_cast<Callbackable::Callback>(&Leases::select_cb));

    oss << "SELECT * FROM " << table << " WHERE oid = " << oid;

    rc = db->exec(oss,this);

    unset_callback();

    if (rc != 0)
    {
        goto error_id;
    }

    return 0;

error_id:
    oss.str("");
    oss << "Error getting leases for network nid: " << oid;

    NebulaLog::log("VNM", Log::ERROR, oss);
    return -1;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

int Leases::drop(SqlDB * db)
{
    ostringstream   oss;

    // Drop all the leases
    oss << "DELETE FROM " << table << " WHERE oid=" << oid;

    return db->exec(oss);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

int Leases::insert(SqlDB * db, string& error_str)
{
    error_str = "Should not access to Leases.insert().";
	NebulaLog::log("VNM", Log::ERROR, error_str);
    return -1;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

int Leases::update(SqlDB * db)
{
	NebulaLog::log("VNM", Log::ERROR, "Should not access to Leases.update()");
    return -1;
}

/* ************************************************************************** */
/* Leases :: Interface Methods                                                */
/* ************************************************************************** */

bool Leases::check(const string& ip)
{
    map<unsigned int,Lease *>::iterator it;

    unsigned int _ip;

    Leases::Lease::ip_to_number(ip,_ip);


    it=leases.find(_ip);

    if (it!=leases.end())
    {
        return it->second->used;
    }
    else
    {
        return false;
    }
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

bool Leases::check(unsigned int ip)
{
    map<unsigned int,Lease *>::iterator it;

    it=leases.find(ip);

    if (it!=leases.end())
    {
        return it->second->used;
    }
    else
    {
        return false;
    }
}

/* ************************************************************************** */
/* Leases :: Misc                                                             */
/* ************************************************************************** */

string& Leases::to_xml(string& xml) const
{
    map<unsigned int, Leases::Lease *>::const_iterator  it;
    ostringstream os;
    string        lease_xml;

    os << "<LEASES>";

    for(it=leases.begin();it!=leases.end();it++)
    {
        os << it->second->to_xml(lease_xml);
    }

    os << "</LEASES>";

    xml = os.str();

    return xml;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

string& Leases::to_str(string& str) const
{
    map<unsigned int, Leases::Lease *>::const_iterator  it;
    ostringstream os;
    string        lease_str;

    for(it=leases.begin();it!=leases.end();it++)
    {
        os << it->second->to_str(lease_str) << endl;
    }

    str = os.str();

    return str;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

ostream& operator<<(ostream& os, Leases& _leases)
{
    string xml;

    os << _leases.to_xml(xml);

    return os;
};

