/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_netctl.c,v $ -- network control interface
 * $Id: mod_netctl.c,v 1.8 2004/02/17 20:40:36 boucman Exp $
 *
 * Copyright (C) by David E. Potter, Ben Reser
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <malloc.h>
#include <netdb.h>
#include <regex.h>

#include "config.h"
#include "irmp3config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_netctl.h"


/*************************************************************************
 * GLOBALS AND STATICS
 */

#define	MOD_NETCTL_BUFSIZE	512

fd_set	 mod_netctl_fdset;
static int	 mod_netctl_listen_socket=0;
int mod_netctl_clients_connected=0;
int mod_netctl_max_clients;
mod_netctl_client_t *clients = NULL;
unsigned int mod_netctl_next_id = 0;

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_netctl = {
	"mod_netctl",
	mod_netctl_deinit,	// deinit
	NULL,			// reload
	&mod_netctl_fdset,	// watch_fdset
	mod_netctl_poll,	// poll
	NULL,			// update
	mod_netctl_message,	// message
	NULL,			// SIGCHLD handler
	mod_netctl_init,
	NULL,			// avoid warning
};

/*************************************************************************
 * write data to client
 */
int mod_netctl_client_write(mod_netctl_client_t *client, char *buf, int nbytes)
{
	if (write(client->socket,buf,nbytes) < 0) {
		mod_netctl_disc_client(client);	// bad write, kill the client
		return 0;
	}
	return 1;
}

/*************************************************************************
 * callback that receives messages and passes them on to clients
 */
void mod_netctl_message (int msgtype, char *msg,const char __attribute__((unused))*sender)
{
	mod_netctl_client_t	*client=0;
	char	buf[MOD_NETCTL_BUFSIZE];

	if(mod_netctl_clients_connected != 0) {
		bzero(buf,sizeof(buf));
		if (msgtype == MSGTYPE_EVENT)   strcpy(buf,"200: ");
		if (msgtype == MSGTYPE_INPUT)   strcpy(buf,"210: ");
		if (msgtype == MSGTYPE_INFO)    strcpy(buf,"220: ");
		if (msgtype == MSGTYPE_PLAYER)  strcpy(buf,"230: ");
		if (msgtype == MSGTYPE_QUERY)  strcpy(buf,"240: ");

		if (strlen(msg) < sizeof(buf)-5) strcat(buf,msg);
		else {
			strncat(buf,msg,sizeof(buf)-5);
			buf[MOD_NETCTL_BUFSIZE]='\0';
		}

		while ((client = mod_netctl_next_client(client))) {
			if (client->socket && ( client->filter.enabled ? !mod_netctl_regex_match(&client->filter.regx,buf) : 1)) {
				if (!mod_netctl_client_write(client,buf,strlen(buf)))
					continue;
				if (!mod_netctl_client_write(client,(char *)"\r\n",(int) 2))
					continue;
			}
		}
	}
}

/*************************************************************************
 * NETWORK POLL FUNCTION
 */
void mod_netctl_poll(int fd)
{

	struct sockaddr_in	cli_addr;
	int					clilen=0,newsockfd,result;
	char				buf[MOD_NETCTL_BUFSIZE];
	char				*tmpbuf=NULL,*ctl_char=NULL;
	char				buffer_overflow[]="Buffer overflow. Clearing buffer.\n\0";
	mod_netctl_client_t		*client;

	if (fd == mod_netctl_listen_socket) {		// new connection?
		log_printf(LOG_DEBUG,"mod_netctl_poll(): client connecting...\n");
		clilen = sizeof(cli_addr);
		newsockfd=accept(fd,(struct sockaddr *) &cli_addr,&clilen);

		if (newsockfd <0) 
			log_printf(LOG_ERROR,"mod_netctl_poll(): accept error.\n");
		else 	mod_netctl_new_client(newsockfd);
	} else {
		client = 0;
		while ((client = mod_netctl_next_client(client))) {
			if(client->socket==fd) {
				
				// client->socket is our socket
				bzero(buf,sizeof(buf));
				result=read(fd,buf,sizeof(buf));
				if (0 >= result)  {
					mod_netctl_disc_client(client);	// bad read, kill the client
					// We must break out of this loop.  We can't coninue to walk the
					// client linked list after deleting an entry.  Any sockets waiting
					// reads should get picked up by the next select anyway.  Which would
					// create a minimal dealay but prevent crashing...
					break; 
				} else {
					log_printf(LOG_NOISYDEBUG,"mod_netctl_poll(): received %d bytes from client %u (fd %d): %s\n",result,client->id,client->socket,buf);
					if (strlen(client->buffer)+result<MOD_NETCTL_BUFSIZE)  // buffer bounds check
						strcat(client->buffer,buf);
					else {		// buffer overflow
						if (!mod_netctl_client_write(client,buffer_overflow,sizeof(buffer_overflow)))
							continue;
						bzero(client->buffer,MOD_NETCTL_BUFSIZE);
						log_printf(LOG_DEBUG,"Buffer overflow for network client %u.\n",client->id);
					}
					while((tmpbuf=memchr(client->buffer,'\r',MOD_NETCTL_BUFSIZE)) || (tmpbuf=memchr(client->buffer,'\n',MOD_NETCTL_BUFSIZE))) {	
							// user pressed enter
						tmpbuf[0]='\0';
						tmpbuf++;
						while(tmpbuf[0]=='\r' || tmpbuf[0]=='\n') // handle cr, lf, and crlf.
							tmpbuf++;
						while((ctl_char=memchr(client->buffer,0xff,strlen(client->buffer)))) 
							// remove telnet ctl chars - always come in sets of 3
							memcpy(client->buffer,ctl_char+3,MOD_NETCTL_BUFSIZE-(ctl_char-client->buffer));
						log_printf(LOG_DEBUG,"mod_netctl_poll(): got command: %s\n",client->buffer);

						if (!strncasecmp(client->buffer,"exit",4)) {
							mod_netctl_disc_client(client);
							// again we must drop out of the loop after deleting a
							// a client.  Prevents us from crashing and burning.
							break;
						}
						// am i receiving a 'network filter' command?
						// if so, handle it and *do not* send it as a message

						// (the hard-coded 14 and 15 in the code below represents 
						// the number of characters in the command "network filter")

						if (!strncasecmp(client->buffer,"network filter",14)) {
							if(client->filter.enabled) regfree(&client->filter.regx);
							if(!regcomp(&client->filter.regx,&(client->buffer[15]),REG_EXTENDED)) client->filter.enabled=1;
								else client->filter.enabled=0;
							log_printf(LOG_NOISYDEBUG,"mod_netctl_poll(): set network filter for client %u to %s.\n",client->id,&(client->buffer[15]));
							
						} else
							mod_sendmsg(MSGTYPE_INPUT,client->buffer);

						if(tmpbuf[0] == '\0' ) {
							bzero(client->buffer,MOD_NETCTL_BUFSIZE);
						} else strcpy(client->buffer,tmpbuf);
					}
				}
			}
		}
	}
}

/*************************************************************************
 * SETUP NEW NETWORK CLIENT BUFFER
 */
void mod_netctl_new_client(int newsockfd)
{
	char 	maxclient[]="Maximum clients already connected.\r\n\0";
	char	welcome[128];

	if (mod_netctl_clients_connected >= mod_netctl_max_clients) {
		write(newsockfd,maxclient,sizeof(maxclient));
		close(newsockfd);
		log_printf(LOG_DEBUG,"mod_netctl_new_client(): connection rejected. Maximum client count (%d) already reached.\n",mod_netctl_max_clients);
	} else {
		mod_netctl_client_t *client = mod_netctl_add_client(newsockfd);

		sprintf(welcome,"100: IRMP3 %s (C) 2000 by Andreas Neuhaus and David Potter.\r\n110:%u You are client %u.\r\n",VERSION,client->id,client->id);
		if (mod_netctl_client_write(client,welcome,strlen(welcome)))
		    log_printf(LOG_DEBUG,"mod_netctl_new_client(): client connection #%u on fd %d.\n",client->id,client->socket);
		else
		    log_printf(LOG_DEBUG,"mod_netctl_new_client(): failed client #%u on fd %d.\n",client->id,client->socket);
	}
}
		

/*************************************************************************
 * TEAR DOWN NETWORK CLIENT BUFFER
 */
void mod_netctl_disc_client(mod_netctl_client_t *client)
{
	log_printf(LOG_DEBUG,"mod_netctl_disc_client(): Disconnecting client #%u on fd %d.\n",client->id,client->socket);

	mod_netctl_remove_client(client);

}


/*************************************************************************
 * NETWORK SETUP FUNCTION
 */
int mod_netctl_net_setup(void)
{
	int			sockfd=0,netport=0;
	struct sockaddr_in	serv_addr;
        int reuse_addr = 1; //We can rebind to our address while a prev. conn. is still in wait-state     
	char *net_intf;

	// open a TCP socket

	if ((sockfd=socket(AF_INET,SOCK_STREAM,0))<0) {
		log_printf(LOG_ERROR,"mod_netctl_net_setup(): cannot open stream socket.\n");
		return(-1);
	}

	// read the desired network port from the logfile

	netport=config_getnum("netctl_port",0);
	if ( !netport) {
		log_printf(LOG_ERROR,"mod_netctl_net_setup(): cannot find network port in irmp3.conf.\n");
		return (-2);
	}
	

	// bind our local address

	bzero((char *) &serv_addr,sizeof(serv_addr));
	serv_addr.sin_family		= AF_INET;
	net_intf = config_getstr("netctl_interface", "all");
	if (strcmp(net_intf,"all")) {
		struct hostent *intf_info;
		intf_info = gethostbyname(net_intf);
		if (!intf_info) {
			log_printf(LOG_ERROR,"mod_netctl_net_setup(): invalid netctl_interface setting (%s) in irmp3.conf.\n",net_intf);
			return(-4);
		} else {
			serv_addr.sin_addr = *(struct in_addr *) intf_info->h_addr;
		}
        } else {
		serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	}
	serv_addr.sin_port		= htons(netport);
        setsockopt(sockfd,SOL_SOCKET, SO_REUSEADDR,&reuse_addr,sizeof(reuse_addr));
	if (bind(sockfd,(struct sockaddr *) &serv_addr,sizeof(serv_addr)) < 0) {
		log_printf(LOG_ERROR,"mod_netctl_net_setup(): cannot bind local address.\n");
		return (-3);
	} else 	log_printf(LOG_DEBUG,"mod_netctl_net_setup(): listening on port %d.\n",netport);

	listen(sockfd,5);

	// handle signals
	signal(SIGPIPE, SIG_IGN);

	return sockfd;

}

/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_netctl_init (void)
{
	// Check to see if netctl support is requested
	if (!strcasecmp(config_getstr("netctl_enabled", "yes"), "no")) {
		log_printf(LOG_DEBUG, "mod_netctl_init(): mod_netctl is disabled.\n");
		return NULL;
	}
	                                
	FD_ZERO(&mod_netctl_fdset);

	mod_netctl_max_clients = config_getnum("netctl_max_clients",5);
	
	//setup the linked list
	clients = NULL;

	// register our module
	mod_netctl_listen_socket=mod_netctl_net_setup();
	if (0 <= mod_netctl_listen_socket) {
		log_printf(LOG_DEBUG, "mod_netctl_init(): initializing on fd %d.\n",mod_netctl_listen_socket);
		FD_SET(mod_netctl_listen_socket,&mod_netctl_fdset);
	}

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_netctl_deinit (void)
{
	mod_netctl_remove_all_clients();
	log_printf(LOG_DEBUG, "mod_netctl_deinit(): deinitialized\n");
}


/*************************************************************************
 * REGEX MATCHING FUNCTION
 *
 * returns positive if match found
 */
int mod_netctl_regex_match(regex_t *preg, char *string)
{
	int 		result=0,eflags=0;
	regmatch_t	pmatch[1];
	size_t		nmatch=0;

	result=regexec(preg,string,nmatch,pmatch,eflags);
	if(!result) return 1;
	else return 0;
}

/*************************************************************************
 * find the next entry in the linked list
 * if passed 0 for the param it returns the first entry...
 */
mod_netctl_client_t *mod_netctl_next_client(mod_netctl_client_t *client)
{
	if (!client) {
		return clients;
	} else {
		return client->next;
	}
}
/*************************************************************************
 * find the last entry in the linked list
 */
mod_netctl_client_t *mod_netctl_last_client(void)
{
	mod_netctl_client_t *client = 0;
	while ((client = mod_netctl_next_client(client))) {
		if (!client->next) {
			return client;
		}
	}
	return NULL; // just in case
}

/*************************************************************************
 * adds a client to the linked list
 */
mod_netctl_client_t *mod_netctl_add_client(int newsockfd)
{
	mod_netctl_client_t *new_client = (mod_netctl_client_t *) malloc(sizeof(mod_netctl_client_t));
	new_client->id = mod_netctl_next_id++;
	new_client->socket = newsockfd;
	new_client->buffer = (char *) malloc(MOD_NETCTL_BUFSIZE);
	new_client->filter.enabled = 0;
	new_client->next = 0;
	bzero(new_client->buffer,MOD_NETCTL_BUFSIZE);
	FD_SET(new_client->socket, &mod_netctl_fdset);

	if (!clients) {
  		// If we don't have any clients make this the top of the linked lists 
		clients = new_client;
		new_client->prev = 0;
  	} else {
		// walk the list until we find one that has 0 as it's next entry
		// and make the link.
		mod_netctl_client_t *last_client = mod_netctl_last_client();
		last_client->next = new_client;
		new_client->prev = last_client;
  	}

	mod_netctl_clients_connected++;
  
	return new_client;  
}

/*************************************************************************
 * removes a client from the linked list
 */
void mod_netctl_remove_client (mod_netctl_client_t *client) {
	FD_CLR(client->socket,&mod_netctl_fdset);
	if (client->socket) {
		close(client->socket);
	}
	free(client->buffer);
	if (client->filter.enabled) {
		regfree(&client->filter.regx);
	}
	if (client->prev) {
		client->prev->next = client->next;
	} else {
		// if we don't have a previous then we're deleting the first
		// entry and we ned to reset the clients pointer.
        	clients = client->next;
	}
	if (client->next) {
		client->next->prev = client->prev;
	}
	mod_netctl_clients_connected--;
}

/*************************************************************************
 * removes all clients from the linked lists.
 */
void mod_netctl_remove_all_clients (void) {
	mod_netctl_client_t *client = 0;
	while ((client = mod_netctl_next_client(client))) {
		mod_netctl_remove_client(client);
	}
}

/*************************************************************************
 * closes all the connections, handy for processes that we fork
 */
void mod_netctl_close_all_clients (void) {
	mod_netctl_client_t *client = 0;
	while ((client = mod_netctl_next_client(client))) {
		if (client->socket) {
			close(client->socket);
		}
	}
}

/*************************************************************************
 * EOF
 */
