#!/usr/local/bin/lua -f

-- Load modules.
--LUADOC_HOME = "/usr/local/luadoc"
LUADOC_HOME = "."
LUADOC_SUB = LUADOC_HOME.."/sub.lua"
LUADOC_CMP = LUADOC_HOME.."/cmp.lua"
assert (dofile (LUADOC_HOME.."/analyze.lua"))
assert (dofile (LUADOC_HOME.."/compose.lua"))

-- Print version number.
function print_version ()
	print ("LuaDoc "..LUADOC_VERSION)
end

-- Print usage message.
function print_help ()
	print ("Usage: "..arg[0]..[[ [options|files]
Extract documentation from files.  Available options are:
  -d path               output directory path
  -f "<find>=<repl>"    define a substitution filter (only string patterns)
  -g "<find>=<repl>"    define a substitution filter (gsub patterns)
  -h, --help            print this help and exit
      --noindexpage     do not generate global index page
  -q, --quiet           suppress all normal output
  -v, --version         print version information]])
end

function off_messages (arg, i, options)
	options.verbose = nil
end

-- Global filters.
FILTERS = {}
-- Process options.
OPTIONS = {
	d = function (arg, i, options)
		local dir = arg[i+1]
		if strsub (dir, -2) ~= "/" then
			dir = dir..'/'
		end
		options.output_dir = dir
		return 1
	end,
	f = function (arg, i, options)
		local sub = arg[i+1]
		local find = gsub (sub, '^([^=]+)%=.*$', '%1')
		find = gsub (find, "(%p)", "%%%1")
		local repl = gsub (sub, '^.-%=([^"]+)$', '%1')
		repl = gsub (repl, "(%p)", "%%%1")
		tinsert (FILTERS, { find, repl })
		return 1
	end,
	g = function (arg, i, options)
		local sub = arg[i+1]
		local find = gsub (sub, '^([^=]+)%=.*$', '%1')
		local repl = gsub (sub, '^.-%=([^"]+)$', '%1')
		tinsert (FILTERS, { find, repl })
		return 1
	end,
	h = print_help,
	help = print_help,
	q = off_messages,
	quiet = off_messages,
	v = print_version,
	version = print_version,
}
DEFAULT_OPTIONS = {
	output_dir = "",
	verbose = 1,
}
function process_options (arg)
	local files = {}
	local options = DEFAULT_OPTIONS
	for i = 1, getn(arg) do
		local argi = arg[i]
		if strsub (argi, 1, 1) ~= '-' then
			tinsert (files, argi)
		else
			local opt = strsub (argi, 2)
			if strsub (opt, 1, 1) == '-' then
				opt = gsub (opt, "%-", "")
			end
			if OPTIONS[opt] then
				if OPTIONS[opt] (arg, i, options) then
					i = i+1
				end
			else
				options[opt] = 1
			end
		end
	end
	return files, options
end 

-- Process options.
local argc = getn(arg)
if argc < 1 then
	print_help ()
end
local files, options = process_options (arg)

-- Process files.
for i = 1, getn(files) do
	local f = files[i]
	local h = gsub (f, "lua$", "html")
	h = options.output_dir..gsub (h, "^.-([%w_]+%.html)$", "%1")
	if options.verbose then
		print ("Processando "..f.." (=> "..h..")")
	end
	compose (analyze (f, LUADOC_SUB), LUADOC_CMP, h)
end

-- Generate index file.
if (getn(files) > 0) and (not options.noindexpage) then
	index (options.output_dir)
end
