/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Collections;

namespace Cowbell.Base
{
	public enum ModificationType
	{
		Add,
		Change,
		Delete
	}

	public class SongDatabase : ArrayList
	{
		/* public events */
		public event DatabaseModifiedHandler Modified;
		public event VoidHandler GlobalDataModified;
		public event VoidHandler MultipleDiscsChanged;
		public event VoidHandler MultipleArtistsChanged;
		
		/* public properties */
		public bool MultipleArtists {
			get { return multipleartists; }
			set {
				multipleartists = value;
				if (MultipleArtistsChanged != null) {
					MultipleArtistsChanged ();
				}
			}
		}

		public bool MultipleDiscs {
			get { return multiplediscs; }
			set {
				multiplediscs = value;
				if (MultipleDiscsChanged != null) {
					MultipleDiscsChanged ();
				}
			}
		}

		public TimeSpan PlayTime {
			get {
				TimeSpan ret = new TimeSpan ();
				for (int i = 0; i < Count; i++)
				{
					if (this[i] is Song) {
						ret += ((Song)this[i]).Length;
					}
				}
				return ret;
			}
		}

		public Song GlobalData {
			get { return globaldata; }
			set {
				globaldata = new Song ();
				globaldata.Artist = value.Artist;
				globaldata.Album = value.Album;
				globaldata.Year = value.Year;
				globaldata.Genre = value.Genre;

				globaldata.Changed += new SongChangedHandler (OnGlobalDataModified);

				OnGlobalDataModified (null);
			}
		}

		public override object this[int index] {
			get { return base[index]; }
			set {
				base[index] = value;

				if (Modified != null) {
					Modified (index, ModificationType.Change);
				}
			}
		}

		/* public methods */
		public SongDatabase () : base (20)
		{
		}

		public override int Add (object o)
		{
			int ret = base.Add (o);

			if (Modified != null) {
				Modified (ret, ModificationType.Add);
			}

			return ret;
		}

		public override void RemoveAt (int location)
		{
			if (Modified != null) {
				Modified (location, ModificationType.Delete);
			}

			base.RemoveAt (location);
		}

		public override void Clear ()
		{
			for (int i = 0; i < Count; i++)
			{
				if (Modified != null) {
					Modified (i, ModificationType.Delete);
				}
			}

			base.Clear ();
		}

		public void RenameAll (string pattern)
		{
			if (pattern == "") {
				return;
			}

			foreach (Song s in this)
			{
				string ext = System.IO.Path.GetExtension (s.Filename);
				string path = System.IO.Path.GetDirectoryName (s.Filename);
				string filename = FilenamePattern.Parse (s, pattern) + ext;

				filename = filename.Replace (System.IO.Path.DirectorySeparatorChar, ' ');

				path = System.IO.Path.Combine (path, filename);
				if (path.IndexOfAny (System.IO.Path.InvalidPathChars) > -1) {
					// Check for any invalid characters and replace them with a _
					string[] tokens = path.Split (System.IO.Path.InvalidPathChars);
					path = String.Join ("_", tokens);
				}

				if (!System.IO.File.Exists (path)) {
					s.Rename (path);
				}
			}
		}

		public void RevertAll ()
		{
			foreach (Song s in this)
			{
				s.Revert ();
			}

			GlobalData = (Song)this[0];
		}

		public bool HasMultipleArtists ()
		{
			if (Count < 1) {
				return false;
			}

			string artist = ((Song)this[1]).Artist;
			for (int i = 0; i < Count; i++)
			{
				if (artist != ((Song)this[i]).Artist) {
					return false;
				}
			}

			return true;
		}

		public bool HasMultipleDiscs ()
		{
			if (Count < 1) {
				return false;
			}

			string album = ((Song)this[1]).Album;
			for (int i = 0; i < Count; i++)
			{
				if (album != ((Song)this[i]).Album) {
					return false;
				}
			}

			return true;
		}

		/* private fields */
		private bool multipleartists;
		private bool multiplediscs;
		private Song globaldata = null;

		/* private methods */
		private void OnGlobalDataModified (Song s)
		{
			Runtime.Debug ("GlobalDataModified called with song:");
			Runtime.Debug (GlobalData);
			if (GlobalDataModified != null) {
				GlobalDataModified ();
			}
		}
	}
}
