// distribution boxbackup-0.11rc2 (svn version: 2072)
//  
// Copyright (c) 2003 - 2008
//      Ben Summers and contributors.  All rights reserved.
//  
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. All use of this software and associated advertising materials must 
//    display the following acknowledgment:
//        This product includes software developed by Ben Summers.
// 4. The names of the Authors may not be used to endorse or promote
//    products derived from this software without specific prior written
//    permission.
// 
// [Where legally impermissible the Authors do not disclaim liability for 
// direct physical injury or death caused solely by defects in the software 
// unless it is modified by a third party.]
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
// ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//  
//  
//  
// --------------------------------------------------------------------------
//
// File
//		Name:    Archive.h
//		Purpose: Backup daemon state archive
//		Created: 2005/04/11
//
// --------------------------------------------------------------------------

#ifndef ARCHIVE__H
#define ARCHIVE__H

#include <vector>
#include <string>
#include <memory>

#include "IOStream.h"
#include "Guards.h"

#define ARCHIVE_GET_SIZE(hdr)		(( ((uint8_t)((hdr)[0])) | ( ((uint8_t)((hdr)[1])) << 8)) >> 2)

#define ARCHIVE_MAGIC_VALUE_RECURSE 0x4449525F
#define ARCHIVE_MAGIC_VALUE_NOOP 0x5449525F

class Archive
{
public:
	Archive(IOStream &Stream, int Timeout)
		: mrStream(Stream)
	{
		mTimeout = Timeout;
	}
private:
	// no copying
	Archive(const Archive &);
	Archive & operator=(const Archive &);
public:
	~Archive()
	{
	}
	//
	//
	//
	void Write(bool Item)
	{
		Write((int) Item);
	}
	void Write(int Item)
	{
		int32_t privItem = htonl(Item);
		mrStream.Write(&privItem, sizeof(privItem));
	}
	void Write(int64_t Item)
	{
		int64_t privItem = box_hton64(Item);
		mrStream.Write(&privItem, sizeof(privItem));
	}
	void Write(uint64_t Item)
	{
		uint64_t privItem = box_hton64(Item);
		mrStream.Write(&privItem, sizeof(privItem));
	}
	void Write(uint8_t Item)
	{
		int privItem = Item;
		Write(privItem);
	}
	void Write(const std::string &Item)
	{
		int size = Item.size();
		Write(size);
		mrStream.Write(Item.c_str(), size);
	}
	//
	//
	//
	void Read(bool &rItemOut)
	{
		int privItem;
		Read(privItem);

		if (privItem)
		{
			rItemOut = true;
		}
		else
		{
			rItemOut = false;
		}
	}
	void Read(int &rItemOut)
	{
		int32_t privItem;
		if(!mrStream.ReadFullBuffer(&privItem, sizeof(privItem), 0 /* not interested in bytes read if this fails */))
		{
			THROW_EXCEPTION(CommonException, ArchiveBlockIncompleteRead)
		}
		rItemOut = ntohl(privItem);
	}
	void Read(int64_t &rItemOut)
	{
		int64_t privItem;
		if(!mrStream.ReadFullBuffer(&privItem, sizeof(privItem), 0 /* not interested in bytes read if this fails */))
		{
			THROW_EXCEPTION(CommonException, ArchiveBlockIncompleteRead)
		}
		rItemOut = box_ntoh64(privItem);
	}
	void Read(uint64_t &rItemOut)
	{
		uint64_t privItem;
		if(!mrStream.ReadFullBuffer(&privItem, sizeof(privItem), 0 /* not interested in bytes read if this fails */))
		{
			THROW_EXCEPTION(CommonException, ArchiveBlockIncompleteRead)
		}
		rItemOut = box_ntoh64(privItem);
	}
	void Read(uint8_t &rItemOut)
	{
		int privItem;
		Read(privItem);
		rItemOut = privItem;
	}
	void Read(std::string &rItemOut)
	{
		int size;
		Read(size);

		// Assume most strings are relatively small
		char buf[256];
		if(size < (int) sizeof(buf))
		{
			// Fetch rest of pPayload, relying on the Protocol to error on stupidly large sizes for us
			if(!mrStream.ReadFullBuffer(buf, size, 0 /* not interested in bytes read if this fails */, mTimeout))
			{
				THROW_EXCEPTION(CommonException, ArchiveBlockIncompleteRead)
			}
			// assign to this string, storing the header and the extra payload
			rItemOut.assign(buf, size);
		}
		else
		{
			// Block of memory to hold it
			MemoryBlockGuard<char*> dataB(size);
			char *ppayload = dataB;

			// Fetch rest of pPayload, relying on the Protocol to error on stupidly large sizes for us
			if(!mrStream.ReadFullBuffer(ppayload, size, 0 /* not interested in bytes read if this fails */, mTimeout))
			{
				THROW_EXCEPTION(CommonException, ArchiveBlockIncompleteRead)
			}
			// assign to this string, storing the header and the extra pPayload
			rItemOut.assign(ppayload, size);
		}
	}
private:
	IOStream &mrStream;
	int mTimeout;
};

#endif // ARCHIVE__H
