// vscreen.cc
//
//  Copyright 1999,2000,2001 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Implementation of vscreen stuff.

// We need this for recursive mutexes.  Should all files be compiled
// with this defined??
#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include "vscreen.h"
#include "curses++.h"
#include "vscreen_widget.h"

#include "vs_editline.h"
#include "vs_menu.h"
#include "vs_menubar.h"
#include "vs_pager.h"
#include "vs_statuschoice.h"
#include "vs_table.h"
#include "vs_text_layout.h"
#include "vs_tree.h"

#include "config/keybindings.h"
#include "config/colors.h"

// For _()
#include "../aptitude.h"

#include <signal.h>

#ifdef HAVE_LIBPTHREAD
#include <pthread.h>
#endif

#include <assert.h>
#include <sys/time.h>

#ifdef HAVE_LIBPTHREAD
static pthread_mutex_t vscreen_mutex;
#endif

using namespace std;

bool curses_avail=false;
bool should_exit=false;

// Used to queue and merge update requests
static bool needs_layout=false;
static bool needs_update=false;
static bool needs_cursorupdate=false;

SigC::Signal0<void> main_hook;

struct timeout_info
{
  SigC::Slot0<void> f;
  int num;
  timeval activate_time; // tells when this timeout should be triggered
  timeout_info(const SigC::Slot0<void> &_f, int _num, timeval &_activate_time)
    :f(_f), num(_num), activate_time(_activate_time)
  {}
};

list<timeout_info> timeouts;
// Timeouts which have been registered.  For optimal speed this should be a
// heap, but there are other considerations...it might be necessary for callers
// to delete arbitrary timeouts, and while maintaining heap order with
// arbitrary deletions might be possible, it's a bit of a nuisance.  Also,
// I don't expect this to ever hold more than a handful of values.
//  In any event, the external interface doesn't depend on details like
// this, so it will be fairly simple to switch to using a more efficient
// data structure internally if this becomes a problem.  I notice that glib
// uses something similar to this, and it's efficient enough..  Optimizing
// stuff that doesn't need to be optimized is a waste of time.
int ntimeouts=0;
// Cheesy way of generating mostly-unique IDs for timeouts -- just incremented 
// once for every timeout that gets added.

static vscreen_widget *toplevel=NULL;
// The widget which is displayed as the root of everything

//static bool resized=false;
static void sigresize(int sig)
{
  vscreen_handleresize();
  //resized=true;
}

// Cleanly shutdown (eg, restore screen settings if possible)
//
// Called on SIGTERM, SIGINT, SIGSEGV, SIGABRT, and SIGQUIT
//
// FIXME: revert to the /previous/ handler, not just SIG_DFL?
static void sigkilled(int sig)
{
  endwin();

  switch(sig)
    {
    case SIGTERM:
      fprintf(stderr, _("Ouch!  Got SIGTERM, dying..\n"));
      break;
    case SIGSEGV:
      fprintf(stderr, _("Ouch!  Got SIGSEGV, dying..\n"));
      break;
    case SIGABRT:
      fprintf(stderr, _("Ouch!  Got SIGABRT, dying..\n"));
      break;
    case SIGQUIT:
      fprintf(stderr, _("Ouch!  Got SIGQUIT, dying..\n"));
      break;
    }

  signal(sig, SIG_DFL);
  raise(sig);
}

///////////////////////////////////////////////////////////////////////////////
// The following function comes from the glibc documentation.
// Subtract the `struct timeval' values X and Y,
// storing the result in RESULT.
// Return 1 if the difference is negative, otherwise 0.

static int
timeval_subtract (timeval *result, timeval *x, timeval *y)
{
  /* Perform the carry for the later subtraction by updating Y. */
  if (x->tv_usec < y->tv_usec)
    {
      int nsec = (y->tv_usec - x->tv_usec) / 1000000 + 1;
      y->tv_usec -= 1000000 * nsec;
      y->tv_sec += nsec;
    }
  if (x->tv_usec - y->tv_usec > 1000000)
    {
      int nsec = (x->tv_usec - y->tv_usec) / 1000000;
      y->tv_usec += 1000000 * nsec;
      y->tv_sec -= nsec;
    }

  /* Compute the time remaining to wait.
     `tv_usec' is certainly positive. */
  result->tv_sec = x->tv_sec - y->tv_sec;
  result->tv_usec = x->tv_usec - y->tv_usec;

  /* Return 1 if result is negative. */
  return x->tv_sec < y->tv_sec;
}
///////////////////////////////////////////////////////////////////////////////

void vscreen_settoplevel(vscreen_widget *w)
{
  if(toplevel)
    toplevel->unfocussed();

  toplevel=w;

  if(curses_avail)
    {
      toplevel->set_owner_window(rootwin, 0, 0, rootwin.getmaxx(), rootwin.getmaxy());
      toplevel->show_all();
      toplevel->focussed();
      vscreen_redraw();
    }
}

void vscreen_init()
{
  keybinding upkey, downkey, leftkey, rightkey, quitkey, homekey, endkey;
  keybinding historynextkey, historyprevkey;
  keybinding delfkey, delbkey, ppagekey, npagekey;
  keybinding undokey, helpkey, researchkey;
  keybinding menutogglekey, cancelkey;

  upkey.push_back(KEY_UP);
  upkey.push_back('k');
  downkey.push_back(KEY_DOWN);
  downkey.push_back('j');
  leftkey.push_back(KEY_LEFT);
  leftkey.push_back('h');
  rightkey.push_back(KEY_RIGHT);
  rightkey.push_back('l');
  quitkey.push_back("q"[0]);

  historyprevkey.push_back(KEY_UP);
  historyprevkey.push_back(KEY_CTRL('p'));
  historynextkey.push_back(KEY_DOWN);
  historynextkey.push_back(KEY_CTRL('n'));

  homekey.push_back(KEY_HOME);
  homekey.push_back(KEY_CTRL("a"[0]));
  endkey.push_back(KEY_END);
  endkey.push_back(KEY_CTRL("e"[0]));

  delfkey.push_back(KEY_DC);
  delfkey.push_back(KEY_CTRL("d"[0]));

  delbkey.push_back(KEY_BACKSPACE);
  delbkey.push_back(KEY_CTRL("h"[0]));

  ppagekey.push_back(KEY_PPAGE);
  ppagekey.push_back(KEY_CTRL("b"[0]));

  npagekey.push_back(KEY_NPAGE);
  npagekey.push_back(KEY_CTRL("f"[0]));

  undokey.push_back(KEY_CTRL("u"[0]));
  undokey.push_back(KEY_CTRL('_'));

  helpkey.push_back('?');
  helpkey.push_back(KEY_CTRL("h"[0]));
  helpkey.push_back(KEY_F(1));

  menutogglekey.push_back(KEY_F(10));
  menutogglekey.push_back(KEY_CTRL(' '));

  cancelkey.push_back(KEY_CTRL("g"[0]));
  cancelkey.push_back('\e');
  cancelkey.push_back(KEY_CTRL('['));

  researchkey.push_back('\\');
  researchkey.push_back('n');

  init_curses();

  mousemask(ALL_MOUSE_EVENTS, NULL);

  curses_avail=true;
  cbreak();
  rootwin.keypad(true);

  global_bindings.set("Quit", quitkey);
  global_bindings.set("Cycle", '\t');
  global_bindings.set("Refresh", KEY_CTRL("l"[0]));

  global_bindings.set("Up", upkey);
  global_bindings.set("Down", downkey);
  global_bindings.set("LevelDown", "J"[0]);
  global_bindings.set("LevelUp", "K"[0]);
  global_bindings.set("Left", leftkey);
  global_bindings.set("Right", rightkey);
  global_bindings.set("HistoryNext", historynextkey);
  global_bindings.set("HistoryPrev", historyprevkey);
  global_bindings.set("Parent", '^');
  global_bindings.set("PrevPage", ppagekey);
  global_bindings.set("NextPage", npagekey);
  global_bindings.set("Begin", homekey);
  global_bindings.set("End", endkey);
  global_bindings.set("Search", '/');
  global_bindings.set("ReSearch", researchkey);
  global_bindings.set("DelBack", delbkey);
  global_bindings.set("DelForward", delfkey);

  global_bindings.set("DelEOL", KEY_CTRL("k"[0]));
  global_bindings.set("DelBOL", KEY_CTRL("u"[0]));

  global_bindings.set("Confirm", KEY_ENTER);
  global_bindings.set("Cancel", cancelkey);
  global_bindings.set("Undo", undokey);
  global_bindings.set("Help", helpkey);
  global_bindings.set("ToggleMenuActive", menutogglekey);
  global_bindings.set("PushButton", ' ');
  global_bindings.set("Yes", _("yes_key")[0]);
  global_bindings.set("No", _("no_key")[0]);

  global_bindings.set("ToggleExpanded", KEY_ENTER);
  global_bindings.set("ExpandAll", '[');
  global_bindings.set("CollapseAll", ']');
  global_bindings.set("SelectParent", '^');

  vs_editline::init_bindings();
  vs_menu::init_bindings();
  vs_menubar::init_bindings();
  vs_pager::init_bindings();
  vs_statuschoice::init_bindings();
  vs_table::init_bindings();
  vs_text_layout::init_bindings();
  vs_tree::init_bindings();

  set_color("Error", COLOR_WHITE, COLOR_RED, A_BOLD);

  set_color("DefaultWidgetBackground", COLOR_WHITE, COLOR_BLACK, 0);

  set_color("ScreenHeaderColor", COLOR_WHITE, COLOR_BLUE, A_BOLD);
  set_color("ScreenStatusColor", COLOR_WHITE, COLOR_BLUE, A_BOLD);

  set_color("MenuEntry", COLOR_WHITE, COLOR_BLUE, 0);
  set_color("MenuBorder", COLOR_WHITE, COLOR_BLUE, A_BOLD);
  set_color("HighlightedMenuEntry", COLOR_WHITE, COLOR_BLUE, A_BOLD|A_REVERSE);
  set_color("DisabledMenuEntry", COLOR_BLACK, COLOR_BLUE, A_DIM);
  set_color("MenuBar", COLOR_WHITE, COLOR_BLUE, A_BOLD);
  set_color("HighlightedMenuBar", COLOR_BLUE, COLOR_WHITE, A_BOLD);

  set_color("EditLine", COLOR_WHITE, COLOR_BLACK, 0);

  set_color("WidgetFrame", COLOR_WHITE, COLOR_BLACK, 0);

  set_color("TreeBackground", COLOR_WHITE, COLOR_BLACK, 0);

  if(toplevel)
    vscreen_settoplevel(toplevel);

  pthread_mutexattr_t vscreen_mutex_attr;
  pthread_mutexattr_init(&vscreen_mutex_attr);
  pthread_mutexattr_settype(&vscreen_mutex_attr, PTHREAD_MUTEX_RECURSIVE);

  pthread_mutex_init(&vscreen_mutex, &vscreen_mutex_attr);

  pthread_mutexattr_destroy(&vscreen_mutex_attr);

  vscreen_install_sighandlers();
}

void vscreen_install_sighandlers()
{
  signal(SIGTERM, sigkilled);
  signal(SIGINT, sigkilled);
  signal(SIGSEGV, sigkilled);
  signal(SIGQUIT, sigkilled);
  signal(SIGABRT, sigkilled);
}

void vscreen_handleresize()
{
  toplevel->set_owner_window(NULL, 0, 0, 0, 0);
  resize();
  toplevel->set_owner_window(rootwin, 0, 0, rootwin.getmaxx(), rootwin.getmaxy());
  vscreen_redraw();
}

void vscreen_checktimeouts()
{
  list<timeout_info>::iterator i,j;
  for(i=timeouts.begin(); i!=timeouts.end(); i=j)
    {
      j=i;
      j++;
      timeval result,curtime;
      gettimeofday(&curtime, 0);

      if(timeval_subtract(&result, &i->activate_time, &curtime)==1 ||
	 result.tv_sec==0 && result.tv_usec<=10)
	// Fudge things a little.. (good idea or not?)
	{
	  i->f();
	  timeouts.erase(i);
	}
    }

  vscreen_tryupdate();
}

int vscreen_mindelay()
  // Returns the number of milliseconds to delay for the first timeout. (-1 for
  // infinite delay)
  // If a timeout should be done now, returns 0.
{
  bool found_one=false;
  timeval mintime;
  mintime.tv_sec=INT_MAX/1000;
  mintime.tv_usec=(INT_MAX%1000)*1000;

  timeval curtime;
  gettimeofday(&curtime, 0);

  list<timeout_info>::iterator i,j;
  for(i=timeouts.begin(); i!=timeouts.end(); i=j)
    {
      j=i;
      j++;
      timeval diff;
      if(timeval_subtract(&diff, &i->activate_time, &curtime)==1 ||
	 diff.tv_sec==0 && diff.tv_usec<=10)
	return 0;
      else
	{
	  if(diff.tv_sec<mintime.tv_sec ||
	     (diff.tv_sec==mintime.tv_sec && diff.tv_usec<mintime.tv_usec))
	  {
	    found_one=true;
	    mintime.tv_sec=diff.tv_sec;
	    mintime.tv_usec=diff.tv_usec;
	  }
	}
    }
  if(found_one)
    return mintime.tv_sec*1000+mintime.tv_usec/1000;
  else
    return -1;
}

void vscreen_updatecursor()
{
  needs_cursorupdate=true;
}

void vscreen_updatecursornow()
{
  if(toplevel->get_cursorvisible())
    {
      point p=toplevel->get_cursorloc();
      toplevel->win.move(p.y, p.x);
      toplevel->win.leaveok(false);
    }
  else
    toplevel->win.leaveok(true);

  needs_cursorupdate=false;
}

void vscreen_update()
{
  needs_update=true;
  needs_cursorupdate=true;
}

void vscreen_updatenow()
{
  if(toplevel)
    {
      toplevel->display();
      toplevel->sync();

      vscreen_updatecursornow();

      doupdate();
    }
  needs_update=false;
}

void vscreen_layoutnow()
{
  toplevel->do_layout();
  needs_layout=false;
}

void vscreen_tryupdate()
{
  if(needs_layout)
    vscreen_layoutnow();

  if(needs_update)
    vscreen_updatenow();

  if(needs_cursorupdate)
    {
      vscreen_updatecursornow();
      refresh();
    }
}

bool vscreen_poll()
  // FIXME: should I take the global lock?  Need to think about this before
  //       using libvscreen for threaded programming
{
  bool rval=false;

  sigset_t signals,prevsignals;

  sigemptyset(&signals);
  sigaddset(&signals, SIGWINCH);

  if(toplevel)
    {
      chtype ch;

      do
	{
	  toplevel->win.nodelay(true);
	  ch=toplevel->win.getch();
	  toplevel->win.nodelay(false);
	} while(ch==(chtype) KEY_RESIZE);

      sigprocmask(SIG_BLOCK, &signals, &prevsignals);

      if(ch!=(chtype) ERR)
	{
	  rval=true;

	  if(ch==KEY_MOUSE)
	    {
	      MEVENT ev;
	      getmouse(&ev);

	      if(toplevel)
		{
		  toplevel->dispatch_mouse(ev.id, ev.x, ev.y, ev.z, ev.bstate);
		  main_hook();
		  vscreen_tryupdate();
		}
	    }
	  else if(global_bindings.key_matches(ch, "Refresh"))
	    vscreen_redraw();
	  else if(toplevel)
	    {
	      toplevel->dispatch_char(ch);
	      main_hook();
	      vscreen_tryupdate();
	    }
	}
    }

  vscreen_checktimeouts();

  sigprocmask(SIG_SETMASK, &signals, &prevsignals);

  return rval;
}

void vscreen_mainloop(int synch)
  // FIXME: can I restructure things so that this just calls vscreen_poll
  // repeatedly?  It'd be a lot cleaner..
{
  static int main_level=0;
  // HACK -- basically, delayed deletion avoids crashes by guaranteeing that
  // things only get deleted in the main loop.  But this breaks if sub-main-loops
  // delete them.  Solution: don't delete things in sub-main-loops..

  main_level++;

  sigset_t signals;

  sigemptyset(&signals);
  sigaddset(&signals, SIGWINCH);

  signal(SIGWINCH, sigresize);

  vscreen_acquirelock();

  // Do this first, so any pending updates from initialization routines get
  // flushed.
  vscreen_tryupdate();

  while(!should_exit && toplevel)
    {
      sigset_t prevsignals;

      assert(toplevel->get_win());

      int prevtimeout=toplevel->timeout(vscreen_mindelay());
      vscreen_widget *curwidget=toplevel;
      // Note that something bad might happen while we don't have the mutex..
      // OTOH, it's perfectly okay to dispatch the character to an unexpected
      // location; I just want to be sure that I reset the timeout of the
      // correct widget. (??)

      chtype ch;

      do
	{
	  vscreen_releaselock();
	  ch=curwidget->get_win().getch();
	  vscreen_acquirelock();
	} while(ch==KEY_RESIZE);

      // used when asynchronous resizing was dangerous
      //if(resized)
      //	{
      //  resized=false;
      //  vscreen_handleresize();
      //}

      curwidget->timeout(prevtimeout);

      sigprocmask(SIG_BLOCK, &signals, &prevsignals);
      // Oddly enough, the part I want to *protect* from signals is my own code
      // -- the ncurses code above is actually graceful about this situation!

      vscreen_checktimeouts();

      if(ch==KEY_MOUSE)
	{
	  MEVENT ev;
	  getmouse(&ev);

	  if(toplevel)
	    {
	      toplevel->dispatch_mouse(ev.id, ev.x, ev.y, ev.z, ev.bstate);
	      main_hook();
	      vscreen_tryupdate();
	    }
	}
      else if(global_bindings.key_matches(ch, "Refresh"))
	vscreen_redraw();
      else if(ch!=(chtype) ERR && toplevel)
	{
	  toplevel->dispatch_char(ch);
	  main_hook();
	  vscreen_tryupdate();
	}

      if(main_level==1)
	vscreen_widget::handle_pending_deletes();

      sigprocmask(SIG_SETMASK, &prevsignals, NULL);
    }

  should_exit=false;

  vscreen_releaselock();

  main_level--;
}

void vscreen_exitmain()
{
  should_exit=1;
}

void vscreen_suspend()
{
  if(toplevel)
    toplevel->set_owner_window(NULL, 0, 0, 0, 0);
  endwin();
  curses_avail=false;
}

void vscreen_resume()
{
  curses_avail=true;
  if(toplevel)
    {
      toplevel->set_owner_window(rootwin, 0, 0, rootwin.getmaxx(), rootwin.getmaxy());
      toplevel->display();
      toplevel->sync();
      doupdate();
    }
  else
    refresh();
}

void vscreen_queuelayout()
{
  needs_layout=true;
  needs_update=true;
  needs_cursorupdate=true;
}

void vscreen_redraw()
{
  if(toplevel)
    {
      toplevel->focussed();
      toplevel->get_win().touch();
      toplevel->get_win().clearok(true);
      toplevel->do_layout();
      toplevel->display();
      vscreen_updatecursornow();
      toplevel->sync();
      doupdate();
      toplevel->get_win().clearok(false);
    }
}

int vscreen_addtimeout(const SigC::Slot0<void> &slot, int msecs)
{
  if(msecs<0)
    return -1;

  int rval=ntimeouts++;
  timeval activate_time;
  gettimeofday(&activate_time, 0);
  activate_time.tv_sec+=msecs/1000;
  activate_time.tv_usec+=(msecs%1000)*1000;
  while(activate_time.tv_usec>1000*1000)
    // Should only run through once
    {
      activate_time.tv_sec++;
      activate_time.tv_usec-=1000*1000;
    }

  timeouts.push_back(timeout_info(slot, rval, activate_time));
  return rval;
}

void vscreen_deltimeout(int num)
{
  for(list<timeout_info>::iterator i=timeouts.begin(); i!=timeouts.end(); i++)
    {
      if(i->num==num)
	{
	  timeouts.erase(i);
	  return;
	}
    }
}

void vscreen_acquirelock()
{
#ifdef HAVE_LIBPTHREAD
  pthread_mutex_lock(&vscreen_mutex);
#endif
}

void vscreen_releaselock()
{
#ifdef HAVE_LIBPTHREAD
  pthread_mutex_unlock(&vscreen_mutex);
#endif
}

