// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/signin/profile_picker_handler.h"

#include <vector>

#include "base/check.h"
#include "base/check_op.h"
#include "base/debug/dump_without_crashing.h"
#include "base/feature_list.h"
#include "base/files/file_path.h"
#include "base/functional/bind.h"
#include "base/functional/callback_helpers.h"
#include "base/json/values_util.h"
#include "base/metrics/histogram_functions.h"
#include "base/notreached.h"
#include "base/ranges/algorithm.h"
#include "base/strings/utf_string_conversions.h"
#include "base/trace_event/trace_event.h"
#include "base/values.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/metrics/first_web_contents_profiler_base.h"
#include "chrome/browser/new_tab_page/chrome_colors/chrome_colors_service.h"
#include "chrome/browser/new_tab_page/chrome_colors/generated_colors_info.h"
#include "chrome/browser/profiles/keep_alive/profile_keep_alive_types.h"
#include "chrome/browser/profiles/keep_alive/scoped_profile_keep_alive.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_attributes_entry.h"
#include "chrome/browser/profiles/profile_attributes_storage.h"
#include "chrome/browser/profiles/profile_avatar_icon_util.h"
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/browser/profiles/profile_statistics.h"
#include "chrome/browser/profiles/profile_statistics_factory.h"
#include "chrome/browser/profiles/profile_window.h"
#include "chrome/browser/profiles/profiles_state.h"
#include "chrome/browser/signin/signin_util.h"
#include "chrome/browser/themes/theme_service.h"
#include "chrome/browser/themes/theme_service_factory.h"
#include "chrome/browser/ui/browser_commands.h"
#include "chrome/browser/ui/browser_finder.h"
#include "chrome/browser/ui/browser_window.h"
#include "chrome/browser/ui/chrome_pages.h"
#include "chrome/browser/ui/color/chrome_color_id.h"
#include "chrome/browser/ui/profiles/profile_colors_util.h"
#include "chrome/browser/ui/profiles/profile_picker.h"
#include "chrome/browser/ui/singleton_tabs.h"
#include "chrome/browser/ui/ui_features.h"
#include "chrome/browser/ui/webui/profile_helper.h"
#include "chrome/browser/ui/webui/signin/login_ui_service.h"
#include "chrome/browser/ui/webui/signin/login_ui_service_factory.h"
#include "chrome/browser/ui/webui/signin/signin_ui_error.h"
#include "chrome/browser/ui/webui/theme_source.h"
#include "chrome/browser/ui/webui/webui_util.h"
#include "chrome/common/pref_names.h"
#include "chrome/common/themes/autogenerated_theme_util.h"
#include "chrome/common/webui_url_constants.h"
#include "components/signin/public/base/signin_switches.h"
#include "components/signin/public/identity_manager/account_info.h"
#include "components/startup_metric_utils/browser/startup_metric_utils.h"
#include "components/supervised_user/core/common/features.h"
#include "content/public/browser/url_data_source.h"
#include "content/public/browser/web_ui.h"
#include "third_party/skia/include/core/SkBitmap.h"
#include "third_party/skia/include/core/SkColor.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/mojom/themes.mojom.h"
#include "ui/base/ui_base_features.h"
#include "ui/base/webui/web_ui_util.h"
#include "ui/color/color_id.h"
#include "ui/color/color_provider.h"
#include "ui/gfx/color_utils.h"
#include "ui/gfx/image/image.h"

#if BUILDFLAG(IS_CHROMEOS_LACROS)
#include "ash/webui/settings/public/constants/routes.mojom.h"
#include "chrome/browser/lacros/account_manager/account_manager_util.h"
#include "chrome/browser/lacros/account_manager/account_profile_mapper.h"
#include "chrome/browser/lacros/identity_manager_lacros.h"
#include "chrome/browser/lacros/lacros_url_handling.h"
#include "chromeos/crosapi/mojom/login.mojom.h"
#include "chromeos/lacros/lacros_service.h"
#include "components/account_manager_core/account.h"
#include "components/account_manager_core/chromeos/account_manager_facade_factory.h"
#endif

namespace {
const size_t kProfileCardAvatarSize = 74;
const size_t kProfileCreationAvatarSize = 100;

constexpr int kDefaultThemeColorId = -1;
constexpr int kManuallyPickedColorId = 0;

// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
enum class ProfilePickerAction {
  kLaunchExistingProfile = 0,
  kLaunchExistingProfileCustomizeSettings = 1,
  kLaunchGuestProfile = 2,
  kLaunchNewProfile = 3,
  kDeleteProfile = 4,
  kMaxValue = kDeleteProfile,
};

std::optional<SkColor> GetChromeColorColorById(int color_id) {
  for (chrome_colors::ColorInfo color_info :
       chrome_colors::kGeneratedColorsInfo) {
    if (color_id == color_info.id) {
      return color_info.color;
    }
  }

  return std::nullopt;
}

void RecordProfilePickerAction(ProfilePickerAction action) {
  base::UmaHistogramEnumeration("ProfilePicker.UserAction", action);
}

void RecordAskOnStartupChanged(bool value) {
  base::UmaHistogramBoolean("ProfilePicker.AskOnStartupChanged", value);
}

base::Value::Dict GetAutogeneratedProfileThemeInfoValue(
    int color_id,
    std::optional<SkColor> color,
    const ui::ColorProvider& color_provider,
    SkColor frame_color,
    SkColor active_tab_color,
    SkColor frame_text_color,
    float scale_factor) {
  base::Value::Dict dict;
  dict.Set("colorId", color_id);
  if (color.has_value()) {
    dict.Set("color", static_cast<int>(*color));
  }
  dict.Set("themeFrameColor", color_utils::SkColorToRgbaString(frame_color));
  dict.Set("themeShapeColor",
           color_utils::SkColorToRgbaString(active_tab_color));
  dict.Set("themeFrameTextColor",
           color_utils::SkColorToRgbaString(frame_text_color));
  DefaultAvatarColors avatar_colors =
      GetDefaultAvatarColors(color_provider, frame_color);
  gfx::Image icon = profiles::GetPlaceholderAvatarIconWithColors(
      /*fill_color=*/avatar_colors.fill_color,
      /*stroke_color=*/avatar_colors.stroke_color,
      kProfileCreationAvatarSize * scale_factor);
  dict.Set("themeGenericAvatar", webui::GetBitmapDataUrl(icon.AsBitmap()));
  return dict;
}

base::Value::Dict CreateDefaultProfileThemeInfo(
    const ui::ColorProvider& color_provider,
    float scale_factor) {
  SkColor frame_color = color_provider.GetColor(ui::kColorFrameActive);
  SkColor active_tab_color = color_provider.GetColor(kColorToolbar);
  SkColor frame_text_color =
      color_provider.GetColor(kColorTabForegroundInactiveFrameActive);
  return GetAutogeneratedProfileThemeInfoValue(
      kDefaultThemeColorId, std::nullopt, color_provider, frame_color,
      active_tab_color, frame_text_color, scale_factor);
}

base::Value::Dict CreateAutogeneratedProfileThemeInfo(
    int color_id,
    SkColor color,
    const ui::ColorProvider& color_provider,
    float scale_factor) {
  auto theme_colors = GetAutogeneratedThemeColors(color);
  SkColor frame_color = theme_colors.frame_color;
  SkColor active_tab_color = theme_colors.active_tab_color;
  SkColor frame_text_color = theme_colors.frame_text_color;
  return GetAutogeneratedProfileThemeInfoValue(color_id, color, color_provider,
                                               frame_color, active_tab_color,
                                               frame_text_color, scale_factor);
}

void OpenOnSelectProfileTargetUrl(Browser* browser) {
  GURL target_page_url = ProfilePicker::GetOnSelectProfileTargetUrl();
  if (target_page_url.is_empty()) {
    return;
  }

  if (target_page_url.spec() == chrome::kChromeUIHelpURL) {
    chrome::ShowAboutChrome(browser);
  } else if (target_page_url.spec() == chrome::kChromeUISettingsURL) {
    chrome::ShowSettings(browser);
  } else if (target_page_url.spec() == ProfilePicker::kTaskManagerUrl) {
    chrome::OpenTaskManager(browser);
  } else {
    ShowSingletonTabOverwritingNTP(browser, target_page_url);
  }
}

base::Value::Dict CreateProfileEntry(const ProfileAttributesEntry* entry,
                                     int avatar_icon_size) {
  base::Value::Dict profile_entry;
  profile_entry.Set("profilePath", base::FilePathToValue(entry->GetPath()));
  profile_entry.Set("localProfileName", entry->GetLocalProfileName());
  profile_entry.Set("isSyncing",
                    entry->GetSigninState() ==
                        SigninState::kSignedInWithConsentedPrimaryAccount);
  profile_entry.Set("needsSignin", entry->IsSigninRequired());
  // GAIA full name/user name can be empty, if the profile is not signed in to
  // chrome.
  profile_entry.Set("gaiaName", entry->GetGAIAName());
  profile_entry.Set("userName", entry->GetUserName());

  if (AccountInfo::IsManaged(entry->GetHostedDomain())) {
    profile_entry.Set("avatarBadge", "cr:domain");
#if BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC) || BUILDFLAG(IS_WIN)
  } else if (base::FeatureList::IsEnabled(
                 supervised_user::kShowKiteForSupervisedUsers) &&
             entry->IsSupervised()) {
    profile_entry.Set("avatarBadge", "cr20:kite");
#endif  // BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC) || BUILDFLAG(IS_WIN)
  } else {
    profile_entry.Set("avatarBadge", "");
  }

  gfx::Image icon =
      profiles::GetSizedAvatarIcon(entry->GetAvatarIcon(avatar_icon_size),
                                   avatar_icon_size, avatar_icon_size);
  std::string icon_url = webui::GetBitmapDataUrl(icon.AsBitmap());
  profile_entry.Set("avatarIcon", icon_url);
#if BUILDFLAG(IS_CHROMEOS_LACROS)
  profile_entry.Set("isPrimaryLacrosProfile",
                    Profile::IsMainProfilePath(entry->GetPath()));
#else
  profile_entry.Set("isPrimaryLacrosProfile", false);
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)
  return profile_entry;
}

#if BUILDFLAG(IS_CHROMEOS_LACROS)
base::FilePath GetCurrentProfilePath(content::WebUI* web_ui) {
  DCHECK(web_ui);
  return web_ui->GetWebContents()->GetBrowserContext()->GetPath();
}

// Returns whether this WebUI page is rendered in a user profile (and not the
// default picker profile). The profile picker is loaded in a user profile
// only in the flow of adding an account to an existing profile.
bool IsUsingExistingProfile(content::WebUI* web_ui) {
  return GetCurrentProfilePath(web_ui) != ProfilePicker::GetPickerProfilePath();
}

SkBitmap GetAvailableAccountBitmap(const gfx::Image& gaia_image) {
  if (!gaia_image.IsEmpty()) {
    return gaia_image.AsBitmap();
  }

  // Return a default avatar.
  const int kAccountPictureSize = 128;
  ProfileThemeColors colors = GetDefaultProfileThemeColors();
  gfx::Image default_image = profiles::GetPlaceholderAvatarIconWithColors(
      colors.default_avatar_fill_color, colors.default_avatar_stroke_color,
      kAccountPictureSize);
  return default_image.AsBitmap();
}

void RunAccountSelectionCallback(
    const std::optional<AccountProfileMapper::AddAccountResult>& result) {
  if (!result.has_value() || result->account.key.account_type() !=
                                 account_manager::AccountType::kGaia) {
    return;
  }

  ProfilePicker::NotifyAccountSelected(result->account.key.id());
  ProfilePicker::Hide();
}
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)

void RecordProfilingFinishReason(
    metrics::StartupProfilingFinishReason finish_reason) {
  base::UmaHistogramEnumeration(
      "ProfilePicker.FirstProfileTime.FirstWebContentsFinishReason",
      finish_reason);
}

class FirstWebContentsProfilerForProfilePicker
    : public metrics::FirstWebContentsProfilerBase {
 public:
  explicit FirstWebContentsProfilerForProfilePicker(
      content::WebContents* web_contents,
      base::TimeTicks pick_time);

  FirstWebContentsProfilerForProfilePicker(
      const FirstWebContentsProfilerForProfilePicker&) = delete;
  FirstWebContentsProfilerForProfilePicker& operator=(
      const FirstWebContentsProfilerForProfilePicker&) = delete;

 protected:
  // FirstWebContentsProfilerBase:
  void RecordFinishReason(
      metrics::StartupProfilingFinishReason finish_reason) override;
  void RecordNavigationFinished(base::TimeTicks navigation_start) override;
  void RecordFirstNonEmptyPaint() override;
  bool WasStartupInterrupted() override;

 private:
  ~FirstWebContentsProfilerForProfilePicker() override;

  const base::TimeTicks pick_time_;
};

FirstWebContentsProfilerForProfilePicker::
    FirstWebContentsProfilerForProfilePicker(content::WebContents* web_contents,
                                             base::TimeTicks pick_time)
    : FirstWebContentsProfilerBase(web_contents), pick_time_(pick_time) {
  DCHECK(!pick_time_.is_null());
}

FirstWebContentsProfilerForProfilePicker::
    ~FirstWebContentsProfilerForProfilePicker() = default;

void FirstWebContentsProfilerForProfilePicker::RecordFinishReason(
    metrics::StartupProfilingFinishReason finish_reason) {
  RecordProfilingFinishReason(finish_reason);
}

void FirstWebContentsProfilerForProfilePicker::RecordNavigationFinished(
    base::TimeTicks navigation_start) {
  // Nothing to record here for Profile Picker startups.
}

void FirstWebContentsProfilerForProfilePicker::RecordFirstNonEmptyPaint() {
  const char histogram_name[] =
      "ProfilePicker.FirstProfileTime.FirstWebContentsNonEmptyPaint";
  base::TimeTicks paint_time = base::TimeTicks::Now();
  base::UmaHistogramLongTimes100(histogram_name, paint_time - pick_time_);
  TRACE_EVENT_NESTABLE_ASYNC_BEGIN_WITH_TIMESTAMP0("startup", histogram_name,
                                                   this, pick_time_);
  TRACE_EVENT_NESTABLE_ASYNC_END_WITH_TIMESTAMP0("startup", histogram_name,
                                                 this, paint_time);
}

bool FirstWebContentsProfilerForProfilePicker::WasStartupInterrupted() {
  // We're assuming that no interruptions block opening an existing profile
  // from the profile picker. We would detect this by observing really high
  // latency on the tracked metric, and can start tracking interruptions if we
  // find that such cases occur.
  return false;
}

}  // namespace

ProfilePickerHandler::ProfilePickerHandler()
#if BUILDFLAG(IS_CHROMEOS_LACROS)
    : identity_manager_lacros_(std::make_unique<IdentityManagerLacros>())
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)
{
}

ProfilePickerHandler::~ProfilePickerHandler() {
  OnJavascriptDisallowed();
}

void ProfilePickerHandler::EnableStartupMetrics() {
  DCHECK(creation_time_on_startup_.is_null());
  content::WebContents* contents = web_ui()->GetWebContents();
  if (contents->GetVisibility() == content::Visibility::VISIBLE) {
    // Only record paint event if the window is visible.
    creation_time_on_startup_ = base::TimeTicks::Now();
    Observe(web_ui()->GetWebContents());
  }
}

void ProfilePickerHandler::RegisterMessages() {
  web_ui()->RegisterMessageCallback(
      "mainViewInitialize",
      base::BindRepeating(&ProfilePickerHandler::HandleMainViewInitialize,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "launchSelectedProfile",
      base::BindRepeating(&ProfilePickerHandler::HandleLaunchSelectedProfile,
                          base::Unretained(this), /*open_settings=*/false));
  web_ui()->RegisterMessageCallback(
      "openManageProfileSettingsSubPage",
      base::BindRepeating(&ProfilePickerHandler::HandleLaunchSelectedProfile,
                          base::Unretained(this), /*open_settings=*/true));
  web_ui()->RegisterMessageCallback(
      "launchGuestProfile",
      base::BindRepeating(&ProfilePickerHandler::HandleLaunchGuestProfile,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "askOnStartupChanged",
      base::BindRepeating(&ProfilePickerHandler::HandleAskOnStartupChanged,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "getNewProfileSuggestedThemeInfo",
      base::BindRepeating(
          &ProfilePickerHandler::HandleGetNewProfileSuggestedThemeInfo,
          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "getProfileThemeInfo",
      base::BindRepeating(&ProfilePickerHandler::HandleGetProfileThemeInfo,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "removeProfile",
      base::BindRepeating(&ProfilePickerHandler::HandleRemoveProfile,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "getProfileStatistics",
      base::BindRepeating(&ProfilePickerHandler::HandleGetProfileStatistics,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "closeProfileStatistics",
      base::BindRepeating(&ProfilePickerHandler::HandleCloseProfileStatistics,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "selectNewAccount",
      base::BindRepeating(&ProfilePickerHandler::HandleSelectNewAccount,
                          base::Unretained(this)));
  // TODO(crbug.com/40144179): Consider renaming this message to
  // 'createLocalProfile' as this is only used for local profiles.
  web_ui()->RegisterMessageCallback(
      "getAvailableIcons",
      base::BindRepeating(&ProfilePickerHandler::HandleGetAvailableIcons,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "continueWithoutAccount",
      base::BindRepeating(&ProfilePickerHandler::HandleContinueWithoutAccount,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "getSwitchProfile",
      base::BindRepeating(&ProfilePickerHandler::HandleGetSwitchProfile,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "confirmProfileSwitch",
      base::BindRepeating(&ProfilePickerHandler::HandleConfirmProfileSwitch,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "cancelProfileSwitch",
      base::BindRepeating(&ProfilePickerHandler::HandleCancelProfileSwitch,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "setProfileName",
      base::BindRepeating(&ProfilePickerHandler::HandleSetProfileName,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "recordSignInPromoImpression",
      base::BindRepeating(
          &ProfilePickerHandler::HandleRecordSignInPromoImpression,
          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "updateProfileOrder",
      base::BindRepeating(&ProfilePickerHandler::HandleUpdateProfileOrder,
                          base::Unretained(this)));
#if BUILDFLAG(IS_CHROMEOS_LACROS)
  web_ui()->RegisterMessageCallback(
      "getAvailableAccounts",
      base::BindRepeating(&ProfilePickerHandler::HandleGetAvailableAccounts,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "openAshAccountSettingsPage",
      base::BindRepeating(
          &ProfilePickerHandler::HandleOpenAshAccountSettingsPage,
          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "selectExistingAccountLacros",
      base::BindRepeating(
          &ProfilePickerHandler::HandleSelectExistingAccountLacros,
          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "openDeviceGuestLinkLacros",
      base::BindRepeating(
          &ProfilePickerHandler::HandleOpenDeviceGuestLinkLacros,
          base::Unretained(this)));
#endif
  Profile* profile = Profile::FromWebUI(web_ui());
  content::URLDataSource::Add(profile, std::make_unique<ThemeSource>(profile));
}

void ProfilePickerHandler::OnJavascriptAllowed() {
  ProfileManager* profile_manager = g_browser_process->profile_manager();
#if BUILDFLAG(IS_CHROMEOS_LACROS)
  account_profile_mapper_observation_.Observe(
      profile_manager->GetAccountProfileMapper());
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)
  profile_attributes_storage_observation_.Observe(
      &profile_manager->GetProfileAttributesStorage());
}
void ProfilePickerHandler::OnJavascriptDisallowed() {
#if BUILDFLAG(IS_CHROMEOS_LACROS)
  account_profile_mapper_observation_.Reset();
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)
  profile_attributes_storage_observation_.Reset();
  weak_factory_.InvalidateWeakPtrs();
}

void ProfilePickerHandler::HandleMainViewInitialize(
    const base::Value::List& args) {
  if (!creation_time_on_startup_.is_null() && !main_view_initialized_) {
    // This function can be called multiple times if the page is reloaded. The
    // histogram is only recorded once.
    main_view_initialized_ = true;
    base::UmaHistogramTimes("ProfilePicker.StartupTime.MainViewInitialized",
                            base::TimeTicks::Now() - creation_time_on_startup_);
  }

  AllowJavascript();
  PushProfilesList();
}

void ProfilePickerHandler::HandleLaunchSelectedProfile(
    bool open_settings,
    const base::Value::List& args) {
  TRACE_EVENT1("browser", "ProfilePickerHandler::HandleLaunchSelectedProfile",
               "args", args.DebugString());
  if (args.empty()) {
    return;
  }
  const base::Value& profile_path_value = args[0];

  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);
  if (!profile_path) {
    return;
  }

  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(*profile_path);
  if (!entry) {
    NOTREACHED_IN_MIGRATION();
    return;
  }

  // If a browser window cannot be opened for profile, load the profile to
  // display a dialog.
  if (entry->IsSigninRequired()
#if BUILDFLAG(IS_CHROMEOS_LACROS)
      || (!profiles::AreSecondaryProfilesAllowed() &&
          !Profile::IsMainProfilePath(*profile_path))
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)
  ) {
    g_browser_process->profile_manager()->LoadProfileByPath(
        *profile_path, /*incognito=*/false,
        base::BindOnce(&ProfilePickerHandler::OnProfileForDialogLoaded,
                       weak_factory_.GetWeakPtr()));
    return;
  }

  if (!creation_time_on_startup_.is_null() &&
      // Avoid overriding the picked time if already recorded. This can happen
      // for example if multiple profiles are picked: https://crbug.com/1277466.
      profile_picked_time_on_startup_.is_null()) {
    profile_picked_time_on_startup_ = base::TimeTicks::Now();
  }
  profiles::SwitchToProfile(
      *profile_path, /*always_create=*/false,
      base::BindOnce(&ProfilePickerHandler::OnSwitchToProfileComplete,
                     weak_factory_.GetWeakPtr(), false, open_settings));
}

void ProfilePickerHandler::OnProfileForDialogLoaded(Profile* profile) {
  if (!profile) {
    return;
  }
#if BUILDFLAG(ENABLE_DICE_SUPPORT)
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile->GetPath());
  DCHECK(entry);
  if (entry->IsSigninRequired()) {
    DCHECK(signin_util::IsForceSigninEnabled());
    if (entry->CanBeManaged()) {
      if (base::FeatureList::IsEnabled(kForceSigninFlowInProfilePicker)) {
        ProfilePicker::SwitchToReauth(
            profile,
            base::BindOnce(&ProfilePickerHandler::DisplayForceSigninErrorDialog,
                           weak_factory_.GetWeakPtr(), profile->GetPath()));
      } else {
        ProfilePickerForceSigninDialog::ShowReauthDialog(
            profile, base::UTF16ToUTF8(entry->GetUserName()));
      }
    } else if (entry->GetActiveTime() != base::Time()) {
      // If force-sign-in is enabled, do not allow users to sign in to a
      // pre-existing locked profile, as this may force unexpected profile data
      // merge. We consider a profile as pre-existing if it has been active
      // previously. A pre-existed profile can still be used if it has been
      // signed in with an email address matched RestrictSigninToPattern policy
      // already.
      if (base::FeatureList::IsEnabled(kForceSigninFlowInProfilePicker)) {
        DisplayForceSigninErrorDialog(
            /*profile_path=*/base::FilePath(),
            ForceSigninUIError::ReauthNotAllowed());
      } else {
        LoginUIServiceFactory::GetForProfile(profile)
            ->SetProfileBlockingErrorMessage();
        ProfilePicker::ShowDialogAndDisplayErrorMessage(profile);
      }
    } else {
      // Fresh sign in via profile picker without existing email address.
      if (base::FeatureList::IsEnabled(kForceSigninFlowInProfilePicker)) {
        ProfilePicker::SwitchToDiceSignIn(
            profile->GetPath(),
            base::BindOnce(&ProfilePickerHandler::OnLoadSigninFinished,
                           weak_factory_.GetWeakPtr()));
      } else {
        ProfilePickerForceSigninDialog::ShowForceSigninDialog(profile);
      }
    }
  }
#endif  // BUILDFLAG(ENABLE_DICE_SUPPORT)
#if BUILDFLAG(IS_CHROMEOS_LACROS)
  if (!profiles::AreSecondaryProfilesAllowed() &&
      !Profile::IsMainProfilePath(profile->GetPath())) {
    LoginUIServiceFactory::GetForProfile(profile)
        ->SetProfileBlockingErrorMessage();
    ProfilePicker::ShowDialogAndDisplayErrorMessage(profile);
  }
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)
}

void ProfilePickerHandler::DisplayForceSigninErrorDialog(
    const base::FilePath& profile_path,
    const ForceSigninUIError& error) {
  AllowJavascript();

  const auto& [title, body] = error.GetErrorTexts();
  FireWebUIListener("display-force-signin-error-dialog", base::Value(title),
                    base::Value(body),
                    base::Value(profile_path.AsUTF16Unsafe()));
}

void ProfilePickerHandler::HandleLaunchGuestProfile(
    const base::Value::List& args) {
  // TODO(crbug.com/40123459): Add check |IsGuestModeEnabled| once policy
  // checking has been added to the UI.
  profiles::SwitchToGuestProfile(
      base::BindOnce(&ProfilePickerHandler::OnSwitchToProfileComplete,
                     weak_factory_.GetWeakPtr(), false, false));
}

void ProfilePickerHandler::HandleAskOnStartupChanged(
    const base::Value::List& list) {
  if (list.empty() || !list[0].is_bool()) {
    return;
  }
  const bool show_on_startup = list[0].GetBool();

  PrefService* prefs = g_browser_process->local_state();
  prefs->SetBoolean(prefs::kBrowserShowProfilePickerOnStartup, show_on_startup);
  RecordAskOnStartupChanged(show_on_startup);
}

void ProfilePickerHandler::HandleGetNewProfileSuggestedThemeInfo(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  const base::Value& callback_id = args[0];

#if BUILDFLAG(IS_CHROMEOS_LACROS)
  if (IsUsingExistingProfile(web_ui())) {
    // The picker offers secondary accounts for a given existing profile.
    // Extract the color from the current profile (where this is rendered).
    ThemeService* theme_service =
        ThemeServiceFactory::GetForProfile(Profile::FromWebUI(web_ui()));
    base::Value::Dict profile_dict;
    if (theme_service->UsingAutogeneratedTheme()) {
      // We'll never use `profile_dict` for showing the color picker so we can
      // pass in kManuallyPickedColorId to simplify the code.
      profile_dict = CreateAutogeneratedProfileThemeInfo(
          kManuallyPickedColorId, theme_service->GetAutogeneratedThemeColor(),
          web_ui()->GetWebContents()->GetColorProvider(),
          web_ui()->GetDeviceScaleFactor());
    } else {
      profile_dict = CreateDefaultProfileThemeInfo(
          web_ui()->GetWebContents()->GetColorProvider(),
          web_ui()->GetDeviceScaleFactor());
    }
    ResolveJavascriptCallback(callback_id, profile_dict);
    return;
  }
#endif
  chrome_colors::ColorInfo color_info = GenerateNewProfileColor();
  base::Value::Dict dict = CreateAutogeneratedProfileThemeInfo(
      color_info.id, color_info.color,
      web_ui()->GetWebContents()->GetColorProvider(),
      web_ui()->GetDeviceScaleFactor());
  ResolveJavascriptCallback(callback_id, dict);
}

void ProfilePickerHandler::HandleGetProfileThemeInfo(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(2U, args.size());
  const base::Value& callback_id = args[0];
  const base::Value::Dict& user_theme_choice = args[1].GetDict();
  int color_id = user_theme_choice.FindInt("colorId").value();
  std::optional<SkColor> color = user_theme_choice.FindDouble("color");
  base::Value::Dict dict;
  switch (color_id) {
    case kDefaultThemeColorId:
      dict = CreateDefaultProfileThemeInfo(
          web_ui()->GetWebContents()->GetColorProvider(),
          web_ui()->GetDeviceScaleFactor());
      break;
    case kManuallyPickedColorId:
      dict = CreateAutogeneratedProfileThemeInfo(
          color_id, *color, web_ui()->GetWebContents()->GetColorProvider(),
          web_ui()->GetDeviceScaleFactor());
      break;
    default:
      dict = CreateAutogeneratedProfileThemeInfo(
          color_id, *GetChromeColorColorById(color_id),
          web_ui()->GetWebContents()->GetColorProvider(),
          web_ui()->GetDeviceScaleFactor());
      break;
  }
  ResolveJavascriptCallback(callback_id, dict);
}

void ProfilePickerHandler::HandleGetAvailableIcons(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  const base::Value& callback_id = args[0];
  ResolveJavascriptCallback(callback_id,
                            profiles::GetCustomProfileAvatarIconsAndLabels());
}

void ProfilePickerHandler::HandleContinueWithoutAccount(
    const base::Value::List& args) {
  CHECK_EQ(1U, args.size());

  // profileColor is undefined for the default theme.
  std::optional<SkColor> profile_color;
  if (args[0].is_int()) {
    profile_color = args[0].GetInt();
  }

  RecordProfilePickerAction(ProfilePickerAction::kLaunchNewProfile);
  ProfileMetrics::LogProfileAddNewUser(
      ProfileMetrics::ADD_NEW_PROFILE_PICKER_LOCAL);
  ProfilePicker::SwitchToSignedOutPostIdentityFlow(
      profile_color, profile_picked_time_on_startup_,
      base::BindOnce(&ProfilePickerHandler::OnProfileCreationFinished,
                     // `OnProfileCreationFinished` is called when we want to
                     // close the profile picker.
                     weak_factory_.GetWeakPtr()));
}

void ProfilePickerHandler::HandleGetSwitchProfile(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  const base::Value& callback_id = args[0];
  int avatar_icon_size =
      kProfileCardAvatarSize * web_ui()->GetDeviceScaleFactor();
  base::FilePath profile_path = ProfilePicker::GetSwitchProfilePath();
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path);
  CHECK(entry);
  base::Value::Dict dict = CreateProfileEntry(entry, avatar_icon_size);
  ResolveJavascriptCallback(callback_id, dict);
}

void ProfilePickerHandler::HandleConfirmProfileSwitch(
    const base::Value::List& args) {
  if (args.empty()) {
    return;
  }
  const base::Value& profile_path_value = args[0];

  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);
  if (!profile_path) {
    return;
  }

  // TODO(crbug.com/40751337): remove the profile used for the sign-in
  // flow.
  profiles::SwitchToProfile(
      *profile_path, /*always_create=*/false,
      base::BindOnce(&ProfilePickerHandler::OnSwitchToProfileComplete,
                     weak_factory_.GetWeakPtr(), false, false));
}

void ProfilePickerHandler::HandleCancelProfileSwitch(
    const base::Value::List& args) {
  ProfilePicker::CancelSignedInFlow();
}

void ProfilePickerHandler::OnProfileCreationFinished(
    bool finished_successfully) {
  FireWebUIListener("create-profile-finished", base::Value());
}

void ProfilePickerHandler::HandleRecordSignInPromoImpression(
    const base::Value::List& /*args*/) {
  signin_metrics::RecordSigninImpressionUserActionForAccessPoint(
      signin_metrics::AccessPoint::ACCESS_POINT_USER_MANAGER);
}

void ProfilePickerHandler::HandleSetProfileName(const base::Value::List& args) {
  CHECK_EQ(2U, args.size());
  const base::Value& profile_path_value = args[0];
  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);

  if (!profile_path) {
    NOTREACHED_IN_MIGRATION();
    return;
  }
  std::u16string profile_name = base::UTF8ToUTF16(args[1].GetString());
  base::TrimWhitespace(profile_name, base::TRIM_ALL, &profile_name);
  CHECK(!profile_name.empty());
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path.value());
  CHECK(entry);
  entry->SetLocalProfileName(profile_name, /*is_default_name=*/false);
}

void ProfilePickerHandler::HandleRemoveProfile(const base::Value::List& args) {
  CHECK_EQ(1U, args.size());
  const base::Value& profile_path_value = args[0];
  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);

  if (!profile_path) {
    NOTREACHED_IN_MIGRATION();
    return;
  }

#if BUILDFLAG(IS_CHROMEOS_LACROS)
  // On Lacros, the primary profile should never be deleted.
  CHECK(!Profile::IsMainProfilePath(*profile_path));
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)

  RecordProfilePickerAction(ProfilePickerAction::kDeleteProfile);
  DCHECK(profile_statistics_keep_alive_);

  // Deleting the profile may delete `this` (see See https://crbug.com/1488267),
  // if the profile picker was shown in a tab. Keep the `ScopedProfileKeepAlive`
  // until the end of the function, to avoid the profile being unloaded and
  // reloaded.
  std::unique_ptr<ScopedProfileKeepAlive> profile_statistics_keep_alive =
      std::move(profile_statistics_keep_alive_);
  webui::DeleteProfileAtPath(*profile_path,
                             ProfileMetrics::DELETE_PROFILE_USER_MANAGER);
  // Do not use `this` after this point, it may be deleted.
}

void ProfilePickerHandler::HandleUpdateProfileOrder(
    const base::Value::List& args) {
  CHECK_EQ(2U, args.size());
  CHECK(args[0].is_int());
  CHECK(args[1].is_int());

  int from_index = args[0].GetInt();
  int to_index = args[1].GetInt();
  CHECK(from_index >= 0 && to_index >= 0);

  g_browser_process->profile_manager()
      ->GetProfileAttributesStorage()
      .UpdateProfilesOrderPref(from_index, to_index);
}

void ProfilePickerHandler::HandleCloseProfileStatistics(
    const base::Value::List& args) {
  CHECK_EQ(0U, args.size());
  DCHECK(profile_statistics_keep_alive_);
  profile_statistics_keep_alive_.reset();
}

void ProfilePickerHandler::HandleGetProfileStatistics(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  const base::Value& profile_path_value = args[0];
  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);
  if (!profile_path) {
    return;
  }

  Profile* profile =
      g_browser_process->profile_manager()->GetProfileByPath(*profile_path);

  if (profile) {
    GatherProfileStatistics(profile);
  } else {
    g_browser_process->profile_manager()->LoadProfileByPath(
        *profile_path, false,
        base::BindOnce(&ProfilePickerHandler::GatherProfileStatistics,
                       weak_factory_.GetWeakPtr()));
  }
}

void ProfilePickerHandler::GatherProfileStatistics(Profile* profile) {
  if (!profile) {
    return;
  }

  profile_statistics_keep_alive_ = std::make_unique<ScopedProfileKeepAlive>(
      profile, ProfileKeepAliveOrigin::kProfileStatistics);

  ProfileStatisticsFactory::GetForProfile(profile)->GatherStatistics(
      base::BindRepeating(&ProfilePickerHandler::OnProfileStatisticsReceived,
                          weak_factory_.GetWeakPtr(), profile->GetPath()));
}

void ProfilePickerHandler::OnProfileStatisticsReceived(
    const base::FilePath& profile_path,
    profiles::ProfileCategoryStats result) {
  base::Value::Dict dict;
  dict.Set("profilePath", base::FilePathToValue(profile_path));
  base::Value::Dict stats;
  // Categories are defined in |kProfileStatisticsCategories|
  // {"BrowsingHistory", "Passwords", "Bookmarks", "Autofill"}.
  for (const auto& item : result) {
    stats.Set(item.category, item.count);
  }
  dict.Set("statistics", std::move(stats));
  FireWebUIListener("profile-statistics-received", dict);
}

void ProfilePickerHandler::HandleSelectNewAccount(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  std::optional<SkColor> profile_color = args[0].GetIfInt();
#if BUILDFLAG(IS_CHROMEOS_LACROS)
  SelectAccountLacrosInternal("", profile_color);
#elif BUILDFLAG(ENABLE_DICE_SUPPORT)
  if (signin_util::IsForceSigninEnabled()) {
    // Force sign-in policy uses a separate flow that doesn't initialize the
    // profile color. Generate a new profile color here.
    profile_color = GenerateNewProfileColor().color;
  }
  ProfilePicker::SwitchToDiceSignIn(
      profile_color, base::BindOnce(&ProfilePickerHandler::OnLoadSigninFinished,
                                    weak_factory_.GetWeakPtr()));
#else
  NOTERACHED();
#endif
}

#if BUILDFLAG(IS_CHROMEOS_LACROS)
void ProfilePickerHandler::HandleSelectExistingAccountLacros(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(2U, args.size());
  std::optional<SkColor> profile_color = args[0].GetIfInt();
  const std::string& gaia_id = args[1].GetString();
  DCHECK(!gaia_id.empty());
  SelectAccountLacrosInternal(gaia_id, profile_color);
}
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)

void ProfilePickerHandler::OnLoadSigninFinished(bool success) {
  AllowJavascript();
  FireWebUIListener("load-signin-finished", base::Value(success));
}

void ProfilePickerHandler::OnSwitchToProfileComplete(bool new_profile,
                                                     bool open_settings,
                                                     Browser* browser) {
  if (!browser || browser->is_delete_scheduled()) {
    // The browser is destroyed or about to be destroyed.
    return;
  }

  DCHECK(browser->window());
  Profile* profile = browser->profile();
  TRACE_EVENT1("browser", "ProfilePickerHandler::OnSwitchToProfileComplete",
               "profile_path", profile->GetPath().AsUTF8Unsafe());

  // Measure startup time to display first web contents if the profile picker
  // was displayed on startup and if the initiating action is instrumented. For
  // example we don't record pick time for profile creations.
  if (!profile_picked_time_on_startup_.is_null()) {
    BeginFirstWebContentsProfiling(browser, profile_picked_time_on_startup_);
  }

  // Only show the profile switch IPH when the user clicked the card, and there
  // are multiple profiles.
  std::vector<ProfileAttributesEntry*> entries =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetAllProfilesAttributes();
  int profile_count =
      base::ranges::count(entries, false, &ProfileAttributesEntry::IsOmitted);
  if (profile_count > 1 && !open_settings &&
      ProfilePicker::GetOnSelectProfileTargetUrl().is_empty()) {
    browser->window()->MaybeShowProfileSwitchIPH();
  }

  if (new_profile) {
    RecordProfilePickerAction(ProfilePickerAction::kLaunchNewProfile);
    ProfilePicker::Hide();
    return;
  }

  if (profile->IsGuestSession()) {
    RecordProfilePickerAction(ProfilePickerAction::kLaunchGuestProfile);
  } else {
    RecordProfilePickerAction(
        open_settings
            ? ProfilePickerAction::kLaunchExistingProfileCustomizeSettings
            : ProfilePickerAction::kLaunchExistingProfile);
  }

  if (open_settings) {
    // User clicked 'Edit' from the profile card menu.
    chrome::ShowSettingsSubPage(browser, chrome::kManageProfileSubPage);
  } else {
    // Opens a target url upon user selecting a pre-existing profile. For
    // new profiles the chrome welcome page is displayed.
    OpenOnSelectProfileTargetUrl(browser);
  }

  ProfilePicker::Hide();
}

void ProfilePickerHandler::PushProfilesList() {
  DCHECK(IsJavascriptAllowed());
  FireWebUIListener("profiles-list-changed", GetProfilesList());
}

void ProfilePickerHandler::SetProfilesOrder(
    const std::vector<ProfileAttributesEntry*>& entries) {
  profiles_order_.clear();
  size_t index = 0;
  for (const ProfileAttributesEntry* entry : entries) {
    profiles_order_[entry->GetPath()] = index++;
  }
}

std::vector<ProfileAttributesEntry*>
ProfilePickerHandler::GetProfileAttributes() {
  std::vector<ProfileAttributesEntry*> ordered_entries =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetAllProfilesAttributesSortedByLocalProfileNameWithCheck();
  std::erase_if(ordered_entries, [](const ProfileAttributesEntry* entry) {
    return entry->IsOmitted();
  });
  size_t number_of_profiles = ordered_entries.size();

  if (profiles_order_.size() != number_of_profiles) {
    // Should only happen the first time the function is called.
    // Profile creation and deletion are handled at
    // 'OnProfileAdded', 'OnProfileWasRemoved'.
    DCHECK(!profiles_order_.size());
    SetProfilesOrder(ordered_entries);
    return ordered_entries;
  }

  // Vector of nullptr entries.
  std::vector<ProfileAttributesEntry*> entries(number_of_profiles);
  for (ProfileAttributesEntry* entry : ordered_entries) {
    DCHECK(profiles_order_.find(entry->GetPath()) != profiles_order_.end());
    size_t index = profiles_order_[entry->GetPath()];
    DCHECK_LT(index, number_of_profiles);
    DCHECK(!entries[index]);
    entries[index] = entry;
  }
  return entries;
}

base::Value::List ProfilePickerHandler::GetProfilesList() {
  base::Value::List profiles_list;
  std::vector<ProfileAttributesEntry*> entries = GetProfileAttributes();
  const int avatar_icon_size =
      kProfileCardAvatarSize * web_ui()->GetDeviceScaleFactor();
  for (const ProfileAttributesEntry* entry : entries) {
    profiles_list.Append(CreateProfileEntry(entry, avatar_icon_size));
  }
  return profiles_list;
}

void ProfilePickerHandler::AddProfileToListAndPushUpdates(
    const base::FilePath& profile_path) {
  size_t number_of_profiles = profiles_order_.size();
  auto it_and_whether_inserted =
      profiles_order_.insert({profile_path, number_of_profiles});
  // We shouldn't add the same profile to the list more than once. Use
  // `insert()` to not corrput the map in case this happens.
  // https://crbug.com/1195784
  DCHECK(it_and_whether_inserted.second);

  MaybeUpdateGuestMode();
  PushProfilesList();
}

void ProfilePickerHandler::RemoveProfileFromListAndPushUpdates(
    const base::FilePath& profile_path) {
  auto remove_it = profiles_order_.find(profile_path);
  // Guest and omitted profiles aren't added to the list.
  // It's possible that a profile gets marked as guest or as omitted after it
  // had been added to the list. In that case, the profile gets removed from the
  // list once in `OnProfileIsOmittedChanged()` but not the second time when
  // `OnProfileWasRemoved()` is called.
  if (remove_it == profiles_order_.end()) {
    return;
  }

  size_t index = remove_it->second;
  profiles_order_.erase(remove_it);
  for (auto& it : profiles_order_) {
    if (it.second > index) {
      --it.second;
    }
  }
  MaybeUpdateGuestMode();
  FireWebUIListener("profile-removed", base::FilePathToValue(profile_path));
}

void ProfilePickerHandler::OnProfileAdded(const base::FilePath& profile_path) {
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path);
  CHECK(entry);
  if (entry->IsOmitted()) {
    return;
  }

  AddProfileToListAndPushUpdates(profile_path);
}

void ProfilePickerHandler::OnProfileWasRemoved(
    const base::FilePath& profile_path,
    const std::u16string& profile_name) {
  DCHECK(IsJavascriptAllowed());
  RemoveProfileFromListAndPushUpdates(profile_path);
}

void ProfilePickerHandler::OnProfileIsOmittedChanged(
    const base::FilePath& profile_path) {
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path);
  CHECK(entry);
  if (entry->IsOmitted()) {
    RemoveProfileFromListAndPushUpdates(profile_path);
  } else {
    AddProfileToListAndPushUpdates(profile_path);
  }
}

void ProfilePickerHandler::OnProfileAvatarChanged(
    const base::FilePath& profile_path) {
  PushProfilesList();
}

void ProfilePickerHandler::OnProfileHighResAvatarLoaded(
    const base::FilePath& profile_path) {
  PushProfilesList();
}

void ProfilePickerHandler::OnProfileNameChanged(
    const base::FilePath& profile_path,
    const std::u16string& old_profile_name) {
  PushProfilesList();
}

void ProfilePickerHandler::OnProfileHostedDomainChanged(
    const base::FilePath& profile_path) {
  PushProfilesList();
}

void ProfilePickerHandler::OnProfileSupervisedUserIdChanged(
    const base::FilePath& profile_path) {
  MaybeUpdateGuestMode();
}

void ProfilePickerHandler::DidFirstVisuallyNonEmptyPaint() {
  DCHECK(!creation_time_on_startup_.is_null());
  auto now = base::TimeTicks::Now();
  base::UmaHistogramTimes("ProfilePicker.StartupTime.FirstPaint",
                          now - creation_time_on_startup_);
  startup_metric_utils::GetBrowser().RecordExternalStartupMetric(
      "ProfilePicker.StartupTime.FirstPaint.FromApplicationStart", now,
      /*set_non_browser_ui_displayed=*/true);
  // Stop observing so that the histogram is only recorded once.
  Observe(nullptr);
}

void ProfilePickerHandler::OnVisibilityChanged(content::Visibility visibility) {
  // If the profile picker is hidden, the first paint will be delayed until the
  // picker is visible again. Stop monitoring the first paint to avoid polluting
  // the metrics.
  if (visibility != content::Visibility::VISIBLE) {
    Observe(nullptr);
  }
}

// static
void ProfilePickerHandler::BeginFirstWebContentsProfiling(
    Browser* browser,
    base::TimeTicks pick_time) {
  content::WebContents* visible_contents =
      metrics::FirstWebContentsProfilerBase::GetVisibleContents(browser);
  if (!visible_contents) {
    RecordProfilingFinishReason(metrics::StartupProfilingFinishReason::
                                    kAbandonNoInitiallyVisibleContent);
    return;
  }

  if (visible_contents->CompletedFirstVisuallyNonEmptyPaint()) {
    RecordProfilingFinishReason(
        metrics::StartupProfilingFinishReason::kAbandonAlreadyPaintedContent);
    return;
  }

  // FirstWebContentsProfilerForProfilePicker owns itself and is also bound to
  // |visible_contents|'s lifetime by observing WebContentsDestroyed().
  new FirstWebContentsProfilerForProfilePicker(visible_contents, pick_time);
}

void ProfilePickerHandler::MaybeUpdateGuestMode() {
#if BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC) || BUILDFLAG(IS_WIN)
  if (!base::FeatureList::IsEnabled(
          supervised_user::kHideGuestModeForSupervisedUsers)) {
    return;
  }
#else
  return;
#endif  // BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC) || BUILDFLAG(IS_WIN)
  CHECK(IsJavascriptAllowed());
  FireWebUIListener("guest-mode-availability-updated",
                    base::Value(profiles::IsGuestModeEnabled()));
}

#if BUILDFLAG(IS_CHROMEOS_LACROS)

void ProfilePickerHandler::HandleOpenAshAccountSettingsPage(
    const base::Value::List& args) {
  std::string settings_url = chrome::kChromeUIOSSettingsURL;
  // TODO(b/326488045) Update Settings path to kPeopleSectionPath when Settings
  // revamp is launched.
  settings_url.append(chromeos::settings::mojom::kMyAccountsSubpagePath);
  lacros_url_handling::NavigateInAsh(GURL(settings_url));
}

void ProfilePickerHandler::HandleOpenDeviceGuestLinkLacros(
    const base::Value::List& args) {
  chromeos::LacrosService* lacros_service = chromeos::LacrosService::Get();
  if (!lacros_service ||
      !lacros_service->IsAvailable<crosapi::mojom::Login>()) {
    return;
  }

  if (lacros_service->GetInterfaceVersion<crosapi::mojom::Login>() <
      int(crosapi::mojom::Login::
              kShowGuestSessionConfirmationDialogMinVersion)) {
    return;
  }

  crosapi::mojom::Login* login_api =
      lacros_service->GetRemote<crosapi::mojom::Login>().get();
  if (!login_api) {
    return;
  }

  login_api->ShowGuestSessionConfirmationDialog();
}

void ProfilePickerHandler::HandleGetAvailableAccounts(
    const base::Value::List& args) {
  AllowJavascript();
  UpdateAvailableAccounts();
}

void ProfilePickerHandler::UpdateAvailableAccounts() {
  AccountProfileMapper* mapper =
      g_browser_process->profile_manager()->GetAccountProfileMapper();

  // For the profile creation flow, show all accounts available.
  // For in profile Sign in/ Turn sync on flows, filter accounts already added.
  base::FilePath profile_path = IsUsingExistingProfile(web_ui())
                                    ? GetCurrentProfilePath(web_ui())
                                    : base::FilePath();

  GetAllAvailableAccounts(
      mapper, profile_path,
      base::BindOnce(&ProfilePickerHandler::GetAvailableAccountsInfo,
                     weak_factory_.GetWeakPtr()));
}

void ProfilePickerHandler::GetAvailableAccountsInfo(
    const std::vector<account_manager::Account>& accounts) {
  // If there's a request in flight, it deletes the current helper and starts a
  // new request.
  lacros_account_info_helper_ = std::make_unique<GetAccountInformationHelper>();

  std::vector<std::string> gaia_ids;
  for (const account_manager::Account& account : accounts) {
    gaia_ids.push_back(account.key.id());
  }
  lacros_account_info_helper_->Start(
      gaia_ids, base::BindOnce(&ProfilePickerHandler::SendAvailableAccounts,
                               weak_factory_.GetWeakPtr()));
}

void ProfilePickerHandler::SendAvailableAccounts(
    std::vector<GetAccountInformationHelper::GetAccountInformationResult>
        accounts) {
  base::Value::List accounts_list;
  for (const GetAccountInformationHelper::GetAccountInformationResult& account :
       accounts) {
    // TODO(https://crbug/1226050): Filter out items with no email as items
    // without an email are impossible to use. The email should be always
    // available, unless the mojo connection fails. This requires more robust
    // unit-tests.
    base::Value::Dict account_dict;
    account_dict.Set("gaiaId", account.gaia);
    account_dict.Set("email", account.email);
    account_dict.Set("name", account.full_name);
    SkBitmap account_bitmap = GetAvailableAccountBitmap(account.account_image);
    account_dict.Set("accountImageUrl",
                     webui::GetBitmapDataUrl(account_bitmap));
    accounts_list.Append(std::move(account_dict));
  }
  FireWebUIListener("available-accounts-changed", accounts_list);
}

void ProfilePickerHandler::OnLacrosSignedInProfileCreated(
    std::optional<SkColor> profile_color,
    Profile* profile) {
  DCHECK(lacros_sign_in_provider_);
  lacros_sign_in_provider_.reset();

  if (!profile) {
    FireWebUIListener("load-signin-finished", base::Value(/*success=*/false));
    return;
  }

  FireWebUIListener("load-signin-finished", base::Value(/*success=*/true));
  ProfilePicker::SwitchToSignedInFlow(profile_color, profile);
}

void ProfilePickerHandler::OnAccountUpserted(
    const base::FilePath& profile_path,
    const account_manager::Account& account) {
  UpdateAvailableAccounts();
}

void ProfilePickerHandler::OnAccountRemoved(
    const base::FilePath& profile_path,
    const account_manager::Account& account) {
  UpdateAvailableAccounts();
}

void ProfilePickerHandler::OnReauthDialogClosed(
    const account_manager::AccountUpsertionResult& result) {
  // After the reauth screen is closed, we can now reuse the profile picker
  // account list to select an account.
  FireWebUIListener("reauth-dialog-closed", base::Value());
  lacros_sign_in_provider_.reset();
}

void ProfilePickerHandler::ShowReauthWithEmail(
    account_manager::AccountManagerFacade::AccountAdditionSource source,
    const std::string& email) {
  account_manager::AccountManagerFacade* account_manager_facade =
      ::GetAccountManagerFacade(GetCurrentProfilePath(web_ui()).value());
  account_manager_facade->ShowReauthAccountDialog(
      source, email,
      base::BindOnce(&ProfilePickerHandler::OnReauthDialogClosed,
                     weak_factory_.GetWeakPtr()));
}

void ProfilePickerHandler::AddExistingAccountToExistingProfile(
    AccountProfileMapper* mapper,
    AccountProfileMapper::AddAccountCallback add_account_callback,
    const std::string& gaia_id) {
  mapper->AddAccount(
      GetCurrentProfilePath(web_ui()),
      account_manager::AccountKey(gaia_id, account_manager::AccountType::kGaia),
      std::move(add_account_callback));
}

void ProfilePickerHandler::AddExistingAccountToNewProfile(
    ProfilePickerLacrosSignInProvider::SignedInCallback signed_in_callback,
    const std::string& gaia_id) {
  lacros_sign_in_provider_->CreateSignedInProfileWithExistingAccount(
      gaia_id, std::move(signed_in_callback));
}

void ProfilePickerHandler::ResultAccountInPersistentError(
    const std::string& gaia_id,
    AddAccountCallback add_account_callback,
    account_manager::AccountManagerFacade::AccountAdditionSource source,
    bool persistent_error) {
  if (persistent_error) {
    // Get email to show reauth dialog.
    identity_manager_lacros_->GetAccountEmail(
        gaia_id, base::BindOnce(&ProfilePickerHandler::ShowReauthWithEmail,
                                weak_factory_.GetWeakPtr(), source));
    return;
  }

  std::move(add_account_callback).Run(gaia_id);
}

void ProfilePickerHandler::SelectAccountLacrosInternal(
    const std::string& gaia_id,
    std::optional<SkColor> profile_color) {
  AllowJavascript();
  if (IsUsingExistingProfile(web_ui())) {
    AccountProfileMapper* mapper =
        g_browser_process->profile_manager()->GetAccountProfileMapper();
    AccountProfileMapper::AddAccountCallback add_account_callback =
        base::BindOnce(&RunAccountSelectionCallback);
    if (gaia_id.empty()) {
      mapper->ShowAddAccountDialog(GetCurrentProfilePath(web_ui()),
                                   account_manager::AccountManagerFacade::
                                       AccountAdditionSource::kOgbAddAccount,
                                   std::move(add_account_callback));
    } else {
      AddAccountCallback add_existing_account_to_existing_profile =
          base::BindOnce(
              &ProfilePickerHandler::AddExistingAccountToExistingProfile,
              weak_factory_.GetWeakPtr(), mapper,
              std::move(add_account_callback));
      identity_manager_lacros_->HasAccountWithPersistentError(
          gaia_id,
          base::BindOnce(&ProfilePickerHandler::ResultAccountInPersistentError,
                         weak_factory_.GetWeakPtr(), gaia_id,
                         std::move(add_existing_account_to_existing_profile),
                         account_manager::AccountManagerFacade::
                             AccountAdditionSource::kContentAreaReauth));
    }
    return;
  }

  DCHECK(!lacros_sign_in_provider_);
  lacros_sign_in_provider_ =
      std::make_unique<ProfilePickerLacrosSignInProvider>(
          /*hidden_profile=*/true);
  ProfilePickerLacrosSignInProvider::SignedInCallback callback =
      base::BindOnce(&ProfilePickerHandler::OnLacrosSignedInProfileCreated,
                     weak_factory_.GetWeakPtr(), profile_color);
  if (gaia_id.empty()) {
    lacros_sign_in_provider_->ShowAddAccountDialogAndCreateSignedInProfile(
        std::move(callback));
  } else {
    AddAccountCallback add_exisiting_account_to_new_profile =
        base::BindOnce(&ProfilePickerHandler::AddExistingAccountToNewProfile,
                       weak_factory_.GetWeakPtr(), std::move(callback));
    identity_manager_lacros_->HasAccountWithPersistentError(
        gaia_id,
        base::BindOnce(&ProfilePickerHandler::ResultAccountInPersistentError,
                       weak_factory_.GetWeakPtr(), gaia_id,
                       std::move(add_exisiting_account_to_new_profile),
                       account_manager::AccountManagerFacade::
                           AccountAdditionSource::kChromeProfileCreation));
  }
}

#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)
