/*
   netrik -- The ANTRIK Internet Viewer
   Copyright (C) Olaf D. Buddenhagen AKA antrik, et al (see AUTHORS)
   Published under the GNU GPL; see LICENSE for details.
*/
/*
 * items.h -- all data structures etc. related to processing the item tree.
 *
 * (C) 2001, 2002, 2003 antrik
 *     2002 Patrice Neff
 */

#ifndef __items_h
#define __items_h

struct Data_string {    /* Pascal-like string (for storing binary data that can contain '\0' as a legal value, e.g. form data) */
   char	*data;    /* char array (*not* zero-terminated string!) */
   int	size;
};

/* update list in dump_items (in render.c) also! */
enum Item_type {
   ITEM_TEXT,    /* text block consisting of one string of continous text */
   ITEM_BOX,    /* box containing other items */
   ITEM_FORM,    /* box with additional form data */
   ITEM_BLANK,    /* blank line */
   ITEM_BLOCK_ANCHOR,    /* virtual block, stored after its children, but at same level */
   ITEM_INLINE_ANCHOR    /* virtual sub-item (string section), stored after its parent (the string item), at same level */
};

/* continuous text stream (possibly changing attributes) */
struct String {
   char				*text;

   int				div_count;    /* number of attribute divisions */
   struct Div {    /* data of divisions */
      int		end;    /* position *after* end (beginning of next div) */
      int		color;    /* color of text in division */
   }				*div;    /* array of divisions */

   int				*line_table;    /* array of positions of line breaks after formating */

   int			link_count;    /* number of links in text block */
   struct Link {    /* link data */
      int		start;    /* starting position (relatively to beginning of string) */
      int		end;
      int		x_start, y_start, x_end, y_end;    /* page coordinates */

      struct Data_string
			value;    /* link URL/form data (NOTE: the "size" component is used *only* for form values presently; for normal links, only the "data" component is uses as a normal C string!) */

      enum Form_control {    /* type of link/form element */
	 FORM_NO=0,    /* no form -- normal link */
	 FORM_TEXT,    /* text input field */
	 FORM_PASS,    /* concealed text input field */
	 FORM_HIDDEN,    /* invisible text field (doesn't make much sense without scripting, but still...) */
	 FORM_CHECKBOX,
	 FORM_RADIO,
	 FORM_OPTION,    /* <select> option */
	 FORM_MULTIOPTION,    /* <select "multiple"> option */
	 FORM_TEXTAREA,
	 FORM_FILE,
	 FORM_SUBMIT    /* button */
      }			form;
      char		*name;    /* "name" in form controls */
      int		enabled;    /* form control can be submitted to server */

      /* hashes for identifying link if page changes */
      int		url_fingerprint;    /* link value */
      int		text_fingerprint;    /* link element content */
   }				*link;    /* array of links */
};

struct Block_anchor {
   struct Item	*virtual_child;    /* points back to first child of the virtual block (the virtual block spans up to the anchor item) */
   char		*id;    /* anchor name */
};

struct Inline_anchor {
   struct Item	*virtual_parent;    /* points back to the virtual parent (text item) containing this anchor */
   int		start;    /* beginning of anchor inside string */
   int		end;
   char		*id;    /* anchor name */
};

struct Form {
   enum Form_method {
      METHOD_GET,    /* as part of URL */
      METHOD_POST_URLENC,    /* in body of HTTP request, URL encoded (application/x-www-form-urlencoded) */
      METHOD_POST_MIMEENC    /* body, multipart MIME encoded (multipart/form-data) */
   }		method;    /* how to submit */
   char 	*url;    /* where to submit ("action") */
};

/* complete item data in structure tree */
struct Item {
   struct Item				*next;    /* next item at same tree level */
   struct Item				*list_next;    /* next item in linear list */
   struct Item				*parent;    /* item in whose area this one is contained */
   struct Item				*first_child;    /* first item in this one's area (others accesed by "next") */

   int					center;
   int					x_start, x_end, y_start, y_end;    /* position in page */

   enum Item_type			type;    /* kind of item */
   union {
      struct String		*string;    /* text data, if ITEM_TEXT */
      struct Block_anchor	*block_anchor;
      struct Inline_anchor	*inline_anchor;
      struct Form		*form;
   }					data;    /* item-type specific data */
};

enum Text_mode {
   TM_NORMAL,
   TM_DIM,    /* "hidden" form controls etc. */
   TM_ITALIC,
   TM_LINK,
   TM_FORM,
   TM_IMG,
   TM_SYS,    /* special text generated by netrik itself */
   TM_ACTIVE    /* highlighted (active) link */
};

struct Item_list {    /* all items present in one line of output page */
   int		count;    /* number of items in line */
   struct Item	**item;    /* array of pointers to items */
};

struct Link_list {    /* all links on page */
   int				count;
   struct Link_ptr {    /* pointers to links */
      struct Item	*item;    /* item containing string with link */
      int		num;    /* number of link inside this string */
   }				*link;    /* array of link pointers */
};

struct Anchor_list {    /* all anchors on page */
   int		count;
   struct Item	**anchor_item;    /* array of pointers to the anchors in the item tree */
};


/* items.c */
int line_start(const struct Item *item, int line);    /* get starting position inside text block */
int line_end(const struct Item *item, int line);    /* get ending position inside text block */

int line_pos(const struct Item *item, int line);    /* get x position on page */

/* pre-render.c */
struct Item_list *pre_render(struct Item *item_tree, int width);    /* place all items on output page */
void free_map(struct Item *item_tree, struct Item_list *page_map);    /* unallocate page usage map */

#endif
