/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
package jalview.gui;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Vector;

import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JToggleButton;
import javax.swing.border.BevelBorder;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;

import org.jibble.epsgraphics.EpsGraphics2D;

import jalview.analysis.AlignmentAnnotationUtils;
import jalview.analysis.AlignmentSorter;
import jalview.analysis.AlignmentUtils;
import jalview.analysis.AnnotationSorter;
import jalview.analysis.AverageDistanceTree;
import jalview.analysis.NJTree;
import jalview.analysis.TreeBuilder;
import jalview.analysis.TreeModel;
import jalview.analysis.AnnotationSorter.SequenceAnnotationOrder;
import jalview.analysis.scoremodels.ScoreModels;
import jalview.api.analysis.ScoreModelI;
import jalview.api.analysis.SimilarityParamsI;
import jalview.bin.Cache;
import jalview.bin.Console;
import jalview.commands.CommandI;
import jalview.commands.OrderCommand;
import jalview.datamodel.Alignment;
import jalview.datamodel.AlignmentAnnotation;
import jalview.datamodel.AlignmentI;
import jalview.datamodel.AlignmentView;
import jalview.datamodel.BinaryNode;
import jalview.datamodel.DBRefEntry;
import jalview.datamodel.HiddenColumns;
import jalview.datamodel.NodeTransformI;
import jalview.datamodel.SequenceFeature;
import jalview.datamodel.SequenceI;
import jalview.datamodel.SequenceNode;
import jalview.gui.ImageExporter.ImageWriterI;
import jalview.io.JalviewFileChooser;
import jalview.io.JalviewFileView;
import jalview.io.NewickFile;
import jalview.io.exceptions.ImageOutputException;
import jalview.jbgui.GTreePanel;
import jalview.util.ImageMaker.TYPE;
import jalview.util.ColorUtils;
import jalview.util.Constants;
import jalview.util.MessageManager;
import jalview.viewmodel.AlignmentViewport;

/**
 * DOCUMENT ME!
 * 
 * @author $author$
 * @version $Revision$
 */
public class TreePanel extends GTreePanel
{
  String treeType;

  String scoreModelName; // if tree computed

  String treeTitle; // if tree loaded

  SimilarityParamsI similarityParams;

  private TreeCanvas treeCanvas;

  TreeModel tree;

  private AlignViewport av;
  
  //if tree is based on annotations like secondary structure
  boolean annotationBased = false; 

  JPanel ssProviderPanel;
  
  private Map<String, Color> secondaryStructureProviderColorMap;
  
  private static final int GAP_WIDTH = 2; //gap between components
  private static final int SS_PROVIDER_PANEL_FONT_SIZE = 10; 
  private static final int TOGGLE_BUTTON_ICON_WIDTH = 10; 
  private static final int TOGGLE_BUTTON_ICON_HEIGHT = 10; 
  private static final int SCROLLBAR_HEIGHT = 8;
  private static final int SCROLL_PANEL_HEIGHT = 30; 
  private static final int TOGGLE_BUTTON_MARGIN = 1; 
  private static final int NO_PADDING = 0;
  private static final int NO_WIDTH_PREFERENCE = 0;

  /**
   * Creates a new TreePanel object.
   * 
   * @param ap
   * @param type
   * @param modelName
   * @param options
   */
  public TreePanel(AlignmentPanel ap, String type, String modelName,
          SimilarityParamsI options)
  {
    super();
    this.setFrameIcon(null);
    this.similarityParams = options;
    initTreePanel(ap, type, modelName, null, null, null, null);

    // We know this tree has distances. JBPNote TODO: prolly should add this as
    // a userdefined default
    // showDistances(true);
  }

  public TreePanel(AlignmentPanel alignPanel, NewickFile newtree,
          String theTitle, AlignmentView inputData)
  {
    this(alignPanel, newtree, theTitle, inputData, null, null);
  }

  /**
   * when true, leaf labels are annotations
   */
  boolean forAnnotation = false;

  public TreePanel(AlignmentPanel alignPanel, NewickFile newtree,
          String theTitle, AlignmentView inputData,
          AlignmentAnnotation[] leafAnnotations, String subTitle)
  {
    super();
    this.forAnnotation = leafAnnotations != null
            && leafAnnotations.length > 0;
    this.setFrameIcon(null);
    this.treeTitle = theTitle;
    initTreePanel(alignPanel, null, null, newtree, inputData, null, null);
  }

  /**
   * columnwise tree associated with positions in aa
   * 
   * @param alignPanel
   * @param fin
   * @param title
   * @param aa
   */
  public TreePanel(AlignmentPanel alignPanel, NewickFile fin,
          AlignmentAnnotation aa, String title)
  {
    super();
    columnWise = true;
    assocAnnotation = aa;
    this.setFrameIcon(null);
    this.treeTitle = title;
    initTreePanel(alignPanel, null, null, fin, null, null, null);
  }
  
  public TreePanel(AlignmentPanel alignPanel, NewickFile newtree,
          String theTitle, AlignmentView inputData,
          AlignmentAnnotation[] leafAnnotations, String subTitle, 
          Map<String, AlignmentAnnotation> annotationIds)
  {
    super();
    this.forAnnotation = leafAnnotations != null
            && leafAnnotations.length > 0;
    this.setFrameIcon(null);
    this.treeTitle = theTitle;
    initTreePanel(alignPanel, null, null, newtree, inputData, null, annotationIds);
  }
  

  boolean columnWise = false;

  AlignmentAnnotation assocAnnotation = null;

  public boolean isColumnWise()
  {
    return columnWise;
  }
  
  public boolean isAnnotationBased()
  {
    return annotationBased;
  }

  public void setAnnotationBased(boolean annotationBased)
  {
    this.annotationBased = annotationBased;
  }

  public AlignmentAnnotation getAssocAnnotation()
  {
    return assocAnnotation;
  }

  public AlignmentI getAlignment()
  {
    return getTreeCanvas().getViewport().getAlignment();
  }

  public AlignmentViewport getViewPort()
  {
    // @Mungo - Why don't we return our own viewport ???
    return getTreeCanvas().getViewport();
  }
  

  public Map<String, Color> getSecondaryStructureProviderColorMap()
  {
    return secondaryStructureProviderColorMap;
  }
  
  /**This method restore the colour map from the loaded project file.
   * It replaces the current colour for the labels present in the 
   * loaded project file.
   * @param colorMap colour map from the loaded project
   */
  public void restoreSecondaryStructureProviderColorMap(Map<String, Color> colorMap)
  {
    secondaryStructureProviderColorMap = colorMap;
    ColorUtils.restoreMyHSBSpacedColours(colorMap);
  }

  void initTreePanel(AlignmentPanel ap, String type, String modelName,
          NewickFile newTree, AlignmentView inputData,
          AlignmentAnnotation[] leafAnnotations, Map<String, AlignmentAnnotation> annotationIds)
  {

    av = ap.av;
    this.treeType = type;
    this.scoreModelName = modelName;

    treeCanvas = new TreeCanvas(this, ap, scrollPane);
    scrollPane.setViewportView(treeCanvas);

    if (isAnnotationBased() || (similarityParams != null
            && similarityParams.getSecondaryStructureSource() != null))
    {
      //setting showSecondaryStructureProviderMenu to true if the 
      //similarity is based on secondary structure
      showSecondaryStructureProviderMenu.setVisible(true);
      sortAnnotationAssocViews.setVisible(true); 
      setAnnotationBased(true);
    }
    else {
      //setting showSecondaryStructureProviderMenu to false if the 
      //similarity is not based on secondary structure
      showSecondaryStructureProviderMenu.setVisible(false);
      sortAnnotationAssocViews.setVisible(false);
      setAnnotationBased(false);
    }
    if (leafAnnotations != null)
    {
      forAnnotation = true;
    }
    if (columnWise)
    {
      bootstrapMenu.setVisible(false);
      placeholdersMenu.setState(false);
      placeholdersMenu.setVisible(false);
      fitToWindow.setState(false);
      sortAssocViews.setVisible(false);
      sortAnnotationAssocViews.setVisible(false);
    }

    addKeyListener(new KeyAdapter()
    {
      @Override
      public void keyPressed(KeyEvent e)
      {
        switch (e.getKeyCode())
        {
        case 27: // escape
          treeCanvas.clearSelectedLeaves();
          e.consume();
          break;

        }

      }
    });
    PaintRefresher.Register(this, ap.av.getSequenceSetId());

    buildAssociatedViewMenu();

    final PropertyChangeListener listener = addAlignmentListener();

    /*
     * remove listener when window is closed, so that this
     * panel can be garbage collected
     */
    addInternalFrameListener(new InternalFrameAdapter()
    {
      @Override
      public void internalFrameClosed(InternalFrameEvent evt)
      {
        if (av != null)
        {
          av.removePropertyChangeListener(listener);
        }
        releaseReferences();
      }
    });

    TreeLoader tl = new TreeLoader(newTree, inputData, leafAnnotations, annotationIds);
    tl.start();

  }  
  
  
  /**
   * This method adds the secondary structure provider panel. The panel includes 
   * scrollable list of secondary structure providers and their corresponding 
   * toggle buttons with color key. Secondary structure providers toggle buttons
   * with color key is shown if user selects the option available in 
   * View->Show secondary structure providers -> As coloured lines. And, list of 
   * secondary structure providers in all other cases. A CardLayout is used to 
   * switch between views.
   */
  public void addSSProviderPanel()
  {
    
    if(secondaryStructureProviderColorMap == null || secondaryStructureProviderColorMap.isEmpty()) {
      return;      
    }
    
    // Panels to display the color map and provider list
    JPanel ssProviderColorMapPanel = new JPanel();
    JScrollPane colorMapScrollPanel;

    // Set layouts to arrange components horizontally
    ssProviderColorMapPanel.setLayout(new BoxLayout(ssProviderColorMapPanel, BoxLayout.X_AXIS));

    // Get the color mapping
    List<Map.Entry<String, Color>> sortedSSProviderColorMap = new ArrayList<>(
            secondaryStructureProviderColorMap.entrySet());
    
    // Sort the providers alphabetically by name
    sortedSSProviderColorMap.sort(Map.Entry.comparingByKey()); 
    
    // Add toggle buttons and provider labels in the panels iteratively for each provider
    for (Map.Entry<String, Color> sortedSSProviderColor : sortedSSProviderColorMap)
    {
      // Add toggle button panel
      JPanel toggleButtonPanel = createToggleButtonForProvider(sortedSSProviderColor);
      ssProviderColorMapPanel.add(toggleButtonPanel);
      ssProviderColorMapPanel.add(Box.createRigidArea(new Dimension(GAP_WIDTH, 0))); // gap
    }

    // Set an empty border for the color map panel with no padding
    ssProviderColorMapPanel.setBorder(BorderFactory.createEmptyBorder(NO_PADDING,
            NO_PADDING, NO_PADDING, NO_PADDING)); // no padding
    ssProviderColorMapPanel.revalidate();
    ssProviderColorMapPanel.repaint();

    // Add panels in scroll pane
    colorMapScrollPanel = initializeColorMapScrollPanel(ssProviderColorMapPanel);    

    // Create a CardLayout panel to switch between the panels
    ssProviderPanel = new JPanel(new CardLayout());
    ssProviderPanel.add(colorMapScrollPanel, "colorMapScrollPanel");

    // Show the panel based on user selection
    CardLayout ssProviderCardLayout = (CardLayout) (ssProviderPanel.getLayout());
    if (showStructureProviderColouredLines.isSelected()) {
        ssProviderCardLayout.show(ssProviderPanel, "colorMapScrollPanel");
        ssProviderPanel.setVisible(true);
    }
    else {
      ssProviderPanel.setVisible(false);
    }
    
    ssProviderPanel.repaint();   
    
    this.add(ssProviderPanel, BorderLayout.NORTH);
    this.revalidate();
    this.repaint();
  }

  /**
   * Creates a toggle button panel for a secondary structure provider.
   *
   * @param colorKey A Map.Entry with the provider name and its colour.
   * @return A JPanel with the toggle button.
   */
  private JPanel createToggleButtonForProvider(Map.Entry<String, Color> colorKey)
  {
    String provider = colorKey.getKey();
    Color color = colorKey.getValue();

    JToggleButton toggleButton = createToggleButton(provider, color);
    JPanel toggleButtonPanel = new JPanel();
    toggleButtonPanel.setLayout(new BoxLayout(toggleButtonPanel, BoxLayout.X_AXIS));
    toggleButtonPanel.add(toggleButton);
    toggleButtonPanel.add(Box.createRigidArea(new Dimension(GAP_WIDTH, 0))); // gap

    return toggleButtonPanel;
  }

  
  /**
   * Creates a toggle button for a secondary structure provider with 
   * corresponding coloured icon. The secondary structure provider lines
   * are coloured if the button is OFF (default). The lines are grayed
   * if the button is ON.
   *
   * @param provider The name of the secondary structure provider.
   * @param color The colour mapped with the secondary structure provider.
   * @return JToggleButton with the secondary structure provider name 
   * and colour icon.
   */
  private JToggleButton createToggleButton(String provider, Color color)
  {
    JToggleButton ssProviderToggleButton = new JToggleButton(provider);
    ssProviderToggleButton.setFont(new Font(getTreeFont().getName(),
            getTreeFont().getStyle(), SS_PROVIDER_PANEL_FONT_SIZE));
    ssProviderToggleButton.setMargin(
            new Insets(TOGGLE_BUTTON_MARGIN, TOGGLE_BUTTON_MARGIN,
                    TOGGLE_BUTTON_MARGIN, TOGGLE_BUTTON_MARGIN));

    // Icon to display the secondary structure provider colour
    Icon toggleButtonColorIcon = new Icon()
    {
      @Override
      public void paintIcon(Component c, Graphics g, int x, int y)
      {
        g.setColor(color);
        g.fillRect(x, y, getIconWidth(), getIconHeight());
      }

      @Override
      public int getIconWidth()
      {
        return TOGGLE_BUTTON_ICON_WIDTH;
      }

      @Override
      public int getIconHeight()
      {
        return TOGGLE_BUTTON_ICON_HEIGHT;
      }
    };

    ssProviderToggleButton.setIcon(toggleButtonColorIcon);
    ssProviderToggleButton
            .setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
    ssProviderToggleButton.setIconTextGap(GAP_WIDTH);
    ssProviderToggleButton.setSelected(false); // Set default to OFF

    // button listener 
    ssProviderToggleButton.addItemListener(e -> {
      boolean selected = (e.getStateChange() == ItemEvent.SELECTED);
      toggleStructureProviderColouredLine(provider, selected);
    });

    return ssProviderToggleButton;
  }

  /**
   * Initializes a scroll panel to wrap the given JPanel to include in the tree panel.
   *
   * @param jPanel The JPanel to wrap in a scroll pane.
   * @return A JScrollPane containing the given JPanel.
   */
  private JScrollPane initializeColorMapScrollPanel(JPanel jPanel)
  {
    JScrollPane scrollPanel = new JScrollPane(jPanel);
    scrollPanel.setPreferredSize(new Dimension(NO_WIDTH_PREFERENCE, SCROLL_PANEL_HEIGHT));
    scrollPanel.setHorizontalScrollBarPolicy(
            JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
    
    JScrollBar horizontalScrollBar = scrollPanel
            .getHorizontalScrollBar();
    horizontalScrollBar.setPreferredSize(
            new Dimension(NO_WIDTH_PREFERENCE, SCROLLBAR_HEIGHT));
    
    return scrollPanel;
  }

  
  
  /**
   * Ensure any potentially large object references are nulled
   */
  public void releaseReferences()
  {
    this.tree = null;
    this.treeCanvas.tree = null;
    this.treeCanvas.nodeHash = null;
    this.treeCanvas.nameHash = null;
  }

  /**
   * @return
   */
  protected PropertyChangeListener addAlignmentListener()
  {
    final PropertyChangeListener listener = new PropertyChangeListener()
    {
      @Override
      public void propertyChange(PropertyChangeEvent evt)
      {
        if (evt.getPropertyName().equals("alignment"))
        {
          if (tree == null)
          {
            jalview.bin.Console.outPrintln("tree is null");
            // TODO: deal with case when a change event is received whilst a
            // tree is still being calculated - should save reference for
            // processing message later.
            return;
          }
          if (evt.getNewValue() == null)
          {
            jalview.bin.Console.outPrintln(
                    "new alignment sequences vector value is null");
          }

          tree.updatePlaceHolders((List<SequenceI>) evt.getNewValue());
          treeCanvas.nameHash.clear(); // reset the mapping between canvas
          // rectangles and leafnodes
          repaint();
        }
      }
    };
    av.addPropertyChangeListener(listener);
    return listener;
  }

  @Override
  public void viewMenu_menuSelected()
  {
    buildAssociatedViewMenu();
  }

  void buildAssociatedViewMenu()
  {
    AlignmentPanel[] aps = PaintRefresher
            .getAssociatedPanels(av.getSequenceSetId());
    if (aps.length == 1 && getTreeCanvas().getAssociatedPanel() == aps[0])
    {
      associateLeavesMenu.setVisible(false);
      return;
    }

    associateLeavesMenu.setVisible(true);

    if ((viewMenu
            .getItem(viewMenu.getItemCount() - 2) instanceof JMenuItem))
    {
      viewMenu.insertSeparator(viewMenu.getItemCount() - 1);
    }

    associateLeavesMenu.removeAll();

    JRadioButtonMenuItem item;
    ButtonGroup buttonGroup = new ButtonGroup();
    int i, iSize = aps.length;
    final TreePanel thisTreePanel = this;
    for (i = 0; i < iSize; i++)
    {
      final AlignmentPanel ap = aps[i];
      item = new JRadioButtonMenuItem(ap.av.getViewName(),
              ap == treeCanvas.getAssociatedPanel());
      buttonGroup.add(item);
      item.addActionListener(new ActionListener()
      {
        @Override
        public void actionPerformed(ActionEvent evt)
        {
          treeCanvas.applyToAllViews = false;
          treeCanvas.setAssociatedPanel(ap);
          treeCanvas.setViewport(ap.av);
          PaintRefresher.Register(thisTreePanel, ap.av.getSequenceSetId());
        }
      });

      associateLeavesMenu.add(item);
    }

    final JRadioButtonMenuItem itemf = new JRadioButtonMenuItem(
            MessageManager.getString("label.all_views"));
    buttonGroup.add(itemf);
    itemf.setSelected(treeCanvas.applyToAllViews);
    itemf.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent evt)
      {
        treeCanvas.applyToAllViews = itemf.isSelected();
      }
    });
    associateLeavesMenu.add(itemf);

  }

  class TreeLoader extends Thread
  {
    private NewickFile newtree;

    private AlignmentView odata = null;

    private AlignmentAnnotation[] leafAnnotations;
    
    private Map<String, AlignmentAnnotation> annotationIds = null;

    public TreeLoader(NewickFile newickFile, AlignmentView inputData,
            AlignmentAnnotation[] leafAnnotations, 
            Map<String, AlignmentAnnotation> annotationIds)
    {
      this.newtree = newickFile;
      this.odata = inputData;
      this.leafAnnotations = leafAnnotations;
      this.annotationIds = annotationIds; 

      if (newickFile != null)
      {
        // Must be outside run(), as Jalview2XML tries to
        // update distance/bootstrap visibility at the same time
        showBootstrap(newickFile.HasBootstrap());
        showDistances(newickFile.HasDistances());
      }
    }

    @Override
    public void run()
    {

      if (newtree != null)
      {
        tree = new TreeModel(av.getAlignment().getSequencesArray(), odata,
                newtree, leafAnnotations);
        if (tree.getOriginalData() == null)
        {
          originalSeqData.setVisible(false);
        }
      }
      else
      {
        ScoreModelI sm = ScoreModels.getInstance().getScoreModel(
                scoreModelName, treeCanvas.getAssociatedPanel());
        TreeBuilder njtree = treeType.equals(TreeBuilder.NEIGHBOUR_JOINING)
                ? new NJTree(av, sm, similarityParams)
                : new AverageDistanceTree(av, sm, similarityParams);
        List<String> labels = njtree.getLabels();
        if(labels != null && labels.size()>0) {
          secondaryStructureProviderColorMap = AlignmentUtils.assignColorsForSecondaryStructureProviders(labels);
          if(secondaryStructureProviderColorMap.size()>0) {
            addSSProviderPanel();
          }
        }
        tree = new TreeModel(njtree);
        // don't display distances for columnwise trees

      }
      
      // Process tree nodes to map annotations if annotation ids are present
      if (annotationIds != null && annotationIds.size() > 0)
      {
        TreePanel.processTreeNodes(tree, annotationIds);
      }
      
      showDistances(!columnWise);
      tree.reCount(tree.getTopNode());
      tree.findHeight(tree.getTopNode());
      treeCanvas.setTree(tree);
      treeCanvas.repaint();
      av.setCurrentTree(tree);
      if (av.getSortByTree())
      {
        sortByTree_actionPerformed();
      }
    }
  }

  public void showDistances(boolean b)
  {
    treeCanvas.setShowDistances(b);
    distanceMenu.setSelected(b);
  }

  public void showBootstrap(boolean b)
  {
    treeCanvas.setShowBootstrap(b);
    bootstrapMenu.setSelected(b);
  }

  public void showPlaceholders(boolean b)
  {
    placeholdersMenu.setState(b);
    treeCanvas.setMarkPlaceholders(b);
  }

  /**
   * DOCUMENT ME!
   * 
   * @return DOCUMENT ME!
   */
  public TreeModel getTree()
  {
    return tree;
  }

  /**
   * DOCUMENT ME!
   * 
   * @param e
   *          DOCUMENT ME!
   */
  @Override
  public void textbox_actionPerformed(ActionEvent e)
  {
    CutAndPasteTransfer cap = new CutAndPasteTransfer();

    String newTitle = getPanelTitle();

    NewickFile fout = new NewickFile(tree.getTopNode());
    try
    {
      cap.setText(fout.print(tree.hasBootstrap(), tree.hasDistances(),
              tree.hasRootDistance()));
      Desktop.addInternalFrame(cap, newTitle, 500, 100);
    } catch (OutOfMemoryError oom)
    {
      new OOMWarning("generating newick tree file", oom);
      cap.dispose();
    }

  }

  /**
   * DOCUMENT ME!
   * 
   * @param e
   *          DOCUMENT ME!
   */
  @Override
  public void saveAsNewick_actionPerformed(ActionEvent e)
  {
    // TODO: JAL-3048 save newick file for Jalview-JS
    JalviewFileChooser chooser = new JalviewFileChooser(
            Cache.getProperty("LAST_DIRECTORY"));
    chooser.setFileView(new JalviewFileView());
    chooser.setDialogTitle(
            MessageManager.getString("label.save_tree_as_newick"));
    chooser.setToolTipText(MessageManager.getString("action.save"));

    int value = chooser.showSaveDialog(null);

    if (value == JalviewFileChooser.APPROVE_OPTION)
    {
      String choice = chooser.getSelectedFile().getPath();
      Cache.setProperty("LAST_DIRECTORY",
              chooser.getSelectedFile().getParent());

      try
      {
        jalview.io.NewickFile fout = new jalview.io.NewickFile(
                tree.getTopNode());
        String output = fout.print(tree.hasBootstrap(), tree.hasDistances(),
                tree.hasRootDistance());
        java.io.PrintWriter out = new java.io.PrintWriter(
                new java.io.FileWriter(choice));
        out.println(output);
        out.close();
      } catch (Exception ex)
      {
        ex.printStackTrace();
      }
    }
  }

  /**
   * DOCUMENT ME!
   * 
   * @param e
   *          DOCUMENT ME!
   */
  @Override
  public void printMenu_actionPerformed(ActionEvent e)
  {
    // Putting in a thread avoids Swing painting problems
    treeCanvas.startPrinting();
  }

  @Override
  public void originalSeqData_actionPerformed(ActionEvent e)
  {
    AlignmentView originalData = tree.getOriginalData();
    if (originalData == null)
    {
      Console.info(
              "Unexpected call to originalSeqData_actionPerformed - should have hidden this menu action.");
      return;
    }
    // decide if av alignment is sufficiently different to original data to
    // warrant a new window to be created
    // create new alignmnt window with hidden regions (unhiding hidden regions
    // yields unaligned seqs)
    // or create a selection box around columns in alignment view
    // test Alignment(SeqCigar[])
    char gc = '-';
    try
    {
      // we try to get the associated view's gap character
      // but this may fail if the view was closed...
      gc = av.getGapCharacter();

    } catch (Exception ex)
    {
    }

    Object[] alAndColsel = originalData.getAlignmentAndHiddenColumns(gc);

    if (alAndColsel != null && alAndColsel[0] != null)
    {
      // AlignmentOrder origorder = new AlignmentOrder(alAndColsel[0]);

      AlignmentI al = new Alignment((SequenceI[]) alAndColsel[0]);
      AlignmentI dataset = (av != null && av.getAlignment() != null)
              ? av.getAlignment().getDataset()
              : null;
      if (dataset != null)
      {
        al.setDataset(dataset);
      }

      if (true)
      {
        // make a new frame!
        AlignFrame af = new AlignFrame(al, (HiddenColumns) alAndColsel[1],
                AlignFrame.DEFAULT_WIDTH, AlignFrame.DEFAULT_HEIGHT);

        // >>>This is a fix for the moment, until a better solution is
        // found!!<<<
        // af.getFeatureRenderer().transferSettings(alignFrame.getFeatureRenderer());

        // af.addSortByOrderMenuItem(ServiceName + " Ordering",
        // msaorder);

        Desktop.addInternalFrame(af, MessageManager.formatMessage(
                "label.original_data_for_params", new Object[]
                { this.title }), AlignFrame.DEFAULT_WIDTH,
                AlignFrame.DEFAULT_HEIGHT);
      }
    }
  }

  /**
   * DOCUMENT ME!
   * 
   * @param e
   *          DOCUMENT ME!
   */
  @Override
  public void fitToWindow_actionPerformed(ActionEvent e)
  {
    treeCanvas.fitToWindow = fitToWindow.isSelected();
    repaint();
  }

  /** This method restores the menu options and view settings 
   * in the tree panel if the tree is based on annotation. 
   * 
   * @param viewName The type of annotation view in the loaded project 
   *                 : None, Labels, Coloured Lines. 
   */
  public void setShowAnnotationAs(String viewName)
  {

    if (isAnnotationBased())
    {
      //setting showSecondaryStructureProviderMenu to true if the 
      //similarity is based on secondary structure
      addSSProviderPanel();
      showSecondaryStructureProviderMenu.setVisible(true);
      setShowAnnotationAsView(viewName);
      sortAnnotationAssocViews.setVisible(true); 

    }
    this.treeCanvas.revalidate();
    revalidate();
    repaint();
  }

  /** This method restore the annotation view based on the view name
   * 
   * @param viewName Name of the view name (for None, Labels, Coloured Lines)
   *  which matches with the text displayed in the button.
   */
  public void setShowAnnotationAsView(String viewName) 
  {
      for (AbstractButton button : Arrays.asList(showStructureProviderLabels, 
              showStructureProviderColouredLines, hideStructureProviders)) {
          if (button.getText().equals(viewName)) {
              button.doClick(); // Triggers the action event
              return;
          }
      }
      // The default view - None
      hideStructureProviders.doClick();
  }
  
  /**
   * This method retrieves the text in the selected view button (for None, 
   * Labels, Coloured Lines) in the tree panel and returns it.
   * @return The text in the selected view button
   */
  public String getShowAnnotationAs() {
    for (AbstractButton button : Arrays.asList(showStructureProviderLabels, showStructureProviderColouredLines, hideStructureProviders)) {
        if (button.isSelected()) {
            return button.getText(); // Gets the text
        }
    }
    return hideStructureProviders.getText();
}


  /**
   * sort the associated alignment view by the current tree.
   * 
   * @param e
   */
  @Override
  public void sortByTree_actionPerformed()
  {

    if (treeCanvas.applyToAllViews)
    {
      final ArrayList<CommandI> commands = new ArrayList<>();
      for (AlignmentPanel ap : PaintRefresher
              .getAssociatedPanels(av.getSequenceSetId()))
      {
        commands.add(sortAlignmentIn(ap.av.getAlignPanel()));
      }
      av.getAlignPanel().alignFrame.addHistoryItem(new CommandI()
      {

        @Override
        public void undoCommand(AlignmentI[] views)
        {
          for (CommandI tsort : commands)
          {
            tsort.undoCommand(views);
          }
        }

        @Override
        public int getSize()
        {
          return commands.size();
        }

        @Override
        public String getDescription()
        {
          return "Tree Sort (many views)";
        }

        @Override
        public void doCommand(AlignmentI[] views)
        {

          for (CommandI tsort : commands)
          {
            tsort.doCommand(views);
          }
        }
      });
      for (AlignmentPanel ap : PaintRefresher
              .getAssociatedPanels(av.getSequenceSetId()))
      {
        // ensure all the alignFrames refresh their GI after adding an undo item
        ap.alignFrame.updateEditMenuBar();
      }
    }
    else
    {
      treeCanvas.getAssociatedPanel().alignFrame.addHistoryItem(
              sortAlignmentIn(treeCanvas.getAssociatedPanel()));
    }

  }
  
  /**
   * This method sort the annotation panel by tree. The sorted annotation rows
   * from tree appear first followed by other annotation rows in the original
   * order. Sort annotation by sequence/label overrides sort annotation by tree.
   */
  @Override
  public void sortAnnotationByTree_actionPerformed()
  {
    if (getAlignment() != null
            && getAlignment().getAlignmentAnnotation() != null
            && getAlignment().getAlignmentAnnotation().length > 0)
    {
      // Initialize annotation sorter with the alignment
      final AnnotationSorter sorter = new AnnotationSorter(getAlignment(),
              av.isShowAutocalculatedAbove());

      // Sort by tree
      sorter.sortByTree(getAlignment().getAlignmentAnnotation(), tree);
      av.setSortAnnotationsBy(SequenceAnnotationOrder.NONE);
      // Repaint the alignment panel after the sort
      av.getAlignPanel().repaint();
    }
  }
  

  public CommandI sortAlignmentIn(AlignmentPanel ap)
  {
    // TODO: move to alignment view controller
    AlignmentViewport viewport = ap.av;
    SequenceI[] oldOrder = viewport.getAlignment().getSequencesArray();
    AlignmentSorter.sortByTree(viewport.getAlignment(), tree);
    CommandI undo;
    undo = new OrderCommand("Tree Sort", oldOrder, viewport.getAlignment());

    ap.paintAlignment(true, false);
    return undo;
  }

  /**
   * DOCUMENT ME!
   * 
   * @param e
   *          DOCUMENT ME!
   */
  @Override
  public void font_actionPerformed(ActionEvent e)
  {
    if (treeCanvas == null)
    {
      return;
    }

    new FontChooser(this);
  }

  public Font getTreeFont()
  {
    return treeCanvas.font;
  }

  public void setTreeFont(Font f)
  {
    if (treeCanvas != null)
    {
      treeCanvas.setFont(f);
    }
  }

  /**
   * DOCUMENT ME!
   * 
   * @param e
   *          DOCUMENT ME!
   */
  @Override
  public void distanceMenu_actionPerformed(ActionEvent e)
  {
    treeCanvas.setShowDistances(distanceMenu.isSelected());
  }
  
  @Override
  public void hideStructureProviders_actionPerformed(ActionEvent e)
  {
    treeCanvas.hideStructureProviders(hideStructureProviders.isSelected());
       
    // Show the panel based on user selection
    CardLayout ssProviderCardLayout = (CardLayout) (ssProviderPanel.getLayout());
    if (showStructureProviderColouredLines.isSelected()) {
        ssProviderCardLayout.show(ssProviderPanel, "colorMapScrollPanel");    
        ssProviderPanel.setVisible(true);
    }
    else {
      ssProviderPanel.setVisible(false);
    }    
    ssProviderPanel.repaint();
  }
  
  @Override
  public void showStructureProviderColouredLines_actionPerformed(ActionEvent e)
  {
    treeCanvas.setShowStructureProviderColouredLines(showStructureProviderColouredLines.isSelected());
    
    // Show the panel based on user selection
    CardLayout ssProviderCardLayout = (CardLayout) (ssProviderPanel.getLayout());
    
    if (showStructureProviderColouredLines.isSelected()) {
        ssProviderCardLayout.show(ssProviderPanel, "colorMapScrollPanel");  
        ssProviderPanel.setVisible(true);
    }
    else {
      ssProviderPanel.setVisible(false);
    }
    ssProviderPanel.repaint();
  }
  
  @Override
  public void showStructureProviderLabels_actionPerformed(ActionEvent e)
  {
    treeCanvas.setShowStructureProviderLabels(showStructureProviderLabels.isSelected());  
    
    // Show the panel based on user selection
    CardLayout ssProviderCardLayout = (CardLayout) (ssProviderPanel.getLayout());
    if (showStructureProviderColouredLines.isSelected()) {
        ssProviderCardLayout.show(ssProviderPanel, "colorMapScrollPanel");
        ssProviderPanel.setVisible(true);
    }
    else {
      ssProviderPanel.setVisible(false);
    }
    ssProviderPanel.repaint();
  }
  

  public void toggleStructureProviderColouredLine(String provider, boolean action)
  {
    treeCanvas.toggleStructureProviderColouredLine(provider, action);
  }

  /**
   * DOCUMENT ME!
   * 
   * @param e
   *          DOCUMENT ME!
   */
  @Override
  public void bootstrapMenu_actionPerformed(ActionEvent e)
  {
    treeCanvas.setShowBootstrap(bootstrapMenu.isSelected());
  }

  /**
   * DOCUMENT ME!
   * 
   * @param e
   *          DOCUMENT ME!
   */
  @Override
  public void placeholdersMenu_actionPerformed(ActionEvent e)
  {
    treeCanvas.setMarkPlaceholders(placeholdersMenu.isSelected());
  }

  /**
   * Outputs the Tree in image format (currently EPS or PNG). The user is
   * prompted for the file to save to, and for EPS (unless a preference is
   * already set) for the choice of Text or Lineart for character rendering.
   */
  @Override
  public void writeTreeImage(TYPE imageFormat)
  {
    int width = treeCanvas.getWidth();
    int height = treeCanvas.getHeight();
    ImageWriterI writer = new ImageWriterI()
    {
      @Override
      public void exportImage(Graphics g) throws Exception
      {
        treeCanvas.draw(g, width, height);
      }
    };
    String tree = MessageManager.getString("label.tree");
    ImageExporter exporter = new ImageExporter(writer, null, imageFormat,
            tree);
    try
    {
      exporter.doExport(null, this, width, height,
              tree.toLowerCase(Locale.ROOT));
    } catch (ImageOutputException ioex)
    {
      Console.error(
              "Unexpected error whilst writing " + imageFormat.toString(),
              ioex);
    }
  }

  /**
   * change node labels to the annotation referred to by labelClass TODO:
   * promote to a datamodel modification that can be undone TODO: make argument
   * one case of a generic transformation function ie { undoStep = apply(Tree,
   * TransformFunction)};
   * 
   * @param labelClass
   */
  public void changeNames(final String labelClass)
  {
    tree.applyToNodes(new NodeTransformI()
    {

      @Override
      public void transform(BinaryNode node)
      {
        if (node instanceof SequenceNode
                && !((SequenceNode) node).isPlaceholder()
                && !((SequenceNode) node).isDummy())
        {
          String newname = null;
          SequenceI sq = (SequenceI) ((BinaryNode) node).element();
          if (sq != null)
          {
            // search dbrefs, features and annotation
            List<DBRefEntry> refs = jalview.util.DBRefUtils
                    .selectRefs(sq.getDBRefs(), new String[]
                    { labelClass.toUpperCase(Locale.ROOT) });
            if (refs != null)
            {
              for (int i = 0, ni = refs.size(); i < ni; i++)
              {
                if (newname == null)
                {
                  newname = new String(refs.get(i).getAccessionId());
                }
                else
                {
                  newname += "; " + refs.get(i).getAccessionId();
                }
              }
            }
            if (newname == null)
            {
              List<SequenceFeature> features = sq.getFeatures()
                      .getPositionalFeatures(labelClass);
              for (SequenceFeature feature : features)
              {
                if (newname == null)
                {
                  newname = feature.getDescription();
                }
                else
                {
                  newname = newname + "; " + feature.getDescription();
                }
              }
            }
          }
          if (newname != null)
          {
            // String oldname = ((SequenceNode) node).getName();
            // TODO : save oldname in the undo object for this modification.
            ((BinaryNode) node).setName(newname);
          }
        }
      }
    });
  }

  /**
   * Formats a localised title for the tree panel, like
   * <p>
   * Neighbour Joining Using BLOSUM62
   * <p>
   * For a tree loaded from file, just uses the file name
   * 
   * @return
   */
  public String getPanelTitle()
  {
    if (treeTitle != null)
    {
      return treeTitle;
    }

    /*
     * i18n description of Neighbour Joining or Average Distance method
     */
    String treecalcnm = MessageManager.getString(
            "label.tree_calc_" + treeType.toLowerCase(Locale.ROOT));

    /*
     * short score model name (long description can be too long)
     */
    String smn = scoreModelName;

    /*
     * put them together as <method> Using <model>
     */
    final String ttl = MessageManager.formatMessage("label.calc_title",
            treecalcnm, smn);
    return ttl;
  }

  /**
   * Builds an EPS image and writes it to the specified file.
   * 
   * @param outFile
   * @param textOption
   *          true for Text character rendering, false for Lineart
   */
  protected void writeEpsFile(File outFile, boolean textOption)
  {
    try
    {
      int width = treeCanvas.getWidth();
      int height = treeCanvas.getHeight();

      FileOutputStream out = new FileOutputStream(outFile);
      EpsGraphics2D pg = new EpsGraphics2D("Tree", out, 0, 0, width,
              height);
      pg.setAccurateTextMode(!textOption);
      treeCanvas.draw(pg, width, height);

      pg.flush();
      pg.close();
    } catch (Exception ex)
    {
      jalview.bin.Console.errPrintln("Error writing tree as EPS");
      ex.printStackTrace();
    }
  }

  public AlignViewport getViewport()
  {
    return av;
  }

  public void setViewport(AlignViewport av)
  {
    this.av = av;
  }

  public TreeCanvas getTreeCanvas()
  {
    return treeCanvas;
  }

  /** This method maps the tree nodes with annotation rows and respective sequences.
   * 
   * @param alignPanel Alignment panel object
   * @param nf Loaded newick file 
   * @param annotationIds Map of annotation ids and respective annotation rows
   * @param treeTitle Title of the tree panel
   * @param input Alignment view object
   * @return Returns tree panel with the nodes mapped with annotation rows and sequences
   */
  public static TreePanel newTreeForAnnotations(AlignmentPanel alignPanel,
          NewickFile nf, Map<String, AlignmentAnnotation> annotationIds,
          String treeTitle, AlignmentView input)
  {
    // Create TreePanel instance
    TreePanel tp = new TreePanel(alignPanel, nf, treeTitle, input, null,
            "Annotation Tree", annotationIds);

    return tp;
  }


  /**
   * Processes tree nodes by applying transformations based on annotations.
   * Maps the nodes with annotation rows and/or sequences
   * @param tree
   * @param annotationIds
   */
  private static void processTreeNodes(TreeModel tree,
          Map<String, AlignmentAnnotation> annotationIds)
  {
    // Initialise the node attribute in the tree model
    // object with the leaf nodes
    BinaryNode topNode = tree.getTopNode();
    Vector<BinaryNode> leafNodes = tree.findLeaves(topNode);
    tree.setNode(leafNodes);

    //Apply transformation: map nodes with annotation rows and sequences
    tree.applyToNodes(node -> {
      Object element = node.element();
      String nodeName = node.getName(); //Annotation id in case of annotation based tree

      // Extract annotation ID
      String annotId = nodeName.replaceAll("[^a-zA-Z0-9]", "");
      
      //Get the annotation object
      AlignmentAnnotation annot = annotationIds.get(annotId);

      if (annot != null)
      {
        // Map the sequence and annotation with the node
        SequenceI seq = annot.sequenceRef;
        node.setElement(seq);
        node.setName(seq.getName());

        // Get annotation provider name from the annotation description
        String ssAnnotProvider = AlignmentAnnotationUtils
                .extractSSSourceFromAnnotationDescription(annot);
        node.setLabel(ssAnnotProvider);
        node.setAlignmentAnnotation(annot);
        
        // Initialise AnnotationDetails if present for the annotation
        if (annot.hasAnnotationDetailsProperty())
        {
          node.setAnnotationDetails(annot.getAnnotationDetailsProperty());
        }

      }
      else if (element instanceof SequenceI)
      {
        //If annotation is absent, set the default label 
        node.setLabel(Constants.STRUCTURE_PROVIDERS.get("None"));
      }
    });
  }
}
