#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Copyright 2019 Daniel Estevez <daniel@destevez.net>
#
# This file is part of gr-satellites
#
# SPDX-License-Identifier: GPL-3.0-or-later
#

from gnuradio import gr, blocks, digital
from ... import snet_deframer as deframer
from ...hier.sync_to_pdu import sync_to_pdu
from ...utils.options_block import options_block

_syncword = '00000100110011110101111111001000'

class snet_deframer(gr.hier_block2, options_block):
    """
    Hierarchical block to deframe the S-NET custom framing.

    The input is a float stream of soft symbols. The output are PDUs
    with frames.

    Args:
        buggy_crc: use buggy CRC implementation of S-NET
        syncword_threshold: number of bit errors allowed in syncword (int)
        options: Options from argparse
    """
    def __init__(self, buggy_crc = True, syncword_threshold = None, options = None):
        gr.hier_block2.__init__(self, "snet_deframer",
            gr.io_signature(1, 1, gr.sizeof_float),
            gr.io_signature(0, 0, 0))
        options_block.__init__(self, options)
        
        self.message_port_register_hier_out('out')

        if syncword_threshold is None:
            syncword_threshold = self.options.syncword_threshold

        self.slicer = digital.binary_slicer_fb()
        self.deframer = sync_to_pdu(packlen = 8 * 512,\
                                    sync = _syncword,\
                                    threshold = syncword_threshold)
        self.fec = deframer(self.options.verbose_fec,
                                buggy_crc = buggy_crc)
        
        self.connect(self, self.slicer, self.deframer)
        self.msg_connect((self.deframer, 'out'), (self.fec, 'in'))
        self.msg_connect((self.fec, 'out'), (self, 'out'))

    _default_sync_threshold = 4
        
    @classmethod
    def add_options(cls, parser):
        """
        Adds S-NET deframer specific options to the argparse parser
        """
        parser.add_argument('--syncword_threshold', type = int, default = cls._default_sync_threshold, help = 'Syncword bit errors [default=%(default)r]')
        parser.add_argument('--verbose_fec', action = 'store_true', help = 'Verbose FEC decoder')
