/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id: AbstractPSPDFBitmapProducer.java 1851806 2019-01-22 11:53:31Z ssteiner $ */

package org.apache.fop.visual;

import java.awt.image.BufferedImage;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.text.MessageFormat;

import javax.xml.transform.Transformer;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.io.IOUtils;

import org.apache.fop.apps.FOUserAgent;
import org.apache.fop.apps.Fop;
import org.apache.fop.apps.FopFactory;
import org.apache.fop.configuration.Configurable;
import org.apache.fop.configuration.Configuration;
import org.apache.fop.configuration.ConfigurationException;
import org.apache.fop.util.DefaultErrorListener;

/**
 * BitmapProducer implementation that uses the PS or PDF renderer and an external converter
 * to create bitmaps.
 * <p>
 * Here's what the configuration element looks like for the class:
 * <p>
 * <pre>
 * <producer classname="<concrete class>">
 *   <converter>mypdf2bmp {0} {1} {2}</converter>
 *   <delete-temp-files>false</delete-temp-files>
 * </producer>
 * </pre>
 * <p>
 * You will notice the three parameters in curly braces (java.util.MessageFormat style) to the
 * converter call:
 * <ul>
 *   <li>0: the input file
 *   <li>1: the output bitmap file
 *   <li>2: the requested resolution in dpi for the generated bitmap
 * </ul>
 * <p>
 * The "delete-temp-files" element is optional and defaults to true.
 */
public abstract class AbstractPSPDFBitmapProducer extends AbstractBitmapProducer
            implements Configurable {

    // configure fopFactory as desired
    private final FopFactory fopFactory;

    private String converter;
    private boolean deleteTempFiles;
    /** the bitmap producer's target format */
    protected String targetFormat;

    public AbstractPSPDFBitmapProducer(URI baseUri) {
        super(baseUri);
        fopFactory = FopFactory.newInstance(baseUri);
    }

    /** {@inheritDoc} */
    public void configure(Configuration cfg) throws ConfigurationException {
        this.converter = cfg.getChild("converter").getValue();
        this.deleteTempFiles = cfg.getChild("delete-temp-files").getValueAsBoolean(true);
        if (cfg.getChild("target-format", false) != null) {
            this.targetFormat = cfg.getChild("target-format").getValue();
        }
    }

    /**
     * Calls an external converter to convert the file generated by FOP to a bitmap.
     * @param inFile the generated output file to be converted
     * @param outFile the target filename for the bitmap
     * @param context context information (required bitmap resolution etc.)
     * @throws IOException in case the conversion fails
     */
    public void convert(File inFile, File outFile, ProducerContext context) throws IOException {
        outFile.delete();

        //Build command-line
        String cmd = MessageFormat.format(converter,
                new Object[] {inFile.toString(), outFile.toString(),
                    Integer.toString(context.getTargetResolution())});
        ConvertUtils.convert(cmd, null, null, log);

        if (!outFile.exists()) {
            throw new IOException("The target file has not been generated");
        }
    }

    /**
     * @return the native extension generated by the output format, ex. "ps" or "pdf".
     */
    protected abstract String getTargetExtension();

    /**
     * @return the output format for the FOP renderer, i.e. a MIME type.
     */
    protected String getTargetFormat() {
        return this.targetFormat;
    }

    /** {@inheritDoc} */
    public BufferedImage produce(File src, int index, ProducerContext context) {
        try {
            FOUserAgent userAgent = fopFactory.newFOUserAgent();
            userAgent.setTargetResolution(context.getTargetResolution());

            File tempOut = new File(context.getTargetDir(),
                    src.getName() + "." + index + "." + getTargetExtension());
            File tempPNG = new File(context.getTargetDir(),
                    src.getName() + "." + index + "." + getTargetExtension() + ".png");
            try {
                OutputStream out = new FileOutputStream(tempOut);
                out = new BufferedOutputStream(out);
                try {
                    Fop fop = fopFactory.newFop(getTargetFormat(), userAgent, out);
                    SAXResult res = new SAXResult(fop.getDefaultHandler());

                    Transformer transformer = getTransformer(context);
                    transformer.setErrorListener(new DefaultErrorListener(log));
                    transformer.transform(new StreamSource(src), res);
                } finally {
                    IOUtils.closeQuietly(out);
                }

                convert(tempOut, tempPNG, context);
                BufferedImage img = BitmapComparator.getImage(tempPNG);
                return img;
            } finally {
                if (deleteTempFiles) {
                    if (!tempOut.delete()) {
                        log.warn("Can't delete " + tempOut);
                        tempOut.deleteOnExit();
                    }
                    if (!tempPNG.delete()) {
                        log.warn("Can't delete " + tempPNG);
                        tempPNG.deleteOnExit();
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e);
            return null;
        }
    }

}
