/*
 * USB quirk handling
 *
 * Copyright (c) 2012 Red Hat, Inc.
 *
 * Red Hat Authors:
 * Hans de Goede <hdegoede@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#ifndef HW_USB_QUIRKS_H
#define HW_USB_QUIRKS_H

/* 1 on 1 copy of linux/drivers/usb/serial/ftdi_sio_ids.h */
#include "quirks-ftdi-ids.h"
/* 1 on 1 copy of linux/drivers/usb/serial/pl2303.h */
#include "quirks-pl2303-ids.h"

struct usb_device_id {
    uint16_t vendor_id;
    uint16_t product_id;
    uint8_t interface_class;
    uint8_t interface_subclass;
    uint8_t interface_protocol;
    uint8_t interface_protocol_used:1,
            terminating_entry:1,
            reserved:6;
};

#define USB_DEVICE(vendor, product) \
    .vendor_id = vendor, .product_id = product, .interface_protocol_used = 0,

#define USB_DEVICE_AND_INTERFACE_INFO(vend, prod, iclass, isubclass, iproto) \
    .vendor_id = vend, .product_id = prod, .interface_class = iclass, \
    .interface_subclass = isubclass, .interface_protocol = iproto, \
    .interface_protocol_used = 1

static const struct usb_device_id usbredir_raw_serial_ids[] = {
    /*
     * Silicon Laboratories CP210x USB to RS232 serial adapter ids
     * copied from linux/drivers/usb/serial/cp210x.c
     *
     * Copyright (C) 2005 Craig Shelley (craig@microtron.org.uk)
     */
    { USB_DEVICE(0x045B, 0x0053) }, /* Renesas RX610 RX-Stick */
    { USB_DEVICE(0x0471, 0x066A) }, /* AKTAKOM ACE-1001 cable */
    { USB_DEVICE(0x0489, 0xE000) }, /* Pirelli Broadband S.p.A, DP-L10 SIP/GSM Mobile */
    { USB_DEVICE(0x0489, 0xE003) }, /* Pirelli Broadband S.p.A, DP-L10 SIP/GSM Mobile */
    { USB_DEVICE(0x0745, 0x1000) }, /* CipherLab USB CCD Barcode Scanner 1000 */
    { USB_DEVICE(0x08e6, 0x5501) }, /* Gemalto Prox-PU/CU contactless smartcard reader */
    { USB_DEVICE(0x08FD, 0x000A) }, /* Digianswer A/S , ZigBee/802.15.4 MAC Device */
    { USB_DEVICE(0x0BED, 0x1100) }, /* MEI (TM) Cashflow-SC Bill/Voucher Acceptor */
    { USB_DEVICE(0x0BED, 0x1101) }, /* MEI series 2000 Combo Acceptor */
    { USB_DEVICE(0x0FCF, 0x1003) }, /* Dynastream ANT development board */
    { USB_DEVICE(0x0FCF, 0x1004) }, /* Dynastream ANT2USB */
    { USB_DEVICE(0x0FCF, 0x1006) }, /* Dynastream ANT development board */
    { USB_DEVICE(0x10A6, 0xAA26) }, /* Knock-off DCU-11 cable */
    { USB_DEVICE(0x10AB, 0x10C5) }, /* Siemens MC60 Cable */
    { USB_DEVICE(0x10B5, 0xAC70) }, /* Nokia CA-42 USB */
    { USB_DEVICE(0x10C4, 0x0F91) }, /* Vstabi */
    { USB_DEVICE(0x10C4, 0x1101) }, /* Arkham Technology DS101 Bus Monitor */
    { USB_DEVICE(0x10C4, 0x1601) }, /* Arkham Technology DS101 Adapter */
    { USB_DEVICE(0x10C4, 0x800A) }, /* SPORTident BSM7-D-USB main station */
    { USB_DEVICE(0x10C4, 0x803B) }, /* Pololu USB-serial converter */
    { USB_DEVICE(0x10C4, 0x8044) }, /* Cygnal Debug Adapter */
    { USB_DEVICE(0x10C4, 0x804E) }, /* Software Bisque Paramount ME build-in converter */
    { USB_DEVICE(0x10C4, 0x8053) }, /* Enfora EDG1228 */
    { USB_DEVICE(0x10C4, 0x8054) }, /* Enfora GSM2228 */
    { USB_DEVICE(0x10C4, 0x8066) }, /* Argussoft In-System Programmer */
    { USB_DEVICE(0x10C4, 0x806F) }, /* IMS USB to RS422 Converter Cable */
    { USB_DEVICE(0x10C4, 0x807A) }, /* Crumb128 board */
    { USB_DEVICE(0x10C4, 0x80C4) }, /* Cygnal Integrated Products, Inc., Optris infrared thermometer */
    { USB_DEVICE(0x10C4, 0x80CA) }, /* Degree Controls Inc */
    { USB_DEVICE(0x10C4, 0x80DD) }, /* Tracient RFID */
    { USB_DEVICE(0x10C4, 0x80F6) }, /* Suunto sports instrument */
    { USB_DEVICE(0x10C4, 0x8115) }, /* Arygon NFC/Mifare Reader */
    { USB_DEVICE(0x10C4, 0x813D) }, /* Burnside Telecom Deskmobile */
    { USB_DEVICE(0x10C4, 0x813F) }, /* Tams Master Easy Control */
    { USB_DEVICE(0x10C4, 0x814A) }, /* West Mountain Radio RIGblaster P&P */
    { USB_DEVICE(0x10C4, 0x814B) }, /* West Mountain Radio RIGtalk */
    { USB_DEVICE(0x10C4, 0x8156) }, /* B&G H3000 link cable */
    { USB_DEVICE(0x10C4, 0x815E) }, /* Helicomm IP-Link 1220-DVM */
    { USB_DEVICE(0x10C4, 0x815F) }, /* Timewave HamLinkUSB */
    { USB_DEVICE(0x10C4, 0x818B) }, /* AVIT Research USB to TTL */
    { USB_DEVICE(0x10C4, 0x819F) }, /* MJS USB Toslink Switcher */
    { USB_DEVICE(0x10C4, 0x81A6) }, /* ThinkOptics WavIt */
    { USB_DEVICE(0x10C4, 0x81A9) }, /* Multiplex RC Interface */
    { USB_DEVICE(0x10C4, 0x81AC) }, /* MSD Dash Hawk */
    { USB_DEVICE(0x10C4, 0x81AD) }, /* INSYS USB Modem */
    { USB_DEVICE(0x10C4, 0x81C8) }, /* Lipowsky Industrie Elektronik GmbH, Baby-JTAG */
    { USB_DEVICE(0x10C4, 0x81E2) }, /* Lipowsky Industrie Elektronik GmbH, Baby-LIN */
    { USB_DEVICE(0x10C4, 0x81E7) }, /* Aerocomm Radio */
    { USB_DEVICE(0x10C4, 0x81E8) }, /* Zephyr Bioharness */
    { USB_DEVICE(0x10C4, 0x81F2) }, /* C1007 HF band RFID controller */
    { USB_DEVICE(0x10C4, 0x8218) }, /* Lipowsky Industrie Elektronik GmbH, HARP-1 */
    { USB_DEVICE(0x10C4, 0x822B) }, /* Modem EDGE(GSM) Comander 2 */
    { USB_DEVICE(0x10C4, 0x826B) }, /* Cygnal Integrated Products, Inc., Fasttrax GPS demonstration module */
    { USB_DEVICE(0x10C4, 0x8293) }, /* Telegesis ETRX2USB */
    { USB_DEVICE(0x10C4, 0x82F9) }, /* Procyon AVS */
    { USB_DEVICE(0x10C4, 0x8341) }, /* Siemens MC35PU GPRS Modem */
    { USB_DEVICE(0x10C4, 0x8382) }, /* Cygnal Integrated Products, Inc. */
    { USB_DEVICE(0x10C4, 0x83A8) }, /* Amber Wireless AMB2560 */
    { USB_DEVICE(0x10C4, 0x83D8) }, /* DekTec DTA Plus VHF/UHF Booster/Attenuator */
    { USB_DEVICE(0x10C4, 0x8411) }, /* Kyocera GPS Module */
    { USB_DEVICE(0x10C4, 0x8418) }, /* IRZ Automation Teleport SG-10 GSM/GPRS Modem */
    { USB_DEVICE(0x10C4, 0x846E) }, /* BEI USB Sensor Interface (VCP) */
    { USB_DEVICE(0x10C4, 0x8477) }, /* Balluff RFID */
    { USB_DEVICE(0x10C4, 0x85EA) }, /* AC-Services IBUS-IF */
    { USB_DEVICE(0x10C4, 0x85EB) }, /* AC-Services CIS-IBUS */
    { USB_DEVICE(0x10C4, 0x8664) }, /* AC-Services CAN-IF */
    { USB_DEVICE(0x10C4, 0x8665) }, /* AC-Services OBD-IF */
    { USB_DEVICE(0x10C4, 0xEA60) }, /* Silicon Labs factory default */
    { USB_DEVICE(0x10C4, 0xEA61) }, /* Silicon Labs factory default */
    { USB_DEVICE(0x10C4, 0xEA70) }, /* Silicon Labs factory default */
    { USB_DEVICE(0x10C4, 0xEA80) }, /* Silicon Labs factory default */
    { USB_DEVICE(0x10C4, 0xEA71) }, /* Infinity GPS-MIC-1 Radio Monophone */
    { USB_DEVICE(0x10C4, 0xF001) }, /* Elan Digital Systems USBscope50 */
    { USB_DEVICE(0x10C4, 0xF002) }, /* Elan Digital Systems USBwave12 */
    { USB_DEVICE(0x10C4, 0xF003) }, /* Elan Digital Systems USBpulse100 */
    { USB_DEVICE(0x10C4, 0xF004) }, /* Elan Digital Systems USBcount50 */
    { USB_DEVICE(0x10C5, 0xEA61) }, /* Silicon Labs MobiData GPRS USB Modem */
    { USB_DEVICE(0x10CE, 0xEA6A) }, /* Silicon Labs MobiData GPRS USB Modem 100EU */
    { USB_DEVICE(0x13AD, 0x9999) }, /* Baltech card reader */
    { USB_DEVICE(0x1555, 0x0004) }, /* Owen AC4 USB-RS485 Converter */
    { USB_DEVICE(0x166A, 0x0201) }, /* Clipsal 5500PACA C-Bus Pascal Automation Controller */
    { USB_DEVICE(0x166A, 0x0301) }, /* Clipsal 5800PC C-Bus Wireless PC Interface */
    { USB_DEVICE(0x166A, 0x0303) }, /* Clipsal 5500PCU C-Bus USB interface */
    { USB_DEVICE(0x166A, 0x0304) }, /* Clipsal 5000CT2 C-Bus Black and White Touchscreen */
    { USB_DEVICE(0x166A, 0x0305) }, /* Clipsal C-5000CT2 C-Bus Spectrum Colour Touchscreen */
    { USB_DEVICE(0x166A, 0x0401) }, /* Clipsal L51xx C-Bus Architectural Dimmer */
    { USB_DEVICE(0x166A, 0x0101) }, /* Clipsal 5560884 C-Bus Multi-room Audio Matrix Switcher */
    { USB_DEVICE(0x16D6, 0x0001) }, /* Jablotron serial interface */
    { USB_DEVICE(0x16DC, 0x0010) }, /* W-IE-NE-R Plein & Baus GmbH PL512 Power Supply */
    { USB_DEVICE(0x16DC, 0x0011) }, /* W-IE-NE-R Plein & Baus GmbH RCM Remote Control for MARATON Power Supply */
    { USB_DEVICE(0x16DC, 0x0012) }, /* W-IE-NE-R Plein & Baus GmbH MPOD Multi Channel Power Supply */
    { USB_DEVICE(0x16DC, 0x0015) }, /* W-IE-NE-R Plein & Baus GmbH CML Control, Monitoring and Data Logger */
    { USB_DEVICE(0x17A8, 0x0001) }, /* Kamstrup Optical Eye/3-wire */
    { USB_DEVICE(0x17A8, 0x0005) }, /* Kamstrup M-Bus Master MultiPort 250D */
    { USB_DEVICE(0x17F4, 0xAAAA) }, /* Wavesense Jazz blood glucose meter */
    { USB_DEVICE(0x1843, 0x0200) }, /* Vaisala USB Instrument Cable */
    { USB_DEVICE(0x18EF, 0xE00F) }, /* ELV USB-I2C-Interface */
    { USB_DEVICE(0x1BE3, 0x07A6) }, /* WAGO 750-923 USB Service Cable */
    { USB_DEVICE(0x1E29, 0x0102) }, /* Festo CPX-USB */
    { USB_DEVICE(0x1E29, 0x0501) }, /* Festo CMSP */
    { USB_DEVICE(0x3195, 0xF190) }, /* Link Instruments MSO-19 */
    { USB_DEVICE(0x3195, 0xF280) }, /* Link Instruments MSO-28 */
    { USB_DEVICE(0x3195, 0xF281) }, /* Link Instruments MSO-28 */
    { USB_DEVICE(0x413C, 0x9500) }, /* DW700 GPS USB interface */

    /*
     * Prolific pl2303 USB to RS232 serial adapter ids
     * copied from linux/drivers/usb/serial/pl2303.c
     *
     * Copyright (C) 2001-2007 Greg Kroah-Hartman (greg@kroah.com)
     * Copyright (C) 2003 IBM Corp.
     */
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_RSAQ2) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_DCU11) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_RSAQ3) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_PHAROS) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_ALDIGA) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_MMX) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_GPRS) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_HCR331) },
    { USB_DEVICE(PL2303_VENDOR_ID, PL2303_PRODUCT_ID_MOTOROLA) },
    { USB_DEVICE(IODATA_VENDOR_ID, IODATA_PRODUCT_ID) },
    { USB_DEVICE(IODATA_VENDOR_ID, IODATA_PRODUCT_ID_RSAQ5) },
    { USB_DEVICE(ATEN_VENDOR_ID, ATEN_PRODUCT_ID) },
    { USB_DEVICE(ATEN_VENDOR_ID2, ATEN_PRODUCT_ID) },
    { USB_DEVICE(ELCOM_VENDOR_ID, ELCOM_PRODUCT_ID) },
    { USB_DEVICE(ELCOM_VENDOR_ID, ELCOM_PRODUCT_ID_UCSGT) },
    { USB_DEVICE(ITEGNO_VENDOR_ID, ITEGNO_PRODUCT_ID) },
    { USB_DEVICE(ITEGNO_VENDOR_ID, ITEGNO_PRODUCT_ID_2080) },
    { USB_DEVICE(MA620_VENDOR_ID, MA620_PRODUCT_ID) },
    { USB_DEVICE(RATOC_VENDOR_ID, RATOC_PRODUCT_ID) },
    { USB_DEVICE(TRIPP_VENDOR_ID, TRIPP_PRODUCT_ID) },
    { USB_DEVICE(RADIOSHACK_VENDOR_ID, RADIOSHACK_PRODUCT_ID) },
    { USB_DEVICE(DCU10_VENDOR_ID, DCU10_PRODUCT_ID) },
    { USB_DEVICE(SITECOM_VENDOR_ID, SITECOM_PRODUCT_ID) },
    { USB_DEVICE(ALCATEL_VENDOR_ID, ALCATEL_PRODUCT_ID) },
    { USB_DEVICE(SAMSUNG_VENDOR_ID, SAMSUNG_PRODUCT_ID) },
    { USB_DEVICE(SIEMENS_VENDOR_ID, SIEMENS_PRODUCT_ID_SX1) },
    { USB_DEVICE(SIEMENS_VENDOR_ID, SIEMENS_PRODUCT_ID_X65) },
    { USB_DEVICE(SIEMENS_VENDOR_ID, SIEMENS_PRODUCT_ID_X75) },
    { USB_DEVICE(SIEMENS_VENDOR_ID, SIEMENS_PRODUCT_ID_EF81) },
    { USB_DEVICE(BENQ_VENDOR_ID, BENQ_PRODUCT_ID_S81) }, /* Benq/Siemens S81 */
    { USB_DEVICE(SYNTECH_VENDOR_ID, SYNTECH_PRODUCT_ID) },
    { USB_DEVICE(NOKIA_CA42_VENDOR_ID, NOKIA_CA42_PRODUCT_ID) },
    { USB_DEVICE(CA_42_CA42_VENDOR_ID, CA_42_CA42_PRODUCT_ID) },
    { USB_DEVICE(SAGEM_VENDOR_ID, SAGEM_PRODUCT_ID) },
    { USB_DEVICE(LEADTEK_VENDOR_ID, LEADTEK_9531_PRODUCT_ID) },
    { USB_DEVICE(SPEEDDRAGON_VENDOR_ID, SPEEDDRAGON_PRODUCT_ID) },
    { USB_DEVICE(DATAPILOT_U2_VENDOR_ID, DATAPILOT_U2_PRODUCT_ID) },
    { USB_DEVICE(BELKIN_VENDOR_ID, BELKIN_PRODUCT_ID) },
    { USB_DEVICE(ALCOR_VENDOR_ID, ALCOR_PRODUCT_ID) },
    { USB_DEVICE(WS002IN_VENDOR_ID, WS002IN_PRODUCT_ID) },
    { USB_DEVICE(COREGA_VENDOR_ID, COREGA_PRODUCT_ID) },
    { USB_DEVICE(YCCABLE_VENDOR_ID, YCCABLE_PRODUCT_ID) },
    { USB_DEVICE(SUPERIAL_VENDOR_ID, SUPERIAL_PRODUCT_ID) },
    { USB_DEVICE(HP_VENDOR_ID, HP_LD220_PRODUCT_ID) },
    { USB_DEVICE(CRESSI_VENDOR_ID, CRESSI_EDY_PRODUCT_ID) },
    { USB_DEVICE(ZEAGLE_VENDOR_ID, ZEAGLE_N2ITION3_PRODUCT_ID) },
    { USB_DEVICE(SONY_VENDOR_ID, SONY_QN3USB_PRODUCT_ID) },
    { USB_DEVICE(SANWA_VENDOR_ID, SANWA_PRODUCT_ID) },
    { USB_DEVICE(ADLINK_VENDOR_ID, ADLINK_ND6530_PRODUCT_ID) },
    { USB_DEVICE(SMART_VENDOR_ID, SMART_PRODUCT_ID) },

    { .terminating_entry = 1 } /* Terminating Entry */
};

static const struct usb_device_id usbredir_ftdi_serial_ids[] = {
    /*
     * FTDI USB to RS232 serial adapter ids
     * copied from linux/drivers/usb/serial/ftdi_sio.c
     *
     * Copyright (C) 2009 - 2010
     *    Johan Hovold (jhovold@gmail.com)
     * Copyright (C) 1999 - 2001
     *    Greg Kroah-Hartman (greg@kroah.com)
     *    Bill Ryder (bryder@sgi.com)
     * Copyright (C) 2002
     *    Kuba Ober (kuba@mareimbrium.org)
     */
    { USB_DEVICE(FTDI_VID, FTDI_ZEITCONTROL_TAGTRACE_MIFARE_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CTI_MINI_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CTI_NANO_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_AMC232_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CANUSB_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CANDAPTER_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_NXTCAM_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCS_DEVICE_0_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCS_DEVICE_1_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCS_DEVICE_2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCS_DEVICE_3_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCS_DEVICE_4_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCS_DEVICE_5_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCS_DEVICE_6_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCS_DEVICE_7_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_USINT_CAT_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_USINT_WKEY_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_USINT_RS232_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ACTZWAVE_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IRTRANS_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IPLUS_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IPLUS2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_DMX4ALL) },
    { USB_DEVICE(FTDI_VID, FTDI_SIO_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_8U232AM_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_8U232AM_ALT_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_232RL_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_8U2232C_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_4232H_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_232H_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_FTX_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MICRO_CHAMELEON_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_RELAIS_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_OPENDCC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_OPENDCC_SNIFFER_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_OPENDCC_THROTTLE_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_OPENDCC_GATEWAY_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_OPENDCC_GBM_PID) },
    { USB_DEVICE(INTERBIOMETRICS_VID, INTERBIOMETRICS_IOBOARD_PID) },
    { USB_DEVICE(INTERBIOMETRICS_VID, INTERBIOMETRICS_MINI_IOBOARD_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SPROG_II) },
    { USB_DEVICE(FTDI_VID, FTDI_LENZ_LIUSB_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_XF_632_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_XF_634_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_XF_547_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_XF_633_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_XF_631_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_XF_635_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_XF_640_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_XF_642_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_DSS20_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_URBAN_0_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_URBAN_1_PID) },
    { USB_DEVICE(FTDI_NF_RIC_VID, FTDI_NF_RIC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_VNHCPCUSB_D_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MTXORB_0_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MTXORB_1_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MTXORB_2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MTXORB_3_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MTXORB_4_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MTXORB_5_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MTXORB_6_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_R2000KU_TRUE_RNG) },
    { USB_DEVICE(FTDI_VID, FTDI_VARDAAN_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0100_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0101_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0102_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0103_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0104_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0105_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0106_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0107_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0108_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0109_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_010A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_010B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_010C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_010D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_010E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_010F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0110_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0111_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0112_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0113_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0114_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0115_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0116_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0117_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0118_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0119_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_011A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_011B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_011C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_011D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_011E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_011F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0120_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0121_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0122_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0123_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0124_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0125_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0126_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0127_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0128_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0129_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_012A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_012B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_012C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_012D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_012E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_012F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0130_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0131_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0132_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0133_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0134_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0135_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0136_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0137_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0138_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0139_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_013A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_013B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_013C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_013D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_013E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_013F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0140_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0141_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0142_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0143_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0144_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0145_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0146_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0147_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0148_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0149_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_014A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_014B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_014C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_014D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_014E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_014F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0150_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0151_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0152_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0153_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0154_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0155_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0156_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0157_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0158_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0159_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_015A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_015B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_015C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_015D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_015E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_015F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0160_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0161_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0162_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0163_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0164_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0165_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0166_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0167_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0168_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0169_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_016A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_016B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_016C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_016D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_016E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_016F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0170_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0171_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0172_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0173_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0174_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0175_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0176_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0177_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0178_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0179_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_017A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_017B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_017C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_017D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_017E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_017F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0180_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0181_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0182_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0183_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0184_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0185_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0186_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0187_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0188_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0189_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_018A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_018B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_018C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_018D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_018E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_018F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0190_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0191_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0192_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0193_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0194_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0195_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0196_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0197_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0198_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_0199_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_019A_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_019B_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_019C_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_019D_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_019E_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_019F_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A0_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A1_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A2_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A3_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A4_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A5_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A6_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A7_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A8_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01A9_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01AA_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01AB_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01AC_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01AD_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01AE_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01AF_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B0_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B1_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B2_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B3_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B4_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B5_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B6_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B7_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B8_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01B9_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01BA_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01BB_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01BC_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01BD_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01BE_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01BF_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C0_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C1_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C2_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C3_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C4_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C5_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C6_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C7_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C8_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01C9_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01CA_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01CB_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01CC_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01CD_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01CE_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01CF_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D0_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D1_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D2_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D3_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D4_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D5_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D6_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D7_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D8_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01D9_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01DA_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01DB_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01DC_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01DD_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01DE_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01DF_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E0_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E1_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E2_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E3_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E4_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E5_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E6_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E7_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E8_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01E9_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01EA_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01EB_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01EC_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01ED_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01EE_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01EF_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F0_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F1_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F2_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F3_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F4_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F5_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F6_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F7_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F8_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01F9_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01FA_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01FB_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01FC_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01FD_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01FE_PID) },
    { USB_DEVICE(MTXORB_VID, MTXORB_FTDI_RANGE_01FF_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_PERLE_ULTRAPORT_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_PIEGROUP_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TNC_X_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_USBX_707_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2101_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2102_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2103_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2104_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2106_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2201_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2201_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2202_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2202_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2203_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2203_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2401_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2401_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2401_3_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2401_4_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2402_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2402_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2402_3_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2402_4_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2403_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2403_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2403_3_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2403_4_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2801_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2801_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2801_3_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2801_4_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2801_5_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2801_6_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2801_7_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2801_8_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2802_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2802_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2802_3_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2802_4_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2802_5_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2802_6_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2802_7_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2802_8_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803_3_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803_4_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803_5_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803_6_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803_7_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803_8_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803R_1_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803R_2_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803R_3_PID) },
    { USB_DEVICE(SEALEVEL_VID, SEALEVEL_2803R_4_PID) },
    { USB_DEVICE(IDTECH_VID, IDTECH_IDT1221U_PID) },
    { USB_DEVICE(OCT_VID, OCT_US101_PID) },
    { USB_DEVICE(OCT_VID, OCT_DK201_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_HE_TIRA1_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_USB_UIRT_PID) },
    { USB_DEVICE(FTDI_VID, PROTEGO_SPECIAL_1) },
    { USB_DEVICE(FTDI_VID, PROTEGO_R2X0) },
    { USB_DEVICE(FTDI_VID, PROTEGO_SPECIAL_3) },
    { USB_DEVICE(FTDI_VID, PROTEGO_SPECIAL_4) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E808_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E809_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E80A_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E80B_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E80C_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E80D_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E80E_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E80F_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E888_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E889_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E88A_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E88B_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E88C_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E88D_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E88E_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GUDEADS_E88F_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UO100_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UM100_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UR100_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_ALC8500_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_PYRAMID_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_FHZ1000PC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IBS_US485_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IBS_PICPRO_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IBS_PCMCIA_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IBS_PK1_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IBS_RS232MON_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IBS_APP70_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IBS_PEDO_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_IBS_PROD_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TAVIR_STK500_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TIAO_UMPA_PID) },
    /*
     * ELV devices:
     */
    { USB_DEVICE(FTDI_VID, FTDI_ELV_USR_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_MSM1_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_KL100_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_WS550_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_EC3000_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_WS888_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_TWS550_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_FEM_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_CLI7000_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_PPS7330_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_TFM100_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UDF77_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UIO88_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UAD8_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UDA7_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_USI2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_T1100_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_PCD200_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_ULA200_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_CSI8_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_EM1000DL_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_PCK100_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_RFP500_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_FS20SIG_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UTP8_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_WS300PC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_WS444PC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_FHZ1300PC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_EM1010PC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_WS500_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_HS485_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_UMS100_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_TFD128_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_FM3RX_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELV_WS777_PID) },
    { USB_DEVICE(FTDI_VID, LINX_SDMUSBQSS_PID) },
    { USB_DEVICE(FTDI_VID, LINX_MASTERDEVEL2_PID) },
    { USB_DEVICE(FTDI_VID, LINX_FUTURE_0_PID) },
    { USB_DEVICE(FTDI_VID, LINX_FUTURE_1_PID) },
    { USB_DEVICE(FTDI_VID, LINX_FUTURE_2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CCSICDU20_0_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CCSICDU40_1_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CCSMACHX_2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CCSLOAD_N_GO_3_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CCSICDU64_4_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CCSPRIME8_5_PID) },
    { USB_DEVICE(FTDI_VID, INSIDE_ACCESSO) },
    { USB_DEVICE(INTREPID_VID, INTREPID_VALUECAN_PID) },
    { USB_DEVICE(INTREPID_VID, INTREPID_NEOVI_PID) },
    { USB_DEVICE(FALCOM_VID, FALCOM_TWIST_PID) },
    { USB_DEVICE(FALCOM_VID, FALCOM_SAMBA_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SUUNTO_SPORTS_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_OCEANIC_PID) },
    { USB_DEVICE(TTI_VID, TTI_QL355P_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_RM_CANVIEW_PID) },
    { USB_DEVICE(ACTON_VID, ACTON_SPECTRAPRO_PID) },
    { USB_DEVICE(CONTEC_VID, CONTEC_COM1USBH_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USOTL4_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USTL4_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USO9ML2_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USOPTL4_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USPTL4_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USO9ML2DR_2_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USO9ML2DR_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USOPTL4DR2_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_USOPTL4DR_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_485USB9F_2W_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_485USB9F_4W_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_232USB9M_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_485USBTB_2W_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_485USBTB_4W_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_TTL5USB9M_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_TTL3USB9M_PID) },
    { USB_DEVICE(BANDB_VID, BANDB_ZZ_PROG1_USB_PID) },
    { USB_DEVICE(FTDI_VID, EVER_ECO_PRO_CDS) },
    { USB_DEVICE(FTDI_VID, FTDI_4N_GALAXY_DE_1_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_4N_GALAXY_DE_2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_4N_GALAXY_DE_3_PID) },
    { USB_DEVICE(FTDI_VID, XSENS_CONVERTER_0_PID) },
    { USB_DEVICE(FTDI_VID, XSENS_CONVERTER_1_PID) },
    { USB_DEVICE(FTDI_VID, XSENS_CONVERTER_2_PID) },
    { USB_DEVICE(FTDI_VID, XSENS_CONVERTER_3_PID) },
    { USB_DEVICE(FTDI_VID, XSENS_CONVERTER_4_PID) },
    { USB_DEVICE(FTDI_VID, XSENS_CONVERTER_5_PID) },
    { USB_DEVICE(FTDI_VID, XSENS_CONVERTER_6_PID) },
    { USB_DEVICE(FTDI_VID, XSENS_CONVERTER_7_PID) },
    { USB_DEVICE(MOBILITY_VID, MOBILITY_USB_SERIAL_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ACTIVE_ROBOTS_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MHAM_KW_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MHAM_YS_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MHAM_Y6_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MHAM_Y8_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MHAM_IC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MHAM_DB9_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MHAM_RS232_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MHAM_Y9_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TERATRONIK_VCP_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TERATRONIK_D2XX_PID) },
    { USB_DEVICE(EVOLUTION_VID, EVOLUTION_ER1_PID) },
    { USB_DEVICE(EVOLUTION_VID, EVO_HYBRID_PID) },
    { USB_DEVICE(EVOLUTION_VID, EVO_RCM4_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ARTEMIS_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ATIK_ATK16_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ATIK_ATK16C_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ATIK_ATK16HR_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ATIK_ATK16HRC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ATIK_ATK16IC_PID) },
    { USB_DEVICE(KOBIL_VID, KOBIL_CONV_B1_PID) },
    { USB_DEVICE(KOBIL_VID, KOBIL_CONV_KAAN_PID) },
    { USB_DEVICE(POSIFLEX_VID, POSIFLEX_PP7000_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TTUSB_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ECLO_COM_1WIRE_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_WESTREX_MODEL_777_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_WESTREX_MODEL_8900F_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_PCDJ_DAC2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_RRCIRKITS_LOCOBUFFER_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ASK_RDR400_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_NZR_SEM_USB_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_1_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_OPC_U_UC_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP2C1_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP2C2_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP2D_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP2VT_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP2VR_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP4KVT_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP4KVR_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP2KVT_PID) },
    { USB_DEVICE(ICOM_VID, ICOM_ID_RP2KVR_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ACG_HFDUAL_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_YEI_SERVOCENTER31_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_THORLABS_PID) },
    { USB_DEVICE(TESTO_VID, TESTO_USB_INTERFACE_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_GAMMA_SCOUT_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TACTRIX_OPENPORT_13M_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TACTRIX_OPENPORT_13S_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TACTRIX_OPENPORT_13U_PID) },
    { USB_DEVICE(ELEKTOR_VID, ELEKTOR_FT323R_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_NDI_HUC_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_NDI_SPECTRA_SCU_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_NDI_FUTURE_2_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_NDI_FUTURE_3_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_NDI_AURORA_SCU_PID) },
    { USB_DEVICE(TELLDUS_VID, TELLDUS_TELLSTICK_PID) },
    { USB_DEVICE(RTSYSTEMS_VID, RTSYSTEMS_SERIAL_VX7_PID) },
    { USB_DEVICE(RTSYSTEMS_VID, RTSYSTEMS_CT29B_PID) },
    { USB_DEVICE(RTSYSTEMS_VID, RTSYSTEMS_RTS01_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_MAXSTREAM_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_PHI_FISCO_PID) },
    { USB_DEVICE(TML_VID, TML_USB_SERIAL_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_ELSTER_UNICOM_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_PROPOX_JTAGCABLEII_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_PROPOX_ISPCABLEIII_PID) },
    { USB_DEVICE(OLIMEX_VID, OLIMEX_ARM_USB_OCD_PID) },
    { USB_DEVICE(OLIMEX_VID, OLIMEX_ARM_USB_OCD_H_PID) },
    { USB_DEVICE(FIC_VID, FIC_NEO1973_DEBUG_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_OOCDLINK_PID) },
    { USB_DEVICE(FTDI_VID, LMI_LM3S_DEVEL_BOARD_PID) },
    { USB_DEVICE(FTDI_VID, LMI_LM3S_EVAL_BOARD_PID) },
    { USB_DEVICE(FTDI_VID, LMI_LM3S_ICDI_BOARD_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_TURTELIZER_PID) },
    { USB_DEVICE(RATOC_VENDOR_ID, RATOC_PRODUCT_ID_USB60F) },
    { USB_DEVICE(FTDI_VID, FTDI_REU_TINY_PID) },

    /* Papouch devices based on FTDI chip */
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_SB485_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_AP485_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_SB422_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_SB485_2_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_AP485_2_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_SB422_2_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_SB485S_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_SB485C_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_LEC_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_SB232_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_TMU_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_IRAMP_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_DRAK5_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_QUIDO8x8_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_QUIDO4x4_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_QUIDO2x2_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_QUIDO10x1_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_QUIDO30x3_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_QUIDO60x3_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_QUIDO2x16_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_QUIDO3x32_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_DRAK6_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_UPSUSB_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_MU_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_SIMUKEY_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_AD4USB_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_GMUX_PID) },
    { USB_DEVICE(PAPOUCH_VID, PAPOUCH_GMSR_PID) },

    { USB_DEVICE(FTDI_VID, FTDI_DOMINTELL_DGQG_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_DOMINTELL_DUSB_PID) },
    { USB_DEVICE(ALTI2_VID, ALTI2_N3_PID) },
    { USB_DEVICE(FTDI_VID, DIEBOLD_BCS_SE923_PID) },
    { USB_DEVICE(ATMEL_VID, STK541_PID) },
    { USB_DEVICE(DE_VID, STB_PID) },
    { USB_DEVICE(DE_VID, WHT_PID) },
    { USB_DEVICE(ADI_VID, ADI_GNICE_PID) },
    { USB_DEVICE(ADI_VID, ADI_GNICEPLUS_PID) },
    { USB_DEVICE_AND_INTERFACE_INFO(MICROCHIP_VID, MICROCHIP_USB_BOARD_PID,
                                    0xff, 0xff, 0x00) },
    { USB_DEVICE(JETI_VID, JETI_SPC1201_PID) },
    { USB_DEVICE(MARVELL_VID, MARVELL_SHEEVAPLUG_PID) },
    { USB_DEVICE(LARSENBRUSGAARD_VID, LB_ALTITRACK_PID) },
    { USB_DEVICE(GN_OTOMETRICS_VID, AURICAL_USB_PID) },
    { USB_DEVICE(FTDI_VID, PI_C865_PID) },
    { USB_DEVICE(FTDI_VID, PI_C857_PID) },
    { USB_DEVICE(PI_VID, PI_C866_PID) },
    { USB_DEVICE(PI_VID, PI_C663_PID) },
    { USB_DEVICE(PI_VID, PI_C725_PID) },
    { USB_DEVICE(PI_VID, PI_E517_PID) },
    { USB_DEVICE(PI_VID, PI_C863_PID) },
    { USB_DEVICE(PI_VID, PI_E861_PID) },
    { USB_DEVICE(PI_VID, PI_C867_PID) },
    { USB_DEVICE(PI_VID, PI_E609_PID) },
    { USB_DEVICE(PI_VID, PI_E709_PID) },
    { USB_DEVICE(PI_VID, PI_100F_PID) },
    { USB_DEVICE(PI_VID, PI_1011_PID) },
    { USB_DEVICE(PI_VID, PI_1012_PID) },
    { USB_DEVICE(PI_VID, PI_1013_PID) },
    { USB_DEVICE(PI_VID, PI_1014_PID) },
    { USB_DEVICE(PI_VID, PI_1015_PID) },
    { USB_DEVICE(PI_VID, PI_1016_PID) },
    { USB_DEVICE(KONDO_VID, KONDO_USB_SERIAL_PID) },
    { USB_DEVICE(BAYER_VID, BAYER_CONTOUR_CABLE_PID) },
    { USB_DEVICE(FTDI_VID, MARVELL_OPENRD_PID) },
    { USB_DEVICE(FTDI_VID, TI_XDS100V2_PID) },
    { USB_DEVICE(FTDI_VID, HAMEG_HO820_PID) },
    { USB_DEVICE(FTDI_VID, HAMEG_HO720_PID) },
    { USB_DEVICE(FTDI_VID, HAMEG_HO730_PID) },
    { USB_DEVICE(FTDI_VID, HAMEG_HO870_PID) },
    { USB_DEVICE(FTDI_VID, MJSG_GENERIC_PID) },
    { USB_DEVICE(FTDI_VID, MJSG_SR_RADIO_PID) },
    { USB_DEVICE(FTDI_VID, MJSG_HD_RADIO_PID) },
    { USB_DEVICE(FTDI_VID, MJSG_XM_RADIO_PID) },
    { USB_DEVICE(FTDI_VID, XVERVE_SIGNALYZER_ST_PID) },
    { USB_DEVICE(FTDI_VID, XVERVE_SIGNALYZER_SLITE_PID) },
    { USB_DEVICE(FTDI_VID, XVERVE_SIGNALYZER_SH2_PID) },
    { USB_DEVICE(FTDI_VID, XVERVE_SIGNALYZER_SH4_PID) },
    { USB_DEVICE(FTDI_VID, SEGWAY_RMP200_PID) },
    { USB_DEVICE(FTDI_VID, ACCESIO_COM4SM_PID) },
    { USB_DEVICE(IONICS_VID, IONICS_PLUGCOMPUTER_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CHAMSYS_24_MASTER_WING_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CHAMSYS_PC_WING_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CHAMSYS_USB_DMX_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CHAMSYS_MIDI_TIMECODE_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CHAMSYS_MINI_WING_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CHAMSYS_MAXI_WING_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CHAMSYS_MEDIA_WING_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CHAMSYS_WING_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCIENCESCOPE_LOGBOOKML_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCIENCESCOPE_LS_LOGBOOK_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_SCIENCESCOPE_HS_LOGBOOK_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_CINTERION_MC55I_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_DOTEC_PID) },
    { USB_DEVICE(ST_VID, ST_STMCLT1030_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_RF_R106) },
    { USB_DEVICE(FTDI_VID, FTDI_DISTORTEC_JTAG_LOCK_PICK_PID) },
    { USB_DEVICE(FTDI_VID, FTDI_LUMEL_PD12_PID) },

    { .terminating_entry = 1 } /* Terminating Entry */
};

#undef USB_DEVICE
#undef USB_DEVICE_AND_INTERFACE_INFO

#endif
