/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.button;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.util.Set;

import javax.swing.AbstractButton;
import javax.swing.Icon;
import javax.swing.border.Border;

import org.jvnet.substance.SubstanceButtonBorder;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.utils.*;

/**
 * Button shaper that returns rectangular buttons with slightly rounded corners
 * (ala Windows XP). This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class ClassicButtonShaper extends BaseButtonShaper implements
		RectangularButtonShaper {
	// /**
	// * The default width of button
	// */
	// public static final int DEFAULT_WIDTH = 70;
	//
	// /**
	// * The default height of button
	// */
	// public static final int DEFAULT_HEIGHT = 20;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.button.SubstanceButtonShaper#getDisplayName()
	 */
	public String getDisplayName() {
		return "Classic";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.button.SubstanceButtonShaper#getButtonOutline(javax.swing.AbstractButton,
	 *      java.awt.Insets, int, int)
	 */
	public GeneralPath getButtonOutline(AbstractButton button, Insets insets,
			int width, int height) {
		Set<SubstanceConstants.Side> straightSides = SubstanceCoreUtilities
				.getSides(button, SubstanceLookAndFeel.BUTTON_SIDE_PROPERTY);

		float radius = this.getCornerRadius(button, insets);
		return getBaseOutline(width, height, radius, straightSides, insets);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.button.SubstanceButtonShaper#getButtonBorder(javax.swing.AbstractButton)
	 */
	public Border getButtonBorder(final AbstractButton button) {
		return new SubstanceButtonBorder(ClassicButtonShaper.class) {
			public Insets getBorderInsets(Component c) {
				int extraPadding = SubstanceSizeUtils
						.getExtraPadding(SubstanceSizeUtils
								.getComponentFontSize(c));
				Set<SubstanceConstants.Side> openSides = SubstanceCoreUtilities
						.getSides(button,
								SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY);
				// if (button instanceof SubstanceSpinnerButton) {
				// // special case - bring the icons closer together instead of
				// // having them centered in the spinner buttons
				// boolean isTopButton = openSides
				// .contains(SubstanceConstants.Side.BOTTOM);
				// int deltaTop = isTopButton ? 3 : 0;
				// int deltaBottom = isTopButton ? 0 : 3;
				// return new Insets(extraPadding + deltaTop,
				// extraPadding + 2, extraPadding + deltaBottom,
				// extraPadding + 2);
				// } else {
				int left = extraPadding
						+ (openSides.contains(SubstanceConstants.Side.LEFT) ? 1
								: 2);
				int right = extraPadding
						+ (openSides.contains(SubstanceConstants.Side.RIGHT) ? 1
								: 2);
				int top = extraPadding
						+ (openSides.contains(SubstanceConstants.Side.TOP) ? 1
								: 2);
				int bottom = extraPadding
						+ (openSides.contains(SubstanceConstants.Side.BOTTOM) ? 1
								: 2);
				return new Insets(top, left, bottom, right);
				// }
			}
		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.button.SubstanceButtonShaper#getPreferredSize(javax.swing.AbstractButton,
	 *      java.awt.Dimension)
	 */
	public Dimension getPreferredSize(AbstractButton button,
			Dimension uiPreferredSize) {
		Dimension result;
		boolean toTweakWidth = false;
		boolean toTweakHeight = false;

		Icon icon = button.getIcon();
		boolean hasIcon = SubstanceCoreUtilities.hasIcon(button);
		boolean hasText = SubstanceCoreUtilities.hasText(button);
		Insets margin = button.getMargin();

		result = uiPreferredSize;

		boolean hasNoMinSizeProperty = SubstanceCoreUtilities
				.hasNoMinSizeProperty(button);
		if ((!hasNoMinSizeProperty) && hasText) {
			int baseWidth = uiPreferredSize.width;
			baseWidth = Math.max(baseWidth, SubstanceSizeUtils
					.getMinButtonWidth(SubstanceSizeUtils
							.getComponentFontSize(button)));
			// if (baseWidth < DEFAULT_WIDTH) {
			// baseWidth = DEFAULT_WIDTH;
			// }
			result = new Dimension(baseWidth, uiPreferredSize.height);
			int baseHeight = result.height;
			baseHeight = Math.max(baseHeight, SubstanceSizeUtils
					.getMinButtonHeight(SubstanceSizeUtils
							.getComponentFontSize(button)));
			// if (baseHeight < DEFAULT_HEIGHT) {
			// baseHeight = DEFAULT_HEIGHT;
			// }
			result = new Dimension(result.width, baseHeight);
		} else {
			if (hasNoMinSizeProperty) {
				if (margin != null) {
					result = new Dimension(result.width + margin.left
							+ margin.right, result.height + margin.top
							+ margin.bottom);
				}
			}
		}

		int extraPadding = SubstanceSizeUtils
				.getExtraPadding(SubstanceSizeUtils
						.getComponentFontSize(button));
		int iconPaddingWidth = 6 + 2 * extraPadding;
		int iconPaddingHeight = 6 + 2 * extraPadding;
		if (margin != null) {
			iconPaddingWidth = Math.max(iconPaddingWidth, margin.left
					+ margin.right);
			iconPaddingHeight = Math.max(iconPaddingHeight, margin.top
					+ margin.bottom);
		}
		if (hasIcon) {
			// check the icon height
			int iconHeight = icon.getIconHeight();
			if (iconHeight > (result.getHeight() - iconPaddingHeight)) {
				result = new Dimension(result.width, iconHeight);
				toTweakHeight = true;
			}
			int iconWidth = icon.getIconWidth();
			if (iconWidth > (result.getWidth() - iconPaddingWidth)) {
				result = new Dimension(iconWidth, result.height);
				toTweakWidth = true;
			}
		}

		if (SubstanceCoreUtilities.isScrollBarButton(button)) {
			toTweakWidth = false;
			toTweakHeight = false;
		}

		if (toTweakWidth) {
			result = new Dimension(result.width + iconPaddingWidth,
					result.height);
		}
		if (toTweakHeight) {
			result = new Dimension(result.width, result.height
					+ iconPaddingHeight);
		}

		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.button.SubstanceButtonShaper#isProportionate()
	 */
	public boolean isProportionate() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.button.RectangularButtonShaper#getCornerRadius(javax.swing.AbstractButton,
	 *      java.awt.Insets)
	 */
	public float getCornerRadius(AbstractButton button, Insets insets) {
		float radius = SubstanceSizeUtils
				.getClassicButtonCornerRadius(SubstanceSizeUtils
						.getComponentFontSize(button));
		if (SubstanceCoreUtilities.isToolBarButton(button)) {
			radius = SubstanceCoreUtilities.getToolbarButtonCornerRadius(
					button, insets);
		} else {
//			if (insets != null) {
//				int max = Math.max(Math.max(insets.left, insets.right), Math
//						.max(insets.top, insets.bottom));
//				if (max >= 2)
//					radius = 1;
//			}
		}
		return radius;
	}
}
