(*
   Knuth-Morris-Pratt algorithm following Sedgewick's book "Algorithms".
   The Caml code was proved correct in the Coq Proof Assistant.
   From a mail to the Caml list by Jean-Christophe FILLIATRE. *)

let initnext p =
  let m = String.length p in
  let next = Array.create m 0 in
  let i = ref 1 and j = ref 0 in
  while !i < m - 1 do
    if p.[!i] = p.[!j] then begin
      incr i; incr j; next.(!i) <- !j
    end else
      if !j = 0 then begin incr i; next.(!i) <- 0 end else j := next.(!j)
  done;
  next;;

let kmp p =
  let next = initnext p
  and m = String.length p in
  fun a ->
    let n = String.length a
    and i = ref 0
    and j = ref 0 in
    while !j < m && !i < n do
      if a.[!i] = p.[!j] then begin
        incr i; incr j
      end else
        if !j = 0 then incr i else j := next.(!j)
    done;
    if !j >= m then !i - m else raise Not_found;;

let usage () =
  print_string
   "Usage: strstr <pat> <str>\m\
    returns the index of the first occurrence of <pat> in string <str>\n";
  exit 2;;

let ststr pat str =
  try
    let i = kmp pat str in
    print_string "Match found at character ";
    print_int i;
    print_newline ();
    exit 0
  with Not_found ->
    print_string "Pattern ";
    print_string pat;
    print_string " does not match string ";
    print_string str;
    print_newline();;

let main () =
  let args = Sys.argv in
  if Array.length args < 2 then usage () else
   let pat = Sys.argv.(1)
   and str = Sys.argv.(2) in
  ststr pat str;;

if !Sys.interactive then () else main ();;
