/*
** Generate a VRML model.
**
** The functions in this module are used by 3Dstats and http-analyze
** to create a VRML model for a WWW access statistics.
**
** Copyright  1996-1999 by Stefan Stapelberg, <stefan@rent-a-guru.de>
**
** $Id: vrml.c,v 2.4 1999/10/30 09:36:08 stefan Stab $
**
*/

#include <stdlib.h>
#include <stdarg.h>		/* for `va_list' (in defs.h) */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>
#include <fcntl.h>
#include <assert.h>

#if defined(unix)
# include <unistd.h>

#else
# if defined(WIN32)
#  include <winsock.h>		/* for the u_int, etc. types */
# endif

# if defined(WIN32) || defined(NETWARE)
#  include <direct.h>		/* for other windows/watcom stuff */
#  include <io.h>		/* for the F_OK, etc. symbolic constants */
# endif
#endif

#include "config.h"
#include "defs.h"

#if defined(VRML)
static void prBackground(FILE *const);
static void prNavInfo(FILE *const, u_short const);

/*
** Bar height and scale factor.
** Don't touch unless you know what you're doing.
*/
#define DIFF_X		0.6
#define DIFF_Z		1.25

#define BAR_HT		1.0
#define SCALEF		10.0
#define AVSCALEF	3.4

/* VRML stuff */
#define BEGIN_GROUP	(void) fprintf(ofp, "Group {\n children [\n")
#define END_GROUP	(void) fprintf(ofp, " ]\n}\n")

/*
** Instancing table
*/
#define MAX_STRINGS	20

typedef struct {
	int defined;
	char *name;
	char *str[MAX_STRINGS];
} INSTANCE;

#define MAT_DEF		0	/* order is important! */
#define MAT_ORANGE	1
#define MAT_VIOLET	2
#define MAT_GREEN	3
#define MAT_YELLOW	4
#define MAT_BLUE	5
#define MAT_BROWN	6
#define MAT_RED		7	/* order is important! */
#define MAT_LINE	8
#define FONTSTYLE1	9
#define FONTSTYLE2	10
#define AUDIO_CLIP	11

static INSTANCE instList[] = {
	{ 0, "Mat",		/* default material */
	  { "Material {",
	    "  ambientIntensity 0",
	    "  diffuseColor 0 0.85 0.85",
	    "  emissiveColor 0 0 0",
	    "  specularColor 0.65 0.65 0.65",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "Mat",		/* orange */
	  { "Material {",
	    "  ambientIntensity 0.25",
	    "  diffuseColor 0.9044 0.2503 0",
	    "  specularColor 0.09559 0.09559 0.09559",
	    "  emissiveColor 0 0 0",
	    "  shininess 0.07812",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "Mat",		/* purple */
	  { "Material {",
	    "  ambientIntensity 0.25",
	    "  diffuseColor 0.6218 0.02092 0.8802",
	    "  specularColor 0.08773 0.06944 0.09559",
	    "  emissiveColor 0 0 0",
	    "  shininess 0.07812",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "Mat",		/* green */
	  { "Material {",
	    "  ambientIntensity 0.25",
	    "  diffuseColor 0.05315 0.818 0.08802",
	    "  specularColor 0.00622 0.09559 0.01029",
	    "  emissiveColor 0 0 0",
	    "  shininess 0.07812",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "Mat",		/* yellow */
	  { "Material {",
	    "  ambientIntensity 0.25",
	    "  diffuseColor 0.9044 0.7747 0.08307",
	    "  specularColor 0.09559 0.09559 0.09559",
	    "  emissiveColor 0 0 0",
	    "  shininess 0.07812",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "Mat",		/* blue */
	  { "Material {",
	    "  ambientIntensity 0.25",
	    "  diffuseColor 0 0.2 0.95",
	    "  specularColor 0 0.06977 0.09559",
	    "  emissiveColor 0 0 0",
	    "  shininess 0.07812",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "Mat",		/* brown */
	  { "Material {",
	    "  ambientIntensity 0.25",
	    "  diffuseColor 0.748 0.2995 0",
	    "  specularColor 0.08364 0.08364 0.08364",
	    "  emissiveColor 0 0 0",
	    "  shininess 0.07812",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "Mat",		/* red */
	  { "Material {",
	    "  ambientIntensity 0.25",
	    "  diffuseColor 0.748 0 0.04009",
	    "  specularColor 0.0897 0.0897 0.0897",
	    "  emissiveColor 0 0 0",
	    "  shininess 0.07812",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "Mat",		/* orange for IndexLineSet */
	  { "Material {",
	    "  ambientIntensity 0",
	    "  diffuseColor 0.521 0.174 0",
	    "  specularColor 0.08 0.08 0.08",
	    "  emissiveColor 0.787 0.262 0",
	    "  shininess 0.7",
	    "  transparency 0",
	    " }", NULL } },

	{ 0, "FS1",		/* font style 1 */
	  { "FontStyle {",
	    " size 1",
	    " family \"SANS\"",
	    " style \"ITALIC\"",
	    /*" justify \"END\"",*/
	    " spacing 1",
	    " }", NULL } },

	{ 0, "FS2",		/* font style 2 */
	  { "FontStyle {",
	    " size 1",
	    " family \"SANS\"",
	    " style \"ITALIC\"",
	    " spacing 1",
	    " }", NULL } },

	{ 0, "MotionClip",	/* audio clip for shelf motion */
	  { "\tAudioClip {",
		"\t url \"../3DshelfMotion.wav\"",
		"\t loop FALSE",
		"\t startTime 0",
		"\t}", NULL } }
};

/*
** Material definitions for weekdays.
** Note that in Germany a new week starts on Monday, not Sunday.
** If you change this, you also have to change the algorithm in mkdtab().
*/
static int daytab[] = { MAT_ORANGE, MAT_GREEN, MAT_VIOLET,
		MAT_YELLOW, MAT_BLUE, MAT_BROWN, MAT_RED };


/*
** Define whatever instance
*/
static void prInstance(FILE *const ofp, int const idx) {
	int j;

	assert((idx >= 0) && ((size_t)idx < TABSIZE(instList)));

	if (instList[idx].defined) {
		(void) fprintf(ofp, "USE %s_%d ", instList[idx].name, idx);
		return;
	}
	(void) fprintf(ofp, "DEF %s_%d ", instList[idx].name, idx);
	for (j=0; j < MAX_STRINGS; j++) {
		if (instList[idx].str[j] != NULL)
			(void) fprintf(ofp, "%s\n", instList[idx].str[j]);
	}
	instList[idx].defined = 1;
	return;
}

/*
** Clear all instances
*/
static int shapeDef[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

static void clrInstance(void) {
	size_t idx;

	for (idx=0; idx < TABSIZE(instList); idx++)
		instList[idx].defined = 0;
	for (idx=0; idx < TABSIZE(shapeDef); idx++)
		shapeDef[idx] = 0;
	return;
}

/*
** Draw a bar
*/
static void drawBar(FILE *const ofp,
	float const xpos, float const ypos, float const zpos,
	float const xscale, float const yscale, int const matidx) {
	u_char sep = (u_char)(!shapeDef[matidx] ? '\n' : ' ');

	(void) fprintf(ofp, "Transform {%c", sep);
	if (!shapeDef[matidx]) {
		(void) fprintf(ofp, 
			"\n children DEF Bar_%d Shape {\n" \
			"  appearance Appearance { material ", matidx);
		prInstance(ofp, matidx);
		(void) fprintf(ofp, "  }\n"
			"  geometry Box { size 0.5 1 0.7 }\n }\n");
		shapeDef[matidx] = 1;
	} else
		(void) fprintf(ofp, " children USE Bar_%d ", matidx);

	(void) fprintf(ofp, " translation %.4g %.4g %.4g%c scale %.4g %.4g 1%c}\n",
		xpos, ypos, -zpos, sep, xscale, yscale, sep);
	return;
}

/*
** Create a model of the year with inlines to monthly stats.
*/
int prVRMLYear(FILE *const ofp,
		char *const srvname, char *const prolog, LOGTIME *const tp) {
	char lbuf[LBUFSIZE];
	char tbuf[MAX_FNAMELEN];
	int exist, cmon = 0;
	FILE *pfp;

	float qrpos[3][4] = {{-150., 0., 1., 0.785}, {0., 150., 1., 3.14}, {150., 0., -1., 0.785}};
	float yrpos[4][4] = {
	  {0., 0., 1., 0.}, {300., 300., -1., 1.57}, {0., 600., 1., 3.14}, {-300., 300., 1., 1.57}};

	errno = 0;		/* Output the VRML prolog */
	if ((pfp = fopen(prolog, "r")) == NULL) {
		prmsg(2, GETMSG(430, "Can't open prolog file `%s' (%s)\n"),
			prolog, strerror(errno));
		return 0;
	}
	if (fgets(lbuf, sizeof lbuf, pfp) == NULL ||
	    strcmp(lbuf, "#VRML V2.0 utf8\n") != 0) {
		prmsg(2, GETMSG(431, "Not a valid VRML 2.0 prolog file: `%s'\n"), prolog);
		(void) fclose(pfp);
		return 0;
	}

	rewind(pfp);
	while (fgets(lbuf, sizeof lbuf, pfp) != NULL)
		(void) fputs(lbuf, ofp);
	(void) fclose(pfp);

	(void) fprintf(ofp, "WorldInfo {\n info [ \"%s %s\",\n",
		GETMSG(432, "3D Access Statistics for"), srvname);

	(void) fprintf(ofp, "\t\"%s %s\" ]\n}\n\n",
		GETMSG(433, "Created by"), creator);

	(void) fputs(
		"DEF NavInfo NavigationInfo {\n"
		"  avatarSize [ 0.1, 0.9, 0.4 ]\n"
		"  speed 2.4\n"
		"  headlight FALSE\n"
		"  type [ \"ANY\", \"WALK\" ]\n"
		"}\n", ofp);

	(void) fprintf(ofp, "Transform {\n children [\n");
	(void) fputs(
		"  Transform {\n"
		"   children World { }\n"
		"   translation	33.09 1.34 9.73\n"
		"   rotation 0 0 1  0\n"
		"   scale 0.994515 0.994515 0.994515\n"
		"  }\n", ofp);
	(void) fputs(
		"  Transform {\n"
		"   children  Inline {\n"
		"     url \"../3Dlogo.wrl.gz\"\n"
		"     bboxCenter 0.49918 1.65182 0.554783\n"
		"     bboxSize 15.158 7.50998 15.158\n"
		"   }\n"
		"   translation 673.654 3.1399 1005.19\n"
		"   rotation 0 1 0  3.14159\n"
		"   scale 0.9 0.9 0.9\n"
		"  }\n", ofp);

	for (cmon=0; cmon < 12; cmon++) {
		int qtr = cmon/3;
		int idx = cmon%3;

		(void) snprintf(tbuf, sizeof tbuf,
			"www%hu/3Dstats%02d%02d.wrl.gz",
			tp->year, cmon+1, EPOCH(tp->year));

		exist = (access(tbuf, F_OK) == 0);

		if (qtr > 0 && idx == 0)
			(void) fprintf(ofp, "Transform {\n children [\n");

		(void) fprintf(ofp, "Transform {\n children [\n"
			"   DEF Building_%d Building {\n"
			"     openTime 0\n"
			"     closeTime 0\n"
			"   }\n", cmon+1);

		(void) fprintf(ofp, "   Transform {\n   children [\n"
			"    DEF Sensor%d ProximitySensor { size 44 20 44 }\n"
			"    Transform {\n     children Sound {\n      source ", cmon+1);
		prInstance(ofp, AUDIO_CLIP);
		(void) fprintf(ofp, "\n"
			"      priority 0.5\n"
			"      location 0 0.5 0\n"
			"      direction 0 0 1\n"
			"      minFront 5\n      maxFront 85\n"
			"      minBack 3\n      maxBack 65\n"
			"     }\n    }\n");

		(void) fprintf(ofp,
			"    LOD {\n"
			"     range 12\n"
			"     level [\n"
			"      Transform {\n"
			"       children [\n");

		if (exist) {
			(void) fprintf(ofp,
				"\t  Transform {\n"
				"\t   children Inline {\n"
				"\t\turl \"3Dstats%02d%02d.wrl.gz\"\n",
				cmon+1, EPOCH(tp->year));

#if (VRML_OPT&OLD_LAYOUT) != OLD_LAYOUT
# define WHICH_BBOXCENTER	"11.492 6.862 -4.433"
# define WHICH_BBOXSIZE		"22.983 14.276 9.267"
# define WHICH_TRANSLATION	"-1.1492 0.05 0.4433"
# define WHICH_MODEL	"0"
#else
# define WHICH_BBOXCENTER	"22.85 12.57 -14.33"
# define WHICH_BBOXSIZE		"45.7 25.7 28.85"
# define WHICH_TRANSLATION	"-2.285 0.05 1.433"
# define WHICH_MODEL	"1"
#endif
			(void) fprintf(ofp,
				"\t\tbboxCenter " WHICH_BBOXCENTER "\n"
				"\t\tbboxSize   " WHICH_BBOXSIZE   "\n"
				"\t   }\n"
				"\t   translation " WHICH_TRANSLATION "\n");

			(void) fprintf(ofp,
				"\t   scale 0.1 0.1 0.1\n"
				"\t   rotation 0 -1 0 0.3925\n\t  }\n");
		} else
			(void) fprintf(ofp, "\t  Transform {\n"
				"\t   translation -1.1492 0.05 0.4433\n"
				"\t   scale 0.1 0.1 0.1\n"
				"\t   rotation 0 -1 0 0.3925\n"
				"\t   children EmptyModel { whichModel " WHICH_MODEL " }\n"
				"\t  }\n");

		(void) fprintf(ofp, "       ]\n      }\n");
		(void) fprintf(ofp, "      Transform {\n"
				"       translation -1.1492 0.05 0.4433\n"
				"       scale 0.1 0.1 0.1\n"
				"       rotation 0 -1 0 0.3925\n"
				"       children EmptyModel { whichModel " WHICH_MODEL " }\n"
				"      }\n");

		(void) fprintf(ofp, "     ]\n    }\n   ]\n   }\n   ]\n"
			"   translation %g 0 %g\n   rotation 0 %g 0  %g\n  }\n",
			qrpos[idx][0], qrpos[idx][1], qrpos[idx][2], qrpos[idx][3]);
		if (qtr > 0 && idx == 2)
			(void) fprintf(ofp, " ]\n translation %g 0 %g\n rotation 0 %g 0  %g\n}\n",
				yrpos[qtr][0], yrpos[qtr][1], yrpos[qtr][2], yrpos[qtr][3]);
	}
	(void) fputs( " ]\n}\n"
		"Transform {\n"
		" children Background {\n"
		"  groundAngle [ 1.5, 1.57 ]\n"
		"  groundColor [ 0 0 0, 0.21 0.21 0.38, 0.3 0.35 0.45 ]\n"
		"  skyAngle [ 1.18, 1.42, 1.55, 1.57 ]\n"
		"  skyColor [ 0 0 0, 0 0 0.38, 0.35 0 0.28, 0.68 0.2 0.1, 0.75 0.35 0 ]\n"
		" }\n}\n", ofp);

	(void) fprintf(ofp, "DEF VP0 Viewpoint {\n"
		"  position -68 0.6 -290\n  orientation 0 1 0  3.29\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", GETMSG(437, "Approach"));
	(void) fprintf(ofp, "DEF VP1 Viewpoint {\n"
		"  position -120 0.6 30\n  orientation 0 1 0  0.785\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[0]);
	(void) fprintf(ofp, "DEF VP2 Viewpoint {\n"
		"  position 0 0.6 107.5\n  orientation 0 1 0  3.14\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[1]);
	(void) fprintf(ofp, "DEF VP3 Viewpoint {\n"
		"  position 120 0.6 30\n  orientation 0 -1 0  0.785\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[2]);
	(void) fprintf(ofp, "DEF VP4 Viewpoint {\n"
		"  position 270 0.6 180\n  orientation 0 -1 0  0.785\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[3]);
	(void) fprintf(ofp, "DEF VP5 Viewpoint {\n"
		"  position 192.5 0.6 300\n  orientation 0 1 0  1.57\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[4]);
	(void) fprintf(ofp, "DEF VP6 Viewpoint {\n"
		"  position 270 0.6 420\n  orientation 0 -1 0  2.355\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[5]);
	(void) fprintf(ofp, "DEF VP7 Viewpoint {\n"
		"  position 120 0.6 570\n  orientation 0 1 0  3.928\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[6]);
	(void) fprintf(ofp, "DEF VP8 Viewpoint {\n"
		"  position 0 0.6 492.5\n  orientation 0 0 1  0\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[7]);
	(void) fprintf(ofp, "DEF VP9 Viewpoint {\n"
		"  position -120 0.6 570\n  orientation 0 1 0  2.355\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[8]);
	(void) fprintf(ofp, "DEF VP10 Viewpoint {\n"
		"  position -270 0.6 180\n  orientation 0 1 0  0.785\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[9]);
	(void) fprintf(ofp, "DEF VP11 Viewpoint {\n"
		"  position -192.5 0.6 300\n  orientation 0 -1 0  1.57\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[10]);
	(void) fprintf(ofp, "DEF VP12 Viewpoint {\n"
		"  position -270 0.6 420\n  orientation 0 1 0  2.355\n"
		"  fieldOfView 0.7854\n  description \"%s\"\n}\n", monnam[11]);

	(void) fputs(
		"DEF Light2 PointLight {\n"
		"  intensity 0.9\n"
		"  ambientIntensity 0.2\n"
		"  location 61.380 552.674 316.160\n"
		"}\n", ofp);

	(void) fputs(
		"DEF Light3 SpotLight {\n"
		"  intensity 0.8\n"
		"  ambientIntensity 0.1\n"
		"  location 671.737 7.83031 1000.4\n"
		"  direction 0.219617 -0.667357 0.711622\n"
		"  beamWidth 0.785398\n"
		"  cutOffAngle 1.5708\n"
		"  }\n", ofp);

	for (cmon=1; cmon <= 12; cmon++)
		(void) fprintf(ofp, 
			"ROUTE Sensor%d.enterTime TO MotionClip_11.set_startTime\n"
			"ROUTE Sensor%d.exitTime TO MotionClip_11.set_startTime\n"
			"ROUTE Sensor%d.isActive TO Building_%d.set_spotLight\n"
			"ROUTE Sensor%d.enterTime TO Building_%d.openTime\n"
			"ROUTE Sensor%d.exitTime TO Building_%d.closeTime\n",
			cmon, cmon, cmon, cmon, cmon, cmon, cmon, cmon);
	clrInstance();
	return 1;
}

/*
** Draw curve
*/
static void prAvgCurve(FILE * const ofp,
	float *const pts, float const stepw, int const numpts, float avg_ht) {
	int idx;

	(void) fprintf(ofp, "# prAvgCurve\n");
	(void) fprintf(ofp, "  children [\n"
		"   Shape {\n" \
		"    appearance Appearance { material ");
	prInstance(ofp, MAT_DEF);
	(void) fprintf(ofp, "   }\n"
		"    geometry IndexedLineSet {\n");

	(void) fprintf(ofp, "     coord Coordinate {\n      point [ ");

	for (idx=0; idx < numpts; idx++) {	/* draw line verticies */
		(void) fprintf(ofp, "%.4g %.4g 0", (float)idx*stepw, pts[idx]);
		if (idx < numpts-1) {
			(void) fprintf(ofp, ", ");
			if ((idx%7) == 6)
				(void) fprintf(ofp, "\n\t");
		}
	}
	(void) fprintf(ofp, " ]\n     }\n");

	(void) fprintf(ofp, "     color Color { color [ "
		"0 0.5 1, 0 1 1, 0 1 0.5, 0 1 0, 0.5 1 0, 1 0.5 0, 1 0 0 ] }\n");

	(void) fprintf(ofp, "     coordIndex [ ");
	for (idx=0; idx < numpts; idx++) {
		(void) fprintf(ofp, "%d", idx);
		if (idx < numpts-1) {
			(void) fprintf(ofp, ", ");
			if ((idx%10) == 9)
				(void) fprintf(ofp, "\n\t");
		}
	}
	(void) fprintf(ofp, " ]\n");
	(void) fprintf(ofp, "     colorIndex [ ");
	for (idx=0; idx < numpts; idx++) {
		if (pts[idx] > (AVSCALEF/7.0)*6.0)
			(void) fputc('6', ofp);
		else if (pts[idx] > (AVSCALEF/7.0)*5.0)
			(void) fputc('5', ofp);
		else if (pts[idx] > (AVSCALEF/7.0)*4.0)
			(void) fputc('4', ofp);
		else if (pts[idx] > (AVSCALEF/7.0)*3.0)
			(void) fputc('3', ofp);
		else if (pts[idx] > (AVSCALEF/7.0)*2.0)
			(void) fputc('2', ofp);
		else if (pts[idx] > AVSCALEF/7.0)
			(void) fputc('1', ofp);
		else	(void) fputc('0', ofp);
		if (idx < numpts-1) {
			(void) fprintf(ofp, ", ");
			if ((idx%10) == 9)
				(void) fprintf(ofp, "\n\t");
		}
	}
	(void) fprintf(ofp, "     ]\n    }\n   }\n");

	(void) fprintf(ofp, "  Shape {\n   appearance Appearance { material ");
	prInstance(ofp, MAT_LINE);
	(void) fprintf(ofp, "   }\n"
			    "   geometry IndexedLineSet {\n"
			    "    coord Coordinate {"
			    " point [ 0 %.4g 0, %.4g %.4g 0 ] }\n"
			    "    coordIndex [ 0, 1, -1 ]\n   }\n  }\n",
			avg_ht, (float)(numpts-1)*stepw, avg_ht);

	(void) fprintf(ofp, "  ]\n");
	return;
}

/*
** Draw a grid around the container
*/
static void prGrid(FILE * const ofp,
	float const width, float const height, int const x_tiles, int const y_tiles) {
	float z_dist = width/(float)x_tiles;
	float y_dist = height/(float)y_tiles;
	int idx;

	(void) fprintf(ofp, "# prGrid\n");
	(void) fprintf(ofp, "  children Shape {\n"
		"   appearance Appearance {\n"
		"    material Material {\n"
		"     diffuseColor 0 0 1\n"
		"     shininess 0.8\n"
		"     transparency 0\n    }\n"
		"    texture NULL\n"
		"    textureTransform NULL\n   }\n"
		"   geometry IndexedLineSet {\n");

	(void) fprintf(ofp, "    coord Coordinate {\n     point [\n");

	for (idx=0; idx <= x_tiles; idx++)	/* vertical line verticies */
		(void) fprintf(ofp, "%.4g %.4g 0, %.4g 0 0,%c",
			z_dist*(float)idx, height, z_dist*(float)idx, (idx%4)==3 ? '\n' : ' ');

	for (idx=0; idx <= y_tiles; idx++)	/* horizontal line verticies */
		(void) fprintf(ofp, "0 %.4g 0, %.4g %.4g 0,%c",
			y_dist*(float)idx, width, y_dist*(float)idx, (idx%4)==3 ? '\n' : ' ');

	(void) fprintf(ofp, " ]\n    }\n");
	(void) fprintf(ofp, "    color Color { color [ 0 0 1, 1 0 0, 0 1 0 ] }\n");

	(void) fprintf(ofp, "    coordIndex [\n");
	for (idx=0; idx <= (x_tiles+y_tiles+1)*2; idx += 2)
		(void) fprintf(ofp, "%d, %d, -1,%c",
			idx, idx+1, (idx%10)==8 ? '\n' : ' ');
	(void) fprintf(ofp, " ]\n");

	(void) fprintf(ofp, "    colorIndex [\n");
	for (idx=0; idx <= (x_tiles+y_tiles+1)*2; idx += 2) {
		if (idx == (x_tiles+11)*2)
			(void) fprintf(ofp, "2, 2, -1,");
		else if (idx == (x_tiles+6)*2)
			(void) fprintf(ofp, "1, 1, -1,");
		else
			(void) fprintf(ofp, "0, 0, -1,");
		(void) fprintf(ofp, "%c", (idx%10)==8 ? '\n' : ' ');
	}
	(void) fprintf(ofp, "    ]\n   }\n  }\n");
	return;
}

/*
** Print average hits
*/
static void prAvgLine(FILE *const ofp, float wid, float const ht, float depth) {
	wid -= 0.05;
	depth -= 0.05;

	(void) fprintf(ofp, "# prAvgLine\n");
	(void) fprintf(ofp, "Transform {\n"
			    " children Shape {\n"
			    "  appearance Appearance { material ");
	prInstance(ofp, MAT_LINE);
	(void) fprintf(ofp, "  }\n"
			    "  geometry IndexedLineSet {\n"
			    "   coord Coordinate {"
			    " point [ 0.05 %.4g 0, 0.05 %.4g %.4g, %.4g %.4g %.4g ] }\n"
			    "   coordIndex [ 0, 1, -1, 1, 2, -1 ]\n"
			    "  }\n }\n}\n", ht, ht, -depth, wid, ht, -depth);
	return;
}

/*
** Draw the vertical labels.
*/
static void VTextLabel(FILE * const ofp,
	int const matidx, char * const string,
	float const xpos, float const ypos, float const zpos) {

	(void) fprintf(ofp, "Transform {\n"
			    " children Shape {\n"
			    "  appearance Appearance { material ");

	prInstance(ofp, matidx);
	(void) fprintf(ofp, "  }\n"
			    "  geometry Text {\n"
			    "   string \"%s\"\n   fontStyle ", string);
	prInstance(ofp, FONTSTYLE1);
	(void) fprintf(ofp, "   length 0\n  }\n }\n");
	(void) fprintf(ofp, " translation %.4g %.4g %.4g\n", xpos, ypos, -zpos);
	(void) fprintf(ofp, " scale 0.5 0.5 0.5\n}\n");
	return;
}

/*
** Draw the horizontal labels.
*/
static void HTextLabel(FILE * const ofp,
	int const matidx, char * const string,
	float const xpos, float const ypos, float const zpos) {

	(void) fprintf(ofp, "Transform {\n"
			    " children Shape {\n"
			    "  appearance Appearance { material ");
	prInstance(ofp, matidx);
	(void) fprintf(ofp, "  }\n"
			    "  geometry Text {\n"
			    "   string \"%s\"\n   fontStyle ", string);
	prInstance(ofp, FONTSTYLE2);
	(void) fprintf(ofp, "   length 0\n  }\n }\n");
	(void) fprintf(ofp, " translation %.4g %.4g %.4g\n", xpos, ypos, -zpos);
	(void) fprintf(ofp, " scale 0.5 0.5 0.5\n" \
			    " rotation -1 0 0 0.785\n}\n");
	return;
}

/*
** Draw a container
*/
static void prBox(FILE * const ofp,
	int const nx, int const nz, int const start, int const stop) {
	float width = (float)nx * 0.6;
	float depth = (float)nz * 1.25;
	float wdist = width / 2.0;
	float ddist = depth / 2.0;
	float strlength = (width*2.0)-0.2;
	int num;

	/* draw floor */
	(void) fprintf(ofp, "Transform {\n"
			    " children Shape {\n"
			    "  appearance Appearance { material Material { diffuseColor 0 0 1 } }\n"
			    "  geometry Box { size %.4g 0.1 %.4g }\n }\n"
			    " translation %.4g -0.1 -%.4g\n}\n", width, depth, wdist, ddist);

	/* add numbers */
	(void) fprintf(ofp, "Transform {\n"
			    " children Shape {\n"
			    "  appearance Appearance {\n"
			    "   material ");
	prInstance(ofp, MAT_DEF);
	(void) fprintf(ofp, "  }\n"
			    "  geometry Text {\n"
			    "   string \"");
	for (num=start; num < stop; num++)	/* generate numbers as one string */
		(void) fprintf(ofp, "%2d ", num);
	(void) fprintf(ofp, "%2d\"\n", stop);
	(void) fprintf(ofp, "   fontStyle FontStyle {\n"
			    "    size 1\n"
			    "    family \"TYPEWRITER\"\n"
			    "    style \"BOLD\"\n"
			    "    justify \"MIDDLE\"\n"
			    "   }\n"
			    "   length %.4g\n"
			    "  }\n }\n"
			    " translation %.4g -0.2 0.2\n"
			    " scale 0.5 0.5 0.5\n}\n", strlength, wdist);

	/* draw grid */
	(void) fprintf(ofp, "Transform {\n");
	prGrid(ofp, depth, 12.0, nz, 12);
	(void) fprintf(ofp, " translation 0 0 %.4g\n"
			    " rotation 0 -1 0 1.57\n}\n", -depth);

	(void) fprintf(ofp, "Transform {\n");
	prGrid(ofp, width, 12.0, nx, 12);
	(void) fprintf(ofp, " translation 0 0 %.4g\n}\n", -depth);
	return;
}

/*
** Draw a header
*/
static void prHead(FILE * const ofp,
	int const oldstyle, char *const srvname, char *const period) {
	int idx;
	char *vr_logo[] = {
	" Anchor {\n" \
		"  children Transform {\n" \
		"   children [\n" \
		"    Transform {\n" \
		"      children Shape {\n" \
		"\tappearance Appearance {\n" \
		"\t  material Material {\n" \
		"\t    ambientIntensity 0.115\n" \
		"\t    diffuseColor 0 0.631 0.748\n" \
		"\t    specularColor 0.915 0.915 0.915\n" \
		"\t    emissiveColor 0 0 0\n" \
		"\t    shininess 0.642\n" \
		"\t    transparency 0\n\t  }\n\t}\n" \
		"\tgeometry Sphere { radius 27.1 }\n" \
		"      }\n" \
		"      translation 0.99 -0.3 -0.17\n" \
		"      rotation 1 0 0  3.14\n" \
		"      scale 0.0246 0.0246 0.00538\n" \
		"      scaleOrientation 0 0 1  0\n" \
		"    }\n",
	"    Transform {\n" \
		"      children Transform {\n" \
		"\tchildren Shape {\n" \
		"\t  appearance Appearance {\n" \
		"\t    material Material {\n" \
		"\t      ambientIntensity 0.0739\n" \
		"\t      diffuseColor 0.585 0.274 0.00686\n" \
		"\t      specularColor 0.429 0.199 0.00296\n" \
		"\t      emissiveColor 0 0 0\n" \
		"\t      shininess 0.0663\n" \
		"\t      transparency 0\n" \
		"\t    }\n" \
		"\t    texture NULL\n" \
		"\t    textureTransform NULL\n" \
		"\t  }\n" \
		"\t  geometry IndexedFaceSet {\n" \
		"\t    coord Coordinate {\n" \
		"\t      point   [ 18.3 -7.94 2.11, 18.6 -5.41 2.2, 16.5 -7.42 2.13,\n" \
		"\t\t        15.3 -5.69 2.19, 13.6 -6.63 2.15, 12.2 -8 2.1,\n" \
		"\t\t        13.5 -9.12 2.07, 11.1 -9.66 2.05, 12.4 -11.3 1.99,\n" \
		"\t\t        10.4 -11.5 1.98, 12.6 -13.3 1.92, 10.7 -14.7 1.87,\n" \
		"\t\t        12.9 -16.6 1.8, 14 -14.5 1.88, 16.2 -16.3 1.81,\n" \
		"\t\t        16 -14.3 1.88, 17.9 -15.4 1.85, 19.3 -14 1.9,\n" \
		"\t\t        20.4 -12.3 1.95, 18.7 -11.8 1.97, 21.1 -10.5 2.02,\n" \
		"\t\t        19.1 -9.63 2.05, 20.8 -7.27 2.13, 5.56 -5.69 2,\n" \
		"\t\t        0.701 -5.69 2, 3.25 -5.9 2, -4.32 -4.69 2,\n" \
		"\t\t        -4.6 -6.63 2, -4.32 -10.2 2, -14.4 -5.69 2,\n" \
		"\t\t        -19.3 -5.69 2, -16.8 -5.9 2, -47.6 -1.12 2,\n" \
		"\t\t        -49.7 2.29 2, -50.4 0.583 2, -43.7 -4.1 2,\n" \
		"\t\t        -43.9 -2.18 2, -44.8 3.11 2, -42 2.5 2,\n" \
		"\t\t        -46 -10.1 2, 5.1 -1.65 2, 7.23 -1.63 2,\n" \
		"\t\t        8.33 -5.69 2, -0.224 -7.39 2, 7.18 -7.82 2,\n" \
		"\t\t        2.58 -7.57 2, 4.63 -7.82 2, 0.0073 -13.1 2,\n" \
		"\t\t        2.22 -12.8 2, 1.16 -16.1 2, 3.25 -16.3 2,\n" \
		"\t\t        4.47 -14.2 2, 5.56 -16.1 2, 6.93 -13.9 2,\n" \
		"\t\t        -21.6 -14.3 2, -22.8 -16.5 2, -26.9 -14.5 2,\n" \
		"\t\t        -27.8 -16.6 2, -29.6 -14.2 2, -31 -15.8 2,\n" \
		"\t\t        -32.9 -13.5 2, -30.4 -11.7 2, -32.4 -10.3 2,\n" \
		"\t\t        -30.1 -7.49 2, -26.4 -6 2, -27.2 -7.87 2,\n" \
		"\t\t        -23.6 -6.64 2, -22.5 -8.18 2, -24.9 -8.75 2,\n" \
		"\t\t        -22.7 -10.3 2, -24.1 -11.5 2, -26.5 -10 2,\n" \
		"\t\t        -27 -11.7 2, -28.3 -10 2, -12 -16.4 2,\n" \
		"\t\t        -11.3 -14.3 2, -7.34 -16.4 2, -3.86 -14.9 2,\n" \
		"\t\t        -6.64 -13.8 2, -2.03 -12.6 2, -2.03 -9.7 2,\n" \
		"\t\t        -6.87 -4.48 2, -6.17 -8.52 2, -10.5 -10.6 2,\n" \
		"\t\t        -7.56 -10.7 2, -14.9 -1.65 2, -12.8 -1.63 2,\n" \
		"\t\t        -11.7 -5.69 2, -20.2 -7.39 2, -12.8 -7.82 2,\n" \
		"\t\t        -17.4 -7.57 2, -15.4 -7.82 2, -20 -13.1 2,\n" \
		"\t\t        -17.8 -12.8 2, -18.8 -16.1 2, -16.8 -16.3 2,\n" \
		"\t\t        -15.5 -14.2 2, -14.4 -16.1 2, -13.1 -13.9 2,\n" \
		"\t\t        45.4 -14.3 2, 44.2 -16.5 2, 40.1 -14.5 2,\n" \
		"\t\t        39.2 -16.6 2, 37.4 -14.2 2, 36 -15.8 2,\n" \
		"\t\t        34.1 -13.5 2, 36.6 -11.7 2, 34.6 -10.3 2,\n" \
		"\t\t        36.9 -7.49 2, 40.6 -6 2, 39.8 -7.87 2,\n" \
		"\t\t        43.4 -6.64 2, 44.5 -8.18 2, 42.1 -8.75 2,\n" \
		"\t\t        44.3 -10.3 2, 42.9 -11.5 2, 40.5 -10 2,\n" \
		"\t\t        40 -11.7 2, 38.7 -10 2, -56.4 -14.3 2,\n" \
		"\t\t        -60.1 -14.3 2, -52.5 -5.59 2, -53.2 2.29 2,\n" \
		"\t\t        -49.9 -2.18 2, -45.3 -0.906 2, -47.8 2.47 2,\n" \
		"\t\t        -39.9 0.409 2, -46 -12.4 2, -39.5 -3.25 2,\n" \
		"\t\t        -44.1 -13.9 2, -42.1 -9.63 2, -41.6 -14.1 2,\n" \
		"\t\t        -41.4 -11.3 2, -36 -14.1 2, -34.9 -11.1 2,\n" \
		"\t\t        24.3 -10.5 2, 21.8 -10.3 2, 25.5 -8.2 2,\n" \
		"\t\t        24.3 -3.95 2, 27.6 -3.74 2, 26.9 -5.65 2,\n" \
		"\t\t        33.4 -5.44 2, 27.8 -8.2 2, 33 -7.83 2,\n" \
		"\t\t        31 -7.99 2, 26.3 -10.6 2, 29.7 -9.69 2,\n" \
		"\t\t        25.7 -13 2, 28.5 -11.8 2, 26.5 -15.5 2,\n" \
		"\t\t        28.3 -16.2 2, 30.3 -14.3 2, 32.4 -16.3 2, 33.1 -14.2 2 ]\n" \
		"\t    }\n",
	"\t    color NULL\n" \
		"\t    coordIndex\t[ 39, 35, 38, -1, 38, 35, 37, -1, 37, 35, 36, -1, 34, 32, 33, -1,\n" \
		"\t\t  31, 29, 30, -1, 28, 26, 27, -1, 25, 23, 24, -1, 22, 0, 1, -1,\n" \
		"\t\t  21, 0, 22, -1, 21, 22, 20, -1, 20, 19, 21, -1, 18, 19, 20, -1,\n" \
		"\t\t  18, 17, 19, -1, 19, 17, 15, -1, 15, 17, 16, -1, 14, 15, 16, -1,\n" \
		"\t\t  13, 15, 14, -1, 14, 12, 13, -1, 12, 10, 13, -1, 12, 11, 10, -1,\n" \
		"\t\t  11, 8, 10, -1, 9, 8, 11, -1, 9, 7, 8, -1, 8, 7, 6, -1,\n" \
		"\t\t  6, 7, 5, -1, 4, 6, 5, -1, 2, 6, 4, -1, 4, 3, 2, -1,\n" \
		"\t\t  2, 3, 1, -1, 1, 0, 2, -1, 153, 152, 151, -1, 150, 151, 152, -1,\n" \
		"\t\t  150, 149, 151, -1, 148, 151, 149, -1, 148, 149, 147, -1, 148, 147, 146, -1,\n" \
		"\t\t  145, 146, 147, -1, 145, 144, 146, -1, 145, 142, 144, -1, 143, 144, 142, -1,\n" \
		"\t\t  143, 142, 141, -1, 142, 137, 141, -1, 140, 141, 137, -1, 140, 137, 139, -1,\n" \
		"\t\t  138, 139, 137, -1, 138, 137, 136, -1, 135, 136, 137, -1, 134, 133, 132, -1,\n" \
		"\t\t  131, 132, 133, -1, 131, 130, 132, -1, 131, 129, 130, -1, 128, 130, 129, -1,\n" \
		"\t\t  128, 129, 127, -1, 128, 127, 126, -1, 39, 126, 127, -1, 39, 38, 126, -1,\n" \
		"\t\t  37, 36, 125, -1, 124, 125, 36, -1, 124, 34, 125, -1, 124, 32, 34, -1,\n" \
		"\t\t  32, 123, 33, -1, 122, 33, 123, -1, 123, 121, 122, -1, 120, 122, 121, -1,\n" \
		"\t\t  121, 119, 120, -1, 117, 106, 118, -1, 118, 116, 117, -1, 115, 117, 116, -1,\n" \
		"\t\t  116, 113, 115, -1, 114, 115, 113, -1, 112, 114, 113, -1, 111, 112, 113, -1,\n" \
		"\t\t  113, 110, 111, -1, 109, 111, 110, -1, 110, 118, 109, -1, 108, 109, 118, -1,\n" \
		"\t\t  107, 108, 118, -1, 118, 106, 107, -1, 105, 107, 106, -1, 104, 105, 106, -1,\n" \
		"\t\t  106, 103, 104, -1, 102, 104, 103, -1, 103, 101, 102, -1, 100, 102, 101, -1,\n" \
		"\t\t  101, 99, 100, -1, 98, 97, 96, -1, 95, 96, 97, -1, 95, 93, 96, -1,\n" \
		"\t\t  95, 94, 93, -1, 92, 93, 94, -1, 92, 91, 93, -1, 92, 90, 91, -1,\n" \
		"\t\t  89, 91, 90, -1, 89, 90, 88, -1, 89, 88, 87, -1, 30, 87, 88, -1,\n" \
		"\t\t  29, 31, 86, -1, 85, 86, 31, -1, 84, 83, 82, -1, 81, 82, 83, -1,\n" \
		"\t\t  81, 26, 82, -1, 28, 82, 26, -1, 80, 28, 27, -1, 80, 79, 28, -1,\n" \
		"\t\t  78, 28, 79, -1, 78, 79, 77, -1, 76, 78, 77, -1, 76, 75, 78, -1,\n" \
		"\t\t  76, 74, 75, -1, 72, 61, 73, -1, 73, 71, 72, -1, 70, 72, 71, -1,\n" \
		"\t\t  71, 68, 70, -1, 69, 70, 68, -1, 67, 69, 68, -1, 66, 67, 68, -1,\n" \
		"\t\t  68, 65, 66, -1, 64, 66, 65, -1, 65, 73, 64, -1, 63, 64, 73, -1,\n" \
		"\t\t  62, 63, 73, -1, 73, 61, 62, -1, 60, 62, 61, -1, 59, 60, 61, -1,\n" \
		"\t\t  61, 58, 59, -1, 57, 59, 58, -1, 58, 56, 57, -1, 55, 57, 56, -1,\n" \
		"\t\t  56, 54, 55, -1, 53, 52, 51, -1, 50, 51, 52, -1, 50, 48, 51, -1,\n" \
		"\t\t  50, 49, 48, -1, 47, 48, 49, -1, 47, 46, 48, -1, 47, 45, 46, -1,\n" \
		"\t\t  44, 46, 45, -1, 44, 45, 43, -1, 44, 43, 42, -1, 24, 42, 43, -1,\n" \
		"\t\t  23, 25, 41, -1, 40, 41, 25, -1 ]\n" \
		"\t    normal NULL\n" \
		"\t    texCoord NULL\n" \
		"\t    solid FALSE\n" \
		"\t    creaseAngle 0.524\n" \
		"\t  }\n\t}\n" \
		"\ttranslation 1.12 -0.2 0\n" \
		"\tscale 0.0182 0.0182 0.0182\n" \
		"      }\n    }\n",
	"    Transform {\n" \
		"      children [\n" \
		"\tTransform {\n" \
		"\t  children Shape {\n" \
		"\t    appearance Appearance {\n" \
		"\t      material DEF _0 Material {\n" \
		"\t\tambientIntensity 0.25\n" \
		"\t\tdiffuseColor 0.00232 0 0.748\n" \
		"\t\tspecularColor 0.452 0.15 0.691\n" \
		"\t\temissiveColor 0 0 0\n" \
		"\t\tshininess 0.894\n" \
		"\t\ttransparency 0\n" \
		"\t      }\n\t    }\n" \
		"\t    geometry Text {\n" \
		"\t      string \"WORLD'S FINEST\"\n" \
		"\t      fontStyle FontStyle {\n" \
		"\t\tsize 10\n" \
		"\t\tfamily \"SANS\"\n" \
		"\t\tstyle \"BOLD ITALIC\"\n" \
		"\t\tspacing 1\n\t      }\n" \
		"\t      length 0\n\t    }\n\t  }\n" \
		"\t  translation 1.14 0 0\n" \
		"\t  scale 0.0117 0.0352 0.0123\n\t}\n" \
		"\tTransform {\n" \
		"\t  children Shape {\n" \
		"\t    appearance Appearance { material USE _0 }\n" \
		"\t    geometry Text {\n" \
		"\t      string \"THE\"\n" \
		"\t      fontStyle FontStyle {\n" \
		"\t\tsize 18\n" \
		"\t\tfamily \"SANS\"\n" \
		"\t\tstyle \"ITALIC\"\n" \
		"\t\tspacing 1\n\t      }\n" \
		"\t      length 0\n\t    }\n\t  }\n" \
		"\t  translation 0.7 0 0\n" \
		"\t  scale 0.0119 0.0362 0.0119\n" \
		"\t}\n      ]\n    }\n   ]\n  }\n" \
		"  url \"http://www.netstore.de/\"\n" \
		"  description \"The World's Finest Netstore\"\n" \
		"  parameter [ \"target=_blank\" ]\n }\n"
	};
	/*******************************************************************
	** Do not remove or change the Anchor node to the Netstore site.  **
	** Any modification of the following lines or any modification    **
	** of the resulting data in the VRML model is a violation of the  **
	** license terms under which the use of this software is granted. **
	** See the header of this file for the license terms.		  **
	********************************************************************/
	(void) fprintf(ofp, 
		"# Do not remove or change the following Anchor node to the Netstore\n" \
		"# site.  Any modification on the next lines is a violation of the\n" \
		"# license terms under which the use of this software is granted. See the\n" \
		"# homepage of 3Dstats/http-analyze for information about license terms.\n");

	/* define collision proxy */
	(void) fprintf(ofp, "Transform {\n"
			" children Collision {\n"
			"  proxy NULL\n  collide FALSE\n"
			"  children Group {\n   children [\n");

	for (idx=0; idx < (int)TABSIZE(vr_logo); idx++)
		(void) fprintf(ofp, "%s", vr_logo[idx]);

	(void) fprintf(ofp, " Transform {\n"
		"  children Shape {\n"
		"   appearance Appearance {\n"
		"    material Material {\n"
		"    ambientIntensity 0.25\n"
		"    diffuseColor 0.90441 0.28375 0\n"
		"    specularColor 0.09559 0.09559 0.09559\n"
		"    emissiveColor 0 0 0\n"
		"    shininess 0.078125\n"
		"    transparency 0\n"
		"    }\n   }\n"
		"  geometry Text {\n   string [\n");

	(void) fprintf(ofp, "\t\"%s %s\",\n", GETMSG(432, "3D Access Statistics for"), srvname);
	(void) fprintf(ofp, "\t\"%s %s\",\n", GETMSG(434, "Summary for"), period);
	(void) fprintf(ofp, "\t\"%s %s\" ]\n", GETMSG(433, "Created by"), creator);

	(void) fprintf(ofp, "   fontStyle FontStyle {\n"
		"     size 1\n"
		"     family \"SANS\"\n"
		"     style \"ITALIC\"\n"
		"     justify \"%s\"\n"
		"     spacing 1.0\n    }\n"
		"    length 0\n   }\n  }\n"
		"  translation %s\n"
		"  scale 0.45 0.45 0.45\n }\n",
			oldstyle ? "MIDDLE" : "LEFT",
			oldstyle ? "1.12567 1.735 -0.1119" : "2.7 0 0");

	(void) fprintf(ofp, "   ]\n  } }\n");
	if (oldstyle)
		(void) fprintf(ofp, 
			" translation 21.5921 18.1751 -12.9402\n"
			" rotation 0 1 0  0.298906\n"
			" scale 3.50787 3.50787 3.50787\n"
			" scaleOrientation 0 -1 0  0.0192237\n");
	else
		(void) fprintf(ofp, " translation 0 13 -8.75\n");

	(void) fprintf(ofp, "}\n");
	return;
}

/*
** Create a switch box.
*/

static void prSwitchBox(FILE *const ofp, float const where) {
	(void) fprintf(ofp, "Transform {\n children [\n"
		"  Transform {\n"
		"   children DEF Btn1 Shape {\n"
		"    appearance Appearance { material Material { } }\n"
		"    geometry IndexedFaceSet {\n"
		"     coord Coordinate {\n"
		"      point [ -0.5 0.5 0.5, -0.5 -0.5 0.5, 0.5 0.5 0.5,\n"
		"\t      0.5 -0.5 0.5, 0.5 0.5 -0.5, 0.5 -0.5 -0.5,\n"
		"\t     -0.5 0.5 -0.5, -0.5 -0.5 -0.5 ]\n     }\n"
		"     color Color { color [ 1 0 0, 1 1 0 ] }\n"
		"     coordIndex [ 0, 1, 3, 2, -1, 4, 5, 7, 6, -1, 6, 7, 1, 0, -1, 2,\n"
		"\t\t 3, 5, 4, -1, 6, 0, 2, 4, -1, 1, 7, 5, 3, -1 ]\n");
	(void) fprintf(ofp,
		"     colorIndex [ 0, 1, 1, 0, -1, 0, 1, 1, 0, -1, 0, 1, 1, 0, -1, 0,\n"
		"\t\t 1, 1, 0, -1, 0, 0, 0, 0, -1, 1, 1, 1, 1, -1 ]\n"
		"     texCoord TextureCoordinate { point [ 0 1, 0 0, 1 1, 1 0 ] }\n"
		"     texCoordIndex [ 0, 1, 3, 2, -1, 0, 1, 3, 2, -1, 0, 1, 3, 2, -1, 0,\n"
		"\t\t    1, 3, 2, -1, 0, 1, 3, 2, -1, 0, 1, 3, 2, -1 ]\n"
		"     creaseAngle 0.5\n"
		"    }\n   }\n"
		"   translation 0 0.5 0\n"
		"   scale 0.2 0.5 0.2\n  }\n");
	(void) fprintf(ofp,
		"  Transform {\n"
		"   children USE Btn1\n"
		"   translation 0.22 0.6 0\n"
		"   scale 0.2 0.6 0.2\n"
		"  }\n ]\n"
		" translation %.4g 12.2 -8.75\n"
		" scale 2 2 2\n}\n", where);
	return;
}

/*
** Create a switch.
*/

static void prSwitch(FILE *const ofp) {
	(void) fprintf(ofp, "DEF ModelSwitch Transform {\n children [\n");

	(void) fprintf(ofp, "  DEF Day Transform {\n"
		"   children [\n"
		"    DEF Model1 TouchSensor { }\n"
		"    Group {\n"
		"     children [\n"
		"      Transform {\n"
		"       children DEF Btn0 Shape {\n"
		"\t appearance Appearance {\n"
		"\t  material Material { }\n"
		"\t  texture NULL\n"
		"\t  textureTransform NULL\n"
		"\t }\n"
		"\t geometry IndexedFaceSet {\n"
		"\t  coord Coordinate {\n"
		"\t   point [ -0.5 0.5 0.5, -0.5 -0.5 0.5, 0.5 0.5 0.5,\n"
		"\t\t   0.5 -0.5 0.5, 0.5 0.5 -0.5, 0.5 -0.5 -0.5,\n"
		"\t\t  -0.5 0.5 -0.5, -0.5 -0.5 -0.5 ]\n"
		"\t  }\n"
		"\t  color Color { color [ 0 1 0, 1 1 0, 1 1 0 ] }\n"
		"\t  coordIndex  [ 0, 1, 3, 2, -1, 4, 5, 7, 6, -1, 6, 7, 1, 0, -1, 2,\n"
		"\t\t\t3, 5, 4, -1, 6, 0, 2, 4, -1, 1, 7, 5, 3, -1 ]\n"
		"\t  colorIndex  [ 1, 0, 0, 1, -1, 2, 0, 0, 1, -1, 1, 0, 0, 1, -1, 1,\n"
		"\t\t\t0, 0, 2, -1, 1, 1, 1, 2, -1, 0, 0, 0, 0, -1 ]\n"
		"\t  normal NULL\n"
		"\t  creaseAngle 0.5\n\t }\n       }\n"
		"       translation 0 0.125 0\n"
		"       scale 0.2 0.25 0.2\n      }\n"
		"      Transform {\n"
		"       children USE Btn0\n"
		"       translation 0.22 0.15 0\n"
		"       scale 0.2 0.3 0.2\n"
		"      }\n     ]\n    }\n");

	(void) fprintf(ofp, "    Transform {\n"
		"     children Shape {\n"
		"      appearance Appearance {\n"
		"       material Material {\n"
		"\tambientIntensity 0.256\n"
		"\tdiffuseColor 0.0291577 0.0262922 0.0269642\n"
		"\tspecularColor 0.981818 0 0\n"
		"\temissiveColor 0 0 0\n"
		"\tshininess 0.030303\n"
		"\ttransparency 0\n"
		"       }\n"
		"       texture NULL\n"
		"       textureTransform NULL\n"
		"      }\n"
		"      geometry Text {\n"
		"       string \"%s\"\n"
		"       fontStyle FontStyle {\n"
		"\tsize 4\n"
		"\tfamily \"SANS\"\n"
		"\tstyle \"BOLD\"\n"
		"\tjustify \"MIDDLE\"\n"
		"\tspacing 1\n"
		"       }\n"
		"       length 0\n"
		"      }\n     }\n"
		"     translation 0.11 0.05 0.12\n"
		"     scale 0.025 0.04 0.02\n"
		"    }\n   ]\n"
		"   translation 0.44 0 0\n"
		"  }\n", GETMSG(435, "By Day"));

	(void) fprintf(ofp, "  DEF Hour Transform {\n"
		"   children [\n"
		"    DEF Model2 TouchSensor { }\n"
		"    Transform {\n"
		"     children USE Btn0\n"
		"     translation 0 0.125 0\n"
		"     scale 0.2 0.25 0.2\n"
		"    }\n"
		"    Transform {\n"
		"     children USE Btn0\n"
		"     translation 0.22 0.15 0\n"
		"     scale 0.2 0.3 0.2\n"
		"    }\n");

	(void) fprintf(ofp,
		"    Transform {\n"
		"     children Shape {\n"
		"      appearance Appearance {\n"
		"       material Material {\n"
		"\tambientIntensity 0.256\n"
		"\tdiffuseColor 0.0291577 0.0262922 0.0269642\n"
		"\tspecularColor 0.981818 0 0\n"
		"\temissiveColor 0 0 0\n"
		"\tshininess 0.030303\n"
		"\ttransparency 0\n"
		"       }\n"
		"       texture NULL\n"
		"       textureTransform NULL\n"
		"      }\n"
		"      geometry Text {\n"
		"       string \"%s\"\n", GETMSG(436, "By Hour"));

	(void) fprintf(ofp,
		"       fontStyle FontStyle {\n"
		"\tsize 4\n"
		"\tfamily \"SANS\"\n"
		"\tstyle \"BOLD\"\n"
		"\tjustify \"MIDDLE\"\n"
		"\tspacing 1\n"
		"       }\n"
		"       length 0\n"
		"      }\n     }\n"
		"     translation 0.11 0.05 0.12\n"
		"     scale 0.025 0.04 0.02\n"
		"    }\n   ]\n"
		"   translation 0.88 0 0\n"
		"  }\n ]\n"
		" translation 16 12.2 -8.75\n"
		" scale 2 2 2\n}\n");

	(void) fprintf(ofp, "DEF Display Transform {\n"
		" children DEF modelDisplay ModelSelect { switchChild 0 }\n}\n"
		"DEF modelSwitchScript Script {\n"
		" eventOut SFInt32 switchChild\n"
		" eventIn  SFBool  set_1\n"
		" eventIn  SFBool  set_2\n"
		" url \"vrmlscript:function set_1(value, time) {\n"
		"\t   if (value==TRUE) switchChild  = 0;\n"
		"\t}\n\tfunction set_2(value, time) {\n"
		"\t   if (value==TRUE) switchChild = 1;\n"
		"\t}\"\n}\n");
	return;
}

/*
** Format size of value in KB, MB or GB
*/
static char *fm_size(float const value) {
	static char tbuf[SMALLSIZE];

	if (value >= 1024.0*1024.0*1024.0)
		(void) sprintf(tbuf, "%3.1f GB", value/(1024.0*1024.0*1024.0));
	else if (value >= 1024.0*1024.0)
		(void) sprintf(tbuf, "%3.1f MB", value/(1024.0*1024.0));
	else if (value >= 1024.0)
		(void) sprintf(tbuf, "%3.1f KB", value/1024.0);
	else
		(void) snprintf(tbuf, sizeof tbuf, "%lu Bytes", (u_long)value);
	return (char *)tbuf;
}

/*
** Print background definition
*/
#if (VRML_OPT&BG_COLOR) == BG_COLOR
static void prBackground(FILE *const ofp) {
	(void) fputs( "Transform {\n"
		" children Background {\n"
		"  groundAngle [ 1.5, 1.57 ]\n"
		"  groundColor [ 0 0 0, 0.21 0.21 0.38, 0.3 0.35 0.45 ]\n"
		"  skyAngle [ 1.18, 1.42, 1.55, 1.57 ]\n"
		"  skyColor [ 0 0 0, 0 0 0.38, 0.35 0 0.28, 0.68 0.2 0.1, 0.75 0.35 0 ]\n"
		" }\n}\n", ofp);
	return;
}
#else
/*ARGSUSED*/
static void prBackground(FILE *const ofp) {
	return;
}
#endif

/*
** Print navigation info
*/
static void prNavInfo(FILE *const ofp, u_short const mon) {
#if (VRML_OPT&OLD_LAYOUT) != OLD_LAYOUT
	(void) fprintf(ofp, "DEF VP%02hu0 Viewpoint {\n"
		" position    18.88 7.222 17.81\n"
		" orientation -0.2264 0.9715 0.07053  0.4124\n"
		" fieldOfView 0.7854\n"
		" description \"%.3s %s\"\n}\n",
		mon, monnam[mon], GETMSG(438, "Entry"));

	(void) fprintf(ofp, "DEF VP%02hu1 Viewpoint {\n"
		" position    8.941 26.54 2.603\n"
		" orientation -0.9996 0.02317 -0.01562  1.165\n"
		" fieldOfView 0.7854\n"
		" description \"%.3s %s\"\n}\n",
		mon, monnam[mon], GETMSG(439, "Top view"));
#else
	(void) fprintf(ofp, "DEF VP%02hu0 Viewpoint {\n"
		" position    33.76 15.25 29.75\n"
		" orientation -0.284177 0.95494 0.0856277 0.359334\n"
		" fieldOfView 0.7854\n"
		" description \"%.3s %s\"\n}\n",
		mon, monnam[mon], GETMSG(438, "Entry"));

	(void) fprintf(ofp, "DEF VP%02hu1 Viewpoint {\n"
		" position    20.3094 7.8565 15.7163\n"
		" orientation -0.2825 0.9567 0.07022 0.5085\n"
		" fieldOfView 0.7854\n"
		" description \"%.3s %s\"\n}\n",
		mon, monnam[mon], GETMSG(440, "Hits by day"));

	(void) fprintf(ofp, "DEF VP%02hu2 Viewpoint {\n"
		" position    45.42 7.601 -4.233\n"
		" orientation -0.3568 0.9317 0.06838  0.4057\n"
		" fieldOfView 0.7854\n"
		" description \"%.3s %s\"\n}\n",
		mon, monnam[mon], GETMSG(441, "Hits by hour"));

	(void) fprintf(ofp,
		"DEF VP%02hu3 Viewpoint {\n"
		" position    51.87 38.81 -5.734\n"
		" orientation -0.5331 0.7752 0.3389  1.374\n"
		" fieldOfView 0.7854\n"
		" description \"%.3s %s\"\n}\n",
		mon, monnam[mon], GETMSG(442, "Top view"));
#endif
	return;
}

/*
** Create the VRML model.
*/
int prVRMLStats(FILE *const ofp,
		char *const srvname, char *const period, LOGTIME *const tp) {
	char label[SMALLSIZE];		/* label buffer */
	u_long other, av_sum;		/* temp, average sum */
	int idx, cday, hour;		/* temp */
	float sf, wid, depth;		/* scale factor, step width, depth */
	float pts[30];			/* coordinates for drawing functions */

	/* create the VRML models: hits by day */
	(void) fprintf(ofp, "#VRML V2.0 utf8\n\n"
		"WorldInfo {\n info [ \"%s %s\",\n",
		GETMSG(432, "3D Access Statistics for"), srvname);

	(void) fprintf(ofp, "\t \"%s %s\" ]\n}\n",
		GETMSG(433, "Created by"), creator);

#if (VRML_OPT&OLD_LAYOUT) != OLD_LAYOUT
	(void) fprintf(ofp, "PROTO ModelSelect [ "
			" exposedField SFInt32 switchChild 0 ] {\n"
			" Switch {\n"
			"  whichChoice IS switchChild\n"
			"  whichChoice -1\n"
			"   choice [\n");
#endif
	BEGIN_GROUP;

#if (VRML_OPT&OLD_LAYOUT) != OLD_LAYOUT
	prSwitchBox(ofp, 16.88);
#endif

	/* define collision proxy */
	(void) fprintf(ofp, "  Transform {\n"
			"   children Collision {\n"
#if (VRML_OPT&COLLISION_DETECT) != COLLISION_DETECT
			"  proxy NULL\n  collide FALSE\n"
#else
			"    proxy Transform {\n"
			"     children Shape { geometry Box { size 18.6 10 8.75 } }"
			"     translation 9.3 5 -4.375\n    }\n"
#endif
			"    children Group {\n     children [\n");

	prBox(ofp, 31, 7, 1, 31);

	wid = 31.0*DIFF_X;
	depth = 7.0*DIFF_Z;
	(void) snprintf(label, sizeof label, "%lu %s",
		max_day.hits, GETMSG(417, "Hits"));
	VTextLabel(ofp, MAT_GREEN, label, wid-0.02, 10.1, depth-0.05);

	(void) snprintf(label, sizeof label, "%lu %s",
		max_day.sessions, GETMSG(420, "Sessions"));
	VTextLabel(ofp, MAT_RED,   label, wid-0.02,  5.1, depth-0.05);

	VTextLabel(ofp, MAT_ORANGE,fm_size(max_day.bytes), wid-0.02,  4.5, depth-0.05);
	HTextLabel(ofp, MAT_GREEN,  GETMSG(443, "Total"), wid+0.2, -0.2, 7.0);
	HTextLabel(ofp, MAT_BLUE,   GETMSG(418, "Files"), wid+0.2, -0.2, 6.0);
	HTextLabel(ofp, MAT_YELLOW, GETMSG(444, "Cached"), wid+1.5, -0.2, 5.5);
	HTextLabel(ofp, MAT_RED,    GETMSG(445, "Other"), wid+3.0, -0.2, 5.0);
	HTextLabel(ofp, MAT_RED,    GETMSG(420, "Sessions"), wid+0.2, -0.2, 3.15);
	HTextLabel(ofp, MAT_ORANGE, GETMSG(446, "Data sent"), wid+0.2, -0.2, 0.65);

	if ((sf = ((float)avg_day.hits/(float)max_day.hits)*SCALEF) > 0.1) {
		prAvgLine(ofp, wid, sf, depth);
		if (sf > 9.3)
			sf = 9.3;
		else if (sf > 3.9 && sf < 5.6)
			sf = (((float)max_day.hits/2.0) < sf) ? 5.6 : 3.9;
		(void) snprintf(label, sizeof label, "%lu %s",
			avg_day.hits, GETMSG(417, "Hits"));
		VTextLabel(ofp, MAT_LINE, label, wid-0.02, sf+0.1, depth-0.05);
	}

	/* draw load by day */
	(void) fprintf(ofp, "  Transform {\n   children [\n");
	for (cday=0; cday < (int)tp->mday; cday++) {
		float lasty = 0.0;

		if ((sf = ((float)daily[cday].hits/(float)max_day.hits)*SCALEF) <= 0.02)
			continue;

		wid = (float)cday*DIFF_X;
		depth = 5.0*DIFF_Z;

		drawBar(ofp, wid, (sf/2.0)*BAR_HT, depth, 1.0, sf, MAT_GREEN);
		if ((sf = ((float)daily[cday].files/(float)max_day.hits)*SCALEF) > 0.02) {
			sf -= 0.005;
			drawBar(ofp, wid, (sf/2.0)*BAR_HT, depth-0.1, 0.4, sf, MAT_BLUE);
			lasty = sf;
		}
		if ((sf = ((float)daily[cday].nomod/(float)max_day.hits)*SCALEF) > 0.02) {
			sf -= 0.005;
			drawBar(ofp, wid, lasty+(sf/2.0)*BAR_HT, depth-0.1, 0.4, sf, MAT_YELLOW);
			lasty += sf;
		}
		other = daily[cday].hits - (daily[cday].files + daily[cday].nomod);
		if ((sf = ((float)other/(float)max_day.hits)*SCALEF) > 0.02) {
			sf -= 0.005;
			drawBar(ofp, wid, lasty+(sf/2.0)*BAR_HT, depth-0.1, 0.4, sf, MAT_RED);
		}
		if ((sf = ((float)daily[cday].sessions/(float)max_day.sessions)*(SCALEF/2.0)) > 0.01)
			drawBar(ofp, wid, (sf/2.0)*BAR_HT, 2.0*DIFF_Z, 1.0, sf, MAT_RED);
		if ((sf = (daily[cday].bytes/max_day.bytes)*(SCALEF/2.0)) > 0.01)
			drawBar(ofp, wid, (sf/2.0)*BAR_HT, 0.0, 1.0, sf, MAT_ORANGE);
	}
	(void) fprintf(ofp, "   ]\n   translation %.4g 0 %.4g\n  }\n",
			DIFF_X/2.0, -(DIFF_Z/2.0));

	(void) fprintf(ofp, "     ]\n    }\n   }\n"
			"   translation 0 0 0\n"
			"  }\n");
	END_GROUP;

	/* load by day and hour */
	BEGIN_GROUP;

#if (VRML_OPT&OLD_LAYOUT) != OLD_LAYOUT
	prSwitchBox(ofp, 17.76);
#endif

	/* define collision proxy */
	(void) fprintf(ofp, "  Transform {\n"
			"   children Collision {\n"
#if (VRML_OPT&COLLISION_DETECT) != COLLISION_DETECT
			"  proxy NULL\n  collide FALSE\n"
#else
			"    proxy Transform {\n"
			"     children Shape { geometry Box { size 14.4 10 8.75 } }"
			"     translation 7.2 5 -4.375\n    }\n"
#endif
			"    children Group {\n     children [\n");

	prBox(ofp, 24, 7, 0, 23);
	wid = 24.0*DIFF_X;
	depth = 7.0*DIFF_Z-0.05;
	(void) snprintf(label, sizeof label, "%lu %s", max_whhits, GETMSG(417, "Hits"));
	VTextLabel(ofp, MAT_GREEN, label,   wid+0.02, 10.1, depth);
	(void) snprintf(label, sizeof label, "%lu %s", max_whhits/2, GETMSG(417, "Hits"));
	VTextLabel(ofp, MAT_RED,   label, wid+0.02, 5.1, depth);
	HTextLabel(ofp, daytab[0], daynam[0], wid+0.2, -0.2, 0.65);
	HTextLabel(ofp, daytab[1], daynam[1], wid+0.2, -0.2, 1.9);
	HTextLabel(ofp, daytab[2], daynam[2], wid+0.2, -0.2, 3.15);
	HTextLabel(ofp, daytab[3], daynam[3], wid+0.2, -0.2, 4.4);
	HTextLabel(ofp, daytab[4], daynam[4], wid+0.2, -0.2, 5.65);
	HTextLabel(ofp, daytab[5], daynam[5], wid+0.2, -0.2, 6.9);
	HTextLabel(ofp, daytab[6], daynam[6], wid+0.2, -0.2, 8.15);

	if ((sf = (((float)avg_day.hits/24.0)/(float)max_whhits)*SCALEF) > 0.1) {
		prAvgLine(ofp, wid, sf, depth);
		if (sf > 9.3)
			sf = 9.3;
		else if (sf > 3.9 && sf < 5.6)
			sf = (((float)max_whhits/2.0) < sf) ? 5.6 : 3.9;
		(void) snprintf(label, sizeof label, "%lu %s",
			avg_day.hits/24L, GETMSG(417, "Hits"));
		VTextLabel(ofp, MAT_LINE, label, wid+0.02, sf+0.25, depth-0.05);
	}

	(void) fprintf(ofp, "  Transform {\n   children [\n");
	for (cday=0; cday < 7; cday++) {
		depth = (float)cday*DIFF_Z;
		for (hour=0; hour < 24; hour++) {
			if ((sf = ((float)wh_hits[cday][hour]/(float)max_whhits)*SCALEF) > 0.01)
				drawBar(ofp, (float)hour*DIFF_X, (sf/2.0)*BAR_HT, depth, 1.0, sf, daytab[cday]);
		}

	}
	(void) fprintf(ofp, "   ]\n   translation %.4g 0 %.4g\n  }\n",
			DIFF_X/2.0, -(DIFF_Z/2.0));

	/* compute pts for average hits */
	for (av_sum=0L, idx=0, cday=0; cday < 7; cday++) {
		pts[idx++] = ((float)avg_wday[cday]/(float)max_avdhits)*AVSCALEF;
		av_sum += avg_wday[cday];
	}
	av_sum /= 7L;
	sf = (float)av_sum/((float)max_avdhits)*AVSCALEF;

	(void) fprintf(ofp, "Transform {\n");
	prAvgCurve(ofp, pts, DIFF_Z, idx, sf);
	(void) fprintf(ofp, " translation 0.05 6 %.4g\n"
			    " rotation 0 1 0 1.57\n}\n", -(DIFF_Z/2.0));

	for (av_sum=0L, idx=0, hour=0; hour < 24; hour++) {
		pts[idx++] = ((float)avg_whour[hour]/(float)max_avhhits)*AVSCALEF;
		av_sum += avg_whour[hour];
	}
	av_sum /= 24L;
	sf = (float)av_sum/((float)max_avhhits)*AVSCALEF;

	(void) fprintf(ofp, "Transform {\n");
	prAvgCurve(ofp, pts, DIFF_X, idx, sf);
	(void) fprintf(ofp, " translation %.4g 6 -8.7\n }\n", DIFF_X/2.0);

	(void) fprintf(ofp, "     ]\n    }\n   }\n"
#if (VRML_OPT&OLD_LAYOUT) != OLD_LAYOUT
			"   translation 0 0 0\n"
#else
			"   translation 30 0 -20\n"
#endif
			"  }\n");
	END_GROUP;

#if (VRML_OPT&OLD_LAYOUT) != OLD_LAYOUT
	(void) fprintf(ofp, "  ]\n }\n}\n");
	prHead(ofp, 0, srvname, period);
	prSwitch(ofp);
#else
	prHead(ofp, 1, srvname, period);
#endif
	prBackground(ofp);
	prNavInfo(ofp, tp->mon);

#if (VRML_OPT&OLD_LAYOUT) != OLD_LAYOUT
	(void) fprintf(ofp,
		"ROUTE modelSwitchScript.switchChild TO modelDisplay.set_switchChild\n"
		"ROUTE Model1.isActive TO modelSwitchScript.set_1\n"
		"ROUTE Model2.isActive TO modelSwitchScript.set_2\n");
#endif
	clrInstance();
	return 1;
}
#endif /* definef(VRML) */

