/*
** Utilities for http-analyze
**
** Copyright  1996-1999 by Stefan Stapelberg, <stefan@rent-a-guru.de>
**
** $Id: utils.c,v 2.4 1999/10/30 09:36:57 stefan Stab $
**
*/

#include <assert.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <fcntl.h>

#if defined(unix)
# include <unistd.h>
# include <sys/wait.h>

#else
# if defined(WIN32)
#  include <winsock.h>		/* for the u_int, etc. types */
#  include <direct.h>		/* for other windows/watcom stuff */
#  include <io.h>		/* for the F_OK, etc. symbolic constants */
#  include <process.h>

# elif defined(NETWARE)
#  include <sys/types.h>
#  include <netdb.h>
#  include <nwthread.h>
#  include <direct.h>		/* for other windows/watcom stuff */
#  include <io.h>		/* for the F_OK, etc. symbolic constants */
# endif
#endif

#include "config.h"
#include "defs.h"

#if !defined(FAST_CTYPE)
# include <ctype.h>
#endif

#if !defined(NO_UNAME)
# include <sys/utsname.h>
#endif

#if defined(USE_STBLKSIZE)
# include <sys/stat.h>
#endif

#if defined(USE_FAST_MALLOC)
# include <malloc.h>
#endif

#if defined(NEED_WAITPID)
static pid_t waitpid (pid_t const, int * const, int const);
#endif

#if !defined(USE_FGETS)
/*
** Read line from given file.
** The optimum I/O size can be specified during run time.
*/
static long blk_size = 0L;		/* best block size for I/O */
static char *fbuf = NULL;		/* dynamically allocated I/O buffer */

int readLine(register char *bp, size_t maxsize, FILE *const lfp) {
	static int saweof = 0;		/* set if EOF reached */
	static int nleft = 0;		/* numbers of chars left in fbuf or EOF */
	static char *next = NULL;	/* ptr to next char in fbuf */
	static int overflow = 0;	/* remember buffer overflows */
	register int cc;
	char *obp = bp;
#if defined(USE_STBLKSIZE)
	struct stat stbuf;		/* obtain best I/O size dynamically */
#endif

	if (saweof) {
		if (fbuf != NULL) {	/* cleanup */
			free(fbuf);
			fbuf = NULL;
			blk_size = 0L;
		}
		return -1;
	}
	if (!blk_size) {
#if defined(USE_STBLKSIZE)
		if (best_iosize)			/* use cmd-line option */
			blk_size = best_iosize;
		else if (fstat(fileno(lfp), &stbuf) == 0)
			blk_size = stbuf.st_blksize;	/* use optimum block size */
		else	blk_size = BEST_IOSIZE;		/* use hardcoded default */
#else
		blk_size = best_iosize ? best_iosize : BEST_IOSIZE;
#endif
		if (blk_size < 1024L)
			blk_size = 1024L;
	}
	if (!fbuf) {
		if ((fbuf=(char *)malloc((size_t)blk_size)) == NULL) {
			prmsg(2, GETMSG(397, "Can't allocate %lu bytes for I/O buffer\n"), (u_long)blk_size);
			saweof = 1;
			return -1;
		}
		if (verbose)
			prmsg(0, GETMSG(398, "Best blocksize for I/O is set to %ld KB\n"), blk_size/1024L);
	}

	maxsize--;			/* preserve space for '\0' byte below */
	do {
		if (nleft == 0) {	/* nothing left, read in a chunk of data */
			if ((nleft = read(fileno(lfp), (void *)fbuf, (size_t)blk_size)) <= 0) {
				if (bp > obp) {		/* output rest from previous chunk */
					saweof = 1;	/* remember EOF */
					nleft = 0;
					break;
				}
				if (fbuf != NULL) {	/* cleanup after EOF */
					free(fbuf);
					fbuf = NULL;
					blk_size = 0L;
				}
				return -1;
			}
			next = fbuf;
		}

		/* copy until newline or buffer overflows */
		while (nleft > 0 && maxsize != 0) {
			nleft--;
			if ((cc = *next++) == '\n')
				break;
			if (!overflow) {
				*bp++ = (char)cc;
				maxsize--;
			}
		}
		/*
		** If a NL was found, the line is complete.
		** If maxsize is zero, the buffer pointed to by bp did overflow.
		** If nleft is zero, the I/O buffer did become empty.
		*/
		if (cc == '\n') {
			if (overflow) {
				overflow = 0;	/* clear overflow condition */
				continue;	/* and proceed with next line */
			}
			*bp = '\0';
			while (*(bp-1) == '\r')	/* strip CR if found */
				*--bp = '\0';
			break;
		}
		if (!maxsize) {			/* terminate line */
			*bp = '\0';
			overflow = 1;		/* skip rest of line at next call */
			break;
		}
	} /*CONSTCOND*/ while (1);
	return (int)(bp-obp);
}
#endif

/*
** Save registration ID.
*/
int saveRegID(char * const lib_dir, char * const company, char * const regID) {
#if defined(WIN32)
	HKEY	rHandle;
	DWORD	rStat;

	if (RegCreateKeyEx(HKEY_LOCAL_MACHINE, REGID_FILE, NULL, NULL,
			   REG_OPTION_NON_VOLATILE, KEY_SET_VALUE,
			   NULL, &rHandle, &rStat) == ERROR_SUCCESS) {
		if (RegSetValueEx(rHandle, "Company", NULL, REG_SZ,
				  company, strlen(company)+1) == ERROR_SUCCESS &&
		    RegSetValueEx(rHandle, "Serial", NULL, REG_SZ,
				  regID, strlen(regID)+1) == ERROR_SUCCESS) {
			RegCloseKey(rHandle);
			return 1;
		}
		RegCloseKey(rHandle);
	}
#else
	char tbuf[MAX_FNAMELEN];
	FILE *fp;

	(void) snprintf(tbuf, sizeof tbuf, "%s/%s", lib_dir, REGID_FILE);
	if ((fp = fopen(tbuf, "w")) != NULL) {
		(void) fprintf(fp, "ORG %s\n", company);
		(void) fprintf(fp, "ID %s\n", regID);
		(void) fclose(fp);
		return 1;
	}
#endif
	return 0;
}

/*
** Read registration ID.
*/
static LIC_INFO license = { NULL, NULL };

LIC_INFO *getRegID(char *const lib_dir, char *const company, char *const regID) {
	char lbuf[LBUFSIZE];
#if defined WIN32
	HKEY	rHandle;
	DWORD	kType, bSize = LBUFSIZE;
#else
	char tbuf[MAX_FNAMELEN];
	FILE *fp = NULL;
	size_t len;
#endif
	if (license.regID && license.company)
		return (LIC_INFO *)&license;

	if (company != NULL && regID != NULL) {
		license.company = strsave(company);
		license.regID = strsave(regID);
		return (LIC_INFO *)&license;
	}
#if defined WIN32
	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, REGID_FILE,
			 NULL, KEY_QUERY_VALUE, &rHandle) == ERROR_SUCCESS) {
		if (RegQueryValueEx(rHandle, "Company", NULL, &kType, lbuf, &bSize) == ERROR_SUCCESS) {
			bSize= LBUFSIZE;
			license.company = strsave(lbuf);
			if (RegQueryValueEx(rHandle, "Serial", NULL, &kType, lbuf, &bSize) == ERROR_SUCCESS)
				license.regID = strsave(lbuf);
		}
		RegCloseKey(rHandle);
	}        
#else
	if (lib_dir != NULL) {
		(void) snprintf(tbuf, sizeof tbuf, "%s/%s", lib_dir, REGID_FILE);
		fp = fopen(tbuf, "r");
	}
	if (fp != NULL) {
		while (fgets(lbuf, sizeof lbuf, fp) != NULL) {
			len = strlen(lbuf);
			if (lbuf[len-1] == '\n')
				lbuf[--len] = '\0';

			if (len > 4 && strneq(lbuf, "ORG ", 4))
				license.company = strsave(lbuf+4);
			else if (len > 3 && strneq(lbuf, "ID ", 3)) {
				license.regID = strsave(lbuf+3);
			}
		}
		(void) fclose(fp);
	}
#endif
#if defined(DEF_REGID)
	else {
		license.company = strsave(DEF_COMPANY);
		license.regID = strsave(DEF_REGID);
	}
#endif
	if (!license.regID || !license.company) {
		license.regID = license.company = NULL;
		return NULL;
	}
	return (LIC_INFO *)&license;
}

/*
** Split string into arguments.
** Fields are separated by one or more tabs.
*/

int getargs(char *str, char **av, int const max) {
	int idx;

	for (idx=0; idx < max-1 && *str != '\0'; idx++) {
		while (*str == '\t' || *str == ' ')
			str++;
		av[idx] = str;
		while (*str && *str != '\t' && (idx || *str != ' '))
			str++;
		if (*str == '\t' || *str == ' ')
			*str++ = '\0';
	}
	av[idx] = NULL;
	return idx;
}

/*
** Parse a date string in format DD/MMM/YYYY:HH:MM:SS
** and fill in the elements of the LOGTIME structure.
*/

int setDate(LOGTIME * const tp, char const cp[]) {
	tp->hour = (u_short) ((cp[12]-'0') * 10 + (cp[13]-'0'));
	tp->min = (u_short) ((cp[15]-'0') * 10 + (cp[16]-'0'));
	tp->sec = (u_short) ((cp[18]-'0') * 10 + (cp[19]-'0'));

	tp->mday = (u_short) ((cp[0]-'0') * 10 + (cp[1]-'0'));
	if (tp->mday > 31 || cp[2] != '/')
		return 0;

	tp->year = (u_short) ((cp[7]-'0') * 1000 + (cp[8]-'0') * 100 +
			      (cp[9]-'0') * 10   + (cp[10]-'0'));

	switch (cp[4]) {
	  case 'a':		/* jan, mar, may */
		switch (cp[5]) {
		  case 'n':	tp->mon = 0;	break;
		  case 'r':	tp->mon = 2;	break;
		  default:	tp->mon = 4;	break;
		}
		break;

	  case 'u':		/* jun, jul, aug */
		switch (cp[5]) {
		  case 'n':	tp->mon = 5;	break;
		  case 'l':	tp->mon = 6;	break;
		  default:	tp->mon = 7;	break;
		}
		break;

	  case 'e':		/* feb, sec, dec */
		switch (cp[3]) {
		  case 'F':	tp->mon = 1;	break;
		  case 'S':	tp->mon = 8;	break;
		  default:	tp->mon = 11;	break;
		}
		break;

	  default:		/* apr, oct, nov */
		switch (cp[3]) {
		  case 'A':	tp->mon = 3;	break;
		  case 'O':	tp->mon = 9;	break;
		  default:	tp->mon = 10;	break;
		}
		break;
	}
	return 1;
}

/*
** Compress a file using gzip.
*/
int compress(char * const file) {
	int status = 0;

#if defined(WIN32) || defined(NETWARE)
	errno = 0;
	if ((status = spawnlp(P_WAIT, "gzip", "gzip", "-f", file, (char *)0)) != 0)
		prmsg(2, GETMSG(399, "Couldn't execute gzip (%s)\n"), strerror(errno));
	return !status;
#else
	pid_t pid;

	(void) fflush(NULL);		/* flush all stdio buffers */

	errno = 0;
	if ((pid = fork()) < 0) {	/* create a new process */
		prmsg(2, GETMSG(400, "Couldn't spawn new process (%s)\n"),
			strerror(errno));
		return 0;
	}
	errno = 0;
	if (pid > 0) {		/* parent process waits for child */
		if ((int)waitpid(pid, &status, 0) < 0) {
			prmsg(1, GETMSG(401, "No children to wait for (%s)?!?\n"),
				strerror(errno));
			return 0;
		}
		return !WEXITSTATUS(status);
	}
	/* child executes gzip */
	(void) execlp("gzip", "gzip", "-f", file, (char *)0);
	prmsg(2, GETMSG(399, "Couldn't execute gzip (%s)\n"), strerror(errno));
	return 0;

#endif
}

/*
** Print a message.
** Level is 0 for informational messages, 1 for warnings, and 2 for fatal errors.
*/
#if !defined(STDERR_FILENO)
# define STDERR_FILENO	2
#endif

/*PRINTFLIKE2*/
void prmsg(int level, char * const fmt, ...) {
	extern char *progname;
	static char *severity[3] = { NULL, NULL, NULL };
	va_list ap;

	if (!severity[0]) {
		severity[0] = GETMSG(402, "[INFO]:");
		severity[1] = GETMSG(403, "[WARNING]:");
		severity[2] = GETMSG(404, "[FATAL]:");
	}

	if (level < 0 || level > 2)
		level = 2;

	if (!isatty(STDERR_FILENO))
		(void) fprintf(stderr, "%s %s ", progname, severity[level]);
	else if (level)
		(void) fprintf(stderr, "%s ", severity[level]);

	va_start(ap, fmt);
	(void) vfprintf(stderr, fmt, ap);
	va_end(ap);
	return;
}
#undef PR_CAT
#undef VPR_CAT

/*
** For maximum portability and for the sake of speed,
** here is our own version of the strdup() function.
*/
char *strsave(register char *cp) {
	register char *np, *ep;

	for (ep=cp; *ep != '\0'; ep++)
		/* no-op */ ;

	if (ep == cp)		/* allocate memory, copy string */
		ep = NULL;
	else if ((ep = malloc((size_t)(ep-cp)+1)) != NULL)
		for (np=ep; (*np = *cp) != '\0'; np++, cp++)
			/* noop */ ;
	return ep;
}

/*
** Get hostname. Use whatever your system provides for this.
** If it does not return the full qualified domain name,
** overwrite the name in the configuration file.
*/

SRVINFO *myHostName(void) {
	static SRVINFO srvinfo = { NULL, NULL, NULL, NULL, NULL };
#if !defined(NO_UNAME)
	static struct utsname utbuf;
#endif
#if !defined(NO_GETHOSTNAME)
	static char hbuf[MEDIUMSIZE];

	if (gethostname(hbuf, sizeof hbuf) == 0) {
		hbuf[MEDIUMSIZE-1] = '\0';
		srvinfo.hostname = hbuf;
	}
#endif

#if !defined(NO_UNAME)
	if (uname(&utbuf) != -1) {
		if (!srvinfo.hostname)
			srvinfo.hostname = utbuf.nodename;
		srvinfo.sysname = utbuf.sysname;
		srvinfo.release = utbuf.release;
		srvinfo.version = utbuf.version;
		srvinfo.machine = utbuf.machine;
	}

/* fake uname(2), leave out hostname to use default */
#elif defined (WIN32)
	srvinfo.sysname = "Windows";
	srvinfo.machine = "PC";
	srvinfo.release = srvinfo.version = NULL;
#elif defined (NETWARE)
	srvinfo.sysname = "Netware";
	srvinfo.machine = "PC";
	srvinfo.release = srvinfo.version = NULL;
#else	/* please specify your platform/OS here */
	srvinfo.sysname = "unknown";
	srvinfo.machine = "unknown";
	srvinfo.release = srvinfo.version = NULL;
#endif
	if (!srvinfo.hostname || !*srvinfo.hostname) {
		char *hn = "local WWW server";
		if ((srvinfo.hostname = GETMSG(405, hn)) == NULL)
			srvinfo.hostname = hn;	/* install default if all fails */
	}
	return (SRVINFO *)&srvinfo;
}

#if defined(NEED_STRERROR)
/*
** Print error string from sys_errlist.
*/
char *strerror(int errcode) {
	if (errcode <= 0 || errcode >= sys_nerr)
		errcode = 0;	/* unknown error */
	return (char *)sys_errlist[errcode];
}
#endif

#if defined(NEED_WAITPID)
/*
** Poor man's implementation of waitpid sufficient for us.
*/
/*ARGSUSED*/
static pid_t waitpid (pid_t const pid, int * const statptr, int const options) {
	pid_t rc;

	while ((rc=wait(statptr)) != pid && errno != ECHILD);
		/* noop */ ;
	return rc == pid ? pid : -1;
}
#endif

#if defined(SUN) && defined(_FILE_OFFSET_BITS)
# undef NEED_SNPRINTF
# undef NEED_VSNPRINTF
#endif

#if defined(SCO) && defined(_SCO_DS_)
# undef NEED_SNPRINTF
#endif

/*
** Copyright (c) 1983, 1995, 1996 Eric P. Allman
** Copyright (c) 1988, 1993
**	The Regents of the University of California.  All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. All advertising materials mentioning features or use of this software
**    must display the following acknowledgement:
**	This product includes software developed by the University of
**	California, Berkeley and its contributors.
** 4. Neither the name of the University nor the names of its contributors
**    may be used to endorse or promote products derived from this software
**    without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

/*
**  SNPRINTF, VSNPRINT -- counted versions of printf
**
**	These versions have been grabbed off the net.  They have been
**	cleaned up to compile properly and support for .precision and
**	%lx has been added.
**
***************************************************************
** Original:
** Patrick Powell Tue Apr 11 09:48:21 PDT 1995
** A bombproof version of doprnt (dopr) included.
** Sigh.  This sort of thing is always nasty do deal with.  Note that
** the version here does not include floating point...
**
** snprintf() is used instead of sprintf() as it does limit checks
** for string length.  This covers a nasty loophole.
**
** The other functions are there to prevent NULL pointers from
** causing nast effects.
***************************************************************
**
** Stefan Stapelberg:	Added `%h' format specifier, make
** sure that value is indeed a short. Made it lint-clean.
*/

#if defined(NEED_SNPRINTF) || defined(NEED_VSNPRINTF)
static char *output;
static char *endptr = NULL;

static void dopr(char *, const char *, va_list);
static void fmtstr(char *, int, int, int);
static void fmtnum(long, int, int, int, int, int);
static void dostr(char *, int);
static void dopr_outch(int);

# if defined(NEED_SNPRINTF)
/* VARARGS3 */
int snprintf(char *str, size_t count, const char *fmt, ...) {
	int len;
	va_list ap;

	va_start(ap, fmt);
	len = vsnprintf(str, count, fmt, ap);
	va_end(ap);
	return len;
}
# endif 

# if defined(NEED_VSNPRINTF)
int vsnprintf(char *str, size_t count, const char *fmt, va_list ap) {
	if (!count)
		return 0;

	*str = '\0';
	endptr = str + (count-1);
	dopr(str, fmt, ap);
	*endptr = '\0';
	return strlen(str);
}
# endif 

/*
** dopr(): poor man's version of doprintf
*/
static void dopr(char *buffer, const char *format, va_list ap) {
	char *strvalue;
	int ch, ljust, len, zpad;
	int shortflag = 0;
	int longflag  = 0;
	int pointflag = 0;
	int maxwidth  = 0;
	long value;

	output = buffer;
	while ((ch = *format++) != '\0') {
		if (ch != '%') {
			dopr_outch(ch);
			continue;
		}

		ljust = len = zpad = maxwidth = 0;
		longflag = pointflag = 0;

	nextch:
		switch (ch = *format++) {
		  case '\0':	dostr("**end of format**" , 0);
				return; /*NOTREACHED*/

		  case '-':	ljust = 1; goto nextch; /*NOTREACHED*/

		  case '0':	/* set zero padding if len not set */
			if (!len && !pointflag)
				zpad = '0'; /*FALLTHROUGH*/
		  case '1': case '2': case '3':
		  case '4': case '5': case '6':
		  case '7': case '8': case '9':
				if (pointflag)
					maxwidth = maxwidth * 10 + (ch - '0');
				else	len = len * 10 + (ch - '0');
				goto nextch;

/*LINTED Expect 'logical expr always false' for lines 660-724 (va_arg) */
		  case '*':	if (pointflag)
					maxwidth = va_arg(ap, int);
				else	len = va_arg(ap, int);
				goto nextch;

		  case '.':	pointflag = 1;
				goto nextch;

		  case 'h':	shortflag = 1;
				goto nextch;

		  case 'l':	longflag = 1;
				goto nextch;

		  case 'u': case 'U':
				if (longflag)
					value = va_arg(ap, long);
				else if (shortflag)
					value = (short)va_arg(ap, int);
				else	value = va_arg(ap, int);
				fmtnum(value, 10, 0, ljust, len, zpad);
				break;

		  case 'o': case 'O':
				if (longflag)
					value = va_arg(ap, long);
				else if (shortflag)
					value = (short)va_arg(ap, int);
				else	value = va_arg(ap, int);
				fmtnum(value, 8, 0, ljust, len, zpad);
				break;

		  case 'd': case 'D':
				if (longflag)
					value = va_arg(ap, long);
				else if (shortflag)
					value = (short)va_arg(ap, int);
				else	value = va_arg(ap, int);
				fmtnum(value, 10, 1, ljust, len, zpad);
				break;

		  case 'x':	if (longflag)
					value = va_arg(ap, long);
				else if (shortflag)
					value = (short)va_arg(ap, int);
				else	value = va_arg(ap, int);
				fmtnum(value, 16, 0, ljust, len, zpad);
				break;

		  case 'X':	if (longflag)
					value = va_arg(ap, long);
				else if (shortflag)
					value = (short)va_arg(ap, int);
				else	value = va_arg(ap, int);
				fmtnum(value, -16, 0, ljust, len, zpad);
				break;

		  case 's':	strvalue = va_arg(ap, char *);
				if (maxwidth > 0 || !pointflag) {
					if (pointflag && len > maxwidth)
						len = maxwidth; /* Adjust padding */
					fmtstr(strvalue, ljust, len, maxwidth);
				}
				break;

		  case 'c':	ch = va_arg(ap, int);
				dopr_outch(ch);
				break;

/*LINTED Expect ptr alignment trouble for lines 660-724 (va_arg) */
		  case '%':	dopr_outch(ch);	
				break;

		  default:	/*assert(0);	** unknown format specifier */
				break;
		}
	}
	*output = '\0';
}

static void fmtstr(char *value, int ljust, int len, int maxwidth) {
	int padlen, slen;		/* amount to pad */

	if (value == NULL)
		value = "(null)";

	for (slen=0; value[slen] != '\0'; slen++)
		/* no-op */ ;
	if (slen > maxwidth && maxwidth)
		slen = maxwidth;
	if ((padlen = len - slen) < 0)
		padlen = 0;
	if (ljust)
		padlen = -padlen;
	while (padlen > 0) {
		dopr_outch(' ');
		--padlen;
	}
	dostr(value, maxwidth);
	while (padlen < 0) {
		dopr_outch(' ');
		++padlen;
	}
}

static void fmtnum(long value, int base,
		   int dosign, int ljust, int len, int zpad) {
	char convert[20];
	unsigned long uvalue;
	int signvalue = 0;
	int place = 0;
	int padlen = 0;
	int caps = 0;

	uvalue = value;
	if (dosign && value < 0) {
		signvalue = '-';
		uvalue = -value;
	}
	if (base < 0) {
		caps = 1;
		base = -base;
	}

	do {
		convert[place++] = (caps ? "0123456789ABCDEF" : "0123456789abcdef")
				   [uvalue % (unsigned)base];
		uvalue = (uvalue / (unsigned)base);
	} while(uvalue);
	convert[place] = '\0';

	padlen = len - place;
	if (padlen < 0)
		padlen = 0;
	if (ljust)
		padlen = -padlen;

	if (zpad && padlen > 0) {
		if (signvalue) {
			dopr_outch(signvalue);
			--padlen;
			signvalue = 0;
		}
		while (padlen > 0) {
			dopr_outch(zpad);
			--padlen;
		}
	}
	while (padlen > 0) {
		dopr_outch(' ');
		--padlen;
	}
	if (signvalue)
		dopr_outch(signvalue);
	while (place > 0)
		dopr_outch(convert[--place]);
	while (padlen < 0) {
		dopr_outch(' ');
		++padlen;
	}
}

static void dostr(char *str, int cut) {
	if (cut)
		while (*str && cut-- > 0)
			dopr_outch(*str++);
	else
		while(*str)
			dopr_outch(*str++);
	return;
}

static void dopr_outch(int c) {
	if (!endptr || output < endptr)
		*output++ = (char)c;
	return;
}
#endif


#if defined(NEED_STRCASECMP)
/*
 * Copyright (c) 1987 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * This array is designed for mapping upper and lower case letter
 * together for a case independent comparison.  The mappings are
 * based upon ascii character sequences.
 */
static u_char charmap[] = {
	'\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
	'\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
	'\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
	'\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
	'\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
	'\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
	'\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
	'\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
	'\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	'\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
	'\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	'\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
	'\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
	'\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
	'\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
	'\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
	'\240', '\241', '\242', '\243', '\244', '\245', '\246', '\247',
	'\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
	'\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
	'\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
	'\300', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
	'\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
	'\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
	'\370', '\371', '\372', '\333', '\334', '\335', '\336', '\337',
	'\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
	'\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
	'\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
	'\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377',
};

int strcasecmp(const char *s1, const char *s2) {
	register u_char	*cm = charmap,
			*us1 = (u_char *)s1,
			*us2 = (u_char *)s2;

	while (cm[*us1] == cm[*us2++])
		if (*us1++ == '\0')
			return(0);
	return(cm[*us1] - cm[*--us2]);
}

int strncasecmp(const char *s1, const char *s2, register size_t n) {
	register u_char	*cm = charmap,
			*us1 = (u_char *)s1,
			*us2 = (u_char *)s2;

	while (--n >= 0 && cm[*us1] == cm[*us2++])
		if (*us1++ == '\0')
			return(0);
	return(n < 0 ? 0 : cm[*us1] - cm[*--us2]);
}
#endif

#if defined(NEED_GETOPT)
/*
 * Copyright (c) 1987 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * get option letter from argument vector
 */
int	opterr = 1,		/* if error message should be printed */
	optind = 1,		/* index into parent argv vector */
	optopt;			/* character checked for validity */
char	*optarg;		/* argument associated with option */

#define	EMSG	""
#define	tell(s)	{ fputs(*nargv, stderr); \
		  fputs(s, stderr); \
		  fputc(optopt, stderr); \
		  fputc((int)'\n', stderr); \
		}

int getopt(int nargc, char * const nargv[], const char * ostr) {
	static char *place = EMSG;		/* option letter processing */
	register char *oli;			/* option letter list index */

	if (!*place) {				/* update scanning pointer */
		if (optind >= nargc ||
		    *(place = nargv[optind]) != '-' || place[1] == '\0')
			return(EOF);		/* end of list or "-" */
		if (*++place == '-') {		/* found "--" */
			++optind;
			return(EOF);
		}
	}					/* option letter okay? */
	if ((optopt = (int)*place++) == (int)':' ||
	    !(oli = strchr(ostr, optopt))) {
		if (!*place)
			++optind;
		if (opterr && *ostr != ':')
			tell(": illegal option -- ");
		return (int)'?';
	}
	if (*++oli != ':') {			/* don't need argument */
		optarg = NULL;
		if (!*place)
			++optind;
	} else {				/* need an argument */
		if (*place)			/* no white space */
			optarg = place;
		else if (nargc <= ++optind) {	/* no arg */
			place = EMSG;
			if (opterr && *ostr != ':')
				tell(": option requires an argument -- ");
			return (int)(*ostr == ':' ? ':' : '?');
		} else				/* white space */
			optarg = nargv[optind];
		place = EMSG;
		++optind;
	}
	return(optopt);				/* dump back option letter */
}
#endif

#if defined(USE_MYCAT)
/*
** Storage area for ptrs to messages.
*/
static char **msgtab = NULL;
static size_t msgidx = 0, msgnum = 0;

/*
** Open the SVR4-style msg catalog, read strings into memory.
** Save a pointer to them in the msgtab array.
*/
#define MEM_CHUNK	256

char *mycatopen(char *const libdir, char *locale) {
	static char stdc[] = "C";
	char lbuf[LBUFSIZE];
	char *s1, *s2;
	FILE *fp;

	if (!locale || (locale[0] == stdc[0] && locale[1] == stdc[1]))
		return (char *)stdc;

	(void) snprintf(lbuf, sizeof lbuf, "%s/msgcat/ha-%s.cat", libdir, locale);
	if ((fp = fopen(lbuf, "r")) == NULL)
		return NULL;

	while (fgets(lbuf, sizeof lbuf, fp) != NULL) {
		int len = strlen(lbuf)-1;
		if (lbuf[len] == '\n')
			lbuf[len] = '\0';

		/* copy message, interpret escape sequences */
		for (s1=s2=lbuf; (*s2 = *s1) != '\0'; s1++, s2++) {
			if (*s1 == '\\' && *(s1+1) != '\0') {
				switch (*++s1) {
				  case 'a': *s2 = '\a'; break;
				  case 'b': *s2 = '\b'; break;
				  case 'n': *s2 = '\n'; break;
				  case 'r': *s2 = '\r'; break;
				  case 't': *s2 = '\t'; break;
				  case 'v': *s2 = '\v'; break;
				  default: 		break;
				}
			}
		}

		/* check whether we need more space */
		if (msgidx == msgnum) {
			char **newcore = !msgnum ? (char **)calloc(MEM_CHUNK, sizeof(char *))
						 : (char **)realloc(msgtab, (msgnum+MEM_CHUNK)*sizeof(char *));
			if (!newcore) {
				prmsg(1, "Not enough memory for message catalog `%s'\n", locale);
				break;
			}
			msgtab = newcore;
			msgnum += MEM_CHUNK;
		}
		if (!len)		/* empty message */
			msgtab[msgidx] = NULL;
		else if ((msgtab[msgidx] = strsave(lbuf)) == NULL) {
			prmsg(1, "Not enough memory for all strings from message catalog `%s'\n", locale);
			break;
		}
		msgidx++;
	}
	(void) fclose(fp);
	return locale;
}
#undef MEM_CHUNK

/*
** Close the message catalog, free all memory.
** Allows message catalogs to be changed on the fly.
*/
void mycatclose(void) {
	size_t idx;

	if (msgtab && msgnum) {
		for (idx=0; idx < msgnum; idx++)
			if (msgtab[idx])
				free(msgtab[idx]);
		free(msgtab);
		msgtab = NULL;
		msgidx = msgnum = 0;
	}
	return;
}

/*
** Retrieve the message defined by `id' from the msg catalog.
** Returns the default message if none found in the catalog.
*/
char *mycatgets(int id, const char *s) {
	if (!msgtab || id < 1 || id > msgidx || !msgtab[id-1])
		return s ? (char *)s : "Message not found!!";
	return msgtab[id-1];
}
#endif

