/* This is for emacs: -*-Mode: C++;-*- */
/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#if !defined(_INC_YEHIA_PLUGIN_H)
#define _INC_YEHIA_PLUGIN_H

#include <string>
#include <list>
#include <map>
#include <stdexcept>

#include <sigcx/thread.h>

#include <yehia/node.h>
#include <yehia/error-handler.h>

namespace Yehia
{

class Plugin;
class PluginManager;
class PluginNode;

/** \addtogroup Plugin
 *@{*/

class PluginInfo
{
  public:
    PluginInfo(const std::string& name, Plugin *plugin);
    ~PluginInfo();

    Plugin *plugin() const { return plugin_; }
    const std::string& name() const { return name_; }

    void set_plugin(Plugin *plugin);
  private:
    std::string name_;
    Plugin *plugin_;
};

/** PluginNode Class.
 *  Represents a node in the plugin tree.
 */
class PluginNode : public G_Node<PluginInfo>
{
    typedef G_Node<PluginInfo> Super;
  public:
    PluginNode(iterator it) : Super(it) { }
    virtual ~PluginNode() { } // for RTTI

    /** Constructor.
     * \param name Name of the plugin.
     * \param p pointer to the plugin. */
    PluginNode(const std::string& name, Plugin *p = 0);

    /** Get full name.
     * \return Name, including names of ancestors. */
    std::string full_name() const;
    
    /** Get name. */
    std::string name() const { return iterator(*this)->name(); }

    /** Get plugin. 
     * \returns Plugin, if loaded, 0 otherwise. */
    Plugin *plugin() const { return iterator(*this)->plugin(); }
    
    /** Set plugin.
     * \param p The plugin */
    void set_plugin(Plugin *p) { iterator(*this)->set_plugin(p); }

    /** Find child by name.
     * \param name Name of the searched child. */
    iterator find(const std::string& name) { return find_(name); }
    /** Find child named by name.
     * \param name Name of the searched child. */
    const_iterator find(const std::string& name) const { return find_(name); }
    
    /** Find plugin named \p name 
     * \return pointer to plugin if found, 0 otherwise. */
    Plugin* operator[](const std::string& name) const {
      const_iterator it = find(name);
      return(it != end() ? (*it).plugin() : 0);
    }
    
    /** Insert a plugin.
     * \return iterator at the inserted node. */
    iterator insert(const std::string& name, Plugin *plugin = 0);

    /** Remove subnode by name.
     * \param name Name of subnode to remove.
     * \return boolean; true if \p name was found. */
    bool erase(const std::string& name);
    void erase(iterator it) { Super::erase(it); }
    void erase(iterator start, iterator end) { Super::erase(start, end); }
  private:
    iterator find_(const std::string& name) const;
};

/** PluginLoader class.
 *
 * A plugin loader is responsible for loading a class of plugins
 * (e.g. there is a plugin loader for loading native C++ plugins and
 * others for the scripting languages).
 */
class PluginLoader : virtual public ErrorHandler
{
  public:
    /** Constructor. 
     * \param parent Parent errorhandler, passed to 
     * ErrorHandler::ErrorHandler(parent). */
    PluginLoader(ErrorHandler *parent) : ErrorHandler(parent) { }

    /** Get identification.
     * \return A free-form string that identifies the plugin loader. */
    virtual std::string id() const = 0;
    /** Load a plugin.
     * \param mgr The calling plugin manager.
     * \param name The name of the plugin to load */
    virtual Plugin *load(PluginManager& mgr, const std::string& name) = 0;
    
    /** Scan for plugins. 
     * \param mgr The calling plugin manager. */
    virtual void scan(PluginManager& mgr) const = 0;
};

/** PluginManager class.
 *
 * The plugin manager is the central component of uCXX. It is
 * responsible for loading plugins using plugin loaders, emitting
 * signals when plugins are loaded and unloaded and for managing
 * scripting languages.
*/
class PluginManager : public ErrorHandler
{
  public:
    /** Constructor.
     * \param parent The parent errorhandler passed to
     *   ErrorHandler::Errorhandler(parent). */
    PluginManager(ErrorHandler *parent = 0);
    /** Destructor. */
    ~PluginManager();
    
    /** Get \c PluginManager instance. 
     * The \c PluginManager class is a singleton, that is, 
     * only one instance exists in a program.
     * \return If a \c PluginManager instance has already been created, 
     *    this will return a reference to that instance, else a new instance 
     *    will be created by invoking the constructor with default 
     *    arguments. */
    static PluginManager& instance();
    
    /** Get architecture independant search paths. 
     * \return A list of strings consisting of the directories searched for 
     * architecture independant plugins (e.g. scripting plugins). */
    const std::list<std::string>& arch_indep_paths();
    /** Get architecture independant search paths. 
     * \return A list of strings consisting of the directories searched for 
     * architecture dependant plugins (e.g. compiled machine code). */
    const std::list<std::string>& arch_dep_paths();

    /** Register a new plugin loader. 
     * \param loader The loader's \c load method will be invoked whenever 
     * a new plugin should be loaded. */
    void register_plugin_loader(PluginLoader& loader);
    /** Unregister a plugin loader.
     * \param loader The loader to be removed from the list of registered 
     *   plugin loaders. */
    void unregister_plugin_loader(PluginLoader& loader);

    /** Load a plugin.
     * \param name Name of the plugin to be loaded.
     * \return A pointer to the plugin on success, 0 otherwise. */
    Plugin *load_plugin(const std::string& name);
    /** Relase a plugin.
     * Free storage asocciated with the plugin.
     * \param name Name of the plugin to release. */
    void release_plugin(const std::string& name);

    /** Set error.
     * This is invoked with an error message from plugin loaders 
     * when they fail. If all loaders registered fail, the last 
     * error message provided to \c set_error will be emitted via 
     * ErrorHandler::emit_error(msg).
     * \param err Error message. */
    void set_error(const std::string& err = std::string());
    
    /** Get tree of plugins.
     * \return The root node of the plugin tree. */
    const PluginNode& plugins() const { return plugin_tree_; }
    
    /** Scan for plugins.
     * This invokes PluginLoader::scan(this) for all registered plugin 
     * loaders, which in turn invoke PluginManager::plugin_found(name) 
     * for all plugins found. */
    void scan();

    /** Indicate a found plugin. 
     * Called by the plugin loaders to tell about available plugins. */
    void plugin_found(const std::string& name);
    
    /** Emitted when a plugin is loaded.
     *  \param id The name of the plugin.
     */
    SigC::Signal1<void, const std::string&> plugin_loaded;

    /** Emitted when a plugin is unloaded.
     *  \param id The name of the plugin.
     */
    SigC::Signal1<void, const std::string&> plugin_unloaded;

    /** Emitted when a new plugin becomes available for loading.
     *  \param id The name of the plugin.
     */
    SigC::Signal1<void, const std::string&> plugin_available;

    /** Emitted when a plugin becomes unavailable.
     *  \param id The name of the plugin.
     */
    SigC::Signal1<void, const std::string&> plugin_unavailable;
  private:
    static PluginManager *instance_;
    
    PluginNode plugin_tree_;
    std::list<PluginLoader *> loaders_;
    std::list<std::string> ad_paths_, ai_paths_;
    std::string last_error_;
    bool scan_time_;
};

/** Plugin class.
 *
 * The \c Plugin class is the base class for all native plugins.
 */
class Plugin : public ErrorHandler
{
  public:
    /** Constructor.
     * \param parent The plugin manager. */
    Plugin(PluginManager& parent) : ErrorHandler(&parent) { }
    /** Destructor */
    virtual ~Plugin();

    /** Get description. 
     * \return A free-form description of the plugin. */
    virtual std::string description() const = 0;

    /** Get plugin manager.
     * \return The plugin manager responsible for this plugin. */
    PluginManager& manager() {
      return dynamic_cast<PluginManager&>(*parent());
    }
 
};

/*@}*/

}

#endif
