/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <sigcx/thread_tunnel.h>

#include <yehia/plugin.h>

#include "python-script.h"
#include "python-loader.h"

using namespace SigC;

using namespace SigCX;
using namespace SigCX::Threads;

using namespace Yehia;
using namespace Yehia::Script;

namespace Yehia
{

namespace Script
{

namespace
{

void thrower(std::exception& e)
{
  throw e;
}

}

class PythonPlugin : public Plugin
{
  public:
    PythonPlugin(PluginManager& mgr);
    virtual ~PythonPlugin();

    virtual std::string name() const { return "python"; }
    virtual std::string description() const { return "Python support"; }
    Language *language() { return lang_; }

    bool single_threaded() const { return single_threaded_; }
    Tunnel *tunnel() { return tunnel_; }
    PluginLoader *loader() { return loader_; }
    void init();

    void run();
  private:
    Language *lang_;
    Plugin *yehia_;
    pythonPluginLoader *loader_;
    pythonObjectFactory obj_factory_;
    pythonObject root_ns_;
    ThreadTunnel *tunnel_;
    Dispatcher *disp_;
    bool single_threaded_;

    // lock for instance data
    Threads::Mutex mutex_; 
};

class PythonDispatcher : public StandardDispatcher
{
  public:
    PythonDispatcher(PythonPlugin& plugin) { }
    virtual ~PythonDispatcher() { }
    
    virtual bool run(bool) { 
      try
      {
        StandardDispatcher::run();
      }
      catch (std::exception& e)
      {
        tunnel<void, std::exception&>(
                slot(&thrower), e, LanguageManager::instance().main_tunnel());
      }
      return true;
    }
};

PythonPlugin::PythonPlugin(PluginManager& mgr)
    : Plugin(mgr), root_ns_(obj_factory_)
{
  if ((yehia_ = mgr.load_plugin("yehia")) == 0)
    throw std::runtime_error("loading yehia plugin failed");
  
  yehia_->reference();
    
  single_threaded_ = (bool)getenv("YEHIA_PYTHON_SINGLE_THREADED");

  lang_ = 0;
  tunnel_ = 0;
  disp_ = 0;
}

PythonPlugin::~PythonPlugin()
{
  mutex_.lock();
  yehia_->unreference();
  manager().unregister_plugin_loader(*loader_);
  //manager()->unregister_language("python");
  if (tunnel_) 
  {
    SigCX::tunnel(slot(*disp_, &Dispatcher::exit), tunnel_, true);
    delete tunnel_;
  }
  mutex_.unlock();
}

void PythonPlugin::init()
{
  if (!Py_IsInitialized())
  {
    char *argv[] = { "", NULL };
    Py_Initialize();
    PySys_SetArgv(1, argv);
  }

  mutex_.lock();
  loader_ = manage(new pythonPluginLoader(manager()));

  lang_ = manage(new Language(*root_ns_.namespace_interface(), 
                              obj_factory_, tunnel_));
  mutex_.unlock();
}

void PythonPlugin::run()
{
  mutex_.lock();
  disp_ = manage(new PythonDispatcher(*this));
  tunnel_ = new ThreadTunnel(*disp_);
  mutex_.unlock();
}

} // Script

} // Yehia

extern "C" Yehia::Plugin *yehia_python_plugin_init(Yehia::PluginManager *mgr)
{
  try
  {
    Script::PythonPlugin *plugin = 
      SigC::manage(new Script::PythonPlugin(*mgr));
    
    plugin->reference();
    if (plugin->single_threaded())
      plugin->init();
    else
    {
      plugin->run();
      tunnel(slot(*plugin, &PythonPlugin::init), plugin->tunnel(), true);
    }
    LanguageManager::instance().register_language("python",  
                                                  *plugin->language());
    mgr->register_plugin_loader(*plugin->loader());
    
    return plugin;
  }
  catch (...)
  {
    mgr->set_error("PythonPlugin initialisation failed");
    return 0;
  }
}


