/*
 * GLX Hardware Device Driver for S3 Savage3D and probably Savage/MX and
 * Savage/IX
 * Copyright (C) 2000 Dominik Behr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * Based on S3 Virge driver by Jim Duchek <jimduchek@ou.edu>
 *
 * Dominik Behr <behr@promail.pl>
 * thanks to Raja Koduri and Tim Roberts   
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <fcntl.h>
#include <signal.h>
#include <dlfcn.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"
#include "fog.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "glx_log.h"
#include "glx_config.h"
#include "glx_symbols.h"
#include "glx_init.h"
#include "glx_clients.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#include "s3savglx.h"
#include "s3savpriv.h"

#define GLX_START                X_GLXDirectPrivateStart
// X_GLXDirectGoDirect:
// X_GLXDirectRelease:
#define X_GLXDirectSwapBuffers   (GLX_START+2)
#define X_GLXDirectDmaFlush      (GLX_START+3)

// function to send data back to x client
static int (*pfnSendVendorPrivate)(int opcode,
				   char *vp,
				   size_t vp_sz,
				   xReply *reply,
				   char *data,
				   size_t data_sz) = 0;

typedef struct S3SAVDIRECTINITtag {
 char aInitFn[20];
// init params for server symbols
   int screenNum;		/* not respected - assumed to be zero */

   int vgaInfoVirtualX;
   int vgaInfoVirtualY;
   int vgaInfoDepth;
   int vgaInfoDisplayWidth;
   int vgaInfoVideoRam;
   int vgaInfoChipID;
   unsigned long vgaInfoPhysBase;

   char vgaInfoChipset[80];   
   int vgaBitsPerPixel;
   int vgaLinearSize;

   int xf86PixmapCacheMemoryStart;
   int xf86PixmapCacheMemoryEnd;
   int xf86PCIFlags;

 int nGLXFirstVisual;
} S3SAVDIRECTINIT, *PS3SAVDIRECTINIT;

extern XSMesaContext XSMesa;
extern int __glXNumClients();

S3SAVDATA stS3Sav;
char aS3SavRenderer[256];

/* helper function to get symbols from server 
   sets any error if any error happened */
static int any_error;

static void *
my_dlsym(void *handle, const char *name)
{
 void *tmp = dlsym(handle, name);
 char *error;
if ((error = dlerror()) != 0) 
   {
    fputs(error, stderr);
    any_error = 1;
    return 0;
   }
 return tmp;
}

typedef volatile unsigned long DWORD;
#define PATTERN 0xDEADBEEFL
/* virtual address should be 4096 byte (page) aligned */
unsigned long
virttophys(unsigned long uVirtAddr)
{
 DWORD *pVirt = (unsigned long *)uVirtAddr;
 unsigned long uStore;
 unsigned long uPage;
 unsigned long uPhysAddr = 0xFFFFFFFF;
 int nFD;
 
 uStore = pVirt[0];
 pVirt[0] = PATTERN;

 nFD = open("/dev/mem", O_RDONLY);

 if (nFD < 0)
    return uPhysAddr;
 
 for (uPage = 0; uPage < 0x100000; uPage++)
     {
      DWORD uTest;
      lseek(nFD, uPage << 12, SEEK_SET);
      if (read(nFD, (void *)&uTest, 4) != 4)
         {
	  close(nFD);
	  return uPhysAddr;
	 }
      if (uTest == PATTERN)
         {
          pVirt[0] = PATTERN ^ 0x12345678;
	  lseek(nFD, uPage << 12, SEEK_SET);
          read(nFD, (void *)&uTest, 4);
          if (uTest == pVirt[0])
	     {
	      uPhysAddr = (uPage << 12);	 
    	      break;	 
	     }
	  else     
	     pVirt[0] = PATTERN;
	 }
     }
 
 close(nFD);
 
 pVirt[0] = uStore;

 return uPhysAddr;     
}
#undef PATTERN

PFNS3SAVDMAFLUSH s3savDMAFlush = 0;
PFNS3SAVFIFOWAIT s3savFIFOWait = 0;

/*******************************************************************************

*******************************************************************************/
hwUI32 
s3savGetEvent(void)
{
 stS3Sav.pSHM->uEvent = (stS3Sav.pSHM->uEvent+1) & 0xFFFF;
 return stS3Sav.pSHM->uEvent;
}

/*******************************************************************************

*******************************************************************************/
hwUI32
s3savInsertEvent(void)
{
 hwUI32 uEvent = s3savGetEvent();
 BCIWR(BCI_CMD_UPDSHADOW | uEvent);
 return uEvent;
}

/*******************************************************************************

*******************************************************************************/
void
s3savWaitEvent(hwUI32 uEvent)
{
 #define noWAITEVENT
 #ifdef WAITEVENT
 hwUI32 uM0;
 hwUI32 uOldM0 = 0xFFFFFFFF;
 hwUI32 uM1;
 hwUI32 uOldM1 = 0xFFFFFFFF;
 #endif

 hwUI32 uVal;
 #ifdef TRACE
 hwUI32 uOldVal = 0xFFFFFFFF;
 hwUI32 uCount = 0;
 fprintf(stderr, "[s3sav] s3savWaitEvent(%04X)\n", uEvent);
 #endif
 // docs says event is low 16 bits
 // my experiments show it is upper 16 bits
 while ((((uVal = *stS3Sav.pStatus1) - uEvent) & 0xFFFF) >= 0x8000)
       {
        #ifdef TRACE
	uCount++;
	if (uVal != uOldVal)
	   {
	    uOldVal = uVal;
	    fprintf(stderr, "[s3sav] status1 = %08X\n", uVal);
	   }  
	#endif   
	#ifdef WAITEVENT
	uM0 = S3SAVRD(S3SAV_STATUS0);
	uM1 = S3SAVRD(S3SAV_STATUS1);
	if (uM0 != uOldM0 || uM1 != uOldM1)
	   {
	    fprintf(stderr, "[s3sav] MMIO %08X %08X\n", uM0, uM1);
	    uOldM0 = uM0;
	    uOldM1 = uM1;
	   }  
	if (uCount == 2000)
	   {
            unsigned int i;
	    fprintf(stderr, "[s3sav] pseudo bci buffer dump %d dwords:\n",
	            stS3Sav.pSHM->nBCI);
	    for (i = 0; i < stS3Sav.pSHM->nBCI; i++)
	        {
		 fprintf(stderr, "%5d: %08X", i, stS3Sav.pSHM->aBCI[i]);
		 if ((stS3Sav.pSHM->aBCI[i] & 0xFF000000) == BCI_CMD_UPDSHADOW)
		    fprintf(stderr, " UPDSHADOW %04X",
		            stS3Sav.pSHM->aBCI[i] & 0xFFFF);
		 fprintf(stderr, "\n");	    
		}
	    fflush(stderr);	
	   }      
	#endif   
        usleep(100);
       }	
 #ifdef TRACE
 fprintf(stderr, "[s3sav] s3savWaitEvent(%04X) OK\n", uEvent);       
 #endif
}

/*******************************************************************************

*******************************************************************************/
void
s3savUpdateShadow(void)
{
 BCIWR(BCI_CMD_UPDSHADOW | stS3Sav.pSHM->uEvent);
}

/*******************************************************************************

*******************************************************************************/
static void
s3savFIFOWaitServer(unsigned int nEntrFree)
{
 unsigned int nSlots = S3SAV_MAXFIFO - nEntrFree;
 hwUI32 uVal;
 
 #ifdef TRACE
 hwUI32 uOldVal = 0xFFFFFFFF;
 fprintf(stderr, "[s3sav] s3savFIFOWaitServer(%d)\n", nEntrFree);
 #endif
 s3savUpdateShadow();
 while (((uVal = *stS3Sav.pStatus0) & 0x1FFFF) > nSlots)
       {
        volatile int i;
        for (i = 0; i < 1000; i++)
	    ;
	#ifdef TRACE
	if (uVal != uOldVal)
 	   {
	    fprintf(stderr, "[s3sav] status0 = %08X\n", uVal);
	    uOldVal = uVal;
	   }    
	#endif
       }
 #if 0       
 // do not allow to overflow fifo       
 while (((uVal = *stS3Sav.pStatus0) & 0x1FFFF) > 0x1000)
       {
        volatile int i;
        for (i = 0; i < 1000; i++)
	    ;
	#ifdef TRACE
	if (uVal != uOldVal)
 	   {
	    fprintf(stderr, "[s3sav] status0 = %08X\n", uVal);
	    uOldVal = uVal;
	   }    
	#endif
       }       
 #endif       
 #ifdef TRACE 
 fprintf(stderr, "[s3sav] s3savFIFOWaitServer(%d) OK\n", nEntrFree);       
 #endif
}

/*******************************************************************************

*******************************************************************************/
static void
s3savFIFOWaitClient(unsigned int nEntrFree)
{
 if (stS3Sav.uBCIPtr + nEntrFree >= S3SAV_PSEUDOBCI_SIZE)
    { 
     s3savDMAFlush();
    }
}

/*******************************************************************************

*******************************************************************************/
void
s3savDMAFlushServer(void)
{
 stS3Sav.uBCIPtr = 0;
}

/*******************************************************************************

*******************************************************************************/
void
s3savDMAFlushClient(void)
{
 xReply	reply;

 if (!stS3Sav.uBCIPtr)
    return;
 s3savUpdateShadow();    
 stS3Sav.pSHM->nBCI = stS3Sav.uBCIPtr;
 if (pfnSendVendorPrivate(X_GLXDirectDmaFlush,
			  0, 0,
			  &reply, 0, 0))
	{
	 if (stS3Sav.pSHM->nBCI)
	    fprintf(stderr, "[s3sav] oops, bci not flushed\n");
	}
 stS3Sav.uBCIPtr = 0;
}

/*******************************************************************************
 hook all symbols we will need from svga server
 symbols should be placed in structure accessed by GLXSYM macro 
*******************************************************************************/
int
s3savHookServerSymbols(void *handle)
{
 unsigned int *ppMmioMem;
 any_error = 0;
 
 ppMmioMem =
 my_dlsym(handle, "s3savMmioMem");
 
 stS3Sav.pS3VPriv = my_dlsym(handle, "s3vPriv");
 if (stS3Sav.pS3VPriv)
    {
     int nChipOK = 1;
     switch (stS3Sav.pS3VPriv->ChipId) {
      case 0x8A20:
       strcpy(aS3SavRenderer, "Savage 3D");
       break;
      case 0x8A21:
       strcpy(aS3SavRenderer, "Savage 3D w/Macrovision");
       break;
      case 0x8C10:
       strcpy(aS3SavRenderer, "Savage/MX w/Macrovision");
       break;
      case 0x8C11:
       strcpy(aS3SavRenderer, "Savage/MX");
       break; 
      case 0x8C12:
       strcpy(aS3SavRenderer, "Savage/IX w/Macrovision");
       break; 
      case 0x8C13:
       strcpy(aS3SavRenderer, "Savage/IX");
       break;
      default:
       strcpy(aS3SavRenderer, "Unknown");
       nChipOK = 0;
     } 
     return nChipOK;
    }
 /*
 fprintf(stderr, "s3priv mmiobase %08X\n",stS3Sav.pS3VPriv->MmioBase);
 fprintf(stderr, "s3priv mmiomem %08X\n",stS3Sav.pS3VPriv->MmioMem);
 */
 return !any_error;
}

/*******************************************************************************

*******************************************************************************/
PMemBlock
s3savMakeRoom(unsigned int nSize,
              unsigned int nAlign)
{
 PMemBlock pRes;
 #ifdef DEBUGx
 fprintf(stderr,
         "[s3sav] allocing %d bytes %dKB vidmem\n",
         nSize, nSize / 1024);
 #endif	 

 pRes = mmAllocMem(stS3Sav.pCardHeap,
                   nSize,
                   nAlign,
                   0);
		   
 if (pRes)
    {
     return pRes;
    }
 else
    {
     PS3SAVTEXTURE pTex, pPrev;
     if (!stS3Sav.pTextures)
        {
         // awww, crap, texture too big ???
         return 0;
        }
     // find last texture in chain
     pPrev = 0;
     for (pTex = stS3Sav.pTextures; pTex->pNext; pTex = pTex->pNext)
         {
          pPrev = pTex;
         }
     /* now this is tricky, we have to detect if texture is
        still in use or is going to be used by commands in 
	cob */
     /* FIXME
        well, it doesnt work too well
	maybe we should use BCI event command ???
     */	
     s3savDMAFlush();
     //s3savWait3DIdle();
     #if 0
     if (pTex->nTextureID != 0xFFFFFFFF)
     {
      int i;
      for (i = 0; i < 10; i++)	
          if (((pTex->nTextureID - s3savGetTextureID()) & 0xFFFF) >= 0x8000)
	     break;
          else 
	     usleep(500);
      if (i == 10)
         {
	  stS3Sav.nTextureID = s3savGetTextureID();
	  fprintf(stderr, "[s3sav] something is broken with texture id %08X hardware %08X\n",
	          pTex->nTextureID,
		  s3savGetTextureID());
	 }	     	     
     }
     #endif
     // remove this texture from memory 
     mmFreeMem(pTex->pMemBlock);  
     pTex->pMemBlock = 0;
       
     /* free mesa's link */
     pTex->pMT->DriverData = 0;
     
     // kill the texture
     free((void *)pTex);
     if (pPrev)
        pPrev->pNext = 0;
     else
        stS3Sav.pTextures = 0;   
     return s3savMakeRoom(nSize, nAlign);
    }                  
}

/******************************************************************************
 initialize GLX module
******************************************************************************/
GLboolean 
s3savInitGLX(void)
{
 fprintf(stderr,
         "[s3sav] s3savInitGLX %d\n",
         __glx_is_server);
 stS3Sav.nSHMID = shmget(S3SAV_SHMKEY, 
                         sizeof(S3SAVSHM), 
			 IPC_CREAT 
			 | SHM_R | SHM_W);
			 //| 0666);
 if (stS3Sav.nSHMID < 0)
    {
     fprintf(stderr,
         "[s3sav] cannot get shared memory\n");
     return GL_FALSE;
    } 
    
 stS3Sav.pSHM = shmat(stS3Sav.nSHMID, 0, 0);			  
 #if 0
 if (__glx_is_server)
    {
     /* mark shared memory for deletion after last detach */
     shmctl(stS3Sav.nSHMID, IPC_RMID, 0);
    }
 #endif    
 
 fprintf(stderr,
         "[s3sav] SHMID %08X SHMADDR %08X\n",
         stS3Sav.nSHMID,
	 (unsigned int)stS3Sav.pSHM);
 //if (!__glx_is_server)
    //return GL_FALSE;

 if (__glx_is_server)    
    {
     stS3Sav.pSHM->uVideoMemorySize = GLXSYM(vga256InfoRec).videoRam;
     stS3Sav.pSHM->uFBPhysAddr = GLXSYM(vga256InfoRec).MemBase;
     stS3Sav.pSHM->uMMIOPhysAddr = stS3Sav.pS3VPriv->MmioBase;
     stS3Sav.pSHM->uShadowPhysAddr = 0;
     if (GLXSYM(xf86AccelInfoRec).Flags & PIXMAP_CACHE)
        {
         stS3Sav.pSHM->uPixmapCacheOffset = GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryStart;
         stS3Sav.pSHM->uPixmapCacheSize = GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryEnd
                                        - stS3Sav.pSHM->uPixmapCacheOffset;
        }
     else
       	{
	 stS3Sav.pSHM->uPixmapCacheOffset = 0;
         stS3Sav.pSHM->uPixmapCacheSize = 0;
	}					
     stS3Sav.pSHM->uWidth = GLXSYM(vga256InfoRec).virtualX;
     stS3Sav.pSHM->uHeight = GLXSYM(vga256InfoRec).virtualY;
     stS3Sav.pSHM->uBPP = (GLXSYM(vga256InfoRec).bitsPerPixel + 7) >> 3;
     stS3Sav.pSHM->uNumClients = 0;
     stS3Sav.pSHM->uBCIFlushes = 0;
     stS3Sav.pSHM->uBCIBytes = 0;
     stS3Sav.pSHM->uFrames = 0;
     stS3Sav.pSHM->uEvent = 0;
     s3savDMAFlush = s3savDMAFlushServer;
     s3savFIFOWait = s3savFIFOWaitServer;
    }
 else
    {
     s3savDMAFlush = s3savDMAFlushClient;
     s3savFIFOWait = s3savFIFOWaitClient;
    }
 stS3Sav.uVideoMemorySize = stS3Sav.pSHM->uVideoMemorySize;
 fprintf(stderr,
         "[s3sav] video memory size %dKB\n",
         stS3Sav.uVideoMemorySize);
         
 stS3Sav.uVideoMemoryPtr = (hwUI32)xf86MapVidMem(0, 
					         LINEAR_REGION, 
					         (pointer)stS3Sav.pSHM->uFBPhysAddr,
					         stS3Sav.uVideoMemorySize * 1024);

 fprintf(stderr,
         "[s3sav] video memory ptr %08X\n",
          stS3Sav.uVideoMemoryPtr);
          
 stS3Sav.stFrontBuffer.nWidth = stS3Sav.pSHM->uWidth;
 stS3Sav.stFrontBuffer.nHeight = stS3Sav.pSHM->uHeight;
 stS3Sav.stFrontBuffer.nBPP = stS3Sav.pSHM->uBPP;
 stS3Sav.stFrontBuffer.nStride = stS3Sav.stFrontBuffer.nWidth * stS3Sav.stFrontBuffer.nBPP;
 stS3Sav.stFrontBuffer.pMemBlock = 0;
 stS3Sav.stFrontBuffer.p = (unsigned char *)stS3Sav.uVideoMemoryPtr;
 
 fprintf(stderr,
         "[s3sav] frontbuffer %dx%dx%d stride %d\n",
         stS3Sav.stFrontBuffer.nWidth,
         stS3Sav.stFrontBuffer.nHeight,
         stS3Sav.stFrontBuffer.nBPP,
         stS3Sav.stFrontBuffer.nStride);
             
 stS3Sav.pCardHeap = mmInit(0, 
                            stS3Sav.uVideoMemorySize * 1024);
    
 /* reserve memory for the front buffer */
 mmReserveMem(stS3Sav.pCardHeap, 
              0, 
              stS3Sav.stFrontBuffer.nStride * stS3Sav.stFrontBuffer.nHeight + 0x1000);
 /* 
  reserve memory for pixmap cache
  it seems from the code of 2d driver that it uses all memory
  after frontbuffer as pixmap cache
  it seems we have to use in device section in XF86Config
  Option "no_pixmap_cache"
  until the 2d driver will be fixed (that is forever)
 */
 if (stS3Sav.pSHM->uPixmapCacheOffset
     && stS3Sav.pSHM->uPixmapCacheSize)
    {
     fprintf(stderr,
             "[s3sav] reserving %08X bytes at %08X for pixmap cache",
	     stS3Sav.pSHM->uPixmapCacheSize,
             stS3Sav.pSHM->uPixmapCacheOffset);
     fprintf(stderr, 
             "[s3sav] WARNING! unless you have modified 2D driver it will\n"
	     "                 use all video memory for pixmap cache\n"
	     "                 and that will break 3D driver\n"
	     "                 disable pixmap cache in XF86Config\n");	     
     mmReserveMem(stS3Sav.pCardHeap, 
                  stS3Sav.pSHM->uPixmapCacheOffset, 
                  stS3Sav.pSHM->uPixmapCacheSize);
    }
 else
    {
     fprintf(stderr, 
             "[s3sav] GREAT! no pixmap cache detected\n");
    }    

 /* FIXME: what other GLXProcs pointers should we change? */
 GLXProcs.CreateContext = s3savGLXCreateContext;
 GLXProcs.DestroyContext = s3savGLXDestroyContext;
 GLXProcs.SwapBuffers = s3savGLXSwapBuffers;
 GLXProcs.CreateImage = s3savGLXCreateImage;
 GLXProcs.DestroyImage = s3savGLXDestroyImage;
 GLXProcs.CreateDepthBuffer = s3savGLXCreateDepthBuffer;
 GLXProcs.MakeCurrent = s3savGLXMakeCurrent;
 GLXProcs.BindBuffer = s3savGLXBindBuffer;
 GLXProcs.VendorPrivate = s3savGLXVendorPrivate;
 GLXProcs.AllowDirect = s3savGLXAllowDirect;

 stS3Sav.pContexts = 0;
 stS3Sav.pTextures = 0;
 stS3Sav.pContext = 0;

 stS3Sav.pDrawBuffer = 0;
 
 if (__glx_is_server)    
    {
     stS3Sav.pBCI = (hwUI32 *)(stS3Sav.pS3VPriv->BciMem);
     fprintf(stderr,
             "[s3sav] BCI mapped at %08X\n",
	     (unsigned int)stS3Sav.pBCI);
    }
 else
    {
     stS3Sav.pBCI = stS3Sav.pSHM->aBCI;
     fprintf(stderr,
             "[s3sav] pseudo BCI in shared memory at %08X\n",
	     (unsigned int)stS3Sav.pBCI);
    } 
 stS3Sav.pfBCI = (float *)stS3Sav.pBCI;
 stS3Sav.uBCIPtr = 0;
 
 {
  unsigned int uMMIOMem;

  uMMIOMem = (hwUI32)xf86MapVidMem(0, 
				   MMIO_REGION, 
				   (pointer)stS3Sav.pSHM->uMMIOPhysAddr,
			           0x80000); // 512kb 
 
  stS3Sav.pRegs = (PS3SAVREGS)(uMMIOMem + S3SAV_3DREGS);
  stS3Sav.pMMIOMem = (hwUI8 *)uMMIOMem;
  #ifdef USE_SHADOW_STATUS
  stS3Sav.pStatus0 = (hwUI32 *)((((unsigned long)stS3Sav.pSHM->aShadowBuf) + 0xFFF) & 0xFFFFF000);
  if (S3SAVRD(S3SAV_STATUS_ADDR) & 1)
     {
      fprintf(stderr, "[s3sav] shadow status already enabled at %08X virt %08X\n",
              stS3Sav.pSHM->uShadowPhysAddr,
	      stS3Sav.pStatus0);
     }
  else
     { 
      hwUI32 uTemp;
      stS3Sav.pStatus0[0] = 0xDEADBEEF;
      stS3Sav.pStatus0[1] = 0xDEADBEEF;
      stS3Sav.pStatus0[2] = 0xDEADBEEF;
      
      if (mlock((void *)stS3Sav.pStatus0, 4096) < 0)
         {
	  fprintf(stderr, "[s3sav] trouble ahead, mlock failed !\n");
	 }
      stS3Sav.pSHM->uShadowPhysAddr = (hwUI32)virttophys((unsigned long)stS3Sav.pStatus0);
      // BCI off
      uTemp = S3SAVRD(0x48C18);
      uTemp |= 2; // 2 is shadow status enable
      S3SAVWR(0x48C18, (uTemp & 0xFFFFFFF7)); // bci enable = 0
      usleep(1000);
      S3SAVWR(S3SAV_STATUS_ADDR, (stS3Sav.pSHM->uShadowPhysAddr | 1));
      S3SAVWR(S3SAV_BUFFER_TRESH, COB_TRESHOLDS);
      usleep(1000);
      // BCI back on
      S3SAVWR(0x48C18, uTemp); 
      usleep(1000);
      s3savUpdateShadow();
      usleep(1000);
      fprintf(stderr, "[s3sav] shadaddr %08X\n", S3SAVRD(S3SAV_STATUS_ADDR));
      fprintf(stderr, "[s3sav] enabling shadow status at %08X virt %08X\n",
              stS3Sav.pSHM->uShadowPhysAddr,
	      stS3Sav.pStatus0);
     }     
  stS3Sav.pStatus1 = stS3Sav.pStatus0 + 1;
  stS3Sav.pStatus2 = stS3Sav.pStatus0 + 2;     
  #else
  {
   stS3Sav.pStatus0 = (hwUI32 *)(uMMIOMem + S3SAV_STATUS0);
   stS3Sav.pStatus1 = (hwUI32 *)(uMMIOMem + S3SAV_STATUS1);
   stS3Sav.pStatus2 = (hwUI32 *)(uMMIOMem + S3SAV_STATUS2);
  }
  #endif
  fprintf(stderr, "[s3sav] status = %08X\n", *stS3Sav.pStatus0);
 } 
 stS3Sav.nTextureID = s3savGetTextureID();
 
 stS3Sav.uChangedFlags = 0xFFFFFFFF;
 stS3Sav.stPALADDR = stS3Sav.pRegs->stTEXPALADDR;
 stS3Sav.stTEXADDR = stS3Sav.pRegs->stTEXADDR;
 stS3Sav.stTEXDESC = stS3Sav.pRegs->stTEXDESC;
 stS3Sav.stTEXCTRL = stS3Sav.pRegs->stTEXCTRL;
 stS3Sav.stFOGCTRL = stS3Sav.pRegs->stFOGCTRL;
 stS3Sav.stDRAWCTRL = stS3Sav.pRegs->stDRAWCTRL;
 stS3Sav.stZBCTRL = stS3Sav.pRegs->stZBCTRL;
 stS3Sav.stZBADDR = stS3Sav.pRegs->stZBADDR;
 stS3Sav.stDESTCTRL = stS3Sav.pRegs->stDESTCTRL;
 stS3Sav.stSCSTART = stS3Sav.pRegs->stSCSTART;
 stS3Sav.stSCEND = stS3Sav.pRegs->stSCEND;
 
 {
  hwUI32 nCOB;
  hwUI32 nCOBSize;
  hwUI32 nCOBAddr;
  
  nCOB = S3SAVRD(S3SAV_COB);
  
  nCOBAddr = (nCOB & 0xFFF) << 11;
  nCOBSize = nCOB >> 29;
  switch (nCOBSize) {
   case 0:
    nCOBSize = 256;
    break;
   case 1:
    nCOBSize = 512;
    break;
   case 2:
    nCOBSize = 1024;
    break;        
   case 3:
    nCOBSize = 2048;
    break;    
   case 4:
    nCOBSize = 4096;
    break;    
   case 5:
    nCOBSize = 8192;
    break;    
   case 6:
    nCOBSize = 16384;
    break;    
   case 7:
    nCOBSize = 32768;
    break;    
  }
  fprintf(stderr,
          "[s3sav] COB addr: %08X size: %d\n",
          nCOBAddr,
	  nCOBSize);
  /* reserve memory for Command Overflow Buffer */
  /* add 4KB before COB for cursor */
  mmReserveMem(stS3Sav.pCardHeap, 
               nCOBAddr - 0x1000,
               nCOBSize*4 + 0x1000);
 }

 /* the remaining memory is available for back buffers, depth
    buffers, and textures */
 mmDumpMemInfo(stS3Sav.pCardHeap);

 

 {
  unsigned int uRegion;
  for (uRegion = 0; uRegion < 6; uRegion++)
      {
       stS3Sav.aTiledApertures[uRegion] = (hwUI8 *)xf86MapVidMem(0,
                                                                 MMIO_REGION,
								 (pointer)(stS3Sav.pSHM->uMMIOPhysAddr + (uRegion + 1) * 0x1000000),
								 0x1000000);
      }
 }
 fprintf(stderr,
         "[s3sav] 3d regs mapped at %08X size %08X status at %08X\n",
         stS3Sav.pRegs,
         sizeof(S3SAVREGS),
	 (unsigned int)stS3Sav.pStatus0);
 fprintf(stderr,
         "[s3sav] s3savInitGLX OK\n");
 return GL_TRUE;
}


/******************************************************************************

******************************************************************************/
void 
s3savGLXCreateDepthBuffer(GLcontext* pCtx)
{
 XSMesaContext xsmesa = (XSMesaContext)pCtx->DriverCtx;
 PS3SAVBUFFER pBuf; 

 if (xsmesa->xsm_buffer->backimage == NULL) 
    {
     fprintf(stderr,
             "[s3sav] no backimage in xsmesa->sxm_buffer\n");
     return;
    }

 pBuf = (PS3SAVBUFFER)xsmesa->xsm_buffer->backimage->devPriv;

 #if DEPTH_BITS != 16
 #error "expects 16 bit z buffer"
 #endif
 pBuf->stZBuffer.nBPP = 2; // always do 16 bit z buffer
 // copy size from backbuffer
 pBuf->stZBuffer.nWidth = pBuf->stBackBuffer.nWidth;
 pBuf->stZBuffer.nHeight = pBuf->stBackBuffer.nHeight;
 pBuf->stZBuffer.nStride = (pBuf->stZBuffer.nWidth * pBuf->stZBuffer.nBPP + 127) & 0xFFFFFF80;
 pBuf->stZBuffer.pMemBlock = s3savMakeRoom(pBuf->stZBuffer.nStride
                                           * ALIGN16(pBuf->stZBuffer.nHeight),
                                           S3SAV_BUFFERALIGN);
 if (pBuf->stZBuffer.pMemBlock)
    {
     pBuf->stZBuffer.p = stS3Sav.aTiledApertures[APERTURE_ZBUFFER];
     fprintf(stderr, "[s3sav] ZBufer VidMem ofs: %08X\n",
             pBuf->stZBuffer.pMemBlock->ofs);
     /*
     pBuf->stZBuffer.p = (unsigned char *)stS3Sav.uVideoMemoryPtr
                       + pBuf->stZBuffer.pMemBlock->ofs;
     */		       
     //pCtx->Buffer->Depth = pBuf->stZBuffer.p;
     //pCtx->Buffer->Width = 4096 / pBuf->stZBuffer.nBPP;
    }
 else
    {
     fprintf(stderr,
             "[s3sav] memory allocation for z buffer failed %d KB\n",
	     (pBuf->stZBuffer.nStride * ALIGN16(pBuf->stZBuffer.nHeight)) / 1024);
     pBuf->stZBuffer.p = malloc(pBuf->stZBuffer.nHeight 
                                * pBuf->stZBuffer.nStride);
     //pCtx->Buffer->Depth = pBuf->stZBuffer.p;
     //pCtx->Buffer->Width = pBuf->stZBuffer.nWidth;				 
    }

 fprintf(stderr,
         "[s3sav] depth buffer %dx%dx%d at %08X stride %d\n",
         pBuf->stZBuffer.nWidth,
         pBuf->stZBuffer.nHeight,
         pBuf->stZBuffer.nBPP,
         pBuf->stZBuffer.p,
         pBuf->stZBuffer.nStride);
 
}

/******************************************************************************

******************************************************************************/
static void
s3savDestroySurface(S3SAVSURFACE *pS)
{
 if (pS->pMemBlock)
    {
     mmFreeMem(pS->pMemBlock);
     pS->pMemBlock = 0;
     pS->p = 0;
     return;
    }
 if (pS->p)
    {
     free(pS->p);
     pS->p = 0;
     return;
    }
}

/******************************************************************************

******************************************************************************/
/*
 * s3savGLXDestroyImage
 */
void 
s3savGLXDestroyImage(GLXImage* image)
{
 PS3SAVBUFFER pBuf;

 fprintf(stderr, "[s3sav] destroying backbuffer\n");
 pBuf = (PS3SAVBUFFER)image->devPriv;
 if (pBuf)
    {
     s3savDestroySurface(&(pBuf->stZBuffer));
     s3savDestroySurface(&(pBuf->stBackBuffer));
    }
 else 
 if (image->data)
    {
     free(image->data);
     image->data = 0;
    }
 xfree(image);
}


/******************************************************************************

******************************************************************************/
/*
 * s3savGLXCreateImage
 *
 * This will always succeed, but if the buffer couldn't be allocated in
 * card memory, HW accel won't be usable.
 */
GLXImage *
s3savGLXCreateImage(WindowPtr pwindow, 
                    GLvisual *visual,
                    int width, 
                    int height, 
                    GLXImage *old_image)
{
 PS3SAVBUFFER pBuf;
 GLXImage *pImage;
 int nDepth;
 
 fprintf(stderr, "[s3sav] s3savGLXCreateImage visual %08X\n",
         visual);
 nDepth = visual->RedBits 
        + visual->GreenBits
        + visual->BlueBits;

 fprintf(stderr,
         "[s3sav] GLXCreateImage %dx%dx%d\n",
         width,
         height,
         nDepth);
	 
 if (old_image)
    s3savGLXDestroyImage(old_image);
    
 pBuf = (PS3SAVBUFFER)malloc(sizeof(S3SAVBUFFER));
 if (!pBuf)
    {
     FatalError( "Malloc for buffer failed" );
    }
 memset((void *)pBuf, 0, sizeof(S3SAVBUFFER));    
 
 pBuf->stZBuffer.p = 0;
 pBuf->stZBuffer.pMemBlock = 0;
 
 pBuf->stBackBuffer.nBPP = (nDepth+7) >> 3;
 /*if (pBuf->stBackBuffer.nBPP == 2)
    pBuf->stBackBuffer.nWidth = ALIGN64(width);
 else
    pBuf->stBackBuffer.nWidth = ALIGN32(width);*/
 pBuf->stBackBuffer.nWidth = width;    
 pBuf->stBackBuffer.nHeight = height;
 
 if (pBuf->stBackBuffer.nBPP != stS3Sav.stFrontBuffer.nBPP)
    {
     fprintf(stderr, "[s3sav] frontbuffer is %d bits, backbuffer is %d bits: hell is going loose!\n",
             stS3Sav.stFrontBuffer.nBPP*8,
	     pBuf->stBackBuffer.nBPP*8);
     pBuf->stBackBuffer.nBPP = stS3Sav.stFrontBuffer.nBPP;
     nDepth = pBuf->stBackBuffer.nBPP*8;
    }
 pBuf->stBackBuffer.nStride = (pBuf->stBackBuffer.nWidth * pBuf->stBackBuffer.nBPP + 127) & 0xFFFFFF80; 
	 
 pBuf->stBackBuffer.pMemBlock = s3savMakeRoom(pBuf->stBackBuffer.nStride *
                                              ALIGN16(pBuf->stBackBuffer.nHeight),
                                              S3SAV_BUFFERALIGN);     
 if (pBuf->stBackBuffer.pMemBlock)
    {
     /*
     pBuf->stBackBuffer.p = (unsigned char *)stS3Sav.uVideoMemoryPtr 
                          + pBuf->stBackBuffer.pMemBlock->ofs;*/
     fprintf(stderr, "[s3sav] backbuffer vidmem ofs: %08X\n",
             pBuf->stBackBuffer.pMemBlock->ofs);			  
     pBuf->stBackBuffer.p = stS3Sav.aTiledApertures[APERTURE_BACKBUFFER];
    }
    
 else
    { 
     fprintf(stderr, "[s3sav] no free video memory for backbuffer %d KB\n",
             (pBuf->stBackBuffer.nStride * ALIGN16(pBuf->stBackBuffer.nHeight)) / 1024);
     // drats, fall back to software
     pBuf->stBackBuffer.p = malloc(pBuf->stBackBuffer.nStride *
                                   pBuf->stBackBuffer.nHeight);
    }
 fprintf(stderr, "[s3sav] backbuffer %dx%dx%d at %08X stride %d\n",
         pBuf->stBackBuffer.nWidth,
	 pBuf->stBackBuffer.nHeight,
	 pBuf->stBackBuffer.nBPP*8,
	 pBuf->stBackBuffer.p,
	 pBuf->stBackBuffer.nStride);    
	
 pImage = (GLXImage *)xalloc(sizeof(GLXImage));
 if (!pImage) 
    {
     FatalError( "Malloc for back ximage failed" );
    }

 pImage->pwin = pwindow;
 pImage->width = width;
 pImage->height = height;
 pImage->bits_per_pixel = nDepth;
 pImage->data = pBuf->stBackBuffer.p;
 pImage->bytes_per_line = 4096; //pBuf->stBackBuffer.nStride;
 pImage->devPriv = pBuf;

 pImage->width = 4096 / pBuf->stBackBuffer.nBPP;//pBuf->stBackBuffer.nStride / pBuf->stBackBuffer.nBPP;

 fprintf(stderr, "[s3sav] s3savGLXCreateImage()=%08X\n", pImage);
 return pImage;
}

/******************************************************************************

******************************************************************************/
static int 
DoMakeCurrent(XSMesaContext c, 
              XSMesaBuffer b)
{
 PS3SAVCONTEXT pCtx;
 PS3SAVBUFFER pBuf;

 stS3Sav.pContext = 0;
 stS3Sav.pDrawBuffer = 0;

 if (!c || !b) 
    {
     return 0;
    }

 pCtx = (PS3SAVCONTEXT)c->hw_ctx;
 if (!pCtx) 
    {
     FatalError("DoMakeCurrent: for null hardware context");
    }

 /* if we don't have a backimage, create one now.  This happens
  * because of the way software GLX doesn't allocate a
  * buffer when front buffer rendering, but we need one
  * anyway
  */
 if (!b->backimage) 
    {
     GLvisual visual;
     fprintf(stderr, "[s3sav] create backimage because there is none\n");
     memset( &visual, 0, sizeof( visual ) );

     if (stS3Sav.pSHM->uBPP == 2)
        {
         visual.RedBits = 5;
         visual.GreenBits = 5;
         visual.BlueBits = 5;
	}
     else
     	{
         visual.RedBits = 8;
         visual.GreenBits = 8;
         visual.BlueBits = 8;
	} 
     visual.DepthBits = 16;

     b->backimage = s3savGLXCreateImage((WindowPtr)b->frontbuffer,
                                        &visual, 
                                        b->width, 
                                        b->height, 
                                        NULL);
    }

 pBuf = (PS3SAVBUFFER)b->backimage->devPriv;
 s3savSetTile(APERTURE_BACKBUFFER, 
              pBuf->stBackBuffer.pMemBlock->ofs,
	      pBuf->stBackBuffer.nStride,
	      pBuf->stBackBuffer.nBPP);
 if (pBuf->stZBuffer.pMemBlock)
    { 
     s3savSetTile(APERTURE_ZBUFFER, 
                  pBuf->stZBuffer.pMemBlock->ofs,
                  pBuf->stZBuffer.nStride,
 	          pBuf->stZBuffer.nBPP);
    }
 pCtx->pDrawBuffer = pBuf;
 stS3Sav.pContext = pCtx;
 stS3Sav.pDrawBuffer = pBuf;
 #ifdef DEBUGx
 fprintf(stderr, "[s3sav] set drawbuffer %08X\n", stS3Sav.pDrawBuffer);
 #endif
 return 0;
}


/******************************************************************************

******************************************************************************/
/*
 * s3savGLXMakeCurrent
 * Note that GLX calls this every batch of commands,
 * so it will show up several times a frame even if
 * the client application isn't issuing makeCurrent calls.
 */
GLboolean 
s3savGLXMakeCurrent(XSMesaContext c)
{
 #ifdef TRACE
 fprintf(stderr,
         "[s3sav] s3savGLXMakeCurrent(%08X)\n",
         c);
 #endif	 
	 
 if (c == XSMesa) 
    {
     return (GL_TRUE);
    }

 /* mesa 3.1 will sometimes leave unflushed vertexes */
 if (XSMesa) 
    {
     // so why are hacking around here instead of fixing it in mesa ?? TCM
     glFlush();
    }

 XSMesa = c;
 if (c)
    {
     DoMakeCurrent(c, c->xsm_buffer);
     gl_make_current(c->gl_ctx,
		     (c->xsm_buffer) ? c->xsm_buffer->gl_buffer : NULL);
    }
 else
    {
     /* detach */
     DoMakeCurrent( NULL, NULL );
     gl_make_current( NULL, NULL );
    }
 return GL_TRUE;
}

/******************************************************************************

******************************************************************************/
/*
 * Bind buffer b to context c and make c the current rendering context.
 */
GLboolean 
s3savGLXBindBuffer(XSMesaContext c, 
                   XSMesaBuffer b)
{ 
 #ifdef TRACE
 fprintf(stderr,
         "[s3sav] s3savGLXBindBuffer(%08X, %08X)\n",
         c,
	 b);
 #endif	 
 DoMakeCurrent(c, b);
 return XSMesaBindBuffer(c, b);
}


/******************************************************************************

******************************************************************************/
PS3SAVCONTEXT 
s3savCreateContext(GLcontext *ctx)
{
 PS3SAVCONTEXT pCtx;

 pCtx = (PS3SAVCONTEXT)malloc(sizeof(S3SAVCONTEXT));

 if (!pCtx)
    return 0;
    
 pCtx->pDrawBuffer = 0;
 pCtx->gl_ctx = ctx; // set mesa context

 ctx->Const.MaxTextureLevels = 9;
 ctx->Const.MaxTextureUnits = 1;
 ctx->Const.MaxTextureSize = 512;
    
 s3savDDExtensionsInit(ctx);
 stS3Sav.pSHM->uNumClients++;

 return pCtx;
}

/******************************************************************************

******************************************************************************/
/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext 
s3savGLXCreateContext(XSMesaVisual v,
                      XSMesaContext share_list)
{
 XSMesaContext c;

 c = (XSMesaContext)calloc(1, sizeof(struct xsmesa_context));
    
 if (!c) 
    {
     return NULL;
    }

 c->gl_ctx = gl_create_context(v->gl_visual,
                               (share_list) ? share_list->gl_ctx : NULL,
                               (void *)c,
                               GL_FALSE /* direct rendering */ ); // dont want direct TCM
	   
 if (!c->gl_ctx) 
    {
     free( c );
     return NULL;
    }
        
 c->xsm_visual = v;
 c->xsm_buffer = NULL;  /* set later by XSMesaMakeCurrent */
 c->pixelformat = v->dithered_pf; /* Dithering is enabled by default */
 c->hw_ctx = (void *)s3savCreateContext(c->gl_ctx);
 if (!c->hw_ctx)
    {
     return NULL;
    }
 c->gl_ctx->Driver.UpdateState = s3sav_setup_DD_pointers;
 
 c->gl_ctx->TriangleCaps |= DD_CLIP_FOG_COORD; // tell mesa to clip fog coords for us

 // modify mesa pipeline stage to generate fog coords in alpha component of specular color array in vb
 {
  int nStage;
  for (nStage = 0; nStage < c->gl_ctx->NrPipelineStages; nStage++)
      if (c->gl_ctx->PipelineStage[nStage].ops == PIPE_OP_FOG)
         { 
	  c->gl_ctx->PipelineStage[nStage] = gl_fog_coord_stage;
	 }
 }
 return c;
}


/******************************************************************************

******************************************************************************/
int 
s3savDestroyContext(PS3SAVCONTEXT pCtx)
{
 hwUI32 uEvent;
 if (!pCtx) 
    {
     return 0;
    }
 uEvent = s3savInsertEvent();    
 s3savDMAFlush();    
 s3savWaitEvent(uEvent);

 if (stS3Sav.pSHM->uBCIFlushes)
    {
     fprintf(stderr, "[s3sav] avg BCI flush %10.3f dwords flushes %d\n",
             (double)stS3Sav.pSHM->uBCIBytes / (double)stS3Sav.pSHM->uBCIFlushes,
	     stS3Sav.pSHM->uBCIFlushes);
    }
 if (stS3Sav.pSHM->uFrames)    
    {
     fprintf(stderr, "[s3sav] avg flushes per frame %10.3f frames %d\n",
             (double)stS3Sav.pSHM->uBCIFlushes / (double)stS3Sav.pSHM->uFrames,
	     stS3Sav.pSHM->uFrames);
    }
 // FIXME what if there are any textures left for this context
 if (pCtx == stS3Sav.pContext) 
    { 
     stS3Sav.pContext = 0;
     stS3Sav.pDrawBuffer = 0;
    }
 stS3Sav.pSHM->uNumClients--;    
 free(pCtx);
 return 0;
}

/******************************************************************************

******************************************************************************/
void 
s3savGLXDestroyContext(XSMesaContext c)
{
 if (s3savDestroyContext((PS3SAVCONTEXT)c->hw_ctx ) != 0) 
    {
    }

 XSMesaDestroyContext(c);
 if (XSMesa == c) 
    {
	 XSMesa = 0;
    }
}

// direct rendering stuff
/******************************************************************************

******************************************************************************/
static void
s3savDirectDone(void)
{
 if (GLXSYM(vgaLinearBase) != (pointer) -1) {
      xf86UnMapVidMem(0,
		      LINEAR_REGION,
		      (pointer) GLXSYM(vga256InfoRec).physBase,
		      GLXSYM(vgaLinearSize));
      GLXSYM(vgaLinearBase) = 0;
   }
}

/******************************************************************************
 initialize GLX module in direct client mode
******************************************************************************/
ClientPtr 
s3savDirectInit(PS3SAVDIRECTINIT hw,
		void *display,
		int screen,
		int (*send_func)(),
		void (**release_hook)( void ) )
{
 fprintf(stderr, 
         "s3savDirectInit(%08X, %08X, %08X, %08X, %08X)\n",
         hw,
	 display,
	 screen,
	 send_func,
	 release_hook);
 if (!glxInstallLocalSVGASymbols())
    return 0;
 // this is how we send messages to the X server
 pfnSendVendorPrivate = send_func;
 strcpy(aS3SavRenderer, "Savage 3D Direct");

   memset(&GLXSYM(vga256InfoRec), 0, sizeof(GLXSYM(vga256InfoRec)));
   memset(&GLXSYM(xf86AccelInfoRec), 0, sizeof(GLXSYM(xf86AccelInfoRec)));
   
   GLXSYM(vgaLinearSize) = hw->vgaLinearSize;
   GLXSYM(vga256InfoRec).virtualX = hw->vgaInfoVirtualX;
   GLXSYM(vga256InfoRec).virtualY = hw->vgaInfoVirtualY;
   GLXSYM(vga256InfoRec).depth = hw->vgaInfoDepth;
   GLXSYM(vga256InfoRec).displayWidth = hw->vgaInfoDisplayWidth;
   GLXSYM(vga256InfoRec).videoRam = hw->vgaInfoVideoRam;
   GLXSYM(vga256InfoRec).chipID = hw->vgaInfoChipID;
   GLXSYM(vga256InfoRec).physBase = hw->vgaInfoPhysBase;
   GLXSYM(vga256InfoRec).chipset = hw->vgaInfoChipset;
   GLXSYM(vgaBitsPerPixel) = hw->vgaBitsPerPixel;
   GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryStart = hw->xf86PixmapCacheMemoryStart;
   GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryEnd = hw->xf86PixmapCacheMemoryEnd;
   GLXSYM(xf86AccelInfoRec).ServerInfoRec = &GLXSYM(vga256InfoRec);
   GLXSYM(xf86PCIFlags) = hw->xf86PCIFlags;
   GLXSYM(vgaLinearBase) = xf86MapVidMem(hw->screenNum, 
				 LINEAR_REGION,
				 (pointer) GLXSYM(vga256InfoRec).physBase,
				 GLXSYM(vgaLinearSize));

   if (GLXSYM(vgaLinearBase) == (pointer) -1) {
      hwError("failed to map vga linear region");
      s3savDirectDone();
      return 0;
   }

 // GLX will call this when the client drops.
 // I think this should be a separate entry point...
 *release_hook = s3savDirectDone;

 return glxInitDirectClient(display, 
                            screen, 
			    hw->nGLXFirstVisual,
			    XF_SVGA, 
			    S3_SAVAGE3D);
}

/******************************************************************************

******************************************************************************/
static int 
s3savGLXGoDirect(ClientPtr client)
{
 xGLXGetStringReply reply;
 PS3SAVDIRECTINIT hw;
 int l;

 /* Allow only one direct client, which must be the only active
  * client, and must be on the local server.
  */
 if ( direct_client || !__glx_is_server ||
     __glXNumClients() != 1 || !LocalClient( client ) ) 
    {	
     return BadAccess;
    }

 direct_client = client;
 
 hw = (PS3SAVDIRECTINIT)malloc(sizeof(S3SAVDIRECTINIT));

 strcpy(hw->aInitFn, "s3savDirectInit");

   hw->screenNum = 0;		/* oh really? */
   hw->vgaInfoVirtualX = GLXSYM(vga256InfoRec).virtualX;
   hw->vgaInfoVirtualY = GLXSYM(vga256InfoRec).virtualY;
   hw->vgaInfoDepth = GLXSYM(vga256InfoRec).depth;
   hw->vgaInfoDisplayWidth = GLXSYM(vga256InfoRec).displayWidth;
   hw->vgaInfoVideoRam = GLXSYM(vga256InfoRec).videoRam;
   hw->vgaInfoChipID = GLXSYM(vga256InfoRec).chipID;
   hw->vgaInfoPhysBase = GLXSYM(vga256InfoRec).physBase;

   strncpy(hw->vgaInfoChipset, GLXSYM(vga256InfoRec).chipset, 80);
   hw->vgaInfoChipset[79] = 0;

   hw->vgaBitsPerPixel = GLXSYM(vgaBitsPerPixel);
   hw->vgaLinearSize = GLXSYM(vgaLinearSize);
   hw->xf86PixmapCacheMemoryStart = GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryStart;
   hw->xf86PixmapCacheMemoryEnd = GLXSYM(xf86AccelInfoRec).PixmapCacheMemoryEnd;   
   hw->xf86PCIFlags = GLXSYM(xf86PCIFlags);

 // whats that ?
 hw->nGLXFirstVisual = __glx_first_visual;

 reply.type = X_Reply;
 reply.sequenceNumber = client->sequence;
 reply.length = 0;
 reply.n = l = (sizeof(*hw)+3)/4;

 WriteToClient( client, sizeof(xGLXGetStringReply), (char*)&reply );
 WriteToClient( client, sizeof(int)*l, (char*)hw );

 return client->noClientException;
}

/******************************************************************************

******************************************************************************/
static int 
s3savGLXReleaseDirect(ClientPtr client)
{
	if ( __glx_is_server && direct_client && direct_client == client ) {
		direct_client = 0;
		return Success;
	}

	return BadAccess;
}

/******************************************************************************

******************************************************************************/
static void 
s3savDoFlush(void)
{
 #ifdef TRACE
 fprintf(stderr, "[s3sav] s3savDoFlush(%08X)\n", stS3Sav.pSHM->nBCI);
 #endif
 #define noDDOSWAP
 if (stS3Sav.pSHM->nBCI)
    {
     #ifdef DDOSWAP
     hwUI32 uVal;
     hwUI32 uOldVal = 0xFFFFFFFF;
     #endif
     unsigned int c = stS3Sav.pSHM->nBCI;
     hwUI32 *pBuf = (hwUI32 *)stS3Sav.pSHM->aBCI;
     stS3Sav.pSHM->uBCIBytes += c;
     stS3Sav.pSHM->uBCIFlushes++;
     stS3Sav.uBCIPtr = 0;
     
     #ifdef DDOSWAP
     uOldVal = uVal = (*stS3Sav.pStatus1) & 0xFFFF;
     fprintf(stderr, "[s3sav] FLEV %04X\n", uVal);
     #endif
     
     s3savFIFOWaitServer(c);
     while (c) {
      BCIWR(*pBuf);
      #ifdef DDOSWAP
      if (((*pBuf) & 0xFF000000) == BCI_CMD_UPDSHADOW)
         fprintf(stderr, "[s3sav] updshad %04X\n",
                 (*pBuf) & 0xFFFF);
      #endif		 
      pBuf++;
      #ifdef DDOSWAP
      uVal = (*stS3Sav.pStatus1) & 0xFFFF;
      if (uVal != uOldVal)
         {
	  uOldVal = uVal;
	  fprintf(stderr, "[s3sav] FLEV %04X\n", uVal);
	 }
      #endif	 
      c--;	 
     }
     #ifdef FLUSHWAITEMPTY
     s3savWaitEvent(s3savInsertEvent()); 
     #endif
     stS3Sav.pSHM->nBCI = 0;
     stS3Sav.uBCIPtr = 0;
    }  
 #ifdef TRACE
 fprintf(stderr, "[s3sav] s3savDoFlush() OK\n");
 #endif      
}

/******************************************************************************
 flush data from pseudo bci buffer in shared memory to real bci
******************************************************************************/
static int 
s3savGLXDirectDMAFlush(ClientPtr client, 
                       xGLXVendorPrivateReq *stuff)
{
 xReply	reply;

 #ifdef TRACE
 fprintf(stderr, "[s3sav] s3savGLXDirectDMAFlush()\n");
 #endif
 
 if (client != direct_client) 
    {
     return BadAccess;
    }

 s3savDoFlush();
 // is this needed ???
 reply.generic.type = X_Reply;
 reply.generic.length = 0;
 reply.generic.sequenceNumber = client->sequence;
 WriteToClient( client, sizeof(xReply), (char *) &reply );

 return client->noClientException;
}

// structure with swap buffer request
typedef struct {
 S3SAVSURFACE stBackBuffer;
 TMemBlock stMemBlock;
 CARD32	nDrawable;
 CARD32	attrib;
 CARD32	flag;
} SwapBufferReq;

typedef struct {
 BYTE type;
 CARD8 xpad;
 CARD16	sequenceNumber;
 CARD32	length;
 CARD16	width;
 CARD16	height;
} SwapBufferReply;

/******************************************************************************
 this function is executed on x serwer in response to swap request
******************************************************************************/
static int 
s3savGLXDirectSwapBuffers(ClientPtr client,
			  xGLXVendorPrivateReq *stuff)
{
 SwapBufferReq *vp = (SwapBufferReq *)(stuff+1);
 xReply	reply;
 SwapBufferReply *rep = (SwapBufferReply *)&reply;
 DrawablePtr pDraw;

 #ifdef TRACE
 fprintf(stderr, "[s3sav] s3savGLXDirectSwapBuffers()\n");
 #endif
 
 if (client != direct_client) 
    {
     return BadAccess;
    }
 s3savDoFlush();
 // find the window we are going to blit to
 pDraw = LookupIDByClass(vp->nDrawable, RC_DRAWABLE);

 if (!pDraw) 
    {
     return GLXBadDrawable;
    }
 vp->stBackBuffer.pMemBlock = &(vp->stMemBlock);
 s3savDoSwap(&(vp->stBackBuffer),
             pDraw);

 rep->type = X_Reply;
 rep->length = 0;
 rep->sequenceNumber = client->sequence;
 rep->width = pDraw->width;
 rep->height = pDraw->height;

 WriteToClient( client, sizeof(xReply), (char *) &reply );
 return client->noClientException;
}

/******************************************************************************
 this function sends request to xserwer to swap buffers
******************************************************************************/
void 
s3savDirectClientSwapBuffers(XSMesaBuffer b)
{
 SwapBufferReq vp;
 xReply	reply;
 PS3SAVBUFFER pBuf;

 if (!b->db_state) 
    {
     hwMsg( 10, "client swap buffers: only single buffered!\n" );
    }

 if ( !b->backimage || !(pBuf = (PS3SAVBUFFER)b->backimage->devPriv) ) 
    {
     fprintf( stderr, "[s3sav] client swap buffers: wtf???\n" );
     return;
    }

 // flush whats left in bci
 stS3Sav.pSHM->nBCI = stS3Sav.uBCIPtr;

 vp.nDrawable = b->frontbuffer->id;
 vp.stBackBuffer = pBuf->stBackBuffer;
 vp.stMemBlock = *pBuf->stBackBuffer.pMemBlock;
 vp.flag = 0;

 if (pfnSendVendorPrivate(X_GLXDirectSwapBuffers,
		          (char *)&vp, sizeof(vp), &reply, 0, 0 ))
    {
     SwapBufferReply *rep = (SwapBufferReply *)&reply;

     b->frontbuffer->width = rep->width;
     b->frontbuffer->height = rep->height;
     stS3Sav.uBCIPtr = 0;
    }
 else 
    {
     FatalError( "clientSwapBuffers failed" );
    }
}

/******************************************************************************
 private requests for direct rendering
******************************************************************************/
int 
s3savGLXVendorPrivate(ClientPtr client,
		      XSMesaContext ctx, 
		      xGLXVendorPrivateReq *stuff)
{
 if (!__glx_is_server) 
    {
     return GLXUnsupportedPrivateRequest;
    }

 switch (stuff->opcode) {
  case X_GLXDirectGoDirect:
   return s3savGLXGoDirect(client);
  case X_GLXDirectRelease:
   return s3savGLXReleaseDirect(client);
  case X_GLXDirectSwapBuffers:
   return s3savGLXDirectSwapBuffers(client, stuff);
  case X_GLXDirectDmaFlush:
   return s3savGLXDirectDMAFlush(client, stuff);
  default:
   hwError( "not-handled case in s3savGLXVendorPrivate" );
   return GLXUnsupportedPrivateRequest;
 } // end of switch opcode
 return GLXUnsupportedPrivateRequest;
}

/******************************************************************************
 we dont want to go direct
******************************************************************************/
int 
s3savGLXAllowDirect(ClientPtr client)
{
 #ifdef DIRECT_ENABLE
 return 1;
 #else
 return 0;
 #endif
}

