 /***************************************************************************\
|*                                                                           *|
|*       Copyright 1993-1999 NVIDIA, Corporation.  All rights reserved.      *|
|*                                                                           *|
|*     NOTICE TO USER:   The source code  is copyrighted under  U.S. and     *|
|*     international laws.  Users and possessors of this source code are     *|
|*     hereby granted a nonexclusive,  royalty-free copyright license to     *|
|*     use this code in individual and commercial software.                  *|
|*                                                                           *|
|*     Any use of this source code must include,  in the user documenta-     *|
|*     tion and  internal comments to the code,  notices to the end user     *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 1993-1999 NVIDIA, Corporation.  All rights reserved.      *|
|*                                                                           *|
|*     NVIDIA, CORPORATION MAKES NO REPRESENTATION ABOUT THE SUITABILITY     *|
|*     OF  THIS SOURCE  CODE  FOR ANY PURPOSE.  IT IS  PROVIDED  "AS IS"     *|
|*     WITHOUT EXPRESS OR IMPLIED WARRANTY OF ANY KIND.  NVIDIA, CORPOR-     *|
|*     ATION DISCLAIMS ALL WARRANTIES  WITH REGARD  TO THIS SOURCE CODE,     *|
|*     INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY, NONINFRINGE-     *|
|*     MENT,  AND FITNESS  FOR A PARTICULAR PURPOSE.   IN NO EVENT SHALL     *|
|*     NVIDIA, CORPORATION  BE LIABLE FOR ANY SPECIAL,  INDIRECT,  INCI-     *|
|*     DENTAL, OR CONSEQUENTIAL DAMAGES,  OR ANY DAMAGES  WHATSOEVER RE-     *|
|*     SULTING FROM LOSS OF USE,  DATA OR PROFITS,  WHETHER IN AN ACTION     *|
|*     OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,  ARISING OUT OF     *|
|*     OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOURCE CODE.     *|
|*                                                                           *|
|*     U.S. Government  End  Users.   This source code  is a "commercial     *|
|*     item,"  as that  term is  defined at  48 C.F.R. 2.101 (OCT 1995),     *|
|*     consisting  of "commercial  computer  software"  and  "commercial     *|
|*     computer  software  documentation,"  as such  terms  are  used in     *|
|*     48 C.F.R. 12.212 (SEPT 1995)  and is provided to the U.S. Govern-     *|
|*     ment only as  a commercial end item.   Consistent with  48 C.F.R.     *|
|*     12.212 and  48 C.F.R. 227.7202-1 through  227.7202-4 (JUNE 1995),     *|
|*     all U.S. Government End Users  acquire the source code  with only     *|
|*     those rights set forth herein.                                        *|
|*                                                                           *|
 \***************************************************************************/

/*
 * Mesa 3-D graphics library
 * Version:  3.0
 * Copyright (C) 1995-1998  Brian Paul
 */

#ifndef MESA31 /* ugh don't need any of this :) */


/*
 * Render points, lines, and polygons.  The only entry point to this
 * file is the gl_render_vb() function.  This function is called after
 * the vertex buffer has filled up or glEnd() has been called.
 *
 * This file basically only makes calls to the clipping functions and
 * the point, line and triangle rasterizers via the function pointers.
 *    context->Driver.PointsFunc()
 *    context->Driver.LineFunc()
 *    context->Driver.TriangleFunc()
 */
/*
 * X includes.
 */
#include "X.h"
#include "Xproto.h"
#include "windowstr.h"
/*
 * Mesa includes.
 */
#include "clip.h"
#include "context.h"
#include "light.h"
#include "lines.h"
#include "macros.h"
#include "matrix.h"
#include "pb.h"
#include "points.h"
#include "types.h"
#include "vb.h"
#include "vbrender.h"
#include "xform.h"
/*
 * GLX includes.
 */
#include "xsmesaP.h"
#include "glx_log.h"
/*
 * Riva includes.
 */
#include "riva_glx.h"

#define CLIP_ALL_BITS    0x3f

/*
 * Render a line segment from VB[v1] to VB[v2] when either one or both
 * endpoints must be clipped.
 */
static void RivaRenderClippedLine( GLcontext *ctx, GLuint v1, GLuint v2 )
{
    GLfloat ndc_x, ndc_y, ndc_z, wInv;
    GLuint provoking_vertex;
    struct vertex_buffer *VB = ctx->VB;

    /* which vertex dictates the color when flat shading: */
    provoking_vertex = v2;

    /*
     * Clipping may introduce new vertices.  New vertices will be stored
     * in the vertex buffer arrays starting with location VB->Free.  After
     * we've rendered the line, these extra vertices can be overwritten.
     */
    VB->Free = VB_MAX;

    /* Clip against user clipping planes */
    if (ctx->Transform.AnyClip)
    {
        GLuint orig_v1 = v1, orig_v2 = v2;
        if (gl_userclip_line( ctx, &v1, &v2 )==0)
            return;
        /* Apply projection matrix:  clip = Proj * eye */
        if (v1!=orig_v1)
        {
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION >= 1)
            TRANSFORM_POINT( VB->Clip[v1], ctx->ProjectionMatrix.m, VB->Eye[v1] );
#else
            TRANSFORM_POINT( VB->Clip[v1], ctx->ProjectionMatrix, VB->Eye[v1] );
#endif
        }
        if (v2!=orig_v2)
        {
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION >= 1)
            TRANSFORM_POINT( VB->Clip[v2], ctx->ProjectionMatrix.m, VB->Eye[v2] );
#else
            TRANSFORM_POINT( VB->Clip[v2], ctx->ProjectionMatrix, VB->Eye[v2] );
#endif
        }
    }

    /* Clip against view volume */
    if (gl_viewclip_line( ctx, &v1, &v2 )==0)
        return;

    /* Transform from clip coords to ndc:  ndc = clip / W */
    if (VB->Clip[v1][3] != 0.0F)
    {
        wInv  = 1.0F / VB->Clip[v1][3];
        ndc_x = VB->Clip[v1][0] * wInv;
        ndc_y = VB->Clip[v1][1] * wInv;
        ndc_z = VB->Clip[v1][2] * wInv;
    }
    else
    {
        /* Can't divide by zero, so... */
        ndc_x = ndc_y = ndc_z = 0.0F;
        wInv = 1.0F;
    }

    /* Map ndc coord to window coords. */
    VB->Win[v1][0] = ndc_x * ctx->Viewport.Sx + ctx->Viewport.Tx;
    VB->Win[v1][1] = ndc_y * ctx->Viewport.Sy + ctx->Viewport.Ty;
    VB->Win[v1][2] = ndc_z * ctx->Viewport.Sz + ctx->Viewport.Tz;
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION >= 1)
    VB->Win[v1][3] = wInv;
#endif

    /* Transform from clip coords to ndc:  ndc = clip / W */
    if (VB->Clip[v2][3] != 0.0F)
    {
        wInv  = 1.0F / VB->Clip[v2][3];
        ndc_x = VB->Clip[v2][0] * wInv;
        ndc_y = VB->Clip[v2][1] * wInv;
        ndc_z = VB->Clip[v2][2] * wInv;
    }
    else
    {
        /* Can't divide by zero, so... */
        ndc_x = ndc_y = ndc_z = 0.0F;
    }

    /* Map ndc coord to window coords. */
    VB->Win[v2][0] = ndc_x * ctx->Viewport.Sx + ctx->Viewport.Tx;
    VB->Win[v2][1] = ndc_y * ctx->Viewport.Sy + ctx->Viewport.Ty;
    VB->Win[v2][2] = ndc_z * ctx->Viewport.Sz + ctx->Viewport.Tz;
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION >= 1)
    VB->Win[v2][3] = wInv;
#endif

    if (ctx->Driver.RasterSetup)
    {
        /* Device driver rasterization setup */
        (*ctx->Driver.RasterSetup)( ctx, v1, v1+1 );
        (*ctx->Driver.RasterSetup)( ctx, v2, v2+1 );
    }
    (*ctx->Driver.LineFunc)( ctx, v1, v2, provoking_vertex );
}
/*
 * Compute Z offsets for a polygon with plane defined by (A,B,C,D)
 * D is not needed.
 */
static void RivaOffsetPolygon( GLcontext *ctx, GLfloat a, GLfloat b, GLfloat c )
{
    GLfloat ac, bc, m;
    GLfloat offset;

    if (c<0.001F && c>-0.001F)
    {
        /* to prevent underflow problems */
        offset = 0.0F;
    }
    else
    {
        ac = a / c;
        bc = b / c;
        if (ac<0.0F)  ac = -ac;
        if (bc<0.0F)  bc = -bc;
        m = MAX2( ac, bc );
        /* m = sqrt( ac*ac + bc*bc ); */
        offset = m * ctx->Polygon.OffsetFactor + ctx->Polygon.OffsetUnits;
    }
    ctx->PointZoffset   = ctx->Polygon.OffsetPoint ? offset : 0.0F;
    ctx->LineZoffset    = ctx->Polygon.OffsetLine  ? offset : 0.0F;
    ctx->PolygonZoffset = ctx->Polygon.OffsetFill  ? offset : 0.0F;
}
/*
 * When glPolygonMode() is used to specify that the front/back rendering
 * mode for polygons is not GL_FILL we end up calling this function.
 */
static void RivaUnfilledPolygon( GLcontext *ctx,
                              GLuint n, GLuint vlist[],
                              GLuint pv, GLuint facing )
{
    GLenum mode = facing ? ctx->Polygon.BackMode : ctx->Polygon.FrontMode;
    struct vertex_buffer *VB = ctx->VB;

    if (mode==GL_POINT)
    {
        GLuint i, j;
        GLboolean edge;

        if (   ctx->Primitive==GL_TRIANGLES
               || ctx->Primitive==GL_QUADS
               || ctx->Primitive==GL_POLYGON)
        {
            edge = GL_FALSE;
        }
        else
        {
            edge = GL_TRUE;
        }

        for (i=0;i<n;i++)
        {
            j = vlist[i];
            if (edge || VB->Edgeflag[j])
            {
                (*ctx->Driver.PointsFunc)( ctx, j, j );
            }
        }
    }
    else if (mode==GL_LINE)
    {
        GLuint i, j0, j1;
        GLboolean edge;

        ctx->StippleCounter = 0;
        if (   ctx->Primitive==GL_TRIANGLES
               || ctx->Primitive==GL_QUADS
               || ctx->Primitive==GL_POLYGON)
        {
            edge = GL_FALSE;
        }
        else
        {
            edge = GL_TRUE;
        }
        /* draw the edges */
        for (i=0;i<n;i++)
        {
            j0 = (i==0) ? vlist[n-1] : vlist[i-1];
            j1 = vlist[i];
            if (edge || VB->Edgeflag[j0])
            {
                (*ctx->Driver.LineFunc)( ctx, j0, j1, pv );
            }
        }
    }
    else
    {
        /* Fill the polygon */
        GLuint j0, i;
        j0 = vlist[0];
        for (i=2;i<n;i++)
        {
            (*ctx->Driver.TriangleFunc)( ctx, j0, vlist[i-1], vlist[i], pv );
        }
    }
}
/*
 * Compute signed area of the n-sided polgyon specified by vertices vb->Win[]
 * and vertex list vlist[].
 * A clockwise polygon will return a negative area.
 * A counter-clockwise polygon will return a positive area.
 */
static GLfloat RivaPolygonArea( const struct vertex_buffer *vb,
                             GLuint n, const GLuint vlist[] )
{
    GLfloat area = 0.0F;
    GLuint i;
    for (i=0;i<n;i++)
    {
        /* area = sum of trapezoids */
        GLuint j0 = vlist[i];
        GLuint j1 = vlist[(i+1)%n];
        GLfloat x0 = vb->Win[j0][0];
        GLfloat y0 = vb->Win[j0][1];
        GLfloat x1 = vb->Win[j1][0];
        GLfloat y1 = vb->Win[j1][1];
        GLfloat trapArea = (x0-x1)*(y0+y1);  /* Note: no divide by two here! */
        area += trapArea;
    }
    return (area * 0.5F);     /* divide by two now! */
}
/*
 * Render a polygon in which doesn't have to be clipped.
 * Input:  n - number of vertices
 *         vlist - list of vertices in the polygon.
 */
static void RivaRenderPolygon( GLcontext *ctx, GLuint n, GLuint vlist[] )
{
    struct vertex_buffer *VB = ctx->VB;
    GLuint pv;

    /* which vertex dictates the color when flat shading: */
    pv = (ctx->Primitive==GL_POLYGON) ? vlist[0] : vlist[n-1];
    /* Compute orientation of polygon, do cull test, offset, etc */
    {
        GLuint facing;   /* 0=front, 1=back */
        GLfloat area = RivaPolygonArea( VB, n, vlist );
        if (area==0.0F && !ctx->Polygon.Unfilled)
        {
            /* polygon has zero area, don't draw it */
            return;
        }
        facing = (area<0.0F) ^ (ctx->Polygon.FrontFace==GL_CW);
        if ((facing+1) & ctx->Polygon.CullBits)
        {
            return;   /* culled */
        }
        if (ctx->Polygon.OffsetAny)
        {
            /* compute plane equation of polygon, apply offset */
            GLuint j0 = vlist[0];
            GLuint j1 = vlist[1];
            GLuint j2 = vlist[2];
            GLuint j3 = vlist[ (n==3) ? 0 : 3 ];
            GLfloat ex = VB->Win[j1][0] - VB->Win[j3][0];
            GLfloat ey = VB->Win[j1][1] - VB->Win[j3][1];
            GLfloat ez = VB->Win[j1][2] - VB->Win[j3][2];
            GLfloat fx = VB->Win[j2][0] - VB->Win[j0][0];
            GLfloat fy = VB->Win[j2][1] - VB->Win[j0][1];
            GLfloat fz = VB->Win[j2][2] - VB->Win[j0][2];
            GLfloat a = ey*fz-ez*fy;
            GLfloat b = ez*fx-ex*fz;
            GLfloat c = ex*fy-ey*fx;
            RivaOffsetPolygon( ctx, a, b, c );
        }
        if (ctx->LightTwoSide)
        {
            if (facing == 1)
            {
                /* use back color or index */
                VB->Color    = VB->Bcolor;
                VB->Index    = VB->Bindex;
                VB->Specular = VB->Bspec;
            }
            else
            {
                /* use front color or index */
                VB->Color    = VB->Fcolor;
                VB->Index    = VB->Findex;
                VB->Specular = VB->Fspec;
            }
        }
        /* Render the polygon! */
        if (ctx->Polygon.Unfilled)
        {
            RivaUnfilledPolygon( ctx, n, vlist, pv, facing );
        }
        else
        {
            /* Draw filled polygon as a triangle fan */
            GLuint i;
            GLuint j0 = vlist[0];
            for (i=2;i<n;i++)
            {
                (*ctx->Driver.TriangleFunc)( ctx, j0, vlist[i-1], vlist[i], pv );
            }
        }
    }
}
/*
 * Render a polygon in which at least one vertex has to be clipped.
 * Input:  n - number of vertices
 *         vlist - list of vertices in the polygon.
 *                 CCW order = front facing.
 */
static void RivaRenderClippedPolygon( GLcontext *ctx, GLuint n, GLuint vlist[] )
{
    GLuint pv;
    struct vertex_buffer *VB = ctx->VB;
    GLfloat wInv;
#ifdef MESA_INVW
    GLfloat (*win)[4] = VB->Win;
#else
    GLfloat (*win)[3] = VB->Win;
#endif

    /* which vertex dictates the color when flat shading: */
    pv = (ctx->Primitive==GL_POLYGON) ? vlist[0] : vlist[n-1];
    /*
     * Clipping may introduce new vertices.  New vertices will be stored
     * in the vertex buffer arrays starting with location VB->Free.  After
     * we've rendered the polygon, these extra vertices can be overwritten.
     */
    VB->Free = VB_MAX;
    /* Clip against user clipping planes in eye coord space. */
    if (ctx->Transform.AnyClip)
    {
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION >= 1)
        GLfloat *proj = ctx->ProjectionMatrix.m;
#else
        GLfloat *proj = ctx->ProjectionMatrix;
#endif
        GLuint i;
        n = gl_userclip_polygon( ctx, n, vlist );
        if (n<3)
            return;
        /* Transform vertices from eye to clip coordinates:  clip = Proj * eye */
        for (i=0;i<n;i++)
        {
            GLuint j = vlist[i];
            TRANSFORM_POINT( VB->Clip[j], proj, VB->Eye[j] );
        }
    }
    /* Clip against view volume in clip coord space */
    n = gl_viewclip_polygon( ctx, n, vlist );
    if (n<3)
        return;
    /* Transform new vertices from clip to ndc to window coords.    */
    /* ndc = clip / W    window = viewport_mapping(ndc)             */
    /* Note that window Z values are scaled to the range of integer */
    /* depth buffer values.                                         */
    {
        GLfloat sx = ctx->Viewport.Sx;
        GLfloat tx = ctx->Viewport.Tx;
        GLfloat sy = ctx->Viewport.Sy;
        GLfloat ty = ctx->Viewport.Ty;
        GLfloat sz = ctx->Viewport.Sz;
        GLfloat tz = ctx->Viewport.Tz;
        GLuint i;
        /* Only need to compute window coords for new vertices */
        for (i=VB_MAX; i<VB->Free; i++)
        {
            if (VB->Clip[i][3] != 0.0F)
            {
                wInv      = 1.0F / VB->Clip[i][3];
                win[i][0] = VB->Clip[i][0] * wInv * sx + tx;
                win[i][1] = VB->Clip[i][1] * wInv * sy + ty;
                win[i][2] = VB->Clip[i][2] * wInv * sz + tz;
#ifdef MESA_INVW
                win[i][3] = wInv;
#endif
            }
            else
            {
                /* Can't divide by zero, so... */
                win[i][0] = win[i][1] = win[i][2] = 0.0F;
#ifdef MESA_INVW
                win[i][3] = 1.0F;
#endif
            }
        }
    }
    /* Compute orientation of polygon, do cull test, offset, etc */
    {
        GLuint facing;   /* 0=front, 1=back */
        GLfloat area = RivaPolygonArea( VB, n, vlist );
        if (area == 0.0F && !ctx->Polygon.Unfilled)
        {
            /* polygon has zero area, don't draw it */
            return;
        }
        facing = (area<0.0F) ^ (ctx->Polygon.FrontFace==GL_CW);
        if ((facing+1) & ctx->Polygon.CullBits)
        {
            return;   /* culled */
        }
        if (ctx->Polygon.OffsetAny)
        {
            /* compute plane equation of polygon, apply offset */
            GLuint j0 = vlist[0];
            GLuint j1 = vlist[1];
            GLuint j2 = vlist[2];
            GLuint j3 = vlist[ (n==3) ? 0 : 3 ];
            GLfloat ex = win[j1][0] - win[j3][0];
            GLfloat ey = win[j1][1] - win[j3][1];
            GLfloat ez = win[j1][2] - win[j3][2];
            GLfloat fx = win[j2][0] - win[j0][0];
            GLfloat fy = win[j2][1] - win[j0][1];
            GLfloat fz = win[j2][2] - win[j0][2];
            GLfloat a = ey*fz-ez*fy;
            GLfloat b = ez*fx-ex*fz;
            GLfloat c = ex*fy-ey*fx;
            RivaOffsetPolygon( ctx, a, b, c );
        }
        if (ctx->LightTwoSide)
        {
            if (facing==1)
            {
                /* use back color or index */
                VB->Color = VB->Bcolor;
                VB->Index = VB->Bindex;
                VB->Specular = VB->Bspec;
            }
            else
            {
                /* use front color or index */
                VB->Color = VB->Fcolor;
                VB->Index = VB->Findex;
                VB->Specular = VB->Fspec;
            }
        }
        /*
         * Reset vertex cache.
         */
        {
            unsigned  long          *pvc = (unsigned long *)(rivaContext.VCache);
            pvc[0] = 0xFFFFFFFF; pvc[1] = 0xFFFFFFFF;
            pvc[2] = 0xFFFFFFFF; pvc[3] = 0xFFFFFFFF;
        }
        /* Render the polygon! */
        if (ctx->Polygon.Unfilled)
        {
            RivaUnfilledPolygon( ctx, n, vlist, pv, facing );
        }
        else
        {
            /* Draw filled polygon as a triangle fan */
            GLuint i;
            GLuint j0 = vlist[0];
            for (i=2;i<n;i++)
            {
                (*ctx->Driver.TriangleFunc)( ctx, j0, vlist[i-1], vlist[i], pv );
            }
        }
    }
}
/*
 * Render an un-clipped triangle.
 * v0, v1, v2 - vertex indexes.  CCW order = front facing
 * pv - provoking vertex
 */
static void RivaRenderTriangle( GLcontext *ctx,
                             GLuint v0, GLuint v1, GLuint v2, GLuint pv )
{
    struct vertex_buffer *VB = ctx->VB;
    GLfloat ex, ey, fx, fy, c;
    GLuint facing;  /* 0=front, 1=back */
#ifdef MESA_INVW
    GLfloat (*win)[4] = VB->Win;
#else
    GLfloat (*win)[3] = VB->Win;
#endif

    /* Compute orientation of triangle */
    ex = win[v1][0] - win[v0][0];
    ey = win[v1][1] - win[v0][1];
    fx = win[v2][0] - win[v0][0];
    fy = win[v2][1] - win[v0][1];
    c = ex*fy-ey*fx;
    if (c==0.0F && !ctx->Polygon.Unfilled)
    {
        /* polygon is perpindicular to view plane, don't draw it */
        return;
    }
    facing = (c<0.0F) ^ (ctx->Polygon.FrontFace==GL_CW);
    if ((facing+1) & ctx->Polygon.CullBits)
    {
        return;   /* culled */
    }
    if (ctx->Polygon.OffsetAny)
    {
        /* finish computing plane equation of polygon, compute offset */
        GLfloat fz = win[v2][2] - win[v0][2];
        GLfloat ez = win[v1][2] - win[v0][2];
        GLfloat a = ey*fz-ez*fy;
        GLfloat b = ez*fx-ex*fz;
        RivaOffsetPolygon( ctx, a, b, c );
    }
    if (ctx->LightTwoSide)
    {
        if (facing==1)
        {
            /* use back color or index */
            VB->Color = VB->Bcolor;
            VB->Index = VB->Bindex;
            VB->Specular = VB->Bspec;
        }
        else
        {
            /* use front color or index */
            VB->Color = VB->Fcolor;
            VB->Index = VB->Findex;
            VB->Specular = VB->Fspec;
        }
    }
    if (ctx->Polygon.Unfilled)
    {
        GLuint vlist[3];
        vlist[0] = v0;
        vlist[1] = v1;
        vlist[2] = v2;
        RivaUnfilledPolygon( ctx, 3, vlist, pv, facing );
    }
    else
    {
        (*ctx->Driver.TriangleFunc)( ctx, v0, v1, v2, pv );
    }
}
/*
 * Render an un-clipped quadrilateral.
 * v0, v1, v2, v3 : CCW order = front facing
 * pv - provoking vertex
 */
static void RivaRenderQuad( GLcontext *ctx, GLuint v0, GLuint v1,
                         GLuint v2, GLuint v3, GLuint pv )
{
    struct vertex_buffer *VB = ctx->VB;
    GLfloat ex, ey, fx, fy, c;
    GLuint facing;  /* 0=front, 1=back */
#ifdef MESA_INVW
    GLfloat (*win)[4] = VB->Win;
#else
    GLfloat (*win)[3] = VB->Win;
#endif

    /* Compute polygon orientation */
    ex = win[v2][0] - win[v0][0];
    ey = win[v2][1] - win[v0][1];
    fx = win[v3][0] - win[v1][0];
    fy = win[v3][1] - win[v1][1];
    c = ex*fy-ey*fx;
    if (c==0.0F && !ctx->Polygon.Unfilled)
    {
        /* polygon is perpindicular to view plane, don't draw it */
        return;
    }
    facing = (c<0.0F) ^ (ctx->Polygon.FrontFace==GL_CW);
    if ((facing+1) & ctx->Polygon.CullBits)
    {
        return;   /* culled */
    }
    if (ctx->Polygon.OffsetAny)
    {
        /* finish computing plane equation of polygon, compute offset */
        GLfloat ez = win[v2][2] - win[v0][2];
        GLfloat fz = win[v3][2] - win[v1][2];
        GLfloat a = ey*fz-ez*fy;
        GLfloat b = ez*fx-ex*fz;
        RivaOffsetPolygon( ctx, a, b, c );
    }
    if (ctx->LightTwoSide)
    {
        if (facing==1)
        {
            /* use back color or index */
            VB->Color = VB->Bcolor;
            VB->Index = VB->Bindex;
            VB->Specular = VB->Bspec;
        }
        else
        {
            /* use front color or index */
            VB->Color = VB->Fcolor;
            VB->Index = VB->Findex;
            VB->Specular = VB->Fspec;
        }
    }
    /* Render the quad! */
    if (ctx->Polygon.Unfilled)
    {
        GLuint vlist[4];
        vlist[0] = v0;
        vlist[1] = v1;
        vlist[2] = v2;
        vlist[3] = v3;
        RivaUnfilledPolygon( ctx, 4, vlist, pv, facing );
    }
    else
    {
        (*ctx->Driver.QuadFunc)( ctx, v0, v1, v2, v3, pv );
    }
}
/*
 * When the vertex buffer is full, we transform/render it.  Sometimes we
 * have to copy the last vertex (or two) to the front of the vertex list
 * to "continue" the primitive.  For example:  line or triangle strips.
 * This function is a helper for that.
 */
static void RivaCopyVertex( struct vertex_buffer *vb, GLuint dst, GLuint src )
{
    COPY_4V( vb->Clip[dst], vb->Clip[src] );
    COPY_4V( vb->Eye[dst], vb->Eye[src] );
#ifdef MESA_INVW
    COPY_4V( vb->Win[dst], vb->Win[src] );
#else
    COPY_3V( vb->Win[dst], vb->Win[src] );
#endif
    COPY_4V( vb->Fcolor[dst], vb->Fcolor[src] );
    COPY_4V( vb->Bcolor[dst], vb->Bcolor[src] );
    /* XXX Specular */
    COPY_4V( vb->TexCoord[dst], vb->TexCoord[src] );
    vb->Findex[dst]       = vb->Findex[src];
    vb->Bindex[dst]       = vb->Bindex[src];
    vb->Edgeflag[dst]     = vb->Edgeflag[src];
    vb->ClipMask[dst]     = vb->ClipMask[src];
    vb->MaterialMask[dst] = vb->MaterialMask[src];
    vb->Material[dst][0]  = vb->Material[src][0];
    vb->Material[dst][1]  = vb->Material[src][1];
}
/*
 * Either the vertex buffer is full (VB->Count==VB_MAX) or glEnd() has been
 * called.  Render the primitives defined by the vertices and reset the
 * buffer. gl_reset_vb() incorporated into this routine.
 *
 * Input:  allDone - GL_TRUE = caller is glEnd()
 *                   GL_FALSE = calling because buffer is full.
 */
GLboolean RivaRenderVB
(
    GLcontext *ctx,
    GLboolean  allDone
)
{
    struct    vertex_buffer *VB  = ctx->VB;
    int       oldCount           = VB->Count;
    GLubyte   clipOrMask         = VB->ClipOrMask;
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
    GLboolean monoMaterial       = VB->MonoMaterial;
#endif    
    GLuint    vertexSizeMask     = VB->VertexSizeMask;
    unsigned  long          *pvc = (unsigned long *)(rivaContext.VCache);
    GLuint    vlist[VB_SIZE];

    /*
     * Reset vertex cache.
     */
    pvc[0] = 0xFFFFFFFF; pvc[1] = 0xFFFFFFFF;
    pvc[2] = 0xFFFFFFFF; pvc[3] = 0xFFFFFFFF;
    /*
     * Render primitive.
     */
    switch (ctx->Primitive)
    {
        case GL_POLYGON:
            if (VB->Count>2)
            {
                if (VB->ClipAndMask & CLIP_ALL_BITS)
                {
                    /* all points clipped by common plane, draw nothing */
                    break;
                }
                if (VB->ClipOrMask)
                {
                    /* need clipping */
                    GLuint i;
                    for (i=0;i<VB->Count;i++)
                    {
                        vlist[i] = i;
                    }
                    RivaRenderClippedPolygon( ctx, VB->Count, vlist );
                }
                else
                {
                    /* no clipping needed */
                    static GLuint const_vlist[VB_SIZE];
                    static GLboolean initFlag = GL_TRUE;
                    if (initFlag)
                    {
                        /* vertex list always the same, never changes */
                        GLuint i;
                        for (i=0;i<VB_SIZE;i++)
                        {
                            const_vlist[i] = i;
                        }
                        initFlag = GL_FALSE;
                    }
                    RivaRenderPolygon( ctx, VB->Count, const_vlist );
                }
            }
            if (!allDone)
            {
                RivaCopyVertex( VB, 1, VB_MAX-1 );
                VB->Start = VB->Count = 2;
                VB->ClipOrMask   = VB->ClipMask[0] | VB->ClipMask[1];
                VB->ClipAndMask  = VB->ClipMask[0] & VB->ClipMask[1];
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = !(VB->MaterialMask[0] | VB->MaterialMask[1]);
#endif
            }
            break;
        case GL_TRIANGLES:
            if (VB->ClipOrMask)
            {
                GLuint i;
                for (i=2;i<VB->Count;i+=3)
                {
                    if (VB->ClipMask[i-2] & VB->ClipMask[i-1]
                        & VB->ClipMask[i] & CLIP_ALL_BITS)
                    {
                        /* all points clipped by common plane */
                        continue;
                    }
                    else if (VB->ClipMask[i-2] | VB->ClipMask[i-1] | VB->ClipMask[i])
                    {
                        vlist[0] = i-2;
                        vlist[1] = i-1;
                        vlist[2] = i-0;
                        RivaRenderClippedPolygon( ctx, 3, vlist );
                    }
                    else
                    {
                        if (ctx->DirectTriangles)
                        {
                            (*ctx->Driver.TriangleFunc)( ctx, i-2, i-1, i, i );
                        }
                        else
                        {
                            RivaRenderTriangle( ctx, i-2, i-1, i, i );
                        }
                    }
                }
            }
            else
            {
                /* no clipping needed */
                GLuint i;
                if (ctx->DirectTriangles)
                {
                    for (i=2;i<VB->Count;i+=3)
                    {
                        (*ctx->Driver.TriangleFunc)( ctx, i-2, i-1, i, i );
                    }
                }
                else
                {
                    for (i=2;i<VB->Count;i+=3)
                    {
                        RivaRenderTriangle( ctx, i-2, i-1, i, i );
                    }
                }
            }
            if (!allDone)
            {
                VB->Start = VB->Count = 0;
                VB->ClipOrMask   = 0;
                VB->ClipAndMask  = CLIP_ALL_BITS;
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = GL_TRUE;
#endif
                VB->MonoNormal   = GL_TRUE;
            }
            break;
        case GL_TRIANGLE_STRIP:
            if (VB->ClipOrMask)
            {
                GLuint i;
                for (i=2;i<VB->Count;i++)
                {
                    if (VB->ClipMask[i-2] & VB->ClipMask[i-1]
                        & VB->ClipMask[i] & CLIP_ALL_BITS)
                    {
                        /* all points clipped by common plane */
                        continue;
                    }
                    else if (VB->ClipMask[i-2] | VB->ClipMask[i-1] | VB->ClipMask[i])
                    {
                        if (i&1)
                        {
                            /* reverse vertex order */
                            vlist[0] = i-1;
                            vlist[1] = i-2;
                            vlist[2] = i-0;
                            RivaRenderClippedPolygon( ctx, 3, vlist );
                        }
                        else
                        {
                            vlist[0] = i-2;
                            vlist[1] = i-1;
                            vlist[2] = i-0;
                            RivaRenderClippedPolygon( ctx, 3, vlist );
                        }
                    }
                    else
                    {
                        if (ctx->DirectTriangles)
                        {
                            (*ctx->Driver.TriangleFunc)( ctx, i-2, i-1, i, i );
                        }
                        else
                        {
                            if (i&1)
                                RivaRenderTriangle( ctx, i, i-1, i-2, i );
                            else
                                RivaRenderTriangle( ctx, i-2, i-1, i, i );
                        }
                    }
                }
            }
            else
            {
                /* no vertices were clipped */
                GLuint i;
                if (ctx->DirectTriangles)
                {
                    for (i=2;i<VB->Count;i++)
                    {
                        (*ctx->Driver.TriangleFunc)( ctx, i-2, i-1, i, i );
                    }
                }
                else
                {
                    for (i=2;i<VB->Count;i++)
                    {
                        if (i&1)
                            RivaRenderTriangle( ctx, i, i-1, i-2, i );
                        else
                            RivaRenderTriangle( ctx, i-2, i-1, i, i );
                    }
                }
            }
            if (!allDone)
            {
                RivaCopyVertex( VB, 0, VB_MAX-2 );
                RivaCopyVertex( VB, 1, VB_MAX-1 );
                VB->Start = VB->Count = 2;
                VB->ClipOrMask   = VB->ClipMask[0] | VB->ClipMask[1];
                VB->ClipAndMask  = VB->ClipMask[0] & VB->ClipMask[1];
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = !(VB->MaterialMask[0] | VB->MaterialMask[1]);
#endif
            }
            break;
        case GL_TRIANGLE_FAN:
            if (VB->ClipOrMask)
            {
                GLuint i;
                for (i=2;i<VB->Count;i++)
                {
                    if (VB->ClipMask[0] & VB->ClipMask[i-1] & VB->ClipMask[i]
                        & CLIP_ALL_BITS)
                    {
                        /* all points clipped by common plane */
                        continue;
                    }
                    else if (VB->ClipMask[0] | VB->ClipMask[i-1] | VB->ClipMask[i])
                    {
                        vlist[0] = 0;
                        vlist[1] = i-1;
                        vlist[2] = i;
                        RivaRenderClippedPolygon( ctx, 3, vlist );
                    }
                    else
                    {
                        if (ctx->DirectTriangles)
                        {
                            (*ctx->Driver.TriangleFunc)( ctx, 0, i-1, i, i );
                        }
                        else
                        {
                            RivaRenderTriangle( ctx, 0, i-1, i, i );
                        }
                    }
                }
            }
            else
            {
                /* no clipping needed */
                GLuint i;
                if (ctx->DirectTriangles)
                {
                    for (i=2;i<VB->Count;i++)
                    {
                        (*ctx->Driver.TriangleFunc)( ctx, 0, i-1, i, i );
                    }
                }
                else
                {
                    for (i=2;i<VB->Count;i++)
                    {
                        RivaRenderTriangle( ctx, 0, i-1, i, i );
                    }
                }
            }
            if (!allDone)
            {
                RivaCopyVertex( VB, 1, VB_MAX-1 );
                VB->Start = VB->Count = 2;
                VB->ClipOrMask   = VB->ClipMask[0] | VB->ClipMask[1];
                VB->ClipAndMask  = VB->ClipMask[0] & VB->ClipMask[1];
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = !(VB->MaterialMask[0] | VB->MaterialMask[1]);
#endif
            }
            break;
        case GL_QUADS:
            if (VB->ClipOrMask)
            {
                GLuint i;
                for (i=3;i<VB->Count;i+=4)
                {
                    if (VB->ClipMask[i-3] & VB->ClipMask[i-2]
                        & VB->ClipMask[i-1] & VB->ClipMask[i] & CLIP_ALL_BITS)
                    {
                        /* all points clipped by common plane */
                        continue;
                    }
                    else if (VB->ClipMask[i-3] | VB->ClipMask[i-2]
                             | VB->ClipMask[i-1] | VB->ClipMask[i])
                    {
                        vlist[0] = i-3;
                        vlist[1] = i-2;
                        vlist[2] = i-1;
                        vlist[3] = i-0;
                        RivaRenderClippedPolygon( ctx, 4, vlist );
                    }
                    else
                    {
                        if (ctx->DirectTriangles)
                        {
                            (*ctx->Driver.QuadFunc)( ctx, i-3, i-2, i-1, i, i );
                        }
                        else
                        {
                            RivaRenderQuad( ctx, i-3, i-2, i-1, i, i );
                        }
                    }
                }
            }
            else
            {
                /* no vertices were clipped */
                GLuint i;
                if (ctx->DirectTriangles)
                {
                    for (i=3;i<VB->Count;i+=4)
                    {
                        (*ctx->Driver.QuadFunc)( ctx, i-3, i-2, i-1, i, i );
                    }
                }
                else
                {
                    for (i=3;i<VB->Count;i+=4)
                    {
                        RivaRenderQuad( ctx, i-3, i-2, i-1, i, i );
                    }
                }
            }
            if (!allDone)
            {
                VB->Start = VB->Count = 0;
                VB->ClipOrMask   = 0;
                VB->ClipAndMask  = CLIP_ALL_BITS;
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = GL_TRUE;
#endif
                VB->MonoNormal   = GL_TRUE;
            }
            break;
        case GL_QUAD_STRIP:
            if (VB->ClipOrMask)
            {
                GLuint i;
                for (i=3;i<VB->Count;i+=2)
                {
                    if (VB->ClipMask[i-2] & VB->ClipMask[i-3]
                        & VB->ClipMask[i-1] & VB->ClipMask[i] & CLIP_ALL_BITS)
                    {
                        /* all points clipped by common plane */
                        continue;
                    }
                    else if (VB->ClipMask[i-2] | VB->ClipMask[i-3]
                             | VB->ClipMask[i-1] | VB->ClipMask[i])
                    {
                        vlist[0] = i-1;
                        vlist[1] = i-3;
                        vlist[2] = i-2;
                        vlist[3] = i-0;
                        RivaRenderClippedPolygon( ctx, 4, vlist );
                    }
                    else
                    {
                        if (ctx->DirectTriangles)
                        {
                            (*ctx->Driver.QuadFunc)( ctx, i-3, i-2, i, i-1, i );
                        }
                        else
                        {
                            RivaRenderQuad( ctx, i-3, i-2, i, i-1, i );
                        }
                    }
                }
            }
            else
            {
                /* no clipping needed */
                GLuint i;
                if (ctx->DirectTriangles)
                {
                    for (i=3;i<VB->Count;i+=2)
                    {
                        (*ctx->Driver.QuadFunc)( ctx, i-3, i-2, i, i-1, i );
                    }
                }
                else
                {
                    for (i=3;i<VB->Count;i+=2)
                    {
                        RivaRenderQuad( ctx, i-3, i-2, i, i-1, i );
                    }
                }
            }
            if (!allDone)
            {
                RivaCopyVertex( VB, 0, VB_MAX-2 );
                RivaCopyVertex( VB, 1, VB_MAX-1 );
                VB->Start = VB->Count = 2;
                VB->ClipOrMask   = VB->ClipMask[0] | VB->ClipMask[1];
                VB->ClipAndMask  = VB->ClipMask[0] & VB->ClipMask[1];
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = !(VB->MaterialMask[0] | VB->MaterialMask[1]);
#endif
            }
            break;
        case GL_POINTS:
            (*ctx->Driver.PointsFunc)( ctx, 0, VB->Count-1 );
            if (!allDone)
            {
                VB->Start = VB->Count = 0;
                VB->ClipOrMask   = 0;
                VB->ClipAndMask  = CLIP_ALL_BITS;
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = GL_TRUE;
#endif
                VB->MonoNormal   = GL_TRUE;
            }
            break;
        case GL_LINES:
            if (VB->ClipOrMask)
            {
                GLuint i;
                for (i=1;i<VB->Count;i+=2)
                {
                    if (VB->ClipMask[i-1] | VB->ClipMask[i])
                    {
                        RivaRenderClippedLine( ctx, i-1, i );
                    }
                    else
                    {
                        (*ctx->Driver.LineFunc)( ctx, i-1, i, i );
                    }
                    ctx->StippleCounter = 0;
                }
            }
            else
            {
                GLuint i;
                for (i=1;i<VB->Count;i+=2)
                {
                    (*ctx->Driver.LineFunc)( ctx, i-1, i, i );
                    ctx->StippleCounter = 0;
                }
            }
            if (!allDone)
            {
                VB->Start = VB->Count = 0;
                VB->ClipOrMask   = 0;
                VB->ClipAndMask  = CLIP_ALL_BITS;
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = GL_TRUE;
#endif
                VB->MonoNormal   = GL_TRUE;
            }
            break;
        case GL_LINE_STRIP:
            if (VB->ClipOrMask)
            {
                GLuint i;
                for (i=1;i<VB->Count;i++)
                {
                    if (VB->ClipMask[i-1] | VB->ClipMask[i])
                    {
                        RivaRenderClippedLine( ctx, i-1, i );
                    }
                    else
                    {
                        (*ctx->Driver.LineFunc)( ctx, i-1, i, i );
                    }
                }
            }
            else
            {
                /* no clipping needed */
                GLuint i;
                for (i=1;i<VB->Count;i++)
                {
                    (*ctx->Driver.LineFunc)( ctx, i-1, i, i );
                }
            }
            if (!allDone)
            {
                RivaCopyVertex( VB, 0, VB->Count-1 );  /* copy last vertex to front */
                VB->Start = VB->Count = 1;
                VB->ClipOrMask   = VB->ClipMask[0];
                VB->ClipAndMask  = VB->ClipMask[0];
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                VB->MonoMaterial = VB->MaterialMask[0] ? GL_FALSE : GL_TRUE;
#endif
            }
            break;
        case GL_LINE_LOOP:
            {
                GLuint i;
                if (VB->Start==0)
                {
                    i = 1;  /* start at 0th vertex */
                }
                else
                {
                    i = 2;  /* skip first vertex, we're saving it until glEnd */
                }
                while (i < VB->Count)
                {
                    if (VB->ClipMask[i-1] | VB->ClipMask[i])
                    {
                        RivaRenderClippedLine( ctx, i-1, i );
                    }
                    else
                    {
                        (*ctx->Driver.LineFunc)( ctx, i-1, i, i );
                    }
                    i++;
                }
                if (allDone)
                {
                    if (VB->ClipMask[VB->Count-1] | VB->ClipMask[0])
                    {
                        RivaRenderClippedLine( ctx, VB->Count-1, 0 );
                    }
                    else
                    {
                        (*ctx->Driver.LineFunc)( ctx, VB->Count-1, 0, 0 );
                    }
                }
                else
                {
                    RivaCopyVertex( VB, 1, VB_MAX-1 );
                    VB->Start = VB->Count = 2;
                    VB->ClipOrMask   = VB->ClipMask[0] | VB->ClipMask[1];
                    VB->ClipAndMask  = VB->ClipMask[0] & VB->ClipMask[1];
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
                    VB->MonoMaterial = !(VB->MaterialMask[0] | VB->MaterialMask[1]);
#endif
                }
            }
            break;
        default:
            /* should never get here */
            break;
    }
    /*
     * Reset VB.
     */
    if (allDone)
    {
        /* glEnd() was called so reset Vertex Buffer to default, empty state */
        VB->Start = VB->Count = 0;
        VB->ClipOrMask     = 0;
        VB->ClipAndMask    = CLIP_ALL_BITS;
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION < 1)
        VB->MonoMaterial   = GL_TRUE;
#endif
        VB->MonoNormal     = GL_TRUE;
        VB->MonoColor      = GL_TRUE;
        VB->VertexSizeMask = VERTEX3_BIT;
        if (VB->TexCoordSize!=2)
        {
            GLint i, n = VB->Count;
            for (i=0;i<n;i++)
            {
                VB->TexCoord[i][2] = 0.0F;
                VB->TexCoord[i][3] = 1.0F;
            }
        }
        if (ctx->Current.TexCoord[2]==0.0F && ctx->Current.TexCoord[3]==1.0F)
        {
            VB->TexCoordSize = 2;
        }
        else
        {
            VB->TexCoordSize = 4;
        }
    }
    if (clipOrMask)
    {
        /* reset clip masks to zero */
        MEMSET( VB->ClipMask + VB->Start, 0,
                (oldCount - VB->Start) * sizeof(VB->ClipMask[0]) );
    }
#if (MESA_MAJOR_VERSION == 3) && (MESA_MINOR_VERSION >= 1)
    VB->LastMaterial = VB->Start;
    VB->MaterialMask[VB->Start] = 0;
#else
    if (!monoMaterial)
    {
        /* reset material masks to zero */
        MEMSET( VB->MaterialMask + VB->Start, 0,
                (oldCount - VB->Start) * sizeof(VB->MaterialMask[0]) );
        gl_update_lighting(ctx);
    }
#endif    
    if (vertexSizeMask!=VERTEX3_BIT)
    {
        /* reset object W coords to one */
        GLint i, n;
        GLfloat (*obj)[4] = VB->Obj + VB->Start;
        n = oldCount - VB->Start;
        for (i=0; i<n; i++)
        {
            obj[i][3] = 1.0F;
        }
    }
    return (GL_TRUE);
}


#endif